/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "realtime.h"
#include "defs.h"


extern bool interactive;
extern bool beep;

/* Return false if nothing was printed, true if everything is ok. */
bool
wf_realtime_parse(const string& line,
		  const string& filename, unsigned int lineno,
		  const list<wf_inmodule*>& input_modules,
		  enum wf_logentries_parsing_strictness strictness,
		  wflogs_filter* myfilter, wf_outmodule* output_module) {
  /* Parse input line. */
  wf_logentry* logentry;
  switch (wf_inmodule_parse(&logentry, line, filename, lineno,
			    input_modules, strictness)) {
  case WF_LOGENTRY_PARSING_OK:
    break;
  case WF_LOGENTRY_PARSING_ERROR:
    wflogs_log.log(RVLOG_ERROR, _("Error: parsing failed."));
    exit(1);
  case WF_LOGENTRY_PARSING_NOHIT:
    return false;
  case WF_LOGENTRY_PARSING_NOMATCH: /* cannot happen */
    ;
  }
  
  /* Filter log line. */
  if (myfilter != NULL) {
    if ((*myfilter)(logentry)) { /* filter matches: don't display */
      delete logentry;
      return false;
    }
  }
  /* Obfuscate log. Should we? ALL@@8 */

  if (interactive) {
    /* erase from start to current position, and move to the beginning */
    cout << "\e[1K\015";
    if (beep)
      cout << "\a";
  }

  /* Output line on stdout. */
  if (output_module->print(logentry, cout) == false) {
    wflogs_log.log(RVLOG_ERROR, _("Error: output for line %i failed."),
		   lineno);
    delete logentry;
    return false;
  }
  cout << endl;
  delete logentry;
  return true;
}
