/* -*-c-*- */
/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

%option prefix="ipfilter"
%option outfile="ipfilter.cc"
%option noyywrap

%{
using namespace std;

#define YY_NO_UNPUT

#define IPLEN 16

#include <netdb.h> /* for getprotobyname */

#include "ipfilter.h"
#include "common.h"
#include "defs.h"

#define IPF_OPT_SRC 4
#define IPF_OPT_DST 8
#define IPF_OPT_RES 16
#define IPF_OPT_PORT 32
#define IPF_OPT_RPORT 64

#define SHORTLEN 32
#define SHORTLEN_S "32"

static void ipfilter_parse_data(char* input);
static void ipfilter_parse_data_count(char* input);
static void ipfilter_parse_ips(char* input, unsigned char mode);
static void ipfilter_parse_proto(char *input);
static void ipfilter_parse_tcpflags(char* input);

static bool parseerror = false;
static wf_logentry* logentry;
%}

MONTH	"Jan"|"Feb"|"Mar"|"Apr"|"May"|"Jun"|"Jul"|"Aug"|"Sep"|"Oct"|"Nov"|"Dec"
STRING	[a-zA-Z-][a-zA-Z0-9._-]*
LOGHOST	[0-9.a-zA-Z_-]*
DIGIT	[0-9]
NUMBER	{DIGIT}+
BYTE	{DIGIT}{1,3}
IPADDR	{BYTE}\.{BYTE}\.{BYTE}\.{BYTE}
PORT	{DIGIT}{1,5}
HEXDIGIT	[0-9a-fA-F]
HEXNUM	"0x"{HEXDIGIT}+
IPFILTER	"ipmon["{NUMBER}"]:"
PROTO		[0-9a-z-]+
TARGET		[SpPbBnL]
FLAGS		"-"[SAFRPU]+

%%

{MONTH}[ ]{1,2}{DIGIT}{1,2}[ ]{DIGIT}{2}:{DIGIT}{2}:{DIGIT}{2}[ ]{LOGHOST}	parse_date(ipfiltertext, logentry);
{IPFILTER}									/* ignore */
"[ID "{NUMBER}" "{STRING}"]"							/* ignore */
{DIGIT}{2}":"{DIGIT}{2}":"{DIGIT}{2}"."{DIGIT}{6}				/* ignore */
{STRING}[ ]"@"{NUMBER}":"{NUMBER}[ ]{TARGET}					ipfilter_parse_data(ipfiltertext);
{NUMBER}"x"[ ]+{STRING}[ ]"@"{NUMBER}":"{NUMBER}[ ]{TARGET}			ipfilter_parse_data_count(ipfiltertext);
{IPADDR}					ipfilter_parse_ips(ipfiltertext, IPF_OPT_SRC);
" -> "{IPADDR}					ipfilter_parse_ips(ipfiltertext + 4, IPF_OPT_DST);
{IPADDR}","{PORT}				ipfilter_parse_ips(ipfiltertext, IPF_OPT_SRC|IPF_OPT_PORT);
" -> "{IPADDR}","{PORT}				ipfilter_parse_ips(ipfiltertext + 4, IPF_OPT_DST|IPF_OPT_PORT);
{STRING}"["{IPADDR}"]"				ipfilter_parse_ips(ipfiltertext, IPF_OPT_SRC|IPF_OPT_RES);
" -> "{STRING}"["{IPADDR}"]"			ipfilter_parse_ips(ipfiltertext + 4, IPF_OPT_DST|IPF_OPT_RES);
{STRING}"["{IPADDR}"],"{PORT}			ipfilter_parse_ips(ipfiltertext, IPF_OPT_SRC|IPF_OPT_RES|IPF_OPT_PORT);
" -> "{STRING}"["{IPADDR}"],"{PORT}		ipfilter_parse_ips(ipfiltertext + 4, IPF_OPT_DST|IPF_OPT_RES|IPF_OPT_PORT);
{IPADDR}","{STRING}				ipfilter_parse_ips(ipfiltertext, IPF_OPT_SRC|IPF_OPT_RPORT);
" -> "{IPADDR}","{STRING}			ipfilter_parse_ips(ipfiltertext + 4, IPF_OPT_DST|IPF_OPT_RPORT);
{STRING}"["{IPADDR}"],"{STRING}			ipfilter_parse_ips(ipfiltertext, IPF_OPT_SRC|IPF_OPT_RES|IPF_OPT_RPORT);
" -> "{STRING}"["{IPADDR}"],"{STRING}		ipfilter_parse_ips(ipfiltertext + 4, IPF_OPT_DST|IPF_OPT_RES|IPF_OPT_RPORT);
"PR "{PROTO}									ipfilter_parse_proto(ipfiltertext + 3);
"len "{NUMBER}[ ][(]?{NUMBER}[)]?						logentry->datalen = atoi(ipfiltertext + 4);
"frag "{NUMBER}"@"{NUMBER}							/* ignore */
"icmp "{DIGIT}{1,2}"/"{DIGIT}{1,2}	{
  int sport, dport;
  if (sscanf(ipfiltertext, "icmp %d/%d", &sport, &dport) == 2) {
    logentry->sport = sport;
    logentry->dport = dport;
  }
}
"icmp echo/0"									logentry->sport = 8; logentry->dport = 0;
"icmp sourcequench/0"								logentry->sport = 4; logentry->dport = 0;
"icmp unreach/host"								logentry->sport = 3; logentry->dport = 1;
"icmp unreach/port"								logentry->sport = 3; logentry->dport = 3;
"icmp unreach/admin_prohibit"							logentry->sport = 3; logentry->dport = 10;
"icmp timxceed/intrans"								logentry->sport = 11; logentry->dport = 1;
"for "{IPADDR}","{PORT}" - "{IPADDR}","{PORT}" PR "{PROTO}" len "{NUMBER}[ ]{NUMBER}	/* ignore */
"for "{IPADDR}","{STRING}" - "{STRING}"["{IPADDR}"],"{STRING}" PR "{PROTO}" len "{NUMBER}[ ]{NUMBER}	/* ignore */
"for "{STRING}"["{IPADDR}"],"{PORT}" - "{IPADDR}","{PORT}" PR "{PROTO}" len "{NUMBER}[ ]{NUMBER}	/* ignore */
"for "{STRING}"["{IPADDR}"],"{PORT}" - "{IPADDR}","{STRING}" PR "{PROTO}" len "{NUMBER}[ ]{NUMBER}	/* ignore */
"for "{STRING}"["{IPADDR}"] - "{STRING}"["{IPADDR}"] PR "{PROTO}" len "{NUMBER}" ("{NUMBER}")"	/* ignore */
"for "{STRING}"["{IPADDR}"],"{PORT}" - "{STRING}"["{IPADDR}"],"{PORT}" PR "{PROTO}" len "{NUMBER}[ ]{NUMBER}	/* ignore */
"for "{STRING}"["{IPADDR}"],"{PORT}" - "{STRING}"["{IPADDR}"],"{STRING}" PR "{PROTO}" len "{NUMBER}[ ]{NUMBER}	/* ignore */
{FLAGS}		ipfilter_parse_tcpflags(ipfiltertext + 1);
"K-S"										/* ignore */
"K-F"										/* ignore */
"IN"										/* ignore */
"OUT"										/* ignore */
{NUMBER}[ ]{NUMBER}[ ]{NUMBER}" IN"						/* ignore */
({HEXDIGIT}{HEXDIGIT}[ ]?)+[ ]+[ -~]+				{
  // parseerror = true; ALL@@7 why ?
}
[ \t]+		/* ignore whitespace */
[\n]		/* ignore */
{STRING}	if (verbose) fprintf(stderr, _("Unrecognized token: `%s'\n"), ipfiltertext);
.		if (verbose) fprintf(stderr, _("Unrecognized character: `%s'\n"), ipfiltertext);

%%

static void
ipfilter_parse_data(char* input) {
  char chainlabel[SHORTLEN + 1], branchname[SHORTLEN + 1], iface[SHORTLEN + 1];

  if (sscanf(input, "%" SHORTLEN_S "s @%" SHORTLEN_S "s %" SHORTLEN_S "s",
	     iface, chainlabel, branchname) != 3)
    return;
  logentry->count = 1;
  logentry->input_iface = iface;
  logentry->chainlabel = chainlabel;
  logentry->branchname = branchname;
}

static void
ipfilter_parse_data_count(char* input) {
  int count;
  char chainlabel[SHORTLEN + 1], branchname[SHORTLEN + 1], iface[SHORTLEN + 1];

  if (sscanf(input, "%dx %" SHORTLEN_S "s @%" SHORTLEN_S "s %" SHORTLEN_S "s",
	     &count, iface, chainlabel, branchname) != 4)
    return;
  logentry->count = count;
  logentry->input_iface = iface;
  logentry->chainlabel = chainlabel;
  logentry->branchname = branchname;
}

static void
ipfilter_parse_ips(char* input, unsigned char mode) {
  int host1, host2, host3, host4, port;
  char ip[IPLEN];

  if (mode == (IPF_OPT_SRC|IPF_OPT_RES|IPF_OPT_RPORT) ||
      mode == (IPF_OPT_DST|IPF_OPT_RES|IPF_OPT_RPORT)) {
    char name[256], *ptr, portname[32];
    ptr = strchr(input, '[');
    *ptr = ' ';
    if (sscanf(input, "%255s %3d.%3d.%3d.%3d],%31s",
	       name, &host1, &host2, &host3, &host4, portname) != 6)
      return;
  }
  else if (mode == (IPF_OPT_SRC|IPF_OPT_RES|IPF_OPT_PORT) ||
	   mode == (IPF_OPT_DST|IPF_OPT_RES|IPF_OPT_PORT)) {
    char name[256], *ptr;
    ptr = strchr(input, '['); *ptr = ' ';
    if (sscanf(input, "%255s %3d.%3d.%3d.%3d],%5d",
	       name, &host1, &host2, &host3, &host4, &port) != 6)
      return;
  }
  else if (mode == (IPF_OPT_SRC|IPF_OPT_RES) ||
	   mode == (IPF_OPT_DST|IPF_OPT_RES)) {
    char name[256], *ptr;
    ptr = strchr(input, '['); *ptr = ' ';
    if (sscanf(input, "%255s %3d.%3d.%3d.%3d]",
	       name, &host1, &host2, &host3, &host4) != 5)
      return;
  }
  else if (mode == (IPF_OPT_SRC|IPF_OPT_RPORT) ||
	   mode == (IPF_OPT_DST|IPF_OPT_RPORT)) {
    char portname[32];
    if (sscanf(input, "%3d.%3d.%3d.%3d,%31s",
	       &host1, &host2, &host3, &host4, portname) != 5)
      return;
  }
  else if (mode == (IPF_OPT_SRC|IPF_OPT_PORT) ||
	   mode == (IPF_OPT_DST|IPF_OPT_PORT)) {
    if (sscanf(input, "%3d.%3d.%3d.%3d,%5d",
	       &host1, &host2, &host3, &host4, &port) != 5)
      return;
  }
  else if (mode == IPF_OPT_SRC || mode == IPF_OPT_DST) {
    if (sscanf(input, "%3d.%3d.%3d.%3d",
	       &host1, &host2, &host3, &host4) != 4)
      return;
  }
  else
    return;

  snprintf(ip, IPLEN, "%d.%d.%d.%d", host1, host2, host3, host4);
  if (mode & IPF_OPT_SRC) {
    /*
    if (convert_ip(ip, &logentry->shost) == false)
      return;
    */
    if (logentry->sipaddr.set(ip) == false)
      return;
    if (mode & IPF_OPT_PORT)
      logentry->sport = port;
  }
  else {
    /*
    if (convert_ip(ip, &logentry->dhost) == false)
      return;
    */
    if (logentry->dipaddr.set(ip) == false)
      return;
    if (mode & IPF_OPT_PORT)
      logentry->dport = port;
  }
}

static void
ipfilter_parse_proto(char* input) {
  if (isdigit(input[0]))
    logentry->protocol = atoi(input);
  else {
    struct protoent* proto = getprotobyname(input);
    if (proto == NULL) {
      parseerror = true;
      fprintf(stderr, _("Unknown protocol, ignoring `%s'\n"), input);
      return;
    }
    logentry->protocol = proto->p_proto;
  }
}

static void
ipfilter_parse_tcpflags(char* input) {
  while (*input != '\0') {
    switch (*input) {
    case 'S':
      logentry->tcpflags |= TCP_SYN;
      break;
    case 'A':
      logentry->tcpflags |= TCP_ACK;
      break;
    case 'F':
      logentry->tcpflags |= TCP_FIN;
      break;
    case 'R':
      logentry->tcpflags |= TCP_RST;
      break;
    case 'P':
      logentry->tcpflags |= TCP_PSH;
      break;
    case 'U':
      logentry->tcpflags |= TCP_URG;
      break;
    }
    input++;
  }
}


wf_inmodule_ipfilter::wf_inmodule_ipfilter() {
}

bool
wf_inmodule_ipfilter::match(const string& line) const {
  return (line.find(" ipmon[") != string::npos);
}

enum wf_logentry_parsing_result
wf_inmodule_ipfilter::parse(wf_logentry** retlogentry,
  const string& line, int linenum) {
  logentry = new wf_logentry();
  if (logentry == NULL)
    return WF_LOGENTRY_PARSING_ERROR;

  ipfilter_scan_string(line.c_str());
  ipfilterlex();

  logentry->format = "ipfilter";

  /*
  if (parser == (IPFILTER_DATE|IPFILTER_DATA|IPFILTER_IPS))
  */
  if (parseerror == false) {
    *retlogentry = logentry;
    return WF_LOGENTRY_PARSING_OK;
  }

  if (verbose)
    fprintf(stderr, _("ipfilter parse error in line %d, ignoring.\n"),
	    linenum);

  delete logentry;
  return WF_LOGENTRY_PARSING_ERROR;
}

extern "C" wf_inmodule*
wf_inmodule_ipfilter_init() {
  return new wf_inmodule_ipfilter();
}
