/*
  libu - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 2002 Malcolm Walker
  Based on code copyright  (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "messagebox.h"
#include "pushbutton.h"
#include "application.h"
#include "multilineedit.h"
#include "rootwindow.h"
#include "box.h"
#include "filler.h"
#include "debug.h"

#include <sigc++/object_slot.h>

#include "gamma.xpm"

using namespace wftk;

class BackGamma : public RootWindow
{
 public:
  BackGamma(int w, int h)
    : RootWindow(w, h),
      timer(0), // no blocking, limit on drawing speed
      parent_surf(Surface::registry.find("background")),
      background(new Surface(*parent_surf)),
      res(new Surface::Resource(background)),
      gamma(0),
      darker(false),
      gamma_func(gamma_xpm)
#ifdef DEBUG
      , prev_tick(Time::now()),
      total_lag(0),
      n_samples(0)
#endif
  {
      gamma_func.setBitShift(-1, -1);
      setBackground(res);
      timer.alarm.connect(SigC::slot(*this, &BackGamma::updateBackground));
  }

  ~BackGamma()
    {
#ifdef DEBUG
      Debug::out << "Average update time of " << (total_lag / n_samples)
	<< " with " << n_samples << " samples" << Debug::endl;
#endif
      res->free();
    }

 private:
  Timer timer;
  const Surface* parent_surf;
  Surface* background;
  Surface::Resource* res;
  char gamma;
  bool darker;
  GammaFunction gamma_func;
#ifdef DEBUG
  Time prev_tick;
  unsigned long total_lag, n_samples;
#endif

  void updateBackground()
    {
#ifdef DEBUG
      Time tick = Time::now();
      int lag = tick - prev_tick;
      Debug::out << "update = " << lag << Debug::endl;
      total_lag += (lag >= 0) ? lag : 0;
      ++n_samples;
      prev_tick = tick;
#endif

      if(!darker) {
        if(++gamma == 127)
          darker = true;
      }
      else {
        if(--gamma == -128)
          darker = false;
      }

      gamma_func.shiftOffset(1, 5);

      parent_surf->blit(*background);
      background->gammaShift(gamma);
      background->gammaShift(gamma_func);
      invalidate();
    }

};

class GammaApp : public Application
{
 public:
    
  // Application startup code 
  GammaApp(int argc, char** argv) : Application(argc, argv)
    {
      Debug::out << "starting up ..." << Debug::endl;
      //do initialization here

      // These font resources are used by various gui elements,
      // so you should always register text_font and button_font when
      // using wftk
      Font::registry.load("text_font","wf_opal.ttf, 17, 0xF0F0F0FF, 0x707070FF");
      Font::registry.load("button_font","wf_opal.ttf, 16, 0xE0E0F0FF, 0x607f70EE");

      // Register a surface
      Surface::registry.load("background","background.png");

      BackGamma* back = new BackGamma(640,480);

      back->setTitle("LIBWFTK Demo");

      Box *main_box = new Box(Box::LEFT_TO_RIGHT);
      back->pack(main_box);

      Box *left_box = new Box(Box::TOP_TO_BOTTOM);
      main_box->packBack(left_box);

      //create the GUI
      left_box->packBack(new MultiLineEdit("wftk Gamma Demo Application\n"
					   "by Ron Steinke\n"
					   "Code originally by Karsten Laux\n",
					   Font::registry.find("button_font")));


      left_box->packBack(new Filler(Filler::BOTH));

      Box *right_box = new Box(Box::TOP_TO_BOTTOM);
      main_box->packBack(right_box);

      Box *logo_box = new Box(Box::LEFT_TO_RIGHT);
      right_box->packBack(logo_box);

      logo_box->packBack(new Filler(Filler::HORIZONTAL));

      logo_box->packBack(new Widget("wftk_logo"));

      right_box->packBack(new Filler(Filler::VERTICAL));

      //a pushbutton
      PushButton* quit_button = new PushButton("Quit");
      right_box->packBack(quit_button);

      //connect this button's click event (SIGNAL) with the application's
      //quit action (SLOT)
      quit_button->clicked.connect(quitSlot());
    }
};

int main (int argc, char **argv)
{
  Debug::init(Debug::GENERIC);

  return GammaApp(argc, argv).exec();
}
