/*
  libwftk - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 2002 Malcolm Walker
  Based on code copyright  (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "application.h"
#include "multilineedit.h"
#include "pushbutton.h"
#include "painter.h"
#include "listbox.h"
#include "resources.h"
#include "rootwindow.h"
#include "sample.h"
#include "mixer.h"
#include "screensurface.h"
#include "debug.h"
#include "box.h"
#include "filler.h"

#include "windowedroot.h"

using namespace wftk;

#include <sigc++/object_slot.h>

#include <iostream>
#include <sstream> // for string formatting ... I miss printf :(
#include <cstdlib> // for rand()

class LinesApp : public Application
{
  
private:
  WindowedRoot* back;
  ListBox* listbox;
  Label * fps_label;
  int count;
	int ticks;
	int last;
	
  Timer quick_idle;

	Timer fps_update;

	const Surface* texture;
	
public:

	// these functions are fired by the timers.
  void timer();
	void fps();
  
  //needs to be there, but do not use it, use startupProc instead
  LinesApp(int argc, char** argv) : 
    Application(argc, argv),
    count(0),
		ticks(0),
		last(0),
    quick_idle(0),
		fps_update(1000)
    {
      std::cout << "starting up ..." << std::endl;
      //do you init here

      //eg fontloading ....
      // These fonts are used by several gui elements,
      // so you should always load text_font and button_font when
      // using widgets from libwftk
      Font::registry.load("text_font","wf_opal.ttf, 16, 0xF0F0F0, 0x808080");
      Font::registry.load("button_font","wf_opal.ttf, 14, 0xF0F0F0, 0x808080");

      if(Mixer::instance()->audioAvailable()) {
        //register this wav file as resources "press"
        //which is played whenever an button gets pressed
        Sample::registry.load("press","press.wav");
        //same but for "release"
        Sample::registry.load("release","release.wav");
      }

      // these resources are used by ListBox:
//       Surface::registry.load("scroller_up","triangle_up.png");
//       Surface::registry.load("scroller_up_pressed","triangle_up_pressed.png");
//       Surface::registry.load("scroller_down","triangle_down.png");
//       Surface::registry.load("scroller_down_pressed","triangle_down_pressed.png");
      Surface::registry.load("button170","button.png");
      Surface::registry.load("button170_pressed","button_pressed.png");
      Surface::registry.load("texture", "logo.png");

      texture = Surface::registry.find("texture");
      if (!texture) {
              texture = Surface::registry.find("wftk_logo");
      }
      assert("texture loaded successfully" && texture);
			
      back = new WindowedRoot(640, 480);

      back->setTitle("LIBWFTK Painter Demo");

      Box* main_box = new Box(Box::LEFT_TO_RIGHT);
      back->pack(main_box);

      Box* left_box = new Box(Box::TOP_TO_BOTTOM);
      main_box->packBack(left_box);

      Box* right_box = new Box(Box::TOP_TO_BOTTOM);
      main_box->packBack(right_box);

      left_box->packBack(new MultiLineEdit("Painter / WindowedRoot Demo\n"
					   "by Malcolm Walker and Ron Steinke\n"
					   "originally by Karsten Laux, February 2000"));

      left_box->packBack(back->widget());

      // A text info for frames-per-second
      fps_label = new Label("FPS:");
      fps_label->setAlign(Label::ALIGN_LEFT);
      right_box->packBack(fps_label);

      right_box->packBack(new Filler(Filler::VERTICAL));

      // The quit button
      PushButton* quit_button = new PushButton("Quit");
      //quit_button->setColor(Color(32,32,128)); 

      listbox = new ListBox();

      listbox->setColor("DarkSlateBlue");
      listbox->setSelectedColor("white");

      right_box->packBack(listbox);

      right_box->packBack(new Filler(Filler::VERTICAL));

      right_box->packBack(quit_button);

      listbox->addElement("Circles");
      listbox->addElement("Ellipses");
      listbox->addElement("Lines");
      listbox->addElement("Textures 1");
      listbox->addElement("Textures 2");

      listbox->setSelected("Circles");
     
      quick_idle.alarm.connect(SigC::slot(*this, &LinesApp::timer));
      fps_update.alarm.connect(SigC::slot(*this, &LinesApp::fps));
    
      //connect this button's click event (SIGNAL) with the application's
      //quit action (SLOT)
      quit_button->clicked.connect(quitSlot());
    }
};

void ellipses(Surface& surf)
{
  int w = surf.width();
  int h = surf.height();

  Painter painter(&surf);
  Color col;
  Point center;
  int rx, ry;

  col = Color(rand()%255,rand()%255,rand()%255);
  painter.setColor(col);
  
  center = Point(rand()%w, rand()%h);
  rx = rand()%h/3;
  ry = rand()%h/3;
  painter.setFilling(false);
  painter.ellipse(center, rx, ry);
  
  center = Point(rand()%w, rand()%h);
   rx = rand()%h/3;
  ry = rand()%h/3; 
  painter.setFilling(true);
  painter.ellipse(center, rx, ry);
   
}

void circles(Surface& surf)
{
  int w = surf.width();
  int h = surf.height();

  Painter painter(&surf);
  Color col;
  Point center;
  int radius;

  col = Color(rand()%255,rand()%255,rand()%255,rand()%255);
  painter.setColor(col);
  
  center = Point(rand()%w, rand()%h);
  radius = rand()%h/2;
  painter.setFilling(false);
  painter.circle(center, radius);
  
  center = Point(rand()%w, rand()%h);
  radius = rand()%h/2;
  painter.setFilling(true);
  painter.circle(center, radius);
   
}

void lines(Surface& surf)
{
  int w = surf.width();
  int h = surf.height();
 
  //create a painter object for this surface

  Painter painter(&surf);
  Color col;
  Point start;
  Point end;

 
  start = Point(rand()%w, rand()%h);
  end = Point(rand()%w, rand()%h);
  
  col = Color(rand()%255,rand()%255,rand()%255);
  
  painter.line(start, end, col);
    
}

void textures1(Surface& surf, const Surface * texture) 
{
  int w = rand() % surf.width() - 25;
	int h = rand() % surf.height() - 25;

	int new_x = (int)((rand()%200 * texture->width()) / 100);
	int new_y = (int)((rand()%200 * texture->height()) / 100);

//	std::cout << w << " " << h << " " << new_x << " " << new_y << std::endl;
	texture->scaledBlit(surf, Rect(w, h, new_x, new_y));
}

// Same code as above ... was used to test sge vs. rotozoom
// will be used later with a custom Surface type that has
// the ability to rotate

// Right now the difference is smoothing vs. not
// textures2 is smoothing the surfaces after scaling
void textures2(Surface& surf, const Surface * texture) 
{
  int w = rand() % surf.width() - 25;
	int h = rand() % surf.height() - 25;

	int new_x = (int)((rand()%200 * texture->width()) / 100);
	int new_y = (int)((rand()%200 * texture->height()) / 100);

	texture->scaledBlit(surf, Rect(w, h, new_x, new_y), true);
}

void LinesApp::timer()
{
  --count;
  ++ticks;

  RootWindow::instance()->screen()->mutex.grab();

  if(count < 0)
    {
      count = 500;
      back->target().fill(Color(0,16,rand()%64));
    }

  std::string wish = listbox->getSelection();

	// TODO: Design some sort of case/switch logic for 
	// this type of handling ... or use an index from 
	// the list box !

  if(wish == "Lines")
    lines(back->target());
  else if(wish == "Ellipses")
    ellipses(back->target());
  else if(wish == "Circles")
    circles(back->target());
	else if(wish == "Textures 1") {
		textures1(back->target(), texture);
	}
	else if(wish == "Textures 2") {
		textures2(back->target(), texture);
	}

  back->update();

  RootWindow::instance()->screen()->mutex.release();
}

void LinesApp::fps() {
	std::stringstream ss;
	ss << "FPS: " << (ticks - last);
	
 	fps_label->setText(ss.str().c_str());	
 	last = ticks;
}
  
int main (int argc, char **argv)
{
  Debug::init(Debug::GENERIC);

  return LinesApp(argc, argv).exec();
}

