/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/


#ifndef _FOCUS_H
#define _FOCUS_H

#include <assert.h>

#include <list>

#include <sigc++/marshal.h>
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#include <sigc++/object.h>
#endif

#include <SDL/SDL_events.h>

#include <wftk/marshal.h>
#include <wftk/timer.h>

namespace wftk {

class Widget;

/// Singleton class for handling the current widget that owns keyboard focus
class Focus
{
 public:
  ///
  Focus() : CursorSparkle(300), owner_(0),
	have_focus_((SDL_GetAppState() & SDL_APPINPUTFOCUS) != 0)
	{assert(!instance_); instance_ = this;}
  ///
  ~Focus() {assert(instance_ = this); instance_ = 0;}
  /// Fetch the singleton instance
  static Focus* instance() {return instance_ ? instance_ : new Focus();}

  /** Keypress Signal object.
     Emitted whenever a key is pressed. Slots connected to this signal may
     swallow the event by returning true, or pass the event to other listeners
     by returning false. bool value is true for pressed, false for released
  */
  SigC::Signal2<bool,const SDL_keysym&,bool,BoolMarshal> keystroke;
  /// application has lost window focus
  SigC::Signal0<bool,BoolMarshal> lostFocus;
  /// application has gained window focus
  SigC::Signal0<bool,BoolMarshal> gotFocus;
  /// CursorSparkle keeps our cursor blinking
  Timer CursorSparkle;

  /// Attempt to release keyboard focus on the Widget \p w.
  bool release(Widget* w) {return grab(0);}
  /// Attempt to bless keyboard focus upon the Widget \p w.
  bool grab(Widget* w);
  /// true if any widgets have keyboard focus
  bool haveFocus() const {return have_focus_;}
  /// returns a pointer to the widget with current focus
  Widget* currentFocus() const {return owner_;}
  /// Pointer to the next widget capable of handling text focus
  Widget* nextFocus();
  /// Pointer to the previous widget capable of handling text focus
  Widget* prevFocus();
  /// Make Focus object aware of a new text / keyboard focus region
  /// (append to the end of the list)
  void addFocusable(Widget*);
  /// Remove the passed widget from the list of possible focus targets
  void removeFocusable(Widget*);

  /// Handle a keyboard event, called from the mainloop
  bool handleEvent(const SDL_Event*);
 
 private:
  ///
  Widget* owner_;
  ///
  bool have_focus_;
  ///
  typedef std::list<Widget*> Foci;
  ///
  Foci focusable_;
  ///
  Foci::iterator find(Widget*);
  ///
  static Focus* instance_;
};


} // namespace

#endif
