/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "font_data.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

FT_Library wftk::FontData::lib_;
unsigned long wftk::FontData::lib_refcount_ = 0;

wftk::FontData::FontData(const std::string& file, unsigned ptsize,
	unsigned face_index) throw(wftk::Font::BadFont) : map_(*this)
{
  reflib();

  if(FT_New_Face(lib_, file.c_str(), face_index, &face_)) {
    unreflib();
    throw Font::BadFont();
  }

  Point dpi = getDPI();

  if(FT_Set_Char_Size(face_, 0, ptsize, dpi.x, dpi.y)) {
    FT_Done_Face(face_);
    unreflib();
    throw Font::BadFont();
  }

  setMetrics();
}

wftk::FontData::FontData(const unsigned char* buffer, unsigned buf_size,
	unsigned ptsize, unsigned face_index) throw(wftk::Font::BadFont) : map_(*this)
{
  reflib();

  if(FT_New_Memory_Face(lib_, (FT_Byte *)buffer, buf_size, face_index, &face_)) {
    unreflib();
    throw Font::BadFont();
  }

  Point dpi = getDPI();

  if(FT_Set_Char_Size(face_, 0, ptsize, dpi.x, dpi.y)) {
    FT_Done_Face(face_);
    unreflib();
    throw Font::BadFont();
  }

  setMetrics();
}

wftk::FontData::~FontData()
{
  FT_Done_Face(face_);
  unreflib();
}

FT_GlyphSlot
wftk::FontData::operator[](unsigned char charnum)
{
  FT_Load_Char(face_, charnum, FT_LOAD_RENDER);
  return face_->glyph;
}

wftk::Point
wftk::FontData::getKerning(unsigned char prev, unsigned char next) const
{
  if(!FT_HAS_KERNING(face_))
    return Point(0, 0);

  FT_Vector delta;

  FT_Get_Kerning(face_, prev, next, ft_kerning_default, &delta);

  return Point(delta.x, delta.y);
}

void
wftk::FontData::reflib() throw(wftk::Font::BadFont)
{
  if(lib_refcount_ == 0 && FT_Init_FreeType(&lib_))
    throw Font::BadFont();

  ++lib_refcount_;
}

void
wftk::FontData::unreflib() throw()
{
  if(--lib_refcount_ == 0)
    FT_Done_FreeType(lib_);
}

void
wftk::FontData::setMetrics()
{
  metrics_.units_per_EM = face_->units_per_EM;
  metrics_.ascender = face_->ascender;
  metrics_.descender = face_->descender;
  metrics_.height = face_->height;

  metrics_.max_advance_width = face_->max_advance_width;
  metrics_.max_advance_height = face_->max_advance_height;

  metrics_.underline_position = face_->underline_position;
  metrics_.underline_thickness = face_->underline_thickness;

  metrics_.xMin = face_->bbox.xMin;
  metrics_.xMax = face_->bbox.xMax;
  metrics_.yMin = face_->bbox.yMin;
  metrics_.yMax = face_->bbox.yMax;
}
