/*    Copyright (C) 1998 XIAO, Gang of Universite de Nice - Sophia Antipolis
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
	/* line input / output routines */

#include <stdarg.h>

void *xmalloc(size_t n)
{
    void *p;
    p=malloc(n);
    if(p==NULL) {
	fprintf(stderr,"Malloc failure."); exit(1);
    }
    return p;
}

  /* dos/mac to unix/linux translation */
void _tolinux(char *p)
{
    char *pp;
    for(pp=strchr(p,13);pp!=NULL;pp=strchr(pp,13)) {
	if(*(pp+1)=='\n' || (pp>p && *(pp-1)=='\n')) strcpy(pp,pp+1);
	else pp++;
    }
}

	/* find matching parenthesis.
	 * The entrance point should be after the opening 
	 * parenthesis.
	 * Returns NULL if unmatched. */
char *find_matching(char *p, char c)
{
    char *pp;
    int parenth, brak, brace;
    if(c=='|') {
	for(pp=p;*pp!=0;pp++) {
	    switch(*pp) {
		case '|': return pp;
		case '(': {
		    pp=find_matching(pp+1,')');
		    if(pp==NULL) return NULL;
		    break;
		}
		case '[': {
		    pp=find_matching(pp+1,']');
		    if(pp==NULL) return NULL;
		    break;
		}
		case '{': {
		    pp=find_matching(pp+1,'}');
		    if(pp==NULL) return NULL;
		    break;
		}
		case ')':
		case ']':
		case '}': return NULL;
		
		default: break;
	    }
	}
	return NULL;
    }
    parenth=brak=brace=0;
    for(pp=p; *pp!=0; pp++) {
	switch(*pp) {
	    case '[': brak++; break;
	    case ']': brak--; break;
	    case '(': parenth++; break;
	    case ')': parenth--; break;
	    case '{': brace++; break;
	    case '}': brace--; break;
	    default: continue;
	}
	if(parenth<0 || brak<0 || brace<0) {
	    if(*pp!=c || parenth>0 || brak>0 || brace>0) return NULL;
	    else break;
	}
    }
    if(*pp!=c) return NULL;
    return pp;
}

	/* Points to the end of the word */
char *find_word_end(char *p)
{
    int i;
    for(i=0;!isspace(*p) && *p!=0 && i<MAX_LINELEN; p++,i++);
    return p;
}

	/* Strips leading spaces */
char *find_word_start(char *p)
{
    int i;
    for(i=0; isspace(*p) && i<MAX_LINELEN; p++,i++);
    return p;
}

	/* find a character in a string, but skipping parentheses. */
char *strparchr(char *p, char c)
{
    char *pp;
    
    for(pp=p;*pp && *pp!=c && pp-p<MAX_LINELEN; pp++) {
	switch(*pp) {
	    case '(': pp=find_matching(pp+1,')'); break;
	    case '[': pp=find_matching(pp+1,']'); break;
	    case '{': pp=find_matching(pp+1,'}'); break;
	}
	if(pp==NULL) return NULL;
    }
    if(*pp==c) return pp; else return NULL;
}

	/* search for string, skipping parentheses */
char *strparstr(char *p, char *fnd)
{
    char *pp, c;
    int n;
    
    if(*fnd==0) return p+strlen(p);
    c=*fnd; n=strlen(fnd);
    for(pp=p;pp-1!=NULL && *pp!=0; pp++) {
	if(*pp==c && (n==1 || memcmp(pp,fnd,n)==0)) return pp;
	switch(*pp) {
	    case '(': pp=find_matching(pp+1,')'); break;
	    case '[': pp=find_matching(pp+1,']'); break;
	    case '{': pp=find_matching(pp+1,'}'); break;
	}
    }
    if(pp-1==NULL) pp=strstr(p,fnd);
    if(pp!=NULL) return pp; 
    else return p+strlen(p);
}
    
	/* Points to the end of an item */
char *find_item_end(char *p)
{
    return strparstr(p,",");
}

char *charchr(char *p,char *w)
{
    return strchr(p,w[0]);
}

	/* Find first occurrence of word */
char *wordchr(char *p, char *w)
{
    char *r;

    if(*w==0) return NULL;
    for(r=strstr(p,w);r!=NULL && 
	( (r>p && !isspace(*(r-1))) || (!isspace(*(r+strlen(w))) && *(r+strlen(w))!=0) );
	r=strstr(r+1,w));
    return r;
}

	/* Find first occurrence of item */
char *itemchr(char *p, char *w)
{
    char *r, *r1, *r2;

    if(*w==0) return NULL;
    for(r=strstr(p,w);r!=NULL; r=strstr(r+1,w)) {
	r1=r-1;while(r1>=p && isspace(*r1)) r1--;
	r2=find_word_start(r+strlen(w));
	if((r1<p || *r1==',') && (*r2==0 || *r2==',')) return r;
    }
    return r;
}

	/* Find first occurrence of line */
char *linechr(char *p, char *w)
{
    char *r;
    int n;

    n=strlen(w); if(n==0) return NULL;
    for(r=strstr(p,w);r!=NULL; r=strstr(r+1,w)) {
	if((r<=p || *(r-1)=='\n') && (*(r+n)==0 || *(r+n)=='\n'))
	  break;
    }
    return r;
}

	/* find a variable in a string (math expression).
	 * Returns the pointer or NULL. */
char *varchr(char *p, char *v)
{
    char *pp; int n=strlen(v);
    for(pp=strstr(p,v); pp!=NULL; pp=strstr(pp+1,v)) {
	if((pp==p || (!isalnum(*(pp-1)) && *(pp-1)!='_')) &&
	   ((!isalnum(*(pp+n)) && *(pp+n)!='_' && *(pp+n)!='\'')
	    || *(pp+n)==0)) break;
    }
    return pp;
}

	/* strip trailing spaces; return string end. */
char *strip_trailing_spaces(char *p)
{
    char *pp;
    if(*p==0) return p;
    for(pp=p+strlen(p)-1; pp>=p && isspace(*pp); *(pp--)=0);
    return pp;
}

        /* Routines for quick search of item in a list. */

	/* Verify whether a list is well-ordered. For debugging uses.
	 * Returns 0 if order is OK, -1 otherwise. */
int verify_order(void *list, int items, size_t item_size)
{
    int i; char *old, **p;
    p=list; old=*p;
    for(i=item_size;i<items*item_size;i+=item_size) {
	p=list+i;
	if(strcmp(*p,old)<0) {
	    fprintf(stderr,"Table disorder: %s > %s",old,*p);
	    exit(1);
	}
	old=*p;
    }
    return 0;
}

	/* searches a list. Returns index if found, -1 if nomatch. 
	 * Uses binary search, list must be sorted. */
int search_list(void *list, int items, size_t item_size, const char *str)
{
    int i1,i2,j,k;
    char **p;
    char c;
    
    if(items<=0) return -1;
    j=0; c=*str;
    p=list;
    k=**p-c; if(k==0) k=strcmp(*p,str);
    if(k==0) return k; if(k>0) return -1;
    p=list+(items-1)*item_size; 
    k=**p-c; if(k==0) k=strcmp(*p,str);
    if(k==0) return items-1; if(k<0) return ~items;
    for(i1=0,i2=items-1;i2>i1+1;) {
	j=i1+(i2-i1)/2;
	p=list+(j*item_size);
	k=**p-c; if(k==0) k=strcmp(*p,str);
	if(k==0) return j;
	if(k>0) {i2=j; continue;}
	if(k<0) {i1=j; continue;}	
    }
    return ~i2;
}

	/* Returns number of lines in string p */
unsigned int linenum(char *p)
{
    int i; char *pp;

    /* Ending blank line will be thus removed. */
    i=strlen(p); if(i>1 && *(p+i-1)=='\n') *(p+i-1)=0;
    if(*p=='\n') i=1; else i=0;
    for(pp=p; pp!=NULL && *pp!=0; pp=strchr(pp+1,'\n'), i++);
    return i;
}

	/* Returns number of items in the list p, comma separated */
unsigned int itemnum(char *p)
{
    int i; char *pp;

    if(*p==0) return 0;
    for(i=0,pp=p; pp==p || *(pp-1)!=0; pp=find_item_end(pp)+1, i++);
    return i;
}

	/* Returns number of words in string p */
unsigned int wordnum(char *p)
{
    int i; char *pp;
    
    for(i=0, pp=find_word_start(p); *pp!=0; i++) {
	while(!isspace(*pp) && *pp!=0) pp++;
	pp=find_word_start(pp);
    }
    return i;
}

	/* This is just to suppress an annoying compiler warning message. */
unsigned int charnum(char *p)
{
    return strlen(p);
}

	/* find n-th line in string p */
char *fnd_line(char *p, int n, char bf[])
{
    char *pp;
    int i;
    
    for(i=1,pp=p; pp-1!=NULL && *pp!=0 && i<n; pp=strchr(pp,'\n')+1, i++);
    fnd_position=pp; if(pp-1==NULL) {
	fnd_position=NULL; *bf=0; return bf;
    }
    for(i=0; *(pp+i)!=0 && *(pp+i)!='\n'; i++) *(bf+i)=*(pp+i);
    *(bf+i)=0; return bf;
}

	/* find n-th item in list p, comma separated */
char *fnd_item(char *p, int n, char bf[])
{
    char *pp, *pe;
    int i;
    
    for(i=1,pp=p; i<n && (pp==p || *(pp-1)!=0); pp=find_item_end(pp)+1, i++);
    fnd_position=pp; if(pp>p && *(pp-1)==0) {
	fnd_position=NULL; *bf=0; return bf;
    }
    pp=find_word_start(pp); pe=find_item_end(pp);
    while(pe>pp && isspace(*(pe-1))) pe--;
    memmove(bf,pp,pe-pp); bf[pe-pp]=0;
    return bf;
}

	/* find n-th word in string p */
char *fnd_word(char *p, int n, char bf[])
{
    char *pp;
    int i;
    
    for(i=1, pp=find_word_start(p); *pp!=0 && i<n ; i++) {
	while(!isspace(*pp) && *pp!=0) pp++;
	pp=find_word_start(pp);
    }
    if(*pp) fnd_position=pp; else fnd_position=NULL;
    for(i=0; *(pp+i)!=0 && !isspace(*(pp+i)); i++) *(bf+i)=*(pp+i);
    *(bf+i)=0; 
    return bf;
}

	/* find n-th char in string p */
char *fnd_char(char *p, int n, char bf[])
{
    int t;
    
    t=strlen(p);
    if(n>t || n<1) *bf=0;
    else {
	*bf=*(p+n-1); *(bf+1)=0;
    }
    return bf;
}

int _separator(char *p,char *parm[], int max, char fs)
{
    int i;
    char *p1, *p2, *p3;
    
    for(i=0;i<max;i++) parm[i]=NULL;
    if(fs=='\n') p1=p; else p1=find_word_start(p);
    for(i=0;i<max && *p1!=0;i++,p1=p2) {
	switch(fs) {
	    case ' ': p2=find_word_end(p1); break;
	    case ',': p2=find_item_end(p1); break;
	    default: p2=strchr(p1,fs); if(p2==NULL) p2=p1+strlen(p1); break;
	}
	p3=p2; if(*p2) {
	    p3=p2; 
	    if(fs=='\n') p2++;
	    else p2=find_word_start(++p2);
	}
	if(fs==',') while(p3>p1 && isspace(*(p3-1))) p3--; 
	*p3=0; parm[i]=p1;
    }
    return i;
}

	/* Separate items in the string p, end each item with 0,
	 * and store item pointers in parm[]. Does not parse past max.
	 * Returns the number of fields. */
int separate_item(char *p, char *parm[], int max)
{    return _separator(p,parm,max,','); }

/*
int separate_line(char *p, char *parm[], int max)
{    return _separator(p,parm,max,'\n'); }

int separate_word(char *p, char *parm[], int max)
{    return _separator(p,parm,max,' '); }
*/

	/* Returns 1 if semicolons changed to new lines */
int rows2lines(char *p)
{
    char *pp, *p2;
    int t;
    if(strchr(p,'\n')!=NULL) return 0;
    for(t=0, pp=p; *pp; pp++) {
	if(*pp==';') {*pp='\n'; t++; continue;}
	if(*pp=='&' && isalpha(*(pp+1))) {
	    for(p2=pp+1; isalpha(*p2) && p2-pp<14; p2++);
	    pp=p2; continue;
	}
	if(*pp=='&' && *(pp+1)=='#') {
	    for(p2=pp+2; isdigit(*p2) && p2-pp<6; p2++);
	    pp=p2; continue;
	}
    }
    return t;
}

void lines2rows(char *p)
{
    char *pp;
    strip_trailing_spaces(p);
    for(pp=strchr(find_word_start(p),'\n'); pp!=NULL; pp=strchr(pp+1,'\n'))
      *pp=';';
}

unsigned int rownum(char *p)
{
    char buf[MAX_LINELEN+1];
    snprintf(buf,sizeof(buf),"%s",p);
    rows2lines(buf);
    return linenum(buf);
}

	/* find n-th row in a matrix p */
char *fnd_row(char *p, int n, char bf[])
{
    rows2lines(p); return fnd_line(p,n,bf);
}

	/* Strip enclosing pairs of parentheses */
void strip_enclosing_par(char *p)
{
    char *p1;
    partest: strip_trailing_spaces(p);
    if(*p=='(') {
	p1=find_matching(p+1,')');
	if(*(p1+1)==0) {
	    *p1=0; strcpy(p,find_word_start(p+1));
	    goto partest;
	}
    }
    if(*p=='[') {
	p1=find_matching(p+1,']');
	if(*(p1+1)==0) {
	    *p1=0; strcpy(p,find_word_start(p+1));
	    goto partest;
	}
    }
    if(*p=='{') {
	p1=find_matching(p+1,'}');
	if(*(p1+1)==0) {
	    *p1=0; strcpy(p,find_word_start(p+1));
	    goto partest;
	}
    }
}

	/* strstr but may have embedde deros. 
	 * Returns memory end if not found.
	 * Supposes memory ends with 0. */
char *memstr(char *s1, char *s2, int len)
{
    char *p, *pp;
    pp=s1;
    for(p=s1; p<s1+len; p=pp) {
	pp=strstr(p,s2); if(pp!=NULL) break;
	pp=p+strlen(p);
	while(pp<s1+len && *pp==0) pp++;
    }
    return pp;
}

	/* Check whether parentheses are balanced in a given string.
	 * Returns 0 if OK. */
	/* style=0: simple check. style<>0: strong check. */
int check_parentheses(char *p, int style)
{
    int i,j,k;
    j=strlen(p);
    if(j>=MAX_LINELEN) return 65535;
    if(style!=0) {
	char buf[MAX_LINELEN+1];
	char *pp, *pe, c;
	for(pp=p;pp<p+j;pp++) {
	    switch (*pp) {
		case ')':
		case ']':
		case '}': return -1;
		case '(': c=')'; goto find;
		case '[': c=']'; goto find;
		case '{': c='}';
		find: {
		    pe=find_matching(pp+1,c);
		    if(pe==NULL) return 1;
		    memcpy(buf,pp+1,pe-pp-1);
		    buf[pe-pp-1]=0;
		    if((k=check_parentheses(buf,1))!=0) return k;
		    else pp=pe;
		}
		default: break;
	    }
	}
	return 0;
    }
    for(i=k=0;i<j && k>=0;i++) {
	if(*(p+i)=='(') k++;
	if(*(p+i)==')') k--;
    }
    return k;
}

	/* change all spaces into ' ', and collapse multiple occurences */
void singlespace(char *p)
{
    char *pp, *p2;
    for(pp=p;*pp;pp++) {
	if(!isspace(*pp)) continue;
	if(*pp!=' ') *pp=' ';
	if(!isspace(*(pp+1))) continue;
	for(pp++,p2=pp;isspace(*p2);p2++);
	strcpy(pp,p2); pp--;
    }
    
}

	/* collapses all space characters in string. */
void nospace(char *p)
{
    char *pp, *p2;
    for(pp=p;*pp;pp++) {
	if(isspace(*pp)) {
	    for(p2=pp+1; isspace(*p2); p2++);
	    strcpy(pp,p2); pp--;
	}
    }
}

	/* change words to items */
void words2items(char *p)
{
    char *pp;
    singlespace(p);
    if(*p==' ') strcpy(p,p+1);
    if(*(p+strlen(p)-1)==' ') *(p+strlen(p)-1)=0;
    for(pp=p;*pp;pp++) if(*pp==' ') *pp=',';
}

	/* change words to lines */
void words2lines(char *p)
{
    char *pp;
    singlespace(p);
    if(*p==' ') strcpy(p,p+1);
    if(*(p+strlen(p)-1)==' ') *(p+strlen(p)-1)=0;
    for(pp=p;*pp;pp++) if(*pp==' ') *pp='\n';
}

	/* change lines to items */
void lines2items(char *p)
{
    char *pp;
    for(pp=p;*pp;pp++) if(*pp=='\n') *pp=',';
}

	/* change lines to words */
void lines2words(char *p)
{
    char *pp;
    for(pp=p;*pp;pp++) if(*pp=='\n') *pp=' ';
}

	/* change items to words */
void items2words(char *p)
{
    char *pp;
    for(pp=strparstr(p,",");*pp;pp=strparstr(pp+1,",")) *pp=' ';
}

	/* change items to lines */
void items2lines(char *p)
{
    char *pp;
    for(pp=strparstr(p,",");*pp;pp=strparstr(pp+1,",")) *pp='\n';
}

char *acctab="",
     *deatab="ceeeeuuuuaaaaaoooooiiiinyCEEEEUUUUAAAAAOOOOOIIIINY";

	/* fold accented letters to unaccented */
void deaccent(char *p)
{
    signed char *sp;
    char *v;
    for(sp=p;*sp;sp++) {
	if(*sp<0 && (v=strchr(acctab,*sp))!=NULL)
	  *sp=*(deatab+(v-acctab));
    }
}

	/* modify a string. Bufferlen must be ast least MAX_LINELEN */
void string_modify(char *start, char *bad_beg, char *bad_end, char *good,...)
{
    char buf[MAX_LINELEN+1];
    int ln, le;
    va_list vp;
    
    va_start(vp,good);
    vsnprintf(buf,sizeof(buf),good,vp); va_end(vp);
    ln=strlen(buf); le=strlen(bad_end);
    if(ln+le+(bad_beg-start)>=MAX_LINELEN) {
	error1("string_too_long"); return;
    }
    if(ln!=bad_end-bad_beg) memmove(bad_beg+ln,bad_end,le+1);
    memmove(bad_beg,buf,ln);
}

void catfile(FILE *outf, char *fn,...)
{
    char buf[4096];
    va_list vp;
    FILE *f;
    int l;
    
    va_start(vp,fn);
    vsnprintf(buf,sizeof(buf),fn,vp); va_end(vp);
    f=fopen(buf,"r"); if(f==NULL) return;
    for(l=fread(buf,1,4096,f);l>0 && l<=4096; l=fread(buf,1,4096,f))
      fwrite(buf,1,l,outf);
    fclose(f);
}

	/* returns -1 if error */
long int filelength(char *fn,...)
{
    char buf[4096];
    va_list vp;
    struct stat st;
    int l;
    
    va_start(vp,fn);
    vsnprintf(buf,sizeof(buf),fn,vp); va_end(vp);
    l=stat(buf,&st); if(l) return -1;
    return st.st_size;
}

