/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */
#include <boost/lexical_cast.hpp>

#include <Wt/WApplication>
#include <Wt/WText>
#include <Wt/WImage>
#include <Wt/WPushButton>

#include "DemoTreeList.h"
#include "TreeNode.h"
#include "IconPair.h"

using namespace Wt;
using std::rand;

DemoTreeList::DemoTreeList(WContainerWidget *parent)
  : WContainerWidget(parent),
    testCount_(0)
{
  addWidget
    (new WText(L"<h2>Wt Tree List example</h2>"
	       L"<p>This is a simple demo of a treelist, implemented using"
	       L" <a href='http://witty.sourceforge.net/'>Wt</a>.</p>"
	       L"<p>The leafs of the tree contain the source code of the "
	       L"tree-list in the classes <b>TreeNode</b> and "
	       L"<b>IconPair</b>, as well as the implementation of this "
	       L"demo itself in the class <b>DemoTreeList</b>.</p>"));

  tree_ = makeTreeMap(L"Examples", 0);
  addWidget(tree_);

  TreeNode *treelist = makeTreeMap(L"Tree List", tree_);
  TreeNode *wstateicon = makeTreeMap(L"class IconPair", treelist);
  makeTreeFile(L"<a href=\"IconPair.h\">IconPair.h</a>", wstateicon);
  makeTreeFile(L"<a href=\"IconPair.C\">IconPair.C</a>", wstateicon);
  TreeNode *wtreenode = makeTreeMap(L"class TreeNode", treelist);
  makeTreeFile(L"<a href=\"TreeNode.h\">TreeNode.h</a>", wtreenode);
  makeTreeFile(L"<a href=\"TreeNode.C\">TreeNode.C</a>", wtreenode);
  TreeNode *demotreelist = makeTreeMap(L"class DemoTreeList", treelist);
  makeTreeFile(L"<a href=\"DemoTreeList.h\">DemoTreeList.h</a>", demotreelist);
  makeTreeFile(L"<a href=\"DemoTreeList.C\">DemoTreeList.C</a>", demotreelist);

  testMap_ = makeTreeMap(L"Test map", tree_);

  /*
   * Buttons to dynamically demonstrate changing the tree contents.
   */
  addWidget
    (new WText(L"<p>Use the following buttons to change the tree "
	       L"contents:</p>"));

  addMapButton_
    = new WPushButton(L"Add map", this);
  addMapButton_->clicked.connect(SLOT(this, DemoTreeList::addMap));

  removeMapButton_
    = new WPushButton(L"Remove map", this);
  removeMapButton_->clicked.connect(SLOT(this, DemoTreeList::removeMap));
  removeMapButton_->disable();

  addWidget
    (new WText(L"<p>Remarks:"
	       L"<ul>"
	       L"<li><p>This is not the instantiation of a pre-defined "
	       L"tree list component, but the full implementation of such "
	       L"a component, in about 350 lines of C++ code !</p> "
	       L"<p>In comparison, the <a href='http://myfaces.apache.org'> "
	       L"Apache MyFaces</a> JSF implementation of tree2, with similar "
	       L"functionality, uses about 2400 lines of Java, and 140 lines "
	       L"of JavaScript code.</p></li>"
	       L"<li><p>Once loaded, the tree list does not require any "
	       L"interaction with the server for handling the click events on "
	       L"the <img src='icons/nav-plus-line-middle.gif' /> and "
	       L"<img src='icons/nav-minus-line-middle.gif' /> icons, "
	       L"because these events have been connected to slots using "
	       L"STATIC connections. Such connections are converted to the "
	       L"appropriate JavaScript code that is inserted into the page. "
	       L"Still, the events are signaled to the server to update the "
	       L"application state.</p></li>"
	       L"<li><p>In contrast, the buttons for manipulating the tree "
	       L"contents use DYNAMIC connections, and thus the update "
	       L"is computed at server-side, and communicated back to the "
	       L"browser (by default using AJAX).</p></li>"
	       L"<li><p>When loading a page, only visible widgets (that are not "
	       L"<b>setHidden(true)</b>) are transmitted. "
	       L"The remaining widgets are loaded in the background after "
	       L"rendering the page. "
	       L"As a result the application is loaded as fast as possible.</p>"
	       L"</li>"
	       L"<li><p>The browser reload button is supported and behaves as "
	       L"expected: the page is reloaded from the server. Again, "
	       L"only visible widgets are transmitted immediately.</p> "
	       L"<p>(For the curious, this is the way to see the actual "
	       L"HTML/JavaScript code !)</p></li>"
	       L"</ul></p>"));
}

void DemoTreeList::addMap()
{
  TreeNode *node
    = makeTreeMap(L"Map " + boost::lexical_cast<std::wstring>(++testCount_),
		  testMap_);
  makeTreeFile(L"File " + boost::lexical_cast<std::wstring>(testCount_),
	       node);

  removeMapButton_->enable();
}

void DemoTreeList::removeMap()
{
  int numMaps = testMap_->childNodes().size();

  if (numMaps > 0) {
    int c = rand() % numMaps;

    TreeNode *child = testMap_->childNodes()[c];
    testMap_->removeChildNode(child);
    delete child;

    if (numMaps == 1)
      removeMapButton_->disable();
  }
}

TreeNode *DemoTreeList::makeTreeMap(const std::wstring name, TreeNode *parent)
{
  IconPair *labelIcon
    = new IconPair("icons/yellow-folder-closed.png",
		   "icons/yellow-folder-open.png",
		   false);

  TreeNode *node = new TreeNode(name, WText::PlainFormatting, labelIcon, 0);
  if (parent)
    parent->addChildNode(node);

  return node;
}

TreeNode *DemoTreeList::makeTreeFile(const std::wstring name,
				      TreeNode *parent)
{
  IconPair *labelIcon
    = new IconPair("icons/document.png", "icons/yellow-folder-open.png",
		   false);

  TreeNode *node = new TreeNode(name, WText::XHTMLFormatting, labelIcon, 0);
  if (parent)
    parent->addChildNode(node);

  return node;
}

WApplication *createApplication(const WEnvironment& env)
{
  WApplication *app = new WApplication(env);
  new DemoTreeList(app->root());

  /*
   * The look & feel of the tree node is configured using a CSS style sheet.
   * If you are not familiar with CSS, you can use the WCssDecorationStyle
   * class ...
   */
  WCssDecorationStyle treeNodeLabelStyle;
  treeNodeLabelStyle.font().setFamily(WFont::Serif, L"Helvetica");
  app->styleSheet().addRule(".treenodelabel", treeNodeLabelStyle);

  /*
   * ... or if you speak CSS fluently, you can add verbatim rules.
   */
  app->styleSheet().addRule(".treenodechildcount",
			    L"color:blue; font-family:Helvetica,serif;");

  return app;
}

int main(int argc, char **argv)
{
   return WRun(argc, argv, &createApplication);
}

