/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */
#include <boost/lexical_cast.hpp>

#include <Wt/WBreak>
#include <Wt/WImage>
#include <Wt/WLabel>
#include <Wt/WGroupBox>
#include <Wt/WLineEdit>
#include <Wt/WPushButton>
#include <Wt/WText>

#include "TreeListExample.h"
#include "TreeNode.h"
#include "IconPair.h"

TreeListExample::TreeListExample(WContainerWidget *parent)
  : WContainerWidget(parent),
    testCount_(0)
{
  tree_ = makeTreeMap("TreeListExample", 0);
  addWidget(tree_);
  tree_->expand();

  TreeNode *treelist = makeTreeMap("Tree List", tree_);
  TreeNode *wstateicon = makeTreeMap("class IconPair", treelist);
  makeTreeFile("IconPair.h", wstateicon);
  makeTreeFile("IconPair.C", wstateicon);
  TreeNode *wtreenode = makeTreeMap("class TreeNode", treelist);
  makeTreeFile("TreeNode.h", wtreenode);
  makeTreeFile("TreeNode.C", wtreenode);
  TreeNode *wtreeexample = makeTreeMap("class TreeListExample", treelist);
  makeTreeFile("TreeListExample.h", wtreeexample);
  makeTreeFile("TreeListExample.C", wtreeexample);

  testMap_ = makeTreeMap("Test map", tree_);

  /*
   * Buttons to demonstrate dynamically changing the tree contents
   * implies no magic at all.
   */
  addWidget
    (new WText("<p>Use the following buttons to change the "
	       "contents of the Test map:</p>"));

  WGroupBox *addBox = new WGroupBox("Add map", this);

  WLabel *mapNameLabel = new WLabel("Map name:", addBox);
  mapNameLabel->setMargin(WLength(1, WLength::FontEx), Right);
  mapNameEdit_ = new WLineEdit("Map", addBox);
  mapNameLabel->setBuddy(mapNameEdit_);

  /*
   * Example of validation: make the map name mandatory, and give
   * feedback when invalid.
   */
  mapNameEdit_->setValidator(new WValidator(true));

  addMapButton_ = new WPushButton("Add map", addBox);
  addMapButton_->clicked.connect(SLOT(this, TreeListExample::addMap));

  new WBreak(this);

  WGroupBox *removeBox = new WGroupBox("Remove map", this);

  removeMapButton_
    = new WPushButton("Remove map", removeBox);
  removeMapButton_->clicked.connect(SLOT(this, TreeListExample::removeMap));
  removeMapButton_->disable();
}

void TreeListExample::addMap()
{
  if (mapNameEdit_->validate() == WValidator::Valid) {
    TreeNode *node
      = makeTreeMap(mapNameEdit_->text() + " "
		    + boost::lexical_cast<std::string>(++testCount_),
		    testMap_);
    makeTreeFile("File " + boost::lexical_cast<std::string>(testCount_),
		 node);

    removeMapButton_->enable();
  } else
    mapNameEdit_->setStyleClass("Wt-invalid");
}

void TreeListExample::removeMap()
{
  int numMaps = testMap_->childNodes().size();

  if (numMaps > 0) {
    int c = rand() % numMaps;

    TreeNode *child = testMap_->childNodes()[c];
    testMap_->removeChildNode(child);
    delete child;

    if (numMaps == 1)
      removeMapButton_->disable();
  }
}

TreeNode *TreeListExample::makeTreeMap(const WString& name,
				       TreeNode *parent)
{
  IconPair *labelIcon
    = new IconPair("icons/yellow-folder-closed.png",
		   "icons/yellow-folder-open.png",
		   false);

  TreeNode *node = new TreeNode(name, WText::PlainFormatting, labelIcon, 0);
  if (parent)
    parent->addChildNode(node);

  return node;
}

TreeNode *TreeListExample::makeTreeFile(const WString& name,
					TreeNode *parent)
{
  IconPair *labelIcon
    = new IconPair("icons/document.png", "icons/yellow-folder-open.png",
		   false);

  TreeNode *node = new TreeNode("<a href=\"" + name + "\" target=\"_blank\">"
				+ name + "</a>", WText::XHTMLFormatting,
				labelIcon, 0);
  if (parent)
    parent->addChildNode(node);

  return node;
}
