/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */

#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string/predicate.hpp>

#include "Wt/WModelIndex"
#include "Wt/WAbstractItemModel"
#include "Wt/WDate"
#include "Wt/WWebWidget"

#include "WtException.h"

namespace Wt {

WAbstractItemModel::WAbstractItemModel(WObject *parent)
  : WObject(parent)
{ }

WAbstractItemModel::~WAbstractItemModel()
{ }

boost::any WAbstractItemModel::data(int row, int column) const
{
  return data(index(row, column));
}

bool WAbstractItemModel::hasIndex(int row, int column) const
{
  return index(row, column).isValid();
}

bool WAbstractItemModel::insertColumns(int column, int count)
{
  return false;
}

bool WAbstractItemModel::insertRows(int row, int count)
{
  return false;
}

bool WAbstractItemModel::removeColumns(int column, int count)
{
  return false;
}

bool WAbstractItemModel::removeRows(int row, int count)
{
  return false;
}

bool WAbstractItemModel::insertColumn(int column)
{
  return insertColumns(column, 1);
}

bool WAbstractItemModel::insertRow(int row)
{
  return insertRows(row, 1);
}

bool WAbstractItemModel::removeColumn(int column)
{
  return removeColumns(column, 1);
}

bool WAbstractItemModel::removeRow(int row)
{
  return removeRows(row, 1);
}

bool WAbstractItemModel::setData(int row, int column, const boost::any& value)
{
  WModelIndex i = index(row, column);

  if (i.isValid())
    return setData(i, value);
  else
    return false;
}

WModelIndex WAbstractItemModel::createIndex(int row, int column) const
{
  return WModelIndex(row, column, this);
}

void WAbstractItemModel::beginInsertColumns(int first, int last)
{
  first_ = first;
  last_ = last;

  columnsAboutToBeInserted.emit(first, last);
}

void WAbstractItemModel::endInsertColumns()
{
  columnsInserted.emit(first_, last_);
}

void WAbstractItemModel::beginInsertRows(int first, int last)
{
  first_ = first;
  last_ = last;

  rowsAboutToBeInserted.emit(first, last);
}

void WAbstractItemModel::endInsertRows()
{
  rowsInserted.emit(first_, last_);
}

void WAbstractItemModel::beginRemoveColumns(int first, int last)
{
  first_ = first;
  last_ = last;

  columnsAboutToBeRemoved.emit(first, last);
}

void WAbstractItemModel::endRemoveColumns()
{
  columnsRemoved.emit(first_, last_);
}

void WAbstractItemModel::beginRemoveRows(int first, int last)
{
  first_ = first;
  last_ = last;

  rowsAboutToBeRemoved.emit(first, last);
}

void WAbstractItemModel::endRemoveRows()
{
  rowsRemoved.emit(first_, last_);
}

void WAbstractItemModel::refresh()
{
  for (int j = 0; j < columnCount(); ++j) {
    boost::any v = headerData(j);
    
    if (v.type() == typeid(WString)) {
      WString& ws = boost::any_cast<WString&>(v);
      if (ws.refresh())
	setHeaderData(j, v);
    }
    
    for (int i = 0; i < rowCount(); ++i) {
      boost::any v = data(i, j);
    
      if (v.type() == typeid(WString)) {
	WString& ws = boost::any_cast<WString&>(v);
	if (ws.refresh())
	  setData(i, j, v);
      }
    }
  }
}

namespace {

bool matchValue(const boost::any& value,
		const boost::any& query,
		MatchFlags flags)
{
  MatchFlags f = MatchFlags(flags & 0x1F);

  if (f == MatchExactly)
    return (query.type() == value.type())
      && asJSLiteral(query) == asJSLiteral(value);
  else {
    std::string query_str = asString(query).toUTF8();
    std::string value_str = asString(value).toUTF8();

    switch (f) {
    case MatchStringExactly:
      return boost::algorithm::iequals(value_str, query_str);
    case MatchStringExactly | MatchCaseSensitive:
      return boost::algorithm::equals(value_str, query_str);

    case MatchStartsWith:
      return boost::algorithm::istarts_with(value_str, query_str);
    case MatchStartsWith | MatchCaseSensitive:
      return boost::algorithm::starts_with(value_str, query_str);

    case MatchEndsWith:
      return boost::algorithm::iends_with(value_str, query_str);
    case MatchEndsWith | MatchCaseSensitive:
      return boost::algorithm::ends_with(value_str, query_str);

    default:
      throw WtException("Not yet implemented: WAbstractItemModel::match with "
			"MatchFlags = "
			+ boost::lexical_cast<std::string>(flags));
    }
  }
}

}

WModelIndexList WAbstractItemModel::match(const WModelIndex& start,
					  const boost::any& value,
					  int hits,
					  MatchFlags flags)
  const
{
  WModelIndexList result;

  const int rc = rowCount();

  for (int i = 0; i < rc; ++i) {
    int row = start.row() + i;

    if (row >= rc)
      if (!(flags & MatchWrap))
	break;
      else
	row -= rc;

    WModelIndex idx = index(row, start.column());
    boost::any v = data(idx);

    if (matchValue(v, value, flags))
      result.push_back(idx);
  }

  return result;
}

std::string asJSLiteral(const boost::any& v)
{
  if (v.empty())
    return std::string("''");
  else if (v.type() == typeid(WString))
    return boost::any_cast<WString>(v).jsStringLiteral();
  else if (v.type() == typeid(std::string))
    return
      WWebWidget::jsStringLiteral(boost::any_cast<std::string>(v), '\'');
  else if (v.type() == typeid(WDate)) {
    const WDate& d = boost::any_cast<WDate>(v);

    return "new Date('" + boost::lexical_cast<std::string>(d.month())
      + "/" + boost::lexical_cast<std::string>(d.day())
      + "/" + boost::lexical_cast<std::string>(d.year())
      + "')";
  }

#define ELSE_LEXICAL_ANY(TYPE) \
  else if (v.type() == typeid(TYPE)) \
    return boost::lexical_cast<std::string>(boost::any_cast<TYPE>(v))

  ELSE_LEXICAL_ANY(short);
  ELSE_LEXICAL_ANY(unsigned short);
  ELSE_LEXICAL_ANY(int);
  ELSE_LEXICAL_ANY(unsigned int);
  ELSE_LEXICAL_ANY(long);
  ELSE_LEXICAL_ANY(unsigned long);
  ELSE_LEXICAL_ANY(long long);
  ELSE_LEXICAL_ANY(unsigned long long);
  ELSE_LEXICAL_ANY(float);
  ELSE_LEXICAL_ANY(double);

#undef ELSE_LEXICAL_ANY

  else
    throw WtException(std::string("WAbstractItemModel: unsupported type ")
		      + v.type().name());
}

WString asString(const boost::any& v)
{
  if (v.empty())
    return WString();
  else if (v.type() == typeid(WString))
    return boost::any_cast<WString>(v);
  else if (v.type() == typeid(std::string))
    return WString(boost::any_cast<std::string>(v));
  else if (v.type() == typeid(WDate)) {
    const WDate& d = boost::any_cast<WDate>(v);

    return d.toString("dd/MM/yy");
  }

#define ELSE_LEXICAL_ANY(TYPE) \
  else if (v.type() == typeid(TYPE)) \
    return WString(boost::lexical_cast<std::string>(boost::any_cast<TYPE>(v)))

  ELSE_LEXICAL_ANY(short);
  ELSE_LEXICAL_ANY(unsigned short);
  ELSE_LEXICAL_ANY(int);
  ELSE_LEXICAL_ANY(unsigned int);
  ELSE_LEXICAL_ANY(long);
  ELSE_LEXICAL_ANY(unsigned long);
  ELSE_LEXICAL_ANY(long long);
  ELSE_LEXICAL_ANY(unsigned long long);
  ELSE_LEXICAL_ANY(float);
  ELSE_LEXICAL_ANY(double);

#undef ELSE_LEXICAL_ANY

  else
    throw WtException(std::string("WAbstractItemModel: unsupported type ")
		      + v.type().name());
}

double asNumber(const boost::any& v)
{
  if (v.empty())
    return std::numeric_limits<double>::signaling_NaN();
  else if (v.type() == typeid(WString))
    try {
      return boost::lexical_cast<double>(boost::any_cast<WString>(v).toUTF8());
    } catch (boost::bad_lexical_cast&) {
      return std::numeric_limits<double>::signaling_NaN();
    }
  else if (v.type() == typeid(std::string))
    try {
      return boost::lexical_cast<double>(boost::any_cast<std::string>(v));
    } catch (boost::bad_lexical_cast&) {
      return std::numeric_limits<double>::signaling_NaN();
    }
  else if (v.type() == typeid(WDate))
    return static_cast<double>(boost::any_cast<WDate>(v).modifiedJulianDay());

#define ELSE_NUMERICAL_ANY(TYPE) \
  else if (v.type() == typeid(TYPE)) \
    return static_cast<double>(boost::any_cast<TYPE>(v))

  ELSE_NUMERICAL_ANY(short);
  ELSE_NUMERICAL_ANY(unsigned short);
  ELSE_NUMERICAL_ANY(int);
  ELSE_NUMERICAL_ANY(unsigned int);
  ELSE_NUMERICAL_ANY(long);
  ELSE_NUMERICAL_ANY(unsigned long);
  ELSE_NUMERICAL_ANY(long long);
  ELSE_NUMERICAL_ANY(unsigned long long);
  ELSE_NUMERICAL_ANY(float);
  ELSE_NUMERICAL_ANY(double);

#undef ELSE_NUMERICAL_ANY

  else
    throw WtException(std::string("WAbstractItemModel: unsupported type ")
		      + v.type().name());
}

boost::any updateFromJS(const boost::any& v, std::string s)
{
  if (v.empty())
    return boost::any(s);
  else if (v.type() == typeid(WString))
    return boost::any(WString::fromUTF8(s));
  else if (v.type() == typeid(std::string))
    return boost::any(s);
  else if (v.type() == typeid(WDate))
    return boost::any(WDate::fromString(WString::fromUTF8(s),
					"ddd MMM dd yyyy"));
#define ELSE_LEXICAL_ANY(TYPE) \
  else if (v.type() == typeid(TYPE)) \
    return boost::any(boost::lexical_cast<TYPE>(s))

  ELSE_LEXICAL_ANY(short);
  ELSE_LEXICAL_ANY(unsigned short);
  ELSE_LEXICAL_ANY(int);
  ELSE_LEXICAL_ANY(unsigned int);
  ELSE_LEXICAL_ANY(long);
  ELSE_LEXICAL_ANY(unsigned long);
  ELSE_LEXICAL_ANY(long long);
  ELSE_LEXICAL_ANY(unsigned long long);
  ELSE_LEXICAL_ANY(float);
  ELSE_LEXICAL_ANY(double);

#undef ELSE_LEXICAL_ANY

  else
    throw WtException(std::string("WAbstractItemModel: unsupported type ")
		      + v.type().name());
}

}
