/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */

#include <boost/lexical_cast.hpp>

#include "Wt/WApplication"
#include "Wt/WLogger"
#include "Wt/WMenu"
#include "Wt/WMenuItem"
#include "Wt/WStackedWidget"
#include "Wt/WTable"
#include "Wt/WTableCell"
#include "Wt/WText"

#include "WtException.h"

using std::find;

namespace Wt {

WMenu::WMenu(WStackedWidget *contentsStack, Orientation orientation,
	     WContainerWidget *parent)
  : WCompositeWidget(parent),
    contentsStack_(contentsStack),
    orientation_(orientation),
    current_(-1)
{
  setRenderAsList(false);
}

void WMenu::setRenderAsList(bool enable)
{
  if (enable) {
    WContainerWidget *c = new WContainerWidget();
    c->setList(true);

    setImplementation(impl_ = c);
  } else {
    setImplementation(impl_ = new WTable());
  }

  renderAsList_ = enable;
}

WMenu::~WMenu()
{
  for (unsigned i = 0; i < items_.size(); ++i)
    delete items_[i];
}

void WMenu::enableBrowserHistory(const std::string& id)
{
  historyScope_ = id;

  wApp->stateChanged.connect(SLOT(this, WMenu::appStateChanged));
}

WMenuItem *WMenu::addItem(const WString& name, WWidget *contents,
			  WMenuItem::LoadPolicy policy)
{
  return addItem(new WMenuItem(name, contents, policy));
}

WMenuItem *WMenu::addItem(WMenuItem *item)
{
  item->setMenu(this);
  items_.push_back(item);

  if (renderAsList_) {
    WContainerWidget *p = dynamic_cast<WContainerWidget *>(impl_);
    WContainerWidget *li = new WContainerWidget(p);
    li->addWidget(item->itemWidget());

    if (orientation_ == Horizontal)
      li->setInline(true);
  } else {
    WTable *layout = dynamic_cast<WTable *>(impl_);
    WTableCell *parent
      = layout->elementAt((orientation_ == Vertical) ? items_.size() : 0, 0);

    parent->addWidget(item->itemWidget());

    // separate horizontal items so wrapping will occur inbetween items.
    if (orientation_ == Horizontal)
      new WText(" ", parent);
  }

  for (unsigned i = 0; i < items_.size(); ++i)
    item->resetLearnedSlots();

  contentsStack_->addWidget(item->contents());

  if (contentsStack_->count() != (int)items_.size())
    throw WtException
      ("WMenu: inconsistency between "
       "the contents stack contents and menu items. "
       "Contents stack size: "
       + boost::lexical_cast<std::string>(contentsStack_->count())
       + ", menu items count: "
       + boost::lexical_cast<std::string>(items_.size()));

  if (items_.size() == 1)
    select(0);
  else
    item->renderSelected(false);

  return item;
}

void WMenu::select(int index)
{
  selectVisual(index);

  items_[index]->loadContents();

  itemSelected.emit(items_[current_]);
}

void WMenu::selectVisual(int index)
{
  previousCurrent_ = current_;
  current_ = index;
  contentsStack_->setCurrentIndex(current_);

  for (unsigned i = 0; i < items_.size(); ++i)
    items_[i]->renderSelected((int)i == current_);

  if (!historyScope_.empty())
    wApp->setState(historyScope_, boost::lexical_cast<std::string>(current_));

  itemSelectRendered.emit(items_[current_]);
}

void WMenu::appStateChanged(const std::string scope, const std::string value)
{
  if (scope == historyScope_)
    setFromState(value);
}

void WMenu::setFromState(const std::string& value)
{
  try {
    int i = value.empty() ? 0 : boost::lexical_cast<int>(value);

    if (i >= 0 && i < static_cast<int>(items_.size()))
      if (i != current_)
	select(i);
  } catch (std::exception& e) {
    wApp->log("error") << "WMenu::setFromState() could not handle: '"
		       << value << "': " << e.what();
  } catch (...) {
    wApp->log("error") << "WMenu::setFromState() error";
  } 
}

void WMenu::select(WMenuItem *item)
{
  select(indexOf(item));
}

void WMenu::selectVisual(WMenuItem *item)
{
  selectVisual(indexOf(item));
}

int WMenu::indexOf(WMenuItem *item)
{
  return find(items_.begin(), items_.end(), item) - items_.begin();
}

void WMenu::undoSelectVisual()
{
  selectVisual(previousCurrent_);
}

WMenuItem *WMenu::currentItem() const
{
  return current_ >= 0 ? items_[current_] : 0;
}

}
