/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */

#include <iostream>

#include "Wt/WApplication"
#include "Wt/WLogger"
#include "Wt/WStandardItemModel"

namespace {
  void warnInvalid(int row, int column) {
    wApp->log("warn") << "WStandardItemModel: (" << row << ","
		      << column << ") is not a valid index.";
  }

  void warnInvalidColumn(int column) {
    wApp->log("warn") << "WStandardItemModel: " << column
		      << " is not a valid column.";
  }
}

namespace Wt {

WStandardItemModel::WStandardItemModel(int rows, int columns, WObject *parent)
  : WAbstractItemModel(parent)
{ 
  insertRows(0, rows);
  insertColumns(0, columns);
}

int WStandardItemModel::columnCount() const
{
  return columnHeaderData_.size();
}

int WStandardItemModel::rowCount() const
{
  return data_.size();
}

boost::any WStandardItemModel::data(const WModelIndex& index) const
{
  if (isValidIndex(index))    
    return data_[index.row()][index.column()];
  else {
    warnInvalid(index.row(), index.column());
    return boost::any();
  }
}

boost::any WStandardItemModel::headerData(int column) const
{
  if (column < columnCount())
    return columnHeaderData_[column];
  else {
    warnInvalidColumn(column);
    return boost::any();
  }
}

WModelIndex WStandardItemModel::index(int row, int column) const
{
  if (row < rowCount()
      && column < columnCount())
    return createIndex(row, column);
  else {
    warnInvalid(row, column);
    return WModelIndex();
  }
}

bool WStandardItemModel::insertColumns(int column, int count)
{
  if (column <= columnCount()) {
    beginInsertColumns(column, column + count - 1);

    columnHeaderData_
      .insert(columnHeaderData_.begin() + column, count, boost::any());

    for (unsigned i = 0; i < data_.size(); ++i)
      data_[i].insert(data_[i].begin() + column, count, boost::any());

    endInsertColumns();
 
    return true;
  } else
    return false;
}

bool WStandardItemModel::insertRows(int row, int count)
{
  if (row <= rowCount()) {
    beginInsertRows(row, row + count - 1);

    data_.insert(data_.begin() + row, count,
		 std::vector<boost::any>(columnCount(), boost::any()));

    endInsertRows();

    return true;
  } else
    return false;
}

void WStandardItemModel::debugPrint()
{
  for (int i = 0; i < rowCount(); ++i) {
    for (int j = 0; j < columnCount(); ++j) {
      if (j != 0)
	std::cerr << ",";
      std::cerr << asString(data(index(i, j)));
    }
    std::cerr << std::endl;
  }
}

bool WStandardItemModel::removeColumns(int column, int count)
{
  if (column + count <= columnCount()) {
    beginRemoveColumns(column, column + count - 1);

    columnHeaderData_.erase(columnHeaderData_.begin() + column,
			    columnHeaderData_.begin() + column + count);
    
    for (unsigned i = 0; i < data_.size(); ++i)
      data_[i].erase(data_[i].begin() + column,
		     columnHeaderData_.begin() + column + count);

    endRemoveColumns();

    return true;
  } else
    return false;
}

bool WStandardItemModel::removeRows(int row, int count)
{
   if (row + count <= rowCount()) {
    beginRemoveRows(row, row + count - 1);

    data_.erase(data_.begin() + row, data_.begin() + row + count);

    endRemoveRows();

    return true;
  } else
    return false; 
}

bool WStandardItemModel::setData(const WModelIndex& index,
			       const boost::any& value)
{
  if (isValidIndex(index)) {
    data_[index.row()][index.column()] = value;

    dataChanged.emit(index, index);

    return true;
  } else {
    warnInvalid(index.row(), index.column());
    return false;
  }
}

bool WStandardItemModel::setHeaderData(int column, const boost::any& value)
{
  if (column < columnCount()) {
    columnHeaderData_[column] = value;

    headerDataChanged.emit(column, column);

    return true;
  } else {
    warnInvalidColumn(column);
    return false;
  }
}

bool WStandardItemModel::isValidIndex(const WModelIndex& index) const
{
  return
    index.isValid()
    && (index.row() < rowCount())
    && (index.column() < columnCount());
}

}
