/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * See the LICENSE file for terms of use.
 */

#include <algorithm>

#include "Wt/WRectF"
#include "Wt/WPointF"

namespace Wt {

WRectF::WRectF()
  : x_(0), y_(0), width_(0), height_(0)
{ }

WRectF::WRectF(double x, double y, double width, double height)
  : x_(x), y_(y), width_(width), height_(height)
{ }

WRectF::WRectF(const WPointF& topLeft, const WPointF& topRight)
  : x_(topLeft.x()),
    y_(topLeft.y()),
    width_(topRight.x() - topLeft.x()),
    height_(topRight.y() - topLeft.y())
{ }

bool WRectF::operator==(const WRectF& rhs) const
{
  return (x_ == rhs.x_ && y_ == rhs.y_
	  && width_ == rhs.width_ && height_ == rhs.height_);
}

bool WRectF::operator!=(const WRectF& rhs) const
{
  return !(*this == rhs);
}

bool WRectF::isNull() const
{
  return x_ == 0 && y_ == 0 && width_ == 0 && height_ == 0;
}

void WRectF::setX(double x)
{
  width_ += (x_ - x);
  x_ = x;
}

void WRectF::setY(double y)
{
  height_ += (y_ - y);
  y_ = y;
}

WPointF WRectF::center() const
{
  return WPointF(x_ + width_/2, y_ + height_/2);
}

WPointF WRectF::topLeft() const
{
  return WPointF(x_, y_);
}

WPointF WRectF::topRight() const
{
  return WPointF(x_ + width_, y_);
}

WPointF WRectF::bottomLeft() const
{
  return WPointF(x_, y_ + height_);
}

WPointF WRectF::bottomRight() const
{
  return WPointF(x_ + width_, y_ + height_);
}

bool WRectF::intersects(const WRectF& other) const
{
  if (isNull() || other.isNull())
    return false;
  else {
    bool intersectX = (other.left() >= left() && other.left() <= right())
      || (other.right() >= left() && other.right() <= right());

    bool intersectY = (other.top() >= top() && other.top() <= bottom())
      || (other.bottom() >= top() && other.bottom() <= bottom());

    return intersectX && intersectY;
  }
}

WRectF WRectF::united(const WRectF& other) const
{
  if (isNull())
    return other;
  else if (other.isNull())
    return *this;
  else {
    double l = std::min(left(), other.left());
    double r = std::max(right(), other.right());
    double t = std::min(top(), other.top());
    double b = std::max(bottom(), other.bottom());

    return WRectF(l, t, r - l, b - t);
  }
}

}
