/*
 * Copyright (C) 2008 Emweb bvba, Kessel-Lo, Belgium.
 *
 * All rights reserved.
 */

#include <boost/lexical_cast.hpp>
#include <boost/filesystem/operations.hpp>

#include "Request.h"
#include "StaticReply.h"
#include "StockReply.h"
#include "MimeTypes.h"

namespace http {
namespace server {

StaticReply::StaticReply(const std::string &full_path,
			 const std::string &extension,
			 const Request& request,
			 const std::string &err_root)
  : Reply(request),
    path_(full_path),
    extension_(extension)
{
  bool stockReply = false;
  bool gzipReply = false;

  if (request.acceptGzipEncoding()) {
    std::string gzipPath = path_ + ".gz";
    stream_.open(gzipPath.c_str(), std::ios::in | std::ios::binary);

    if (stream_) {
      path_ = gzipPath;
      gzipReply = true;
    } else {
      stream_.clear();
      stream_.open(path_.c_str(), std::ios::in | std::ios::binary);
    }
  } else
    stream_.open(path_.c_str(), std::ios::in | std::ios::binary);

  if (!stream_) {
    stockReply = true;
    setRelay(ReplyPtr(new StockReply(request, StockReply::not_found,
				     "", err_root)));
  }

  if (!stockReply) {
    /*
     * Check if can send a 304 not modified reply.
     */
    Request::HeaderMap::const_iterator i
      = request.headerMap.find("If-Modified-Since");

    if (i != request.headerMap.end()) {
      try {
	time_t t = boost::filesystem::last_write_time(path_);
	struct tm td;
	gmtime_r(&t, &td);
	char buffer[100];
	strftime(buffer, 100, "%a, %d %b %Y %H:%M:%S GMT", &td);

	if (i->second == buffer) {
	  stockReply = true;
	  setRelay(ReplyPtr(new StockReply(request, StockReply::not_modified)));
	}
      } catch (...) {
      }
    }
  }

  if (!stockReply) {
    try {
      fileSize_ = boost::filesystem::file_size(path_);

      /*
       * Add a Last-Modified header
       */
      time_t t = boost::filesystem::last_write_time(path_);
      struct tm td;
      gmtime_r(&t, &td);
      char buffer[100];
      strftime(buffer, 100, "%a, %d %b %Y %H:%M:%S GMT", &td);

      addHeader("Last-Modified", buffer);
    } catch (...) {
      stockReply = true;
      setRelay(ReplyPtr(new StockReply(request, StockReply::not_found,
				       "", err_root)));
    }
  }

  if (gzipReply)
    addHeader("Content-Encoding", "gzip");
}

void StaticReply::consumeRequestBody(Buffer::const_iterator begin,
				     Buffer::const_iterator end,
				     bool endOfRequest)
{
  if (endOfRequest)
    transmitMore();
}

Reply::status_type StaticReply::responseStatus()
{
  return ok;
}

std::string StaticReply::contentType()
{
  return mime_types::extensionToType(extension_);
}

boost::intmax_t StaticReply::contentLength()
{
  return fileSize_;
}

asio::const_buffer StaticReply::nextContentBuffer()
{
  if (request_.method == "HEAD")
    return emptyBuffer;
  else {
    stream_.read(buf_, sizeof(buf_));

    if (stream_.gcount() > 0) {
      return asio::buffer(buf_, stream_.gcount());
    } else
      return emptyBuffer;
  }
}

}
}
