#!/usr/bin/env python

################################################################################
# Copyright (c) 2005 Georg Neis <gn@oglaroon.de>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
################################################################################


"""Print wmii's key bindings.

This program prints the key bindings of the window manager wmii to
stdout.  wmii has to be running.  If no keymap (aka mode) is specified
then the currently active one is used.  The format of the output is
quite simple:

    <header>
    <empty line>
    <key combination> -> <associated command>
    <key combination> -> <associated command>
    <key combination> -> <associated command>
    [...]

The ixplib module (shipped with wmii) is used to access the pseudo
filesystem of wmifs.  If ixplib is not installed printkeys will fall
back to the wmir tool.  For usage information type 'printkeys.py -h'.

Python 2.3 or later required.
"""


from optparse import OptionParser
import os
import sys

# check if ixplib is available:
try:
    import ixplib
    ixplib_avail = True
except:
    ixplib_avail = False


MODEDIR = '/keys/mode/'
LOOKUPFILE = '/keys/lookup'
WMIRREAD = 'wmir -s %s read '


def abort(reason, exitcode):
    program = os.path.basename(sys.argv[0])
    print >>sys.stderr, '%s: error: %s' % (program, reason)
    sys.exit(exitcode)

def ixp_list_modes(ixp, base):
    """Print available key modes (uses ixplib)."""
    fd = ixp.open(base)
    for line in ixp.read(fd).splitlines():
        print line[:-1]
    ixp.close(fd)

def wmir_list_modes(socket, base):
    """Print available key modes (uses wmir)."""
    for line in os.popen(WMIRREAD % socket + base).read().splitlines()[:-1]:
        print line[:-1]

def ixp_list_bindings(ixp, dir, prefixlength):
    """Print key bindings defined in 'dir' (uses ixplib)."""
    fd = ixp.open(dir)
    content = ixp.read(fd).splitlines()
    ixp.close(fd)
    for entry in content:
        if entry.endswith('/'):
            # entry is a directory, step deeper
            ixp_list_bindings(ixp, dir + entry, prefixlength)
        else:
            # entry is a file, get key binding
            print dir[prefixlength:] + entry, '->',  # key
            fd = ixp.open(dir + entry)
            cmd = ixp.read(fd)
            ixp.close(fd)
            print cmd

def wmir_list_bindings(socket, dir, prefixlength):
    """Print key bindings defined in 'dir' (uses wmir)."""
    for entry in os.popen(WMIRREAD % socket + dir):
        if entry == '\n':
            continue
        elif entry.endswith('/\n'):
            # entry is a directory, step deeper
            wmir_list_bindings(socket, dir + entry[:-1], prefixlength)
        else:
            # entry is a file, print key binding
            print dir[prefixlength:] + entry[:-1], '->',  # key
            print os.popen(WMIRREAD % socket + dir + entry[:-1]).read(),  # cmd


if __name__ == '__main__':
    cli = OptionParser(usage='%prog [options]\nprint wmii\'s key bindings',
                       version='%prog 0.5')
    cli.add_option('-l', '--list-modes',
                   action='store_true', default=False,
                   help='list available key modes')
    cli.add_option('-m', '--mode',
                   action='store', default=None,
                   help='print key bindings of this mode '
                        '[default: current mode]')
    cli.add_option('-s', '--socket',
                   action='store', default=os.getenv('WMIR_SOCKET'),
                   help='socket file of wmifs '
                        '[default: WMIR_SOCKET environment variable]')
    options, args = cli.parse_args()

    program = os.path.basename(sys.argv[0])

    if options.socket is None:
        cli.print_usage(sys.stderr)
        abort('no socket file specified', 1)
    if options.mode is None:
        # no mode specified, so get current one
        if ixplib_avail:
            try:
                ixp = ixplib.Client(options.socket)
            except ixplib.ClientError, e:
                abort(e, 2)
            fd = ixp.open(LOOKUPFILE)
            options.mode = ixp.read(fd)[(len(MODEDIR) - MODEDIR.index('/', 1)):]
            ixp.close(fd)
            del ixp
        else:
            options.mode = os.popen(WMIRREAD % options.socket +
                                    LOOKUPFILE).read().rstrip()
            options.mode = options.mode[(len(MODEDIR) - MODEDIR.index('/', 1)):]
    if options.list_modes:
        if ixplib_avail:
            try:
                ixp = ixplib.Client(options.socket)
            except ixplib.ClientError, e:
                abort(e, 2)
            ixp_list_modes(ixp, MODEDIR)
        else:
            wmir_list_modes(options.socket, MODEDIR)
    else:  # print key bindings
        print 'KEY BINDINGS OF THE %s MODE:\n' % options.mode
        dir = MODEDIR + options.mode + '/'
        if ixplib_avail:
            try:
                ixp = ixplib.Client(options.socket)
            except ixplib.ClientError, e:
                abort(e, 2)
            ixp_list_bindings(ixp, dir, len(dir))
        else:
            wmir_list_bindings(options.socket, dir, len(dir))
