################################################################################
# Copyright (c) 2004-2005 Christoph Wegscheider <cw@wegi.net>                  #
#                                                                              #
# Permission is hereby granted, free of charge, to any person obtaining a copy #
# of this software and associated documentation files (the "Software"), to     #
# deal in the Software without restriction, including without limitation the   #
# rights to use, copy, modify, merge, publish, distribute, sublicense, and/or  #
# sell copies of the Software, and to permit persons to whom the Software is   #
# furnished to do so, subject to the following conditions:                     #
#                                                                              #
# The above copyright notice and this permission notice shall be included in   #
# all copies or substantial portions of the Software.                          #
#                                                                              #
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR   #
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,     #
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE  #
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER       #
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      #
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS #
# IN THE SOFTWARE.                                                             #
################################################################################

# ToDo
# + IXPServer
#     + implement File class
#     + implement callbacks 
# + some time in the future: rewrite the whole thing with native python C/API
#   to get rid of the pyrex build-dependency
#

import types

ctypedef unsigned long size_t

cdef extern from "stdio.h":
	int printf(char *format,...)

cdef extern from "string.h":
	char *strncpy(char *, char *, int)
	int strlen(char *)

cdef extern from "Python.h":
	object PyString_FromStringAndSize(char *, int)
	int PyString_Size(PyObject)
	long PyLong_AsLong(PyObject)

cdef extern from "ixp.h":
	ctypedef enum IXPRunlevel:
		HALT
		SHUTDOWN
		RUNNING

# IXPClient
	ctypedef struct IXPClient:
		int fd
		char *errstr
		void (*create)(IXPClient *, char *path)
		void (*remove)(IXPClient *, char *path)
		int (*open)(IXPClient *, char *path)
		void (*close)(IXPClient *, int fd)
		size_t (*read)(IXPClient *, int fd, char *out_buf, size_t out_buf_len)
		void (*write)(IXPClient *, int fd, char *content, size_t in_len)

	IXPClient *init_client(char *sockfile)
	void deinit_client(IXPClient *c)


# IXPServer
	ctypedef struct IXPServer:
		IXPRunlevel runlevel
	
	IXPServer *init_server(char *sockfile, void (*cleanup)())
	void run_server(IXPServer *s)
	void deinit_server(IXPServer *s)


# File
	ctypedef struct File:
		char *name
		void *content
		size_t size
		int lock
		int bind
		File *parent
		File *next
		void (*after_write)(File *)
		void (*before_read)(File *)
		void (*before_remove)(File *)

	File *ixp_walk(IXPServer *s, char *path)
	File *ixp_create(IXPServer *s, char *path)
	File *ixp_open(IXPServer *s, char *path)
	size_t ixp_read(IXPServer *s, int fd, size_t offset, void *out_buf, size_t out_buf_len)
	void ixp_write(IXPServer *s, int fd, size_t offset, void *content, size_t in_len)
	void ixp_close(IXPServer *s, int fd)
	void ixp_remove(IXPServer *s, char *path)



############### Client ###############

class ClientError(Exception):

	def __init__(self, value):
		self.value = value

	def __str__(self):
		return self.value


cdef class Client:
	cdef IXPClient *client

	def __new__(self, sockfile):
		self.client = init_client(sockfile)
		if self.client == NULL:
			raise ClientError("can't connect to sockfile: " + sockfile)

	def __dealloc__(self):
		if self.client != NULL:
			deinit_client(self.client)

	def open(self, path):
		"""opens a file"""

		fd = self.client.open(self.client, path)
		if self.client.errstr:
			raise ClientError("open: " + self.client.errstr)

		return fd

	def close(self, fd):
		"""closes a file"""

		self.client.close(self.client, fd)
		if self.client.errstr:
			raise ClientError("close: " + self.client.errstr)

	def read(self, fd, length=2147483647):
		"""read a file"""
		cdef size_t out_len
		cdef size_t size
		cdef size_t diff
		cdef char buf[512]

		diff = out_len = size = PyLong_AsLong(length)
		content = ''

		while out_len == size and diff > 0:
			if  512 < diff:
				size = 512
			else:
				size = diff

			out_len = self.client.read(self.client, fd, buf, size)
			if self.client.errstr:
				raise ClientError("read: " + self.client.errstr)
			diff = diff - out_len
			content = content + PyString_FromStringAndSize(buf, out_len)

		return content

	def write(self, fd, content=""):
		"""write a file"""

		self.client.write(self.client, fd, content, PyString_Size(content))
		if self.client.errstr:
			raise ClientError("write: " + self.client.errstr)

	def remove(self, path):
		"""removes a file"""

		self.client.remove(self.client, path)
		if self.client.errstr:
			raise ClientError("remove: " + self.client.errstr)

	def create(self, path):
		"""creates a file"""

		self.client.create(self.client, path)
		if self.client.errstr:
			raise ClientError("create: " + self.client.errstr)



############### Server ###############

class ServerError(Exception):

	def __init__(self, value):
		self.value = value

	def __str__(self):
		return repr(self.value)


cdef class Server:
	cdef IXPServer *server

	def __new__(self, sockfile):
		self.server = init_server(sockfile, NULL)
		if self.server == NULL:
			raise ServerError("can't use sockfile: " + sockfile)

	def __dealloc__(self):
		if self.server != NULL:
			deinit_server(self.server)
	
	def start(self):
		run_server(self.server)

	def stop(self):
		self.server.runlevel = SHUTDOWN

	def addfile(self, path):
		ixp_create(self.server, path)


