/*
 * (C)opyright MMIV-MMV Anselm R. Garbe <garbeam at gmail dot com>
 * See LICENSE file for license details.
 */

#include <stdio.h>
#include <string.h>
#include "blitz.h"

static XFontStruct *
get_font(Display * dpy, char *fontstr)
{
    XFontStruct *font;
    font = XLoadQueryFont(dpy, fontstr);
    if(!font) {
        font = XLoadQueryFont(dpy, "fixed");
        if(!font) {
            fprintf(stderr, "%s", "wmii: error, cannot load fixed font\n");
            return 0;
        }
    }
    return font;
}

static unsigned long
load_color(Display * dpy, int mon, char *colstr)
{

    XColor color;
    char col[8];

    STRLCPY(col, colstr, sizeof(col));
    col[7] = '\0';
    XAllocNamedColor(dpy, DefaultColormap(dpy, mon), col, &color, &color);
    return color.pixel;
}

static void
draw_bg(Display * dpy, Draw * d, unsigned long bg)
{
    XRectangle rect[4];
    XSetForeground(dpy, d->gc, bg);
    if(!d->notch) {
        XFillRectangles(dpy, d->drawable, d->gc, &d->rect, 1);
        return;
    }
    rect[0] = d->rect;
    rect[0].height = d->notch->y;
    rect[1] = d->rect;
    rect[1].y = d->notch->y;
    rect[1].width = d->notch->x;
    rect[1].height = d->notch->height;
    rect[2].x = d->notch->x + d->notch->width;
    rect[2].y = d->notch->y;
    rect[2].width = d->rect.width - (d->notch->x + d->notch->width);
    rect[2].height = d->notch->height;
    rect[3] = d->rect;
    rect[3].y = d->notch->y + d->notch->height;
    rect[3].height = d->rect.height - (d->notch->y + d->notch->height);
    XFillRectangles(dpy, d->drawable, d->gc, rect, 4);
}

static void
_draw_border(Display * dpy, Draw * d)
{
    unsigned long color[4];
    int i;

    for(i = 0; i < 4; i++)
        color[i] = load_color(dpy, d->screen_num, d->border[i].colstr);
    XSetLineAttributes(dpy, d->gc, 1, LineSolid, CapButt, JoinMiter);
    XSetForeground(dpy, d->gc, color[0]);
    XDrawLine(dpy, d->drawable, d->gc, d->rect.x, d->rect.y,
              d->rect.x + d->rect.width - 1, d->rect.y);
    XSetForeground(dpy, d->gc, color[1]);
    XDrawLine(dpy, d->drawable, d->gc, d->rect.x + d->rect.width - 1,
              d->rect.y, d->rect.x + d->rect.width - 1,
              d->rect.y + d->rect.height);
    XSetForeground(dpy, d->gc, color[2]);
    XDrawLine(dpy, d->drawable, d->gc, d->rect.x,
              d->rect.y + d->rect.height - 1,
              d->rect.x + d->rect.width - 1,
              d->rect.y + d->rect.height - 1);
    XSetForeground(dpy, d->gc, color[3]);
    XDrawLine(dpy, d->drawable, d->gc, d->rect.x, d->rect.y,
              d->rect.x, d->rect.y + d->rect.height - 1);
    XFreeColors(dpy, DefaultColormap(dpy, d->screen_num), color, 4, 0);
}

static void
draw_text(Display * dpy, Draw * d, unsigned long bg, unsigned long fg)
{
    XFontStruct *font;
    int x, y, w, h, i, shortened = 0;
    size_t len = 0;
    static char text[2048];

    if(!d->text)
        return;
    len = strlen(d->text);
    STRLCPY(text, d->text, sizeof(text));
    if((font = get_font(dpy, d->fnt.font)) == 0)
        return;

    XSetFont(dpy, d->gc, font->fid);
    h = font->ascent + font->descent;
    y = d->rect.y + d->rect.height / 2 - h / 2 + font->ascent;
    w = XTextWidth(font, text, len);

    /* shorten text if necessary */
    for(i = 1; (len - (i * 3) > 0) && (w > d->rect.width); i++) {
        len -= i * 3;
        text[len - 1] = '\0';
        w = XTextWidth(font, text, len);
        shortened = 1;
    }
    if(w > d->rect.width)
        return;
    if(shortened && (len > 3)) {
        text[len - 4] = '.';
        text[len - 3] = '.';
        text[len - 2] = '.';
    }

    /* calculate x offset */
    switch (d->fnt.align) {
    case WEST:
        x = d->rect.x + h / 2;
        break;
    case EAST:
        x = d->rect.x + d->rect.width - (h / 2 + w);
        break;
    default:                   /* CENTER */
        x = d->rect.x + (d->rect.width - w) / 2;
        break;
    }

    XSetBackground(dpy, d->gc, bg);
    XSetForeground(dpy, d->gc, fg);
    XDrawString(dpy, d->drawable, d->gc, x, y, text, strlen(text));
    XFreeFont(dpy, font);
}

/* draws meter */
void
draw_meter(Display * dpy, Draw * d)
{
    int w = d->text ? d->rect.height / 2 : d->rect.width - 4;
    int offy, mh;
    unsigned long color[4];
    int level = 2;

    color[0] = load_color(dpy, d->screen_num, d->bg.colstr);
    color[1] = load_color(dpy, d->screen_num, d->fill.colstr);
    draw_bg(dpy, d, color[0]);
    _draw_border(dpy, d);

    /* draw bg gradient */
    mh = ((d->rect.height - 4) * d->val) / 100;
    offy = d->rect.y + d->rect.height - 2 - mh;
    XSetForeground(dpy, d->gc, color[1]);
    XFillRectangle(dpy, d->drawable, d->gc, d->rect.x + 2, offy, w, mh);

    if(d->text) {
        color[2] = load_color(dpy, d->screen_num, d->bg.colstr);
        color[3] = load_color(dpy, d->screen_num, d->fg.colstr);
        level = 4;
        d->rect.x += w + 4;
        d->rect.width -= (w + 4);
        draw_text(dpy, d, color[2], color[3]);
        /* restore */
        d->rect.x -= w + 4;
        d->rect.width += (w + 4);
    }
    XFreeColors(dpy, DefaultColormap(dpy, d->screen_num), color, level, 0);
}

static void
_draw_label(Display * dpy, Draw * d)
{
    unsigned long color[2];
    int level = 1;
    color[0] = load_color(dpy, d->screen_num, d->bg.colstr);
    draw_bg(dpy, d, color[0]);
    if(d->text) {
        color[1] = load_color(dpy, d->screen_num, d->fg.colstr);
        level = 2;
        draw_text(dpy, d, color[0], color[1]);
    }
    XFreeColors(dpy, DefaultColormap(dpy, d->screen_num), color, level, 0);
}

/* draws label */
void
draw_label(Display * dpy, Draw * d)
{
    _draw_label(dpy, d);
    _draw_border(dpy, d);
}

void
draw_label_noborder(Display * dpy, Draw * d)
{
    _draw_label(dpy, d);
}

int
text_width(Display * dpy, char *fontstr, double size,
           unsigned char *utf8txt)
{
    XFontStruct *font;
    int result;

    if((font = get_font(dpy, fontstr)) == 0)
        return 0;
    if(!utf8txt)
        return 0;

    result = XTextWidth(font, (char *) utf8txt, strlen((char *) utf8txt));
    XFreeFont(dpy, font);
    return result;
}
