/*
 * (C)opyright MMIV-MMV Anselm R. Garbe <garbeam at gmail dot com>
 * See LICENSE file for license details.
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <sys/wait.h>

#include "blitz.h"

void
str_to_align(Align * result, char *val)
{
    *result = CENTER;
    if(!strncmp(val, "west", 4))
        *result = WEST;
    else if(!strncmp(val, "nwest", 5))
        *result = NWEST;
    else if(!strncmp(val, "north", 5))
        *result = NORTH;
    else if(!strncmp(val, "neast", 5))
        *result = NEAST;
    else if(!strncmp(val, "east", 4))
        *result = EAST;
    else if(!strncmp(val, "seast", 5))
        *result = SEAST;
    else if(!strncmp(val, "south", 5))
        *result = SOUTH;
    else if(!strncmp(val, "swest", 5))
        *result = SWEST;
    else if(!strncmp(val, "center", 6))
        *result = CENTER;
    /* note, resize allows syntax like "east-20", this we
     * cannot do include zero termination in strncmp checking! */
}

/* free the result manually! */
char *
button_to_str(unsigned int button)
{
    char result[8];
    result[0] = '\0';

    switch (button) {
    case Button1:
        STRLCPY(result, "Button1", sizeof(result));
        break;
    case Button2:
        STRLCPY(result, "Button2", sizeof(result));
        break;
    case Button3:
        STRLCPY(result, "Button3", sizeof(result));
        break;
    case Button4:
        STRLCPY(result, "Button4", sizeof(result));
        break;
    case Button5:
        STRLCPY(result, "Button5", sizeof(result));
        break;
    default:
        break;
    }

    return strdup(result);
}

unsigned int
str_to_button(char *val)
{
    if(!strncmp(val, "Button1", 8))
        return Button1;
    else if(!strncmp(val, "Button2", 8))
        return Button2;
    else if(!strncmp(val, "Button3", 8))
        return Button3;
    else if(!strncmp(val, "Button4", 8))
        return Button4;
    else if(!strncmp(val, "Button5", 8))
        return Button5;
    return 0;
}

/* free the result manually! */
char *
mod_to_str(unsigned long mod)
{
    char result[60];
    result[0] = '\0';

    if(mod & ShiftMask)
        STRLCAT(result, "shift+", sizeof(result));
    if(mod & LockMask)
        STRLCAT(result, "lock+", sizeof(result));
    if(mod & ControlMask)
        STRLCAT(result, "ctrl+", sizeof(result));
    if(mod & Mod1Mask)
        STRLCAT(result, "mod1+", sizeof(result));
    if(mod & Mod2Mask)
        STRLCAT(result, "mod2+", sizeof(result));
    if(mod & Mod3Mask)
        STRLCAT(result, "mod3+", sizeof(result));
    if(mod & Mod4Mask)
        STRLCAT(result, "mod4+", sizeof(result));
    if(mod & Mod5Mask)
        STRLCAT(result, "mod5+", sizeof(result));
    return strdup(result);
}

void
str_to_color4(Color c[4], char *val)
{
    int colnum = count_toks(val, ' ');
    char **cols = malloc(sizeof(char *) * colnum);
    tokenize(cols, val, ' ');

    switch (colnum) {
    case 4:
        str_to_color(&c[3], cols[0]);
        str_to_color(&c[0], cols[1]);
        str_to_color(&c[2], cols[2]);
        str_to_color(&c[1], cols[3]);
        break;
    case 2:
        str_to_color(&c[0], cols[0]);
        str_to_color(&c[3], cols[0]);
        str_to_color(&c[1], cols[1]);
        str_to_color(&c[2], cols[1]);
        break;
    case 1:
        str_to_color(&c[0], cols[0]);
        str_to_color(&c[1], cols[0]);
        str_to_color(&c[2], cols[0]);
        str_to_color(&c[3], cols[0]);
        break;
    default:
        fprintf(stderr, "blitz: invalid color tupel '%s'\n", val);
        break;
    }
    free_toks(cols, colnum);
}

/* assumes string of type #RRGGBBAA */
void
str_to_color(Color * c, char *val)
{
    int len = strlen(val);
    static double fact_ff = 1.0 / 255.0;
    char buf[8];
    int i, j;

    if(!val || val[0] != '#' || (len != 7 && len != 4)) {
        fprintf(stderr,
                "liblitz: cannot convert bogus color value '%s'\n", val);
        return;
    }

    if(len == 7)
        STRLCPY(buf, val, sizeof(buf));
    else {
        /* #RGB format */
        buf[0] = val[0];        /* # */
        j = 1;
        for(i = 1; i < 4; i++) {
            buf[j] = buf[j + 1] = val[i];
            j += 2;
        }
        buf[7] = '\0';
    }
    STRLCPY(c->colstr, buf, sizeof(buf));
    c->b = fact_ff * (double) strtol(&buf[5], 0, 16);
    buf[5] = '\0';
    c->g = fact_ff * (double) strtol(&buf[3], 0, 16);
    buf[3] = '\0';
    c->r = fact_ff * (double) strtol(&buf[1], 0, 16);
}

/**
 * Basic Syntax: <x>,<y>,<width>,<height>
 * Each component can be of following format:
 * <...> = [+|-]0..n|<alignment>[[+|-]0..n]
 */
void
str_to_rect(Display * dpy, XRectangle * root, XRectangle * r, char *val)
{
    char buf[64];
    char *x, *y, *w, *h;
    char *p;
    int sx, sy, sw, sh;

    if(!val) {
        return;
    }
    sx = sy = sw = sh = 0;
    x = y = w = h = 0;
    STRLCPY(buf, val, sizeof(buf));

    x = strtok_r(buf, ",", &p);
    if(x) {
        y = strtok_r(0, ",", &p);
        if(y) {
            w = strtok_r(0, ",", &p);
            if(w) {
                h = strtok_r(0, "", &p);
            }
        }
    }

    if(x && (sx = (x[0] >= '0') && (x[0] <= '9')))
        r->x = atoi(x);
    if(y && (sy = (y[0] >= '0') && (y[0] <= '9')))
        r->y = atoi(y);
    if(w && (sw = (w[0] >= '0') && (w[0] <= '9')))
        r->width = atoi(w);
    if(h && (sh = (h[0] >= '0') && (h[0] <= '9')))
        r->height = atoi(h);

    if(!sx && !sw && x && w
       && x[0] != '-' && x[0] != '+' && w[0] != '-' && w[0] != '+') {
        Align ax, aw;
        str_to_align(&ax, x);
        str_to_align(&aw, w);
        if((ax == CENTER) && (aw == EAST)) {
            r->x = root->x + root->width / 2;
            r->width = root->width / 2;
        } else {
            r->x = root->x;
            if(aw == CENTER) {
                r->width = root->width / 2;
            } else {
                r->width = root->width;
            }
        }
    } else if(!sx && x && x[0] != '-' && x[0] != '+') {
        Align ax;
        str_to_align(&ax, x);
        if(ax == CENTER) {
            r->x = root->x + (root->width / 2) - (r->width / 2);
        } else if(ax == EAST) {
            r->x = root->x + root->width - r->width;
        } else {
            r->x = root->x;
        }
    } else if(!sw && w && w[0] != '-' && w[0] != '+') {
        Align aw;
        str_to_align(&aw, w);
        if(aw == CENTER) {
            r->width = (root->width / 2) - r->x;
        } else {
            r->width = root->width - r->x;
        }
    }

    if(!sy && !sh && y && h
       && y[0] != '-' && y[0] != '+' && h[0] != '-' && h[0] != '+') {
        Align ay, ah;
        str_to_align(&ay, y);
        str_to_align(&ah, h);
        if((ay == CENTER) && (ah == SOUTH)) {
            r->y = root->y + root->height / 2;
            r->height = root->height / 2;
        } else {
            r->y = root->y;
            if(ah == CENTER) {
                r->height = root->height / 2;
            } else {
                r->height = root->height;
            }
        }
    } else if(!sy && y && y[0] != '-' && y[0] != '+') {
        Align ay;
        str_to_align(&ay, y);
        if(ay == CENTER) {
            r->y = root->y + (root->height / 2) - (r->height / 2);
        } else if(ay == SOUTH) {
            r->y = root->y + root->height - r->height;
        } else {
            r->y = root->y;
        }
    } else if(!sh && h && h[0] != '-' && h[0] != '+') {
        Align ah;
        str_to_align(&ah, h);
        if(ah == CENTER) {
            r->height = (root->height / 2) - r->y;
        } else {
            r->height = root->height - r->y;
        }
    }

    /* now do final calculations */
    if(x) {
        p = strchr(x, '-');
        if(p)
            r->x -= atoi(++p);
        p = strchr(x, '+');
        if(p)
            r->x += atoi(++p);
    }
    if(y) {
        p = strchr(y, '-');
        if(p)
            r->y -= atoi(++p);
        p = strchr(y, '+');
        if(p)
            r->y += atoi(++p);
    }
    if(w) {
        p = strchr(w, '-');
        if(p)
            r->width -= atoi(++p);
        p = strchr(w, '+');
        if(p)
            r->width += atoi(++p);
    }
    if(h) {
        p = strchr(h, '-');
        if(p)
            r->height -= atoi(++p);
        p = strchr(h, '+');
        if(p)
            r->height += atoi(++p);
    }

}

unsigned long
str_to_mod(char *val)
{
    unsigned long mod = 0;
    if(strstr(val, "shift"))
        mod |= ShiftMask;
    if(strstr(val, "lock"))
        mod |= LockMask;
    if(strstr(val, "ctrl"))
        mod |= ControlMask;
    if(strstr(val, "alt"))
        mod |= Mod1Mask;
    if(strstr(val, "mod1"))
        mod |= Mod1Mask;
    if(strstr(val, "mod2"))
        mod |= Mod2Mask;
    if(strstr(val, "mod3"))
        mod |= Mod3Mask;
    if(strstr(val, "mod4"))
        mod |= Mod4Mask;
    if(strstr(val, "mod5"))
        mod |= Mod5Mask;
    return mod;
}

/* chomps whitespces from val */
char *
chomp(char *val)
{
    char *p = strdup(val);
    char *fp = p;
    char *result = strdup(p);
    int i = 0;

    while(*p != '\0') {
        if(*p != ' ' && *p != '\t') {
            result[i++] = *p;
        }
        p++;
    }
    free(fp);
    result[i] = '\0';
    return result;
}

int
is_point_in_rect(int x, int y, XRectangle * r)
{
    return (x >= r->x) && (x <= r->x + r->width)
        && (y >= r->y) && (y <= r->y + r->height);
}

int
count_toks(char *str, char delim)
{
    int tokc = 1;               /* \0 */
    char *p;

    if(!str)
        return 0;
    for(p = str; *p != '\0'; p++) {
        if(*p == delim)
            tokc++;
    }

    return tokc;
}

void
tokenize(char **result, const char *str, char delim)
{
    char *p, *n;
    char *buf = strdup(str);
    int i;

    p = n = buf;
    for(i = 0; *n != '\0';) {
        if(*n == delim) {
            *n = '\0';
            result[i++] = strdup(p);
            p = ++n;
        } else
            n++;
    }
    result[i] = strdup(p);
    free(buf);
}

void
free_toks(char **toks, int num_toks)
{
    int i;
    for(i = 0; i < num_toks; i++)
        free(toks[i]);
    free(toks);
}

int
rect_touches_rect(XRectangle * first, XRectangle * second)
{
    int result = 0;

    if(((first->x > second->x) && (first->x < second->width))
       || ((first->x + first->width > second->x)
           && (first->x + first->width < second->width))) {
        result = (first->y < second->y + second->height) &&
            (first->y + first->height > second->y);
    }
    if(((first->y > second->y) && (first->y < second->height))
       || ((first->y + first->height > second->y)
           && (first->y + first->height < second->height))) {
        result = (first->x < second->x + second->width) &&
            (first->x + first->width > second->x);
    }
    return result;
}

int
distance(XRectangle * origin, XRectangle * target)
{
    int ox = origin->x + origin->width / 2;
    int oy = origin->y + origin->height / 2;
    int tx = target->x + target->width / 2;
    int ty = target->y + target->height / 2;

    return (int) sqrt((double) (((ox - tx) * (ox - tx)) +
                                ((oy - ty) * (oy - ty))));
}

static void
do_spawn(void *dpy, char *cmd)
{
    if(fork() == 0) {
        int argc = count_toks(cmd, ' ') + 1;
        char **argv = malloc(sizeof(char *) * argc);
        tokenize(argv, cmd, ' ');
        argv[argc - 1] = 0;
        setsid();
        close(ConnectionNumber((Display *) dpy));
        execvp(argv[0], argv);
        perror("failed");
        exit(1);
    }
}

void
spawn(void *dpy, char *cmd)
{
    if(fork() == 0) {
        do_spawn(dpy, cmd);
        exit(0);
    }
    wait(0);
}

void
spawn_wait(void *dpy, char *cmd)
{
    do_spawn(dpy, cmd);
    wait(0);
}

void
swap(void **p1, void **p2)
{
    void *tmp = *p1;
    *p1 = *p2;
    *p2 = tmp;
}

int
int_val(void *file_content)
{
    if(file_content)
        return atoi(file_content);
    return 0;
}
