/*
 * WMMail - Window Maker Mail
 *
 * Copyright (c) 1996, 1997, 1998  Per Liden
 * Copyright (c) 1997, 1998  Bryan Chan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * maildir.c: functions to handle MailDir directories
 *
 * $Id: maildir.c,v 1.1 2000/07/02 20:37:59 bryan.chan Exp $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <ctype.h>
#include <dirent.h>
#include <time.h>
#include <unistd.h>
#include "wmmail.h"
#include "wmutil.h"

#ifdef MAILDIR_SUPPORT

/* 
 * Supported key(s) in the Options dictionary:
 *
 *   Path
 *
 * See mbox.c for information on states and transitions.
 *
 */

/* internal function */
static int count_mail(char *, int *, int *);


#define GET_ENTRY(x,y,z) { proplist_t sub_key = PLMakeString(z); \
                           y = PLGetDictionaryEntry(x, sub_key); \
                           PLRelease(sub_key); }

int MAILDIR_check(Mailbox *mailbox, int *beep, int *redraw, int *run)
{
  proplist_t  path;
  char       *mailbox_path;
  int         prev_status, 
              prev_new_mail_count;

  prev_status = mailbox->status;
  prev_new_mail_count = mailbox->new_mail_count;

  GET_ENTRY(mailbox->options, path, "Path");
  if (path == NULL)
  {
    croak("mailbox \"%s\" missing option \"Path\"; ignored", mailbox->name);
    return False;
  }
  else if (!PLIsString(path))
  {
    croak("mailbox \"%s\" has invalid path; ignored", mailbox->name);
    return False;
  }
  else
    mailbox_path = expand_path(PLGetString(path));

  if ( count_mail(mailbox_path,
                  &mailbox->total_mail_count,
                  &mailbox->new_mail_count)   )
  {
    if (mailbox->total_mail_count == 0)
    {
      /* there is no mail in the mailbox */
      mailbox->status = NO_MAIL;
    }
    else if (mailbox->new_mail_count == 0)
    {
      /* there are no new mails */
      mailbox->status = OLD_MAIL;
    }
    else if (mailbox->new_mail_count > prev_new_mail_count)
    {
      /* new mails have arrived! */
      if (mailbox->status == NEW_MAIL && always_new_mail_exec)
        *run |= True;
      else if (mailbox->status == NEW_MAIL)
        *run |= False;
      else
        *run |= True;

      *beep = True;

      mailbox->status = NEW_MAIL;
    }

    /* else no change */
  }
  else
  {
    /* no such mailbox */
    croak("error reading MailDir folder \"%s\"; ignored", mailbox->name);
    wfree(mailbox_path);
    return False;
  }

  *redraw |= (prev_status != mailbox->status);

  mailbox->last_update = time(NULL);

  wfree(mailbox_path);
  return True;
}


static int count_mail(char *path, int *total_mail_count, int *new_mail_count)
{
  struct dirent *entry;
  DIR           *folder_new,
                *folder_cur;
  char          *buf;
  int            new_mail = 0,
                 all_mail = 0;

  buf = wmalloc((strlen(path) + 5) * sizeof(char));

  strcpy(buf, path); strcat(buf, "/new");

  if ( !(folder_new = opendir(buf)) )
  {
    croak("cannot open directory %s", buf);
    free(buf);
    return False;
  }

  strcpy(buf, path); strcat(buf, "/cur");

  if ( !(folder_cur = opendir(buf)) )
  {
    croak("cannot open directory %s", buf);
    free(buf);
    return False;
  }

  free(buf);

  while ((entry = readdir(folder_new)) != NULL) {
    if (entry->d_name && *(entry->d_name) != '.') {
      new_mail++;
      all_mail++;
    }
  }

  while ((entry = readdir(folder_cur)) != NULL) {
    if (entry->d_name && *(entry->d_name) != '.')
      all_mail++;
  }

  closedir(folder_new);
  closedir(folder_cur);

  *total_mail_count = all_mail;
  *new_mail_count = new_mail;

  return True;
}

#endif
