/* aguix.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2004 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: aguix.cc,v 1.72 2005/05/23 18:44:44 ralf Exp $ */

#include "aguix.h"
#include "awindow.h"
#include <locale.h>

#ifdef USE_AGUIXTIMER
#include <signal.h>
#endif

#ifdef USE_XIM

#ifndef HAVE_XSETIMVALUES_PROTOTYPE
extern "C" {
  extern char *XSetIMValues( XIM, ... );
}
#endif

#include <X11/Xlocale.h>

/* ChosseButterStyle is based on an example program I found somewhere
   But there's was no copyright and the orig codepiece also
   doesn't work...
*/
XIMStyle ChooseBetterStyle( XIMStyle style1, XIMStyle style2 )
{
  XIMStyle s, t;
  XIMStyle preedit = XIMPreeditArea |
    XIMPreeditCallbacks |
    XIMPreeditPosition |
    XIMPreeditNothing |
    XIMPreeditNone;
  XIMStyle status = XIMStatusArea |
    XIMStatusCallbacks |
    XIMStatusNothing |
    XIMStatusNone;

  if ( style1 == 0 ) return style2;
  if ( style2 == 0 ) return style1;
  if ( ( style1 & ( preedit | status ) ) == ( style2 & ( preedit | status ) ) )
    return style1;

  s = style1 & preedit;
  t = style2 & preedit;
  if ( s != t ) {
    if ( ( s | t ) & XIMPreeditCallbacks ) {
      return ( s == XIMPreeditCallbacks ) ? style1 : style2;
    } else if ( ( s | t ) & XIMPreeditPosition )
      return ( s == XIMPreeditPosition ) ? style1 : style2;
    else if ( ( s | t ) & XIMPreeditArea )
      return ( s == XIMPreeditArea ) ? style1 : style2;
    else if ( ( s | t ) & XIMPreeditNothing )
      return ( s == XIMPreeditNothing ) ? style1 : style2;
  } else {
    s = style1 & status;
    t = style2 & status;
    if ( ( s | t ) & XIMStatusCallbacks)
      return ( s == XIMStatusCallbacks ) ? style1 : style2;
    else if ( ( s | t ) & XIMStatusArea )
      return ( s == XIMStatusArea ) ? style1 : style2;
    else if ( ( s | t ) & XIMStatusNothing )
      return ( s == XIMStatusNothing ) ? style1 : style2;
  }
  return 0;
}

#ifdef XIM_XREGISTER_OKAY
void AGUIX_im_inst_callback( Display *calldsp, XPointer client_data, XPointer call_data )
{
  if ( client_data != NULL ) {
    ((AGUIX*)client_data)->IMInstCallback( calldsp );
  }
}

void AGUIX_im_dest_callback( XIM callim, XPointer client_data, XPointer call_data )
{
  if ( client_data != NULL ) {
    ((AGUIX*)client_data)->IMDestCallback();
  }
}
#endif //XIM_XREGISTER_OKAY

#endif //USE_XIM

long AGUIX::timerEvent = 0;

#ifdef USE_AGUIXTIMER
void aguix_timer( int s )
{
  if ( s == SIGALRM ) {
    AGUIX::timerEvent++;
  }
}
#endif

AGUIX::AGUIX()
{
#ifdef USE_AGUIXTIMER
  struct sigaction sig_ac;
#endif

  initOK=False;
  classname=NULL;
  myfont=NULL;
  mainfont=NULL;
  dsp=NULL;
  colors=0;
  wins=new List();
  messages=new List();
  fonts=new List();
  privatecmap=false;
  cutstart=NULL;
  cutbuffer=NULL;
  pastestart = NULL;
  backpm = None;
  scr = -1;
#ifdef USE_XIM
  im_style = 0;
  inputmethod = NULL;
#endif
  keybuf_len = 10;
  keybuf = (char*)_allocsafe( keybuf_len );
  transientwindow = NULL;
  lastkeyrelease = None;
  lastmouserelease = 0;
  msgLockElement = NULL;
  timerEnabled = false;
  lastTimerEventNr = 0;

#ifdef USE_AGUIXTIMER
  memset( &sig_ac, 0, sizeof( sig_ac ) );
  sig_ac.sa_handler = aguix_timer;
  sigaction( SIGALRM, &sig_ac, NULL );
#endif
}
  
int AGUIX::initX( int targc, char **targv, const char *tclassname )
{
#ifdef DEBUG
  printf("Opening display...");
#endif
  char *displayname=getenv("DISPLAY");
  dsp=XOpenDisplay(displayname);
  if(dsp==NULL) {
#ifdef DEBUG
    printf("failed\n");
#endif
    return 5;
  }
#ifdef DEBUG
  printf("Ok\n");
#endif
  this->classname=dupstring(tclassname);
  scr=XDefaultScreen(dsp);
  cmap=DefaultColormap(dsp,scr);
  white=WhitePixel(dsp,scr);
  black=BlackPixel(dsp,scr);

  gc=XCreateGC(dsp,RootWindow(dsp,scr),0,0);
  XGCValues gc_values;
  gc_values.graphics_exposures=False;
  gc_values.foreground=black;
  gc_values.background=white;
  gc_values.line_width=0;
  gc_values.cap_style=CapButt;
#ifdef DEBUG
  printf("Loading font...");
#endif
  myfont=XLoadQueryFont(dsp,"fixed");
  if(myfont==NULL) {
    printf("no font, trying default: ");
    myfont=XLoadQueryFont(dsp,"fixed");
    if(myfont==NULL) panic("failed");
#ifndef DEBUG
    printf("Ok\n");
#endif
  }
#ifdef DEBUG
  printf("Ok\n");
#endif
  CharHeight=myfont->ascent+myfont->descent;
#if 0
  CharWidth=myfont->max_bounds.rbearing-myfont->min_bounds.lbearing;
#else
  CharWidth=XTextWidth(myfont,"M",1);
#endif
  gc_values.font=myfont->fid;
  unsigned long gc_valuemask=GCGraphicsExposures|GCForeground|GCBackground|GCLineWidth|GCFont|GCCapStyle;
  XChangeGC(dsp,gc,gc_valuemask,&gc_values);

  dotted_gc = XCreateGC( dsp, RootWindow( dsp, scr ), 0, 0 );
  XCopyGC( dsp, gc, ~0, dotted_gc );
  gc_values.line_style = LineOnOffDash;
  XChangeGC( dsp, dotted_gc, GCLineStyle, &gc_values );

  dashxor_gc = XCreateGC( dsp, RootWindow( dsp, scr ), 0, 0 );
  XCopyGC( dsp, gc, ~0, dashxor_gc );
  gc_values.line_style = LineOnOffDash;
  gc_values.function = GXxor;
  gc_values.plane_mask = ~0;
  gc_values.foreground = 0xffffffff;
  XChangeGC( dsp, dashxor_gc, GCLineStyle | GCFunction | GCPlaneMask | GCForeground, &gc_values );

  dashdouble_gc = XCreateGC( dsp, RootWindow( dsp, scr ), 0, 0 );
  XCopyGC( dsp, gc, ~0, dashdouble_gc );
  gc_values.line_style = LineDoubleDash;
  gc_values.foreground = black;
  gc_values.background = white;
  XChangeGC( dsp, dashdouble_gc, GCLineStyle | GCBackground | GCForeground, &gc_values );

  if(getFont("fixed")==NULL) printf("Opps, no \"fixed\"\n");
  WM_delete_window=XInternAtom(dsp,"WM_DELETE_WINDOW",False);
  this->argc=targc;
  this->argv=targv;
  createGroupWin();

  cursors[WAIT_CURSOR]=XCreateFontCursor(dsp,XC_watch);
  cursors[SCROLLH_CURSOR]=XCreateFontCursor(dsp,XC_sb_h_double_arrow);
  cursors[SCROLLV_CURSOR]=XCreateFontCursor(dsp,XC_sb_v_double_arrow);

  unsigned int rd,rbw;
  int rx, ry;
  Window rootw;
  XGetGeometry( dsp, DefaultRootWindow( dsp ), &rootw, &rx, &ry, &rootWindowWidth, &rootWindowHeight, &rbw, &rd );

  if ( setlocale( LC_ALL, "" ) == NULL ) {
    fprintf( stderr, "Worker Warning: locale not supported\n" );
  }
  if ( XSupportsLocale() == False ) {
    fprintf( stderr, "Worker Warning: X doesn't support current locale, setting locale to C\n" );
    setlocale( LC_ALL, "C" );
  }
  if ( XSetLocaleModifiers( "" ) == NULL ) {
    fprintf( stderr, "Worker Warning: Cannot set locale modifiers\n" );
  }
#ifdef USE_XIM
  openIM();
#endif
  initOK=True;
  return 0;
}

AGUIX::~AGUIX()
{
  AGMessage *agmsg;

  disableTimer();
  while((agmsg=getAGMsg())!=NULL) ReplyMessage(agmsg);
  if(checkX()==True) {
    if(dsp!=NULL) {
      closeX();
    }
  }
  cancelCut();
  if(cutbuffer!=NULL) _freesafe(cutbuffer);
  delete wins;
  delete messages;
  delete fonts;
  _freesafe( keybuf );
}

void AGUIX::panic(const char *msg)
{
  fprintf(stderr,"%s\n",msg);
  exit(1);
}

void AGUIX::closeX()
{
  destroyGroupWin();
  int id=fonts->initEnum();
  AGUIXFont *tf=(AGUIXFont*)fonts->getFirstElement(id);
  while(tf!=NULL) {
    delete tf;
    tf=(AGUIXFont*)fonts->getNextElement(id);
  }
  fonts->closeEnum(id);
  if(classname!=NULL) _freesafe(classname);
  if(myfont!=NULL) XFreeFont(dsp,myfont);
  if(privatecmap==true) {
    XFreeColormap(dsp,cmap);
  }
  XFreeCursor(dsp,cursors[WAIT_CURSOR]);
  XFreeCursor(dsp,cursors[SCROLLH_CURSOR]);
  XFreeCursor(dsp,cursors[SCROLLV_CURSOR]);
  XFreeGC( dsp, gc );
  XFreeGC( dsp, dotted_gc );
  XFreeGC( dsp, dashxor_gc );
  XFreeGC( dsp, dashdouble_gc );
  if ( backpm != None ) XFreePixmap( dsp, backpm );
#ifdef USE_XIM
  closeIM();
#endif
#ifdef DEBUG
  printf("Closing display\n");
#endif
  XCloseDisplay(dsp);
  dsp=NULL;
}

int AGUIX::checkX()
{
  return initOK;
}

int AGUIX::getDepth() const
{
  return DefaultDepth(dsp,scr);
}

Display *AGUIX::getDisplay() const
{
  return dsp;
}

int AGUIX::getScreen() const
{
  return scr;
}

int AGUIX::getCharHeight() const
{
  return CharHeight;
}

int AGUIX::getCharWidth() const
{
  return CharWidth;
}

void AGUIX::setFG(int color)
{
  if(color<0) color=0;
  if(color<colors) {
    XSetForeground(dsp,gc,ColBuf[color]);
  } else {
    XSetForeground(dsp,gc,ColBuf[colors-1]);
  }
}

void AGUIX::setFG(GC tgc,int color)
{
  if(tgc==0) setFG(color);
  else {
    if(color<0) color=0;
    if(color<colors) {
      XSetForeground(dsp,tgc,ColBuf[color]);
    } else {
      XSetForeground(dsp,tgc,ColBuf[colors-1]);
    }
  }
}

void AGUIX::setBG(int color)
{
  if(color<0) color=0;
  if(color<colors) {
    XSetBackground(dsp,gc,ColBuf[color]);
  } else {
    XSetBackground(dsp,gc,ColBuf[colors-1]);
  }
}

void AGUIX::setBG(GC tgc,int color)
{
  if(tgc==0) setBG(color);
  else {
    if(color<0) color=0;
    if(color<colors) {
      XSetBackground(dsp,tgc,ColBuf[color]);
    } else {
      XSetBackground(dsp,tgc,ColBuf[colors-1]);
    }
  }
}

void AGUIX::FillRectangle(Drawable buffer,int x,int y,int w,int h)
{
  if ( w < 1 ) w = 1;
  if ( h < 1 ) h = 1;
  XFillRectangle(dsp,buffer,gc,x,y,w,h);
}

void AGUIX::FillRectangle(Drawable buffer,GC tgc,int x,int y,int w,int h)
{
  if ( w < 1 ) w = 1;
  if ( h < 1 ) h = 1;

  if(tgc==0) FillRectangle(buffer,x,y,w,h);
  else XFillRectangle(dsp,buffer,tgc,x,y,w,h);
}

void AGUIX::DrawText(Drawable buffer,const char *text,int x,int y)
{
  XDrawString(dsp,buffer,gc,x,y+getFontBaseline(),text,strlen(text));
}

void AGUIX::DrawText(Drawable buffer,AGUIXFont *tf,const char *text,int x,int y)
{
  XDrawString(dsp,buffer,tf->getGC(),x,y+tf->getBaseline(),text,strlen(text));
}

int AGUIX::getFontBaseline() const
{
  if(mainfont!=NULL) return mainfont->getBaseline();
  return myfont->ascent;
}

int AGUIX::AddColor(int red,int green,int blue)
{
  XColor col;
  if(colors<256) {
    col.flags=DoRed|DoGreen|DoBlue;
    col.red=red*256;
    col.green=green*256;
    col.blue=blue*256;
    if(!XAllocColor(dsp,cmap,&col)) {
      // error
      if(privatecmap==true) return -1;
      else {
        // now try to go to a private colormap
	changeColormap();
	return AddColor(red,green,blue);
      }
    }
    ColBuf[colors++]=col.pixel;
  }
  return colors-1;
}

void AGUIX::removeLastColor()
{
  if(colors>0) {
    XFreeColors(dsp,cmap,ColBuf+colors-1,1,0);
    colors--;
  }
}

int AGUIX::getMaxCols() const
{
  return colors;
}

char *AGUIX::getClassname() const
{
  char *tstr;
  tstr=dupstring(classname);
  return tstr;
}

unsigned long AGUIX::getPixel(int color) const
{
  if((color<colors)&&(color>=0)) {
    return ColBuf[color];
  } else return 0;
}

Atom *AGUIX::getCloseAtom()
{
  return &WM_delete_window;
}

void AGUIX::insertWindow(AWindow *win)
{
  if(win==NULL) return;
  wins->addElement(win);
}

void AGUIX::removeWindow(AWindow *win)
{
  if(win==NULL) return;
  wins->removeElement(win);
  if ( win == transientwindow ) setTransientWindow();
}

Message *AGUIX::wait4mess( int mode )
{
  Message *newmsg;
  KeySym J;
  int count;
  AWindow *msgawin;
#ifdef USE_XIM
  Window msgwin;
  int id;
  Status status;
#endif
  
  if(mode==MES_GET) {
    if(XEventsQueued(dsp,QueuedAfterFlush)==0) return NULL;
  }
  XNextEvent(dsp,&LastEvent);

#ifdef USE_XIM
  switch ( LastEvent.type ) {
    case KeyPress:
      id = wins->initEnum();
      msgwin = LastEvent.xkey.window;
      
      msgawin = (AWindow*)wins->getFirstElement( id );
      while ( msgawin != NULL ) {
	if ( msgawin->isTopLevel() == true ) {
	  if ( msgawin->isParent( msgwin, false ) == true ) break;
	}
	msgawin = (AWindow*)wins->getNextElement( id );
      }
      wins->closeEnum( id );
      break;
    default:
      msgawin = NULL;
      break;
  }
  if ( XFilterEvent( &LastEvent, ( msgawin != NULL ) ? msgawin->getWindow() : None ) == True ) return NULL;
#else
  msgawin = NULL;
#endif

  newmsg = AGUIX_allocMessage();
  newmsg->type=LastEvent.type;
  newmsg->gadgettype=NON_GADGET;
  newmsg->gadget=NULL;
  newmsg->window=0;
  newmsg->time=0;

  newmsg->lockElement = msgLockElement;
  newmsg->ack = false;
  newmsg->loop = 0;

  switch(LastEvent.type) {
    case MapNotify:
    case UnmapNotify:
      newmsg->window = LastEvent.xmap.window;
      break;
    case MappingNotify:
      XRefreshKeyboardMapping((XMappingEvent*)&LastEvent);
      break;
    case KeyPress:
    case KeyRelease:
      if ( ( LastEvent.type == KeyPress ) && ( msgawin != NULL ) ) {
#ifdef USE_XIM
	if ( msgawin->getXIC() != NULL ) {
	  count = XmbLookupString( msgawin->getXIC(), &(LastEvent.xkey), keybuf, keybuf_len - 1, &J, &status );
	  if ( status == XBufferOverflow ) {
	    _freesafe( keybuf );
	    keybuf_len = count + 1;
	    keybuf = (char*)_allocsafe( keybuf_len );
	    count = XmbLookupString( msgawin->getXIC(), &(LastEvent.xkey), keybuf, keybuf_len - 1, &J, &status );
	  }
	  switch ( status ) {
	    case XLookupBoth:
	      keybuf[count] = '\0';
	      break;
	    case XLookupKeySym:
	      keybuf[0] = '\0';
	      break;
	    case XLookupChars:
	      J = XK_VoidSymbol;
	      keybuf[count] = '\0';
	      break;
	    default:
	      keybuf[0] = '\0';
	      J = XK_VoidSymbol;
	      break;
	  }
	  newmsg->key = J;
	} else {
	  count = XLookupString( &(LastEvent.xkey), keybuf, keybuf_len - 1, &J, NULL );
	  keybuf[ count ] = '\0';
	  newmsg->key=J;
	}
#endif
      } else {
	count = XLookupString( &(LastEvent.xkey), keybuf, keybuf_len - 1, &J, NULL );
	keybuf[ count ] = '\0';
	newmsg->key=J;
      }
      newmsg->keystate=LastEvent.xkey.state;
      newmsg->window=LastEvent.xkey.window;
      newmsg->mousex=LastEvent.xkey.x;
      newmsg->mousey=LastEvent.xkey.y;
      newmsg->keybuf = dupstring( keybuf );
      break;
    case ButtonPress:
    case ButtonRelease:
      newmsg->button = LastEvent.xbutton.button;
      newmsg->mousex = LastEvent.xbutton.x;
      newmsg->mousey = LastEvent.xbutton.y;
      newmsg->window = LastEvent.xbutton.window;
      newmsg->time = LastEvent.xbutton.time;
      break;
    case MotionNotify:
      /* I no longer use XQueryPointer here because of error if the window
         this message is for no longer exists and then XQueryPointer fails
	 Instead use queryPointer when get AG_MOUSEMOVE */
      newmsg->mousex = LastEvent.xmotion.x;
      newmsg->mousey = LastEvent.xmotion.y;
      newmsg->mouserx = LastEvent.xmotion.x_root;
      newmsg->mousery = LastEvent.xmotion.y_root;
      newmsg->window = LastEvent.xmotion.window;
      break;
    case Expose:
      newmsg->window = LastEvent.xexpose.window;
      newmsg->x = LastEvent.xexpose.x;
      newmsg->y = LastEvent.xexpose.y;
      newmsg->width = LastEvent.xexpose.width;
      newmsg->height = LastEvent.xexpose.height;
      break;
    case ConfigureNotify:
      // first try to find further configure notifies for
      // this window to improve handling of many resizes due to
      // solid resize used by many WMs
      while ( XCheckTypedWindowEvent( dsp,
				      LastEvent.xconfigure.window,
				      ConfigureNotify,
				      &LastEvent ) == True );
      newmsg->window=LastEvent.xconfigure.window;
      newmsg->mousex=LastEvent.xconfigure.x;
      newmsg->mousey=LastEvent.xconfigure.y;
      newmsg->width = LastEvent.xconfigure.width;
      newmsg->height = LastEvent.xconfigure.height;
      newmsg->x = LastEvent.xconfigure.x;
      newmsg->y = LastEvent.xconfigure.y;
      break;
    case ClientMessage:
      if(LastEvent.xclient.data.l[0]==(long)WM_delete_window) {
        newmsg->gadgettype=CLOSE_GADGET;
        newmsg->gadget=NULL;
        newmsg->window=LastEvent.xclient.window;
      }
      break;
    case EnterNotify:
    case LeaveNotify:
      newmsg->window=LastEvent.xcrossing.window;
      newmsg->mousex=LastEvent.xcrossing.x;
      newmsg->mousey=LastEvent.xcrossing.y;
      break;
    case SelectionClear:
      if(cutstart!=NULL) {
	if( ( LastEvent.xselectionclear.window==cutstart->getWindow() ) &&
	    ( LastEvent.xselection.selection==XA_PRIMARY ) ) {
	  cancelCut();
	}
      }
      break;
    case SelectionNotify:
      if( (LastEvent.xselection.selection==XA_PRIMARY) &&
	  (LastEvent.xselection.property!=None) ) {
	Atom ret_type;
	int ret_format;
	unsigned long ret_len,ret_after;
	unsigned char *ret_prop;
	XGetWindowProperty(dsp,
			   LastEvent.xselection.requestor,
			   LastEvent.xselection.property,
			   0,
			   8192,
			   False,
			   LastEvent.xselection.target,
			   &ret_type,
			   &ret_format,
			   &ret_len,
			   &ret_after,
			   &ret_prop);
	if(ret_len>0) {
	  //	  ret_prop[ret_len-1]=0;
	  if(pastestart!=NULL) pastestart->paste(ret_prop);
	  XFree(ret_prop);
	  XDeleteProperty(dsp,LastEvent.xselection.requestor,LastEvent.xselection.property);
	} else {
	  if(pastestart!=NULL) pastestart->cancelpaste();
	  pastestart=NULL;
	}
      } else {
	if(pastestart!=NULL) pastestart->cancelpaste();
	pastestart=NULL;
      }
      break;
    case SelectionRequest:
      XEvent ev;
      ev.type=SelectionNotify;
      ev.xselection.requestor=LastEvent.xselectionrequest.requestor;
      ev.xselection.selection=LastEvent.xselectionrequest.selection;
      ev.xselection.target=LastEvent.xselectionrequest.target;
      ev.xselection.time=LastEvent.xselectionrequest.time;
      ev.xselection.property=LastEvent.xselectionrequest.property;
      if( ( LastEvent.xselectionrequest.selection==XA_PRIMARY ) &&
	  ( LastEvent.xselectionrequest.target==XA_STRING ) ) {
	XChangeProperty(dsp,
			ev.xselection.requestor,
			ev.xselection.property,
			ev.xselection.target,
			8,
			PropModeReplace,
			(unsigned char*)cutbuffer,
			strlen(cutbuffer));
      } else {
	ev.xselection.property=None;
      }
      XSendEvent(dsp,ev.xselection.requestor,False,0,&ev);
      break;
    case FocusIn:
    case FocusOut:
      newmsg->window = LastEvent.xfocus.window;
      break;
  }
  return newmsg;
}

void AGUIX::buildAGMessage( Message *msg )
{
  AGMessage *agmsg[2];

  agmsg[0] = AGUIX_allocAGMessage();  // Hauptmessage
  agmsg[1] = NULL;           // eventl. Nebenmessage;
  switch( msg->type ) {
    case KeyPress:
      agmsg[0]->type = AG_KEYPRESSED;
    case KeyRelease:
      if ( agmsg[0]->type == AG_NONE ) agmsg[0]->type = AG_KEYRELEASED;
      agmsg[0]->key.window = msg->window;
      agmsg[0]->key.key = msg->key;
      agmsg[0]->key.keystate = msg->keystate;
      agmsg[0]->key.keybuf = dupstring( msg->keybuf );
      break;
    case ButtonPress:
      agmsg[0]->type = AG_MOUSEPRESSED;
    case ButtonRelease:
      if ( agmsg[0]->type == AG_NONE ) {
        agmsg[0]->type = AG_MOUSERELEASED;
        agmsg[1] = AGUIX_allocAGMessage();
        agmsg[1]->type = AG_MOUSECLICKED;
        agmsg[1]->mouse.window = msg->window;
        agmsg[1]->mouse.button = msg->button;
        agmsg[1]->mouse.x = msg->mousex;
        agmsg[1]->mouse.y = msg->mousey;
        agmsg[1]->mouse.time = msg->time;
      }
      agmsg[0]->mouse.window = msg->window;
      agmsg[0]->mouse.button = msg->button;
      agmsg[0]->mouse.x = msg->mousex;
      agmsg[0]->mouse.y = msg->mousey;
      agmsg[0]->mouse.time = msg->time;
      break;
    case MotionNotify:
      agmsg[0]->type = AG_MOUSEMOVE;
      agmsg[0]->mouse.window = msg->window;
      /* button is wrong -> use queryPointer */
      agmsg[0]->mouse.button = 0;
      agmsg[0]->mouse.x = msg->mousex;
      agmsg[0]->mouse.y = msg->mousey;
      break;
    case Expose:
      agmsg[0]->type = AG_EXPOSE;
      agmsg[0]->expose.window = msg->window;
      agmsg[0]->expose.x = msg->x;
      agmsg[0]->expose.y = msg->y;
      agmsg[0]->expose.w = msg->width;
      agmsg[0]->expose.h = msg->height;
      break;
    case ClientMessage:
      if ( msg->gadgettype == CLOSE_GADGET ) {
        agmsg[0]->type = AG_CLOSEWINDOW;
        agmsg[0]->closewindow.window = msg->window;
      }
      break;
    case EnterNotify:
      agmsg[0]->type = AG_ENTERWINDOW;
    case LeaveNotify:
      if ( agmsg[0]->type == AG_NONE ) agmsg[0]->type = AG_LEAVEWINDOW;
      agmsg[0]->cross.window = msg->window;
      break;
  }
  // agmsg in FIFO speichern
  putAGMsg( agmsg[0] );
  if ( agmsg[1] != NULL ) putAGMsg( agmsg[1] );
}

int AGUIX::msgHandler( int mode, AWindow *parent, bool onlyExpose )
{
  Message *tmsg;
  int waitmode;
  long timerEventNr;
#ifndef USE_AGUIXTIMER
  struct timeval curTime;
  long t1;
#endif

  // do not limit to special window if there is a guielement
  // holding the lock
  if ( msgLockElement != NULL ) parent = NULL;

  for (;;) {
    waitmode = mode;

    // do not wait when timer is enabled
    if ( ( waitmode == MES_WAIT ) &&
	 ( msgLockElement != NULL ) &&
	 ( timerEnabled == true ) ) {
      waitmode = MES_GET;
    }

    if ( ( msgLockElement != NULL ) &&
	 ( timerEnabled == true ) ) {
      onlyExpose = false;
    }
    tmsg = wait4mess( waitmode );
    if ( tmsg != NULL ) {
      if ( ( onlyExpose == false ) || ( ( onlyExpose == true ) && ( tmsg->type == Expose ) ) ) {
	if ( ReactMessage( tmsg, parent ) == false ) {
          if ( msgLockElement == NULL ) {
            // only build msg for application when no element holds the lock
            //TODO: at the time of this writing (20031115) it would be okay
            //      to build the msg in any case, AGUIX can handle this
            //      but I keep this conforming to previous version
            buildAGMessage( tmsg );
          }
        }
      }
      if ( tmsg->type == ButtonRelease ) {
	lastmouserelease = tmsg->button;
      } else if ( tmsg->type == KeyRelease ) {
	lastkeyrelease = tmsg->key;
      }
      AGUIX_freeMessage( tmsg );
    }

    if ( XEventsQueued( dsp, QueuedAfterFlush ) > 0 ) continue;
    // and check whether we hit a timer limit

    // break loop when timer isn't activated
    if ( ! ( ( msgLockElement != NULL ) &&
	     ( timerEnabled == true ) ) ) {
      break;
    }
    //TODO Because it's possible to break the loop
    //     the msgLockElement could be destroyed already
    //     ReactMessage will reset msgLockElement if it doesn't
    //     set ack, but when there's no msg, we can reach this
    //     code
    //     possible solution: check every window using contains()
    if ( msgLockElement->getAllowTimerEventBreak() == true ) {
      if ( msgLockElement->getNrOfLastEventBreak() != lastTimerEventNr ) {
	msgLockElement->setNrOfLastEventBreak( lastTimerEventNr );
	break;
      }
    }

    // now wait some time
#ifndef USE_AGUIXTIMER
    gettimeofday( &curTime, NULL );
    //TODO das koennte ein Precision-Problem sein, wenn zu lange im Timer-Mode
    //     verblieben wird
    t1 = ( ( lastTimerEventNr + 1 ) * 1000 / TIMER_TICKS ) - ldiffgtod_m( &curTime, &timerStart );
    if ( ( tmsg == NULL ) && ( t1 > 1 ) ) waittime( 5 );
#else
    if ( tmsg == NULL ) waittime( 5 );
#endif

#ifndef USE_AGUIXTIMER
    gettimeofday( &curTime, NULL );
    timerEventNr = ( ldiffgtod_m( &curTime, &timerStart ) * TIMER_TICKS ) / 1000;
#else
    timerEventNr = timerEvent;
#endif

    if ( timerEventNr != lastTimerEventNr ) {
      // send timer event msg
      tmsg = AGUIX_allocMessage();
      tmsg->type = ClientMessage;
      tmsg->specialType = Message::TIMEREVENT;
      tmsg->time = timerEventNr;
      tmsg->window = None;
      
      tmsg->lockElement = msgLockElement;
      tmsg->ack = false;
      tmsg->loop = 0;
      
      ReactMessage( tmsg, parent );
      AGUIX_freeMessage( tmsg );
      lastTimerEventNr = timerEventNr;
    }
  }
  return XEventsQueued( dsp, QueuedAfterFlush );
}

AGMessage *AGUIX::GetMessage(AWindow *parent)
{
  msgHandler( MES_GET, parent, false );
  return getAGMsg();
}

AGMessage *AGUIX::WaitMessage(AWindow *parent)
{
  while(messages->size()==0) {
    msgHandler( MES_WAIT, parent, false );
  }
  return getAGMsg();
}

void AGUIX::copyArea(Drawable source,Drawable dest,int s_x,int s_y,int width,int height,int d_x,int d_y)
{
  if ( ( width < 1 ) || ( height < 1 ) ) return;  // do nothing when no valid width/height
  XCopyArea(dsp,source,dest,gc,s_x,s_y,width,height,d_x,d_y);
}

void AGUIX::DrawLine(Drawable buffer,int px1,int py1,int px2,int py2)
{
  XDrawLine(dsp,buffer,gc,px1,py1,px2,py2);
}

void AGUIX::DrawLine(Drawable buffer,GC tgc,int px1,int py1,int px2,int py2)
{
  if(tgc==0) DrawLine(buffer,px1,py1,px2,py2);
  else XDrawLine(dsp,buffer,tgc,px1,py1,px2,py2);
}

void AGUIX::DrawPoint(Drawable buffer,int px,int py)
{
  XDrawPoint(dsp,buffer,gc,px,py);
}

void AGUIX::DrawPoint(Drawable buffer,GC tgc,int px,int py)
{
  if(tgc==0) DrawPoint(buffer,px,py);
  else XDrawPoint(dsp,buffer,tgc,px,py);
}

bool AGUIX::ReactMessage(Message *msg,AWindow *parent)
{
  // als erstes das Fenster ausfindig machen
  // dann ReactMessage des Fenster aufrufen
  bool returnvalue=false;
  AWindow *win;
  if ( ( msg->type == ClientMessage ) &&
       ( msg->specialType == Message::NONE ) ) {
    return returnvalue;
  }
  int id;
  id=wins->initEnum();
  while ( msg->loop < 2 ) {
    win=(AWindow*)wins->getFirstElement(id);
    while((win!=NULL)&&(returnvalue==false)) {
      if((parent==NULL)||(msg->type==Expose)) {
	returnvalue = win->handleMessage( &LastEvent, msg );
      } else {
	if(parent->isParent(win->getWindow(),false)==true) {
	  returnvalue = win->handleMessage( &LastEvent, msg );
	}
      }
      // with lockElement don't quit this loop
      if ( msg->lockElement != NULL ) returnvalue = false;
      win=(AWindow*)wins->getNextElement(id);
    }
    
    if ( msg->lockElement != NULL ) {
      // there is a lock element
      // check whether the element accepted this msg
      // ack should be false only when the element doesn't exists anymore
      if ( msg->ack == true ) {
	// okay, element accepted msg
	break;
      } else {
	// lost element
	msgLockElement = NULL;
	msg->lockElement = NULL;
	fprintf( stderr, "Worker: msg lock element lost!\n" );
	// no repeat
      }
    } else {
      // no lock element => just quit loop
      break;
    }
    msg->loop++;
  }
  wins->closeEnum(id);
  return returnvalue;
}

void AGUIX::Flush()
{
  XFlush(dsp);
}

int AGUIX::getargc() const
{
  return argc;
}

char **AGUIX::getargv() const
{
  return argv;
}

void AGUIX::putAGMsg(AGMessage *msg)
{
  messages->addElement(msg);
}

AGMessage *AGUIX::getAGMsg()
{
  AGMessage *msg=(AGMessage*)messages->getFirstElement();
  messages->removeFirstElement();
  return msg;
}

void AGUIX::ReplyMessage(AGMessage *msg)
{
  AGUIX_freeAGMessage( msg );
}

void AGUIX::ClearWin(Window win)
{
  XClearWindow(dsp,win);
}

void AGUIX::SetWindowBG(Window win,int color)
{
  if(color<0) color=0;
  if(color<colors) {
    XSetWindowBackground(dsp,win,ColBuf[color]);
  } else {
    XSetWindowBackground(dsp,win,ColBuf[color-1]);
  }
}

void AGUIX::WindowtoBack(Window win)
{
  XLowerWindow(dsp,win);
}

void AGUIX::WindowtoFront(Window win)
{
  XRaiseWindow(dsp,win);
}

int AGUIX::changeColor(int index2,int red,int green,int blue)
{
  XColor col;
#ifdef DEBUG
  printf("changeColor:%d/%d\n",index2,colors);
#endif
  XFreeColors(dsp,cmap,ColBuf+index2,1,0);
  col.flags=DoRed|DoGreen|DoBlue;
  col.red=red*256;
  col.green=green*256;
  col.blue=blue*256;
  if(!XAllocColor(dsp,cmap,&col)) return -1;
  ColBuf[index2]=col.pixel;
  return 0;
}

AGUIXFont::AGUIXFont(AGUIX *parent)
{
  name=NULL;
  gc=0;
  font=NULL;
  aguix=parent;
}

AGUIXFont::~AGUIXFont()
{
  removeFont();
}

void AGUIXFont::removeFont()
{
  if(gc!=0) {
    XFreeGC(aguix->getDisplay(),gc);
    gc=0;
  }
  if(font!=NULL) {
    XFreeFont(aguix->getDisplay(),font);
    font=NULL;
  }
  if(name!=NULL) {
    _freesafe(name);
    name=NULL;
  }
}

int AGUIXFont::setFont( const char *fontname )
{
  Display *dsp=aguix->getDisplay();
  int scr=aguix->getScreen();
  unsigned long white=WhitePixel(dsp,scr);
  unsigned long black=BlackPixel(dsp,scr);
  gc=XCreateGC(dsp,RootWindow(dsp,scr),0,0);
  XGCValues gc_values;
  gc_values.graphics_exposures=False;
  gc_values.foreground=black;
  gc_values.background=white;
  gc_values.line_width=0;
  gc_values.cap_style=CapButt;
  font=XLoadQueryFont(dsp,fontname);
  if(font==NULL) {
    XFreeGC(dsp,gc);
    gc=0;
    return 1;
  }
  charHeight=font->ascent+font->descent;
#if 0
  charWidth=font->max_bounds.rbearing-font->min_bounds.lbearing;
#else
  charWidth=XTextWidth(font,"M",1);
#endif
  gc_values.font=font->fid;
  unsigned long gc_valuemask=GCGraphicsExposures|GCForeground|GCBackground|GCLineWidth|GCFont|GCCapStyle;
  XChangeGC(dsp,gc,gc_valuemask,&gc_values);
  if(name!=NULL) _freesafe(name);
  name=dupstring(fontname);
  return 0;
}

const char *AGUIXFont::getName() const
{
  return name;
}

GC AGUIXFont::getGC() const
{
  return gc;
}

int AGUIXFont::getCharWidth() const
{
  return charWidth;
}

int AGUIXFont::getCharHeight() const
{
  return charHeight;
}

AGUIXFont *AGUIX::getFont( const char *name )
{
  int id;
  AGUIXFont *tf;

  id = fonts->initEnum();
  tf = (AGUIXFont*)fonts->getFirstElement( id );
  while ( tf != NULL ) {
    if ( strcmp( tf->getName(), name ) == 0 ) break;
    tf = (AGUIXFont*)fonts->getNextElement( id );
  }
  fonts->closeEnum( id );

  if ( tf == NULL ) {
    tf = new AGUIXFont( this );
    if ( tf->setFont( name ) != 0 ) {
      delete tf;
      return NULL;
    }
    fonts->addElement( tf );
  }
  return tf;
}

int AGUIXFont::getBaseline() const
{
  return font->ascent;
}

void AGUIX::ExposeHandler(Message *msg)
{
  AWindow *win;
  int id;
  if(msg->window!=0) {
    id=wins->initEnum();
    win=(AWindow*)wins->getFirstElement(id);
    while(win!=NULL) {
      if(msg->type==Expose) {
        win->handleMessage(&LastEvent,msg);
      }
      win=(AWindow*)wins->getNextElement(id);
    }
    wins->closeEnum(id);
  }
}

AWindow *AGUIX::findAWindow(Window win)
{
  AWindow *awin;
  int id=wins->initEnum();
  awin=(AWindow*)wins->getFirstElement(id);
  while(awin!=NULL) {
    if(awin->isParent(win,true)==true) break;
    awin=(AWindow*)wins->getNextElement(id);
  }
  wins->closeEnum(id);
  return awin;
}

char *AGUIX::getNameOfKey(KeySym key,unsigned int mod) const
{
  char *tstr,*tstr2;
  tstr2=XKeysymToString(key); // Nicht freilassen
  if(tstr2==NULL) return NULL;
  tstr=dupstring(tstr2);
  if((mod&Mod1Mask)==Mod1Mask) {
    tstr2=(char*)_allocsafe(11+strlen(tstr)+2);
    sprintf(tstr2,"Left Alt + %s",tstr);
    _freesafe(tstr);
    tstr=tstr2;
  }
  if((mod&ControlMask)==ControlMask) {
    tstr2=(char*)_allocsafe(11+strlen(tstr)+2);
    sprintf(tstr2,"Control + %s",tstr);
    _freesafe(tstr);
    tstr=tstr2;
  }
  if((mod&ShiftMask)==ShiftMask) {
    tstr2=(char*)_allocsafe(11+strlen(tstr)+2);
    sprintf(tstr2,"Shift + %s",tstr);
    _freesafe(tstr);
    tstr=tstr2;
  }
  return tstr;
}

Window AGUIX::getGroupWin() const
{
  return groupwin;
}

void AGUIX::createGroupWin()
{
  XTextProperty windowname,iconname;
  Visual *vis=DefaultVisual(dsp,scr);
  unsigned long mask=CWEventMask;
  XSetWindowAttributes attr;
  char *tstr;
  attr.event_mask=0;
  groupwin=XCreateWindow(dsp,RootWindow(dsp,scr),0,0,10,10,0,getDepth(),InputOutput,vis,mask,&attr);
  if(!groupwin) {
    return;
  }
  XClassHint classhint;
  tstr=getClassname();
  classhint.res_name=tstr;
  classhint.res_class=tstr;
  XSetClassHint(dsp,groupwin,&classhint);
  XSetWMProtocols(dsp,groupwin,getCloseAtom(),1);
  XSizeHints *SizeHints;
  XWMHints *WMHints;
  SizeHints=XAllocSizeHints();
  WMHints=XAllocWMHints();
  SizeHints->flags=PSize;
  SizeHints->width=10;
  SizeHints->height=10;
  WMHints->input=True;

  WMHints->window_group=groupwin;

  WMHints->flags=InputHint|WindowGroupHint;
  XStringListToTextProperty(&tstr,1,&windowname);
  XStringListToTextProperty(&tstr,1,&iconname);
  XSetWMName(dsp,groupwin,&windowname);
  XSetWMIconName(dsp,groupwin,&iconname);
  XSetWMHints(dsp,groupwin,WMHints);
  XSetWMNormalHints(dsp,groupwin,SizeHints);
  XStoreName(dsp,groupwin,tstr);
  XFree(SizeHints);
  XFree(WMHints);
  _freesafe(tstr);
  XSetCommand(dsp,groupwin,getargv(),getargc());
  XFree( windowname.value );
  XFree( iconname.value );
}

void AGUIX::destroyGroupWin()
{
  XDestroyWindow(dsp,groupwin);
  groupwin=0;
}

int
AGUIX::queryPointer(Window win,int *x,int *y)
{
  Window usewin,root,child;
  int root_x,root_y;
  unsigned int keys_buttons;

  if ( ( x == NULL ) || ( y == NULL ) ) return -1;

  if(win==0) usewin=DefaultRootWindow(dsp);
  else usewin=win;
  XQueryPointer(dsp,usewin,&root,&child,&root_x,&root_y,x,y,&keys_buttons);
  return 0;
}

int
AGUIX::queryPointer(Window win,int *x,int *y,unsigned int *buttons)
{
  Window usewin,root,child;
  int root_x,root_y;

  if ( ( x == NULL ) || ( y == NULL ) ) return -1;

  if(win==0) usewin=DefaultRootWindow(dsp);
  else usewin=win;
  XQueryPointer(dsp,usewin,&root,&child,&root_x,&root_y,x,y,buttons);
  return 0;
}

int
AGUIX::queryRootPointer(int *x,int *y)
{
  Window usewin,root,child;
  int root_x,root_y;
  unsigned int keys_buttons;

  if ( ( x == NULL ) || ( y == NULL ) ) return -1;

  usewin=DefaultRootWindow(dsp);
  XQueryPointer(dsp,usewin,&root,&child,&root_x,&root_y,x,y,&keys_buttons);
  return 0;
}

int
AGUIX::setFont( const char *newfont )
{
  Flush();
  AGUIXFont *afont=getFont(newfont);
  if(afont==NULL) return 1;
  mainfont=afont;

  XGCValues gc_values;
  XFontStruct *xfont;
  xfont=mainfont->getFontStruct();
  
  CharHeight=mainfont->getCharHeight();
  CharWidth=mainfont->getCharWidth();
  gc_values.font=xfont->fid;
  unsigned long gc_valuemask=GCFont;
  XChangeGC(dsp,gc,gc_valuemask,&gc_values);
  Flush();
  return 0;
}

XFontStruct*
AGUIXFont::getFontStruct() const
{
  return font;
}

void
AGUIX::changeColormap()
{
  Colormap newcmap;
  int id;
  AWindow *win;
  if(privatecmap==false) {
#ifdef DEBUG
    printf("go to private cmap\n");
#endif
    newcmap=XCopyColormapAndFree(dsp,cmap);
    cmap=newcmap;
    privatecmap=true;
    id=wins->initEnum();
    win=(AWindow*)wins->getFirstElement(id);
    while(win!=NULL) {
      XSetWindowColormap(dsp,win->getWindow(),cmap);
      win=(AWindow*)wins->getNextElement(id);
    }
    wins->closeEnum(id);
  }
}

Colormap
AGUIX::getColormap() const
{
  return cmap;
}

void AGUIX::setCursor(Window win,int type)
{
  if((type>=WAIT_CURSOR)&&(type<MAXCURSORS)) {
    XDefineCursor(dsp,win,cursors[type]);
  }
}

void AGUIX::unsetCursor(Window win)
{
  XUndefineCursor(dsp,win);
}

int AGUIX::startCut(GUIElement *elem,const char *buffer)
{
  XSetSelectionOwner(dsp,XA_PRIMARY,elem->getWindow(),CurrentTime);
  if(elem->getWindow()==XGetSelectionOwner(dsp,XA_PRIMARY)) {
    cutstart=elem;
    if(cutbuffer!=NULL) _freesafe(cutbuffer);
    cutbuffer=dupstring(buffer);
  } else {
    cancelCut();
    return 1;
  }
  return 0;
}

void AGUIX::cancelCut()
{
  if(cutstart!=NULL) cutstart->cancelcut();
  cutstart=NULL;
  /*  if(cutbuffer!=NULL) _freesafe(cutbuffer);
      cutbuffer=NULL;*/
}

int AGUIX::startPaste(GUIElement *elem)
{
  pastestart=elem;
  return 0;
}

bool AGUIX::amiOwner() const
{
  if(cutstart!=NULL) return true;
  return false;
}

const char *AGUIX::getCutBuffer() const
{
  return cutbuffer;
}

void AGUIX::requestCut(Window win)
{
  Atom myproperty=XInternAtom(dsp,"Worker Paste Property",False);
  XConvertSelection(dsp,
		    XA_PRIMARY,
		    XA_STRING,
		    myproperty,
		    win,
		    CurrentTime);
}

void AGUIX::DrawTriangleFilled(Drawable buffer, int px1, int py1, int px2, int py2, int px3, int py3)
{
  DrawTriangleFilled( buffer, 0, px1, py1, px2, py2, px3, py3 );
}

void AGUIX::DrawTriangleFilled(Drawable buffer, GC tgc, int px1, int py1, int px2, int py2, int px3, int py3)
{
  GC usegc;
  XPoint points[3];

  if( tgc == 0 )
    usegc = gc;
  else
    usegc = tgc;

  points[0].x = px1;
  points[0].y = py1;
  points[1].x = px2;
  points[1].y = py2;
  points[2].x = px3;
  points[2].y = py3;

  XFillPolygon( dsp, buffer, usegc, points, 3, Convex, CoordModeOrigin );
  XDrawLines( dsp, buffer, usegc, points ,3, CoordModeOrigin );
}

void AGUIX::cancelCutPaste(GUIElement *elem)
{
  if(cutstart==elem) {
    cancelCut();
  }
  if(pastestart==elem) {
    pastestart=NULL;
  }
}

bool AGUIX::isDoubleClick(struct timeval *t1,struct timeval *t2) const
{
  int dt=0;
  int s,us;

  s = abs( (int)( t1->tv_sec - t2->tv_sec ) );
  if ( s > 2 ) return false;
  us=t1->tv_usec-t2->tv_usec;
  if(us<0) {
    us+=1000000;
    s--;
  }
  dt=us+s*1000000;
  if(dt<350000) return true;
  return false;
}

bool AGUIX::isDoubleClick( Time t1, Time t2) const
{
  if ( abs( (int)( t2 - t1 ) ) < 350 ) return true;
  return false;
}

#define ADJUST_SPACE()				\
{						\
  w1 = 0;					\
  for ( i = 0; i < nr; i++ ) {			\
    w1 += elems[i]->getWidth();			\
  }						\
  						\
  sp = wantedWidth - w1 - borderwidth * 2;	\
  						\
  last = borderwidth;				\
  for ( i = 0; i < nr; i++ ) {			\
    widths[i] = elems[i]->getWidth();		\
    xpos[i] = last;				\
    if ( i < ( nr - 1 ) )			\
      tsp = sp / ( nr - 1 - i );		\
    else					\
      tsp = 0;					\
    sp -= tsp;					\
    last += widths[i] + tsp;			\
  }						\
}

#define ARRANGE_MINSPACE()			\
{						\
  last = borderwidth;				\
  for ( i = 0; i < nr; i++ ) {			\
    widths[i] = elems[i]->getWidth();		\
    xpos[i] = last;				\
    last += widths[i];				\
    if ( minSpace > 0 ) last += minSpace;	\
  }						\
}

/*
 * resize and pos all elems to fit given width
 *
 * args:
 *   wantedWidth: width for resize or <1 for any
 *   borderwidth: left and right border
 *                <0 means 0
 *   minSpace: minimal space between elems
 *             <0 means any
 *   maxSpace: maximal space between elems
 *             <0 means any
 *   allowShrink: true when shrinking is allowed
 *   allowStretch: true when stretching is allowed
 *   elems: Array of elements
 *   minWidths: minimal widths of the elements
 *   nr: Arraysize
 *
 * returnvalue:
 *   used width
 * used width = wantedWidth is NOT guaranteed
 *
 */
int AGUIX::scaleElementsW( int wantedWidth,
			   int borderwidth,
			   int minSpace,
			   int maxSpace,
			   bool allowShrink,
			   bool allowStretch,
			   GUIElement **elems,
			   int *minWidths,
			   int nr )
{
  //  GUIElement **elems;
  int cw, i, last, nw;
  int *widths, *xpos;
  int w1, w2, sp, tsp;
  int usedWidth = -1;

  // for ,... notation
  // nr = 0;
  // elems = &elem;
  // while ( elems[nr] != NULL )
  //   nr++;

  if ( elems == NULL ) return -1;
  if ( nr < 1 ) return -1;
  if ( ( minSpace >= 0 ) && ( maxSpace >= 0 ) && ( minSpace > maxSpace ) ) return -1;

  widths = new int[nr];
  xpos = new int[nr];

  if ( borderwidth < 0 ) borderwidth = 0;

  // calc the currently needed width
  cw = 2 * borderwidth;
  if ( nr > 1 )
    cw += ( nr - 1 ) * ( ( minSpace < 0 ) ? 0 : minSpace );
  
  for ( i = 0; i < nr; i++ ) {
    cw += elems[i]->getWidth();
  }
  
  if ( nr == 1 ) {
    // special case
    // center the only element
    // 2 possibilities (when cw<wantedWidth):
    //   1.:stretch never
    //   2.:stretch when allowStretch == true
    // currently I use the second one
    if ( wantedWidth < 1 ) {
      // align to border
      widths[0] = elems[0]->getWidth();
      xpos[0] = borderwidth;
    } else {
      if ( cw < wantedWidth ) {
	// stretching needed
	if ( allowStretch == false ) {
	  // place it in the center
	  widths[0] = elems[0]->getWidth();
	  xpos[0] = ( wantedWidth / 2 ) - ( widths[0] / 2 );
	  
	  // we cannot calc the used width later so set it here
	  usedWidth = wantedWidth;
	} else {
	  widths[0] = wantedWidth - 2 * borderwidth;
	  if ( widths[0] < 4 ) widths[0] = 4;
	  xpos[0] = borderwidth;
	}
      } else if ( cw > wantedWidth ) {
	// shrinking needed
	if ( allowShrink == false ) {
	  // not allowed
	  widths[0] = elems[0]->getWidth();
	  xpos[0] = borderwidth;
	} else {
	  widths[0] = wantedWidth - 2 * borderwidth;
	  if ( widths[0] < 4 ) widths[0] = 4;
	  if ( minWidths != NULL ) {
	    if ( widths[0] < minWidths[0] )
	      widths[0] = minWidths[0];
	  }
	  xpos[0] = borderwidth;
	}
      } else {
	// no changes needed
	widths[0] = elems[0]->getWidth();
	xpos[0] = borderwidth;
      }
    }
  } else {
    if ( wantedWidth < 1 ) {
      // means any width so just arrange them with minSpace
      ARRANGE_MINSPACE();
    } else {
      // try to get the elems to fit wantedWidth
      // prio: 1.change space between elems
      //       2.always stretch or shrink, not both
      
      if ( cw == wantedWidth ) {
	// nothing to do
	last = borderwidth;
	for ( i = 0; i < nr; i++ ) {
	  widths[i] = elems[i]->getWidth();
	  xpos[i] = last;
	  last += widths[i];
	  if ( minSpace > 0 ) last += minSpace;
	}
      } else if ( cw < wantedWidth ) {
	// stretching needed
	if ( maxSpace < 0 ) {
	  // no maxSpace set so just adjust space
	  ADJUST_SPACE();
	} else {
	  // first test if we need to stretch when using maxSpace
	  
	  w1 = 2 * borderwidth;
	  if ( nr > 1 )
	    w1 += ( nr - 1 ) * maxSpace;
	  
	  for ( i = 0; i < nr; i++ ) {
	    w1 += elems[i]->getWidth();
	  }
	  // w1 is now the width when using maxSpace
	  
	  if ( w1 > wantedWidth ) {
	    // we can adjust the space
	    ADJUST_SPACE();
	  } else {
	    // maxSpace isn't enough
	    // we have to stretch some elems
	    if ( allowStretch == false ) {
	      // not allowed
	      // we cannot give wantedWidth so use maxSpace
	      last = borderwidth;
	      for ( i = 0; i < nr; i++ ) {
		widths[i] = elems[i]->getWidth();
		xpos[i] = last;
		last += widths[i] + maxSpace;
	      }
	    } else {
	      bool *finished, change;
	      int tnr;

              finished = new bool[nr];
	      w1 = wantedWidth - ( nr - 1 ) * maxSpace;
	      w1 -= 2 * borderwidth;
	      tnr = nr;

	      for ( i = 0; i < nr; i++ ) {
		widths[i] = elems[i]->getWidth();
		finished[i] = false;
	      }
	      do {
		w2 = w1 / tnr;
		change = false;
		for ( i = 0; i < nr; i++ ) {
		  if ( finished[i] == false ) {
		    if ( widths[i] > w2 ) {
		      finished[i] = true;
		      w1 -= widths[i];
		      tnr--;
		      change = true;
		    }
		  }
		}
	      } while ( change == true );
	      for ( i = 0; i < nr; i++ ) {
		if ( finished[i] == false ) {
		  w2 = w1 / tnr;
		  widths[i] = w2;
		  w1 -= w2;
		  tnr--;
		}
	      }
	      // now calc the xpos
	      last = borderwidth;
	      for ( i = 0; i < nr; i++ ) {
		xpos[i] = last;
		last += widths[i] + maxSpace;
	      }
              delete [] finished;
	    }
	  }
	}
      } else {
	// shrinking needed
	// works different from stretch-case
	// because we calced the width need when using minSpace ( or 0 when <0)
	// so in any case we have to shrink the elems
	// maxSpace doesn't count at all
	if ( allowShrink == false ) {
	  // nothing we can do to fit wantedWidth
	  // just arrange them with minSpace
	  ARRANGE_MINSPACE();
	} else {
	  // again only shrink elems which are not smaller then the needed average

	  bool *finished, change;
	  int tnr;
          
          finished = new bool[nr];
	  w1 = wantedWidth - ( nr - 1 ) * ( minSpace < 0 ? 0 : minSpace );
	  w1 -= 2 * borderwidth;
	  
	  tnr = nr;
	  
	  for ( i = 0; i < nr; i++ ) {
	    widths[i] = elems[i]->getWidth();
	    finished[i] = false;
	  }

	  do {
	    w2 = w1 / tnr;
	    change = false;
	    for ( i = 0; i < nr; i++ ) {
	      if ( finished[i] == false ) {
		if ( widths[i] < w2 ) {
		  finished[i] = true;
		  w1 -= widths[i];
		  tnr--;
		  change = true;
		}
	      }
	    }
	  } while ( change == true );
	  for ( i = 0; i < nr; i++ ) {
	    if ( finished[i] == false ) {
	      w2 = w1 / tnr;
	      widths[i] = w2;
	      w1 -= w2;
	      tnr--;
	    }
	  }

	  if ( minWidths != NULL ) {
	    // now check for minWidths
	    // this algo is stupid, I know, but because we only have few elements
	    // I think it's okay

	    bool *finished2;
	    int *mws;
	    int nf, dis, tnf, tdis, t;
	    
            finished2 = new bool[nr];
            mws = new int[nr];
	    for ( i = 0; i < nr; i++ ) {
	      mws[i] = minWidths[i];
	      if ( mws[i] < 4 )
		mws[i] = 4;
	      finished2[i] = false;
	    }
	    
	    nf = nr;
	    do {
	      dis = 0;
	      for ( i = 0; i < nr; i++ ) {
		if ( widths[i] < mws[i] ) {
		  dis += mws[i] - widths[i];
		  widths[i] = mws[i];
		  finished2[i] = true;
		  nf--;
		}
	      }
	      if ( dis > 0 ) {
		// now distribute dis at all elements finished == false
		tdis = dis;
		tnf = nf;
		for ( i = 0; i < nr; i++ ) {
		  if ( finished2[i] == false ) {
		    t = tdis / tnf;
		    widths[i] -= t;
		    tnf--;
		    tdis -= t;
		  }
		}
		// now dis is distributed to all widths
		// but this means that some new elements could be too small
	      }
	    } while ( dis > 0 );
	    // dis will only be > 0 is there are too small elements
	    // because in each loop atleast one will be corrected this will terminated after n loops
            delete [] finished2;
            delete [] mws;
	  }

	  // now calc the xpos
	  last = borderwidth;
	  for ( i = 0; i < nr; i++ ) {
	    xpos[i] = last;
	    last += widths[i];
	    if ( minSpace > 0 ) last += minSpace;
	  }
          delete [] finished;
	}
      }
    } 
  }
  
  // now arrange the elements
  for ( i = 0; i < nr; i++ ) {
    elems[i]->move( xpos[i], elems[i]->getY() );
    elems[i]->resize( widths[i], elems[i]->getHeight() );
  }
  
  delete [] widths;
  delete [] xpos;

  if ( usedWidth >= 0 ) {
    return usedWidth;
  } else {
    nw = elems[nr - 1]->getX() + elems[nr - 1]->getWidth() + borderwidth;
    return nw;
  }
}

#undef ADJUST_SPACE
#undef ARRANGE_MINSPACE

int AGUIX::centerElementsY( GUIElement *element,
			    GUIElement *center2element )
{
  int h1,h2;

  if ( ( element == NULL ) || ( center2element == NULL ) )
    return -1;

  h1 = element->getHeight();
  h2 = center2element->getHeight();

  element->move( element->getX(),
		 center2element->getY() +
		 h2 / 2 -
		 h1 / 2 );
  return 0;
}

int AGUIX::getRootWindowWidth() const
{
  return rootWindowWidth;
}

int AGUIX::getRootWindowHeight() const
{
  return rootWindowHeight;
}

Pixmap AGUIX::createPixmap( Drawable d, int width, int height )
{
  if ( ( width < 1 ) || ( height < 1 ) || ( d == 0 ) ) return 0;
  
  return XCreatePixmap( dsp, d, width, height, DefaultDepth( dsp, scr ) );
}

void AGUIX::freePixmap( Pixmap p )
{
  if ( p == 0 ) return;
  XFreePixmap ( dsp, p );
}

void AGUIX::DrawDottedLine( Drawable buffer, int px1, int py1, int px2, int py2 )
{
  XDrawLine( dsp, buffer, dotted_gc, px1, py1, px2, py2 );
}

void AGUIX::DrawDottedRectangle( Drawable buffer, int x, int y, int w, int h )
{
  if ( ( w < 1 ) || ( h < 1 ) ) return;
  DrawDottedLine( buffer, x, y, x + w - 1, y );
  DrawDottedLine( buffer, x + w - 1, y, x + w - 1, y + h - 1 );
  DrawDottedLine( buffer, x + w - 1, y + h - 1, x, y + h - 1 );
  DrawDottedLine( buffer, x, y + h - 1, x, y );
}

void AGUIX::setDottedFG( int color )
{
  setFG( dotted_gc, color );
}

void AGUIX::DrawDashXorLine( Drawable buffer, int px1, int py1, int px2, int py2 )
{
  XDrawLine( dsp, buffer, dashxor_gc, px1, py1, px2, py2 );
}

void AGUIX::DrawDashXorRectangle( Drawable buffer, int x, int y, int w, int h )
{
  if ( ( w < 1 ) || ( h < 1 ) ) return;
  DrawDashXorLine( buffer, x, y, x + w - 1, y );
  DrawDashXorLine( buffer, x + w - 1, y, x + w - 1, y + h - 1 );
  DrawDashXorLine( buffer, x + w - 1, y + h - 1, x, y + h - 1 );
  DrawDashXorLine( buffer, x, y + h - 1, x, y );
}

void AGUIX::DrawDashDLine( Drawable buffer, int px1, int py1, int px2, int py2 )
{
  XDrawLine( dsp, buffer, dashdouble_gc, px1, py1, px2, py2 );
}

void AGUIX::DrawDashDRectangle( Drawable buffer, int x, int y, int w, int h )
{
  if ( ( w < 1 ) || ( h < 1 ) ) return;
  DrawDashDLine( buffer, x, y, x + w - 1, y );
  DrawDashDLine( buffer, x + w - 1, y, x + w - 1, y + h - 1 );
  DrawDashDLine( buffer, x + w - 1, y + h - 1, x, y + h - 1 );
  DrawDashDLine( buffer, x, y + h - 1, x, y );
}

void AGUIX::setDashDFG( int color )
{
  setFG( dashdouble_gc, color );
}

void AGUIX::setDashDBG( int color )
{
  setBG( dashdouble_gc, color );
}

bool AGUIX::isModifier( KeySym key )
{
  if ( IsModifierKey( key ) == true ) return true;

  // some additional modifiers not in IsModifierKey
  switch ( key ) {
  case XK_Multi_key:
    return true;
    break;
  }
  return false;
}

char *AGUIX::getStringForKeySym( KeySym key )
{
  char *tstr;

  tstr = XKeysymToString( key );
  if ( tstr != NULL ) {
    return dupstring( tstr );
  }
  return NULL;
}

KeySym AGUIX::getKeySymForString( const char *str1 )
{
  if ( str1 == NULL ) return NoSymbol;

  return XStringToKeysym( str1 );
}

void AGUIX::rebuildBackgroundPixmap()
{
  if ( backpm != None ) XFreePixmap( dsp, backpm );
  backpm = XCreatePixmap( dsp, DefaultRootWindow( dsp ), 2, 2, getDepth() );
  setFG( 0 );
  FillRectangle( backpm, 0, 0, 2, 2 );
  setFG( 2 );
  DrawPoint( backpm, 0, 1 );
  DrawPoint( backpm, 1, 0 );
}

void AGUIX::setWindowBackgroundPixmap( Window win )
{
  if ( backpm == None ) {
    rebuildBackgroundPixmap();
  }
  XSetWindowBackgroundPixmap( dsp, win, backpm );
}

void AGUIX::doXMsgs( AWindow *parent, bool onlyexpose )
{
  doXMsgs( MES_GET, parent, onlyexpose );
}

void AGUIX::doXMsgs( int mode, AWindow *parent, bool onlyexpose )
{
  if ( ( mode != MES_GET ) && ( mode != MES_WAIT ) ) mode = MES_GET;
  while ( msgHandler( mode, parent, onlyexpose ) > 0 );
}

bool AGUIX::noMoreMessages() const
{
  if ( messages->size() == 0 ) return true;
  return false;
}

#ifdef USE_XIM
XIMStyle AGUIX::getXIMStyle() const
{
  return im_style;
}

XIM AGUIX::getXIM() const
{
  return inputmethod;
}

int AGUIX::openIM()
{
  XIMStyle supported_styles, style;
  XIMStyles *im_supported_styles;
  int i;
  int id;
  AWindow *awin;
  
  inputmethod = XOpenIM( dsp, NULL, NULL, NULL );
  if ( inputmethod == NULL ) {
    fprintf( stderr, "Worker Warning: Cannot open input method\n" );
#ifdef XIM_XREGISTER_OKAY
    XRegisterIMInstantiateCallback( dsp, NULL, NULL, NULL, AGUIX_im_inst_callback, (XPointer)this );
#endif
  } else {
#ifdef XIM_XREGISTER_OKAY
    XIMCallback destCB;

    destCB.callback = AGUIX_im_dest_callback;
    destCB.client_data = (XPointer)this;
    if ( XSetIMValues( inputmethod, XNDestroyCallback, &destCB, NULL ) != NULL ) {
      fprintf( stderr, "Worker Warning: Couldn't set IM destroy callback\n" );
    }
#endif
    
    /* I could also "support" XIM...None but with these styles dead keys
       will be completly ignored (at least at my machine)
       But my code will fallback to normal XLookupString when my wanted styles
       are not supported and dead keys will be loockuped to their normal character
    */
    supported_styles = XIMPreeditNothing | XIMStatusNothing;
    XGetIMValues( inputmethod, XNQueryInputStyle, &im_supported_styles, NULL );
    im_style = 0;
    for ( i = 0; i < im_supported_styles->count_styles; i++ ) {
      style = im_supported_styles->supported_styles[i];
      if ( ( style & supported_styles ) == style ) {
	im_style = ChooseBetterStyle( style, im_style );
      }
    }
    
    if ( im_style == 0 ) {
#ifdef DEBUG
      fprintf( stderr, "Worker Warning: Wanted inputstyle not available\n");
#endif
    } else {
      id = wins->initEnum();
      awin = (AWindow*)wins->getFirstElement( id );
      while ( awin != NULL ) {
	awin->createXIC();
	awin = (AWindow*)wins->getNextElement( id );
      }
      wins->closeEnum( id );
    }
    XFree( im_supported_styles );
  }
  return ( ( ( inputmethod != NULL ) && ( im_style != 0 ) ) ? 0 : 1 );
}

void AGUIX::closeIM()
{
  int id;
  AWindow *awin;

  id = wins->initEnum();
  awin = (AWindow*)wins->getFirstElement( id );
  while ( awin != NULL ) {
    awin->closeXIC();
    awin = (AWindow*)wins->getNextElement( id );
  }
  wins->closeEnum( id );

  im_style = 0;
  if ( inputmethod != NULL ) {
    XCloseIM( inputmethod );
    inputmethod = NULL;
  }
}

void AGUIX::IMInstCallback( Display *calldsp )
{
  if ( calldsp != dsp ) return; // not my display

#ifdef XIM_XREGISTER_OKAY
  XUnregisterIMInstantiateCallback( dsp, NULL, NULL, NULL, AGUIX_im_inst_callback, (XPointer)this );
#endif
  openIM();
}

void AGUIX::IMDestCallback()
{
  int id;
  AWindow *awin;

  id = wins->initEnum();
  awin = (AWindow*)wins->getFirstElement( id );
  while ( awin != NULL ) {
    awin->XICdestroyed();
    awin = (AWindow*)wins->getNextElement( id );
  }
  wins->closeEnum( id );

  im_style = 0;
  inputmethod = NULL;
}

#endif

/*
 * set transient window variable
 *
 * apply only when the window is registered
 * clear variable with NULL arg
 */
void AGUIX::setTransientWindow( AWindow *twin )
{
  if ( twin == NULL ) {
    transientwindow = NULL;
  } else {
    if ( wins->getIndex( twin ) >= 0 ) {
      // window found
      transientwindow = twin;
    }
  }
}

const AWindow* AGUIX::getTransientWindow() const
{
  return transientwindow;
}

int AGUIX::addDefaultColors()
{
  if ( AddColor( 160, 160, 160 ) < 0 ) return -1;
  if ( AddColor( 0, 0, 0 ) < 0 ) return -1;
  if ( AddColor( 255, 255, 255 ) < 0 ) return -1;
  if ( AddColor( 0, 85, 187) < 0 ) return -1;
  if ( AddColor( 204, 34, 0 ) < 0 ) return -1;
  if ( AddColor( 50, 180, 20 ) < 0 ) return -1;
  if ( AddColor( 119, 0, 119 ) < 0 ) return -1;
  return AddColor( 238, 170, 68 );
}

/*
 * the following two functions returns the last key/mouse and reset
 * the variables
 * the variables are set by doXMsg currently
 */
KeySym AGUIX::getLastKeyRelease()
{
  KeySym k;

  k = lastkeyrelease;
  lastkeyrelease = None;
  return k;
}

unsigned int AGUIX::getLastMouseRelease()
{
  unsigned int m;

  m = lastmouserelease;
  lastmouserelease = 0;
  return m;
}

void AGUIX::msgLock( GUIElement *e )
{
  msgLockElement = e;
}

void AGUIX::msgUnlock( const GUIElement *e )
{
  if ( msgLockElement != e ) {
    fprintf( stderr, "Worker:wrong msgUnlock\n" );
  }
  msgLockElement = NULL;
}

void AGUIX::enableTimer()
{
#ifdef USE_AGUIXTIMER
  struct itimerval itv;
#endif

  if ( timerEnabled == false ) {
    timerEnabled = true;
#ifndef USE_AGUIXTIMER
    gettimeofday( &timerStart, NULL );
    lastTimerEventNr = 0;
#else
    itv.it_value.tv_sec = itv.it_interval.tv_sec = 0;
    itv.it_value.tv_usec = itv.it_interval.tv_usec = 1000000 / TIMER_TICKS;
    setitimer( ITIMER_REAL, &itv, NULL );
    lastTimerEventNr = timerEvent;
#endif
  }
}

void AGUIX::disableTimer()
{
#ifdef USE_AGUIXTIMER
  struct itimerval itv;
#endif

  if ( timerEnabled == true ) {
#ifdef USE_AGUIXTIMER
    itv.it_value.tv_sec = itv.it_interval.tv_sec = 0;
    itv.it_value.tv_usec = itv.it_interval.tv_usec = 0;
    setitimer( ITIMER_REAL, &itv, NULL );
#endif
  }
  timerEnabled = false;
}
