/* wcfiletype.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2006-2007 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "wcfiletype.hh"
#include <aguix/aguix.h>
#include <aguix/util.h>
#include "wcglobalflags.hh"
#include "datei.h"
#include "functionproto.h"
#include "simplelist.hh"
 
WCFiletype::WCFiletype()
{
  int i;
  name=dupstring("");
  namelen=0;
  pattern=dupstring("");
  usepattern=false;
  usefiledesc=false;
  for(i=1;i<64;i++) filedesc[i]=-1;
  filedesc[0]=-2;
  dndaction=new List();
  doubleclickaction=new List();
  showaction=new List();
  rawshowaction=new List();
  for(i=0;i<10;i++) useraction[i]=new List();
  dndactiongf=new WCGlobalFlags();
  doubleclickactiongf=new WCGlobalFlags();
  showactiongf=new WCGlobalFlags();
  rawshowactiongf=new WCGlobalFlags();
  for(i=0;i<10;i++) useractiongf[i]=new WCGlobalFlags();
  internID = NORMALTYPE;

  patternIgnoreCase = false;
  patternUseRegExp = false;
  patternUseFullname = false;
#ifdef HAVE_REGEX
  patternRegCompiled = false;
#endif

  extCond = dupstring( "" );
  useExtCond = false;

  subtypes = NULL;
  parentType = NULL;

  colmode = DEFAULTCOL;
  for ( i = 0; i < 4; i++ ) {
    colfg[i] = 1;
    colbg[i] = 0;
  }
  colext = NULL;
}

int WCFiletype::getinternID() const
{
  return internID;
}

void WCFiletype::setinternID(int newid)
{
  if((newid<NORMALTYPE)||(newid>DIRTYPE)) return;
  internID=newid;
}

WCFiletype::~WCFiletype()
{
  int i;
  if(name!=NULL) _freesafe(name);

#ifdef HAVE_REGEX
  freeCompPatternReg();
#endif
  if(pattern!=NULL) _freesafe(pattern);
  freecoms(dndaction);
  freecoms(doubleclickaction);
  freecoms(showaction);
  freecoms(rawshowaction);
  for(i=0;i<10;i++) freecoms(useraction[i]);
  delete dndaction;
  delete doubleclickaction;
  delete showaction;
  delete rawshowaction;
  for(i=0;i<10;i++) delete useraction[i];
  delete dndactiongf;
  delete doubleclickactiongf;
  delete showactiongf;
  delete rawshowactiongf;
  for(i=0;i<10;i++) delete useractiongf[i];

  if ( extCond != NULL ) _freesafe( extCond );

  if ( subtypes != NULL ) {
    while ( subtypes->size() > 0 ) {
      WCFiletype *tt = subtypes->back();
      subtypes->pop_back();
      delete tt;
    }
    delete subtypes;
  }

  if ( colext != NULL ) _freesafe( colext );
}

WCFiletype *WCFiletype::duplicate() const
{
  int i;
  WCFiletype *tft=new WCFiletype();
  tft->setName(name);
  tft->setPattern(pattern);
  tft->setUsePattern(usepattern);
  tft->setFiledesc(filedesc);
  tft->setUseFiledesc(usefiledesc);
  tft->setDNDActions(dndaction);
  tft->setDoubleClickActions(doubleclickaction);
  tft->setShowActions(showaction);
  tft->setRawShowActions(rawshowaction);
  for(i=0;i<10;i++) tft->setUserActions(i,useraction[i]);
  tft->setDNDActionsGF(dndactiongf);
  tft->setDoubleClickActionsGF(doubleclickactiongf);
  tft->setShowActionsGF(showactiongf);
  tft->setRawShowActionsGF(rawshowactiongf);
  for(i=0;i<10;i++) tft->setUserActionsGF(i,useractiongf[i]);
  tft->setinternID(internID);
  tft->setPatternIgnoreCase( patternIgnoreCase );
  tft->setPatternUseRegExp( patternUseRegExp );
  tft->setPatternUseFullname( patternUseFullname );
  tft->setExtCond( extCond );
  tft->setUseExtCond( useExtCond );
  
  if ( subtypes != NULL ) {
    std::list<WCFiletype*>::iterator it1;
    
    for ( it1 = subtypes->begin(); it1 != subtypes->end(); it1++ ) {
      tft->addSubType( (*it1)->duplicate() );
    }
  }

  tft->setColorMode( colmode );
  tft->setCustomColor( 0, colfg );
  tft->setCustomColor( 1, colbg );
  tft->setColorExt( colext );

  return tft;
}

void WCFiletype::setName(const char *nname)
{
  if(name!=NULL) _freesafe(name);
  if(nname==NULL) {
    name=dupstring("");
    namelen=0;
  } else {
    name=dupstring(nname);
    namelen=strlen(name);
  }
}

void WCFiletype::setPattern(const char *npattern)
{
  if(pattern!=NULL) _freesafe(pattern);
#ifdef HAVE_REGEX
  freeCompPatternReg();
#endif
  if(npattern==NULL) {
    pattern=dupstring("");
  } else {
    pattern=dupstring(npattern);
  }
}

char *WCFiletype::getName()
{
  return name;
}

char *WCFiletype::getPattern()
{
  return pattern;
}

void WCFiletype::setUsePattern(bool nup)
{
  usepattern=nup;
}

bool WCFiletype::getUsePattern() const
{
  return usepattern;
}

void WCFiletype::setUseFiledesc(bool nufd)
{
  usefiledesc=nufd;
}

bool WCFiletype::getUseFiledesc() const
{
  return usefiledesc;
}

void WCFiletype::setFiledesc( const short *nfd)
{
  for(int i=0;i<64;i++) filedesc[i]=nfd[i];
}

short *WCFiletype::getFiledesc()
{
  return filedesc;
}

List *WCFiletype::getDNDActions()
{
  return dndaction;
}

List *WCFiletype::getDoubleClickActions()
{
  return doubleclickaction;
}

List *WCFiletype::getShowActions()
{
  return showaction;
}

List *WCFiletype::getRawShowActions()
{
  return rawshowaction;
}

List *WCFiletype::getUserActions(int id)
{
  if((id<0)||(id>9)) return NULL;
  return useraction[id];
}

void WCFiletype::setDNDActions(List *ndnda)
{
  freecoms(dndaction);
  FunctionProto *fp;
  int id=ndnda->initEnum();
  fp=(FunctionProto*)ndnda->getFirstElement(id);
  while(fp!=NULL) {
    dndaction->addElement(fp->duplicate());
    fp=(FunctionProto*)ndnda->getNextElement(id);
  }
  ndnda->closeEnum(id);
}

void WCFiletype::addComDNDAction( const FunctionProto *fp )
{
  if ( fp == NULL ) return;
  dndaction->addElement( fp->duplicate() );
}

void WCFiletype::setDoubleClickActions(List *ndca)
{
  freecoms(doubleclickaction);
  FunctionProto *fp;
  int id=ndca->initEnum();
  fp=(FunctionProto*)ndca->getFirstElement(id);
  while(fp!=NULL) {
    doubleclickaction->addElement(fp->duplicate());
    fp=(FunctionProto*)ndca->getNextElement(id);
  }
  ndca->closeEnum(id);
}

void WCFiletype::addComDoubleClickAction( const FunctionProto *fp )
{
  if ( fp == NULL ) return;
  doubleclickaction->addElement( fp->duplicate() );
}

void WCFiletype::setShowActions(List *nsa)
{
  freecoms(showaction);
  FunctionProto *fp;
  int id=nsa->initEnum();
  fp=(FunctionProto*)nsa->getFirstElement(id);
  while(fp!=NULL) {
    showaction->addElement(fp->duplicate());
    fp=(FunctionProto*)nsa->getNextElement(id);
  }
  nsa->closeEnum(id);
}

void WCFiletype::addComShowAction( const FunctionProto *fp )
{
  if ( fp == NULL ) return;
  showaction->addElement( fp->duplicate() );
}

void WCFiletype::setRawShowActions(List *nrsa)
{
  freecoms(rawshowaction);
  FunctionProto *fp;
  int id=nrsa->initEnum();
  fp=(FunctionProto*)nrsa->getFirstElement(id);
  while(fp!=NULL) {
    rawshowaction->addElement(fp->duplicate());
    fp=(FunctionProto*)nrsa->getNextElement(id);
  }
  nrsa->closeEnum(id);
}

void WCFiletype::addComRawShowAction( const FunctionProto *fp )
{
  if ( fp == NULL ) return;
  rawshowaction->addElement( fp->duplicate() );
}

void WCFiletype::setUserActions(int uid,List *nua)
{
  if((uid<0)||(uid>9)) return;
  freecoms(useraction[uid]);
  FunctionProto *fp;
  int id=nua->initEnum();
  fp=(FunctionProto*)nua->getFirstElement(id);
  while(fp!=NULL) {
    useraction[uid]->addElement(fp->duplicate());
    fp=(FunctionProto*)nua->getNextElement(id);
  }
  nua->closeEnum(id);
}

void WCFiletype::addComUserAction( int i, const FunctionProto * fp )
{
  if ( fp == NULL ) return;
  if ( ( i < 0 ) || ( i > 9 ) ) return;
  useraction[i]->addElement( fp->duplicate() );
}

WCGlobalFlags *WCFiletype::getDNDActionsGF()
{
  return dndactiongf;
}

WCGlobalFlags *WCFiletype::getDoubleClickActionsGF()
{
  return doubleclickactiongf;
}

WCGlobalFlags *WCFiletype::getShowActionsGF()
{
  return showactiongf;
}

WCGlobalFlags *WCFiletype::getRawShowActionsGF()
{
  return rawshowactiongf;
}

WCGlobalFlags *WCFiletype::getUserActionsGF(int index2)
{
  if((index2<0)||(index2>9)) return NULL;
  return useractiongf[index2];
}

void WCFiletype::setDNDActionsGF(WCGlobalFlags *nf)
{
  if(nf==NULL) dndactiongf->setFlags(0);
  else dndactiongf->setFlags(nf->getFlags());
}

void WCFiletype::setDoubleClickActionsGF(WCGlobalFlags *nf)
{
  if(nf==NULL) doubleclickactiongf->setFlags(0);
  else doubleclickactiongf->setFlags(nf->getFlags());
}

void WCFiletype::setShowActionsGF(WCGlobalFlags *nf)
{
  if(nf==NULL) showactiongf->setFlags(0);
  else showactiongf->setFlags(nf->getFlags());
}

void WCFiletype::setRawShowActionsGF(WCGlobalFlags *nf)
{
  if(nf==NULL) rawshowactiongf->setFlags(0);
  else rawshowactiongf->setFlags(nf->getFlags());
}

void WCFiletype::setUserActionsGF(int index2,WCGlobalFlags *nf)
{
  if((index2<0)||(index2>9)) return;
  if(nf==NULL) useractiongf[index2]->setFlags(0);
  else useractiongf[index2]->setFlags(nf->getFlags());
}

void WCFiletype::setPatternIgnoreCase( bool nv )
{
#ifdef HAVE_REGEX
  freeCompPatternReg();
#endif
  patternIgnoreCase = nv;
}

bool WCFiletype::getPatternIgnoreCase() const
{
  return patternIgnoreCase;
}

void WCFiletype::setPatternUseRegExp( bool nv )
{
  patternUseRegExp = nv;
}

bool WCFiletype::getPatternUseRegExp() const
{
  return patternUseRegExp;
}

#ifdef HAVE_REGEX
void WCFiletype::freeCompPatternReg()
{
  if ( patternRegCompiled == true ) {
    regfree( &patternReg );
    patternRegCompiled = false;
  }
}

void WCFiletype::compilePatternReg()
{
  int erg;

  if ( patternRegCompiled == false ) {
    erg = regcomp( &patternReg, pattern, REG_NOSUB | ( ( patternIgnoreCase == true ) ? REG_ICASE : 0 ) );
    if ( erg == 0 ) {
      patternRegCompiled = true;
    } else {
      // error
    }
  }
}
#endif

bool WCFiletype::patternMatchString( const char *mstr )
{
  bool matched = false;
#ifdef HAVE_REGEX
  int erg;
#endif
  char *lstr, *pstr;
  int i;

  if ( strlen( pattern ) < 1 ) return false;
  if ( mstr == NULL ) return false;
  if ( strlen( mstr ) < 1 ) return false;
  
#ifdef HAVE_REGEX
  if ( patternUseRegExp == false ) {
#else
  if ( 1 ) {
#endif
    lstr = dupstring( mstr );
    if ( patternIgnoreCase == true ) {
      for ( i = 0; lstr[i] != '\0'; i++ )
        lstr[i] = tolower( lstr[i] );
    }
    pstr = dupstring( pattern );
    if ( patternIgnoreCase == true ) {
      for ( i = 0; pstr[i] != '\0'; i++ )
        pstr[i] = tolower( pstr[i] );
    }
    
    if ( fnmatch( pstr, lstr, 0 ) == 0 ) matched = true;
    _freesafe( lstr );
    _freesafe( pstr );
  } else {
#ifdef HAVE_REGEX
    compilePatternReg();
    if ( patternRegCompiled == true ) {
      erg = regexec( &patternReg, mstr, 0, NULL, 0 );
      if ( erg == 0 ) matched = true;
    }
#endif
  }
  return matched;
}

int WCFiletype::sortfunction( void *p1, void *p2, int mode )
{
  WCFiletype *ft1, *ft2;
  int erg = 0;
  
  if ( ( p1 == NULL ) || ( p2 == NULL ) ) return 0;
  
  ft1 = (WCFiletype*)p1;
  ft2 = (WCFiletype*)p2;

  switch ( mode ) {
    default:
      erg = strcasecmp( ft1->getName(), ft2->getName() );
      break;
  }

  return erg;
}

void WCFiletype::clearFiledesc()
{
  int i;
  
  for ( i = 1; i < 64; i++ ) filedesc[i] = -1;
  filedesc[0] = -2;
}

void WCFiletype::setFiledesc( int pos, short v )
{
  int i;

  if ( ( pos < 0 ) || ( pos > 63 ) ) return;
  if ( ( v < -2 ) || ( v > 255 ) ) return;
  
  for ( i = 0; i < pos; i++ ) {
    if ( filedesc[i] == -2 ) filedesc[i] = -1;
  }
  filedesc[pos] = v;
  for ( i = 63; i >= 0; i-- ) {
    if ( filedesc[i] >= 0 ) {
      if ( i < 63 ) filedesc[i + 1] = -2;
      break;
    }
  }
  if ( i < 0 ) if ( filedesc[0] < 0 ) filedesc[0] = -2;
}

void WCFiletype::setPatternUseFullname( bool nv )
{
  patternUseFullname = nv;
}

bool WCFiletype::getPatternUseFullname() const
{
  return patternUseFullname;
}

void WCFiletype::setExtCond( const char *tstr )
{
  if ( extCond != NULL ) _freesafe( extCond );
  extCond = dupstring ( tstr );
}

void WCFiletype::setUseExtCond( bool nv )
{
  useExtCond = nv;
}

const char *WCFiletype::getExtCond() const
{
  return extCond;
}

bool WCFiletype::getUseExtCond() const
{
  return useExtCond;
}

void WCFiletype::addSubType( WCFiletype *newtype )
{
  if ( newtype == NULL ) return;
  if ( subtypes == NULL ) subtypes = new std::list<WCFiletype*>;
  subtypes->push_back( newtype );
  newtype->setParentType( this );
}

void WCFiletype::removeSubType( WCFiletype *remtype )
{
  std::list<WCFiletype*>::iterator it1;

  if ( remtype == NULL ) return;
  if ( subtypes == NULL ) return;
  for ( it1 = subtypes->begin(); it1 != subtypes->end(); it1++ ) {
    if ( *it1 == remtype ) {
      subtypes->erase( it1 );
      remtype->setParentType( NULL );
      break;
    }
  }
}

const std::list<WCFiletype*> *WCFiletype::getSubTypeList()
{
  return subtypes;
}

void WCFiletype::setParentType( WCFiletype *pt )
{
  parentType = pt;
}

WCFiletype *WCFiletype::getParentType() const
{
  return parentType;
}

std::vector<unsigned int> *WCFiletype::getTypePos()
{
  std::vector<unsigned int> *v;

  v = new std::vector<unsigned int>;
  if ( parentType != NULL ) parentType->getTypePos( this, v );
  return v;
}

void WCFiletype::getTypePos( const WCFiletype *child,
			     std::vector<unsigned int> *v )
{
  std::list<WCFiletype*>::iterator it1;
  unsigned int pos;
  
  if ( ( v == NULL ) || ( child == NULL ) ) return;

  for ( it1 = subtypes->begin(), pos = 0; it1 != subtypes->end(); it1++, pos++ ) {
    if ( *it1 == child ) {
      v->push_back( pos );
      break;
    }
  }
  if ( parentType != NULL ) parentType->getTypePos( this, v );
}

bool WCFiletype::sortfunctionType( WCFiletype *ft1, WCFiletype *ft2 ) {
  int erg = WCFiletype::sortfunction( ft1, ft2, 0 );
  if ( erg < 0 ) return true;
  return false;
}

void WCFiletype::sortSubTypeList()
{
  if ( subtypes == NULL ) return;

  subtypes->sort( sortfunctionType );
}

WCFiletype::colordef_t WCFiletype::getColorMode() const
{
  return colmode;
}

void WCFiletype::setColorMode( colordef_t newmode )
{
  colmode = newmode;
}

const int *WCFiletype::getCustomColor( int m ) const
{
  if ( m == 0 ) return colfg;
  else if ( m == 1 ) return colbg;
  return NULL;
}

void WCFiletype::setCustomColor( int m, const int *c )
{
  int i;

  if ( c == NULL ) return;
  if ( m == 0 ) {
    for ( i = 0; i < 4; i++ )
      colfg[i] = c[i];
  } else if ( m == 1 ) {
    for ( i = 0; i < 4; i++ )
      colbg[i] = c[i];
  }
}

void WCFiletype::setCustomColor( int m, int pos, int c )
{
  if ( m == 0 ) {
    if ( ( pos >= 0 ) && ( pos < 4 ) )
      colfg[pos] = c;
  } else if ( m == 1 ) {
    if ( ( pos >= 0 ) && ( pos < 4 ) )
      colbg[pos] = c;
  }
}

const char *WCFiletype::getColorExt() const
{
  return colext;
}

void WCFiletype::setColorExt( const char *s )
{
  if ( colext != NULL ) _freesafe( colext );
  
  if ( s == NULL ) colext = NULL;
  else colext = dupstring( s );
}

bool WCFiletype::save(Datei *fh) const
{
  int i;
  std::string str1;
  char buf[A_BYTESFORNUMBER(int)];
  if(fh==NULL) return false;

  fh->configPutPairString( "title", name );
  fh->configPutPairString( "pattern", pattern );
  fh->configPutPairBool( "usepattern", usepattern );
  fh->configPutPairBool( "usecontent", usefiledesc );
  
  fh->configOpenSection( "content" );
  for ( i = 0; i < 64; i++ ) {
    if ( filedesc[i] >= 0 ) {
      sprintf( buf, "%d", i );
      str1 = buf;
      str1 += " = ";
      sprintf( buf, "%d", filedesc[i] );
      str1 += buf;
      str1 += ";";
      fh->configPutInfo( str1.c_str(), false );
    }
  }
  fh->configCloseSection(); //content

  switch ( internID ) {
    case NOTYETTYPE:
      fh->configPutPair( "type", "notyetchecked" );
      break;
    case UNKNOWNTYPE:
      fh->configPutPair( "type", "unknown" );
      break;
    case VOIDTYPE:
      fh->configPutPair( "type", "noselect" );
      break;
    case DIRTYPE:
      fh->configPutPair( "type", "dir" );
      break;
    default:
      fh->configPutPair( "type", "normal" );
      break;
  }

  fh->configPutPairBool( "patternignorecase", patternIgnoreCase );
  fh->configPutPairBool( "patternuseregexp", patternUseRegExp );
  fh->configPutPairBool( "patternusefullname", patternUseFullname );

  fh->configPutPairString( "extcond", extCond );
  fh->configPutPairBool( "useextcond", useExtCond );
  //TODO: Aus Platzgruenden nur die nicht-leeren speichern?
  //      Aber was ist, wenn es mal Flags gibt und der User trotz
  //      leerer Liste (wo ja dann flags keinen Einflu haben)
  //      etwas auswhlt, sollte dann trotzdem gespeichert werden
  fh->configOpenSection( "ftcommands" );

  fh->configOpenSection( "dnd" );
  int id = dndaction->initEnum();
  FunctionProto *f1 = (FunctionProto*)dndaction->getFirstElement( id );
  while ( f1 != NULL ) {
    FunctionProto::presave( fh, f1 );
    f1 = (FunctionProto*)dndaction->getNextElement( id );
  }
  dndaction->closeEnum( id );
  fh->configOpenSection( "flags" );
  dndactiongf->save( fh );
  fh->configCloseSection(); //flags
  fh->configCloseSection(); //dnd

  fh->configOpenSection( "doubleclick" );
  id = doubleclickaction->initEnum();
  f1 = (FunctionProto*)doubleclickaction->getFirstElement( id );
  while ( f1 != NULL ) {
    FunctionProto::presave( fh, f1 );
    f1 = (FunctionProto*)doubleclickaction->getNextElement( id );
  }
  doubleclickaction->closeEnum( id );
  fh->configOpenSection( "flags" );
  doubleclickactiongf->save( fh );
  fh->configCloseSection(); //flags
  fh->configCloseSection(); //doubleclick

  fh->configOpenSection( "show" );
  id = showaction->initEnum();
  f1 = (FunctionProto*)showaction->getFirstElement( id );
  while ( f1 != NULL ) {
    FunctionProto::presave( fh, f1 );
    f1 = (FunctionProto*)showaction->getNextElement( id );
  }
  showaction->closeEnum( id );
  fh->configOpenSection( "flags" );
  showactiongf->save( fh );
  fh->configCloseSection(); //flags
  fh->configCloseSection(); //show

  fh->configOpenSection( "rawshow" );
  id = rawshowaction->initEnum();
  f1 = (FunctionProto*)rawshowaction->getFirstElement( id );
  while ( f1 != NULL ) {
    FunctionProto::presave( fh, f1 );
    f1 = (FunctionProto*)rawshowaction->getNextElement( id );
  }
  rawshowaction->closeEnum( id );
  fh->configOpenSection( "flags" );
  rawshowactiongf->save( fh );
  fh->configCloseSection(); //flags
  fh->configCloseSection(); //rawshow

  for ( i = 0; i < 10; i++ ) {
    sprintf( buf, "%d", i );
    str1 = "user ";
    str1 += buf;
    fh->configOpenSection( str1.c_str() );
    id = useraction[i]->initEnum();
    f1 = (FunctionProto*)useraction[i]->getFirstElement( id );
    while ( f1 != NULL ) {
      FunctionProto::presave( fh, f1 );
      f1 = (FunctionProto*)useraction[i]->getNextElement( id );
    }
    useraction[i]->closeEnum( id );
    fh->configOpenSection( "flags" );
    useractiongf[i]->save( fh );
    fh->configCloseSection(); //flags
    fh->configCloseSection(); //user x
  }

  fh->configCloseSection(); //ftcommands

  fh->configOpenSection( "subtype" );
  if ( subtypes != NULL ) {
    std::list<WCFiletype*>::iterator it1;
    
    for ( it1 = subtypes->begin(); it1 != subtypes->end(); it1++ ) {
      fh->configOpenSection( "filetype" );
      (*it1)->save( fh );
      fh->configCloseSection(); //filetype
    }
  }
  fh->configCloseSection(); //subtype

  switch ( colmode ) {
    case CUSTOMCOL:
      fh->configPutPair( "colormode", "custom" );
      break;
    case EXTERNALCOL:
      fh->configPutPair( "colormode", "extern" );
      break;
    case PARENTCOL:
      fh->configPutPair( "colormode", "parent" );
      break;
    default:
      fh->configPutPair( "colormode", "default" );
      break;
  }
  if ( colext != NULL ) fh->configPutPairString( "colorexternprog", colext );
  fh->configPutPairNum( "unselectcolor", colfg[0], colbg[0] );
  fh->configPutPairNum( "selectcolor", colfg[1], colbg[1] );
  fh->configPutPairNum( "unselectactivecolor", colfg[2], colbg[2] );
  fh->configPutPairNum( "selectactivecolor", colfg[3], colbg[3] );

  return true;
}

int WCFiletype::getNameLen() const
{
  return namelen;
}
