/* awindow.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2004 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: awindow.cc,v 1.56 2004/07/18 19:15:03 ralf Exp $ */

#include "awindow.h"

#include "text.h"
#include "button.h"

AWindow::AWindow(AGUIX *parent)
{
  this->aguix=parent;
  title=NULL;
  name=NULL;
  iconname=NULL;
  gui=new List();
  subwins=new List();
  subawins=new List();
  onScreen=false;
  parentawindow=NULL;
  focusOwner=NULL;
  border = 5;
  win = 0;
  created = false;
  focusElements = new List();
  doTabCycling = false;
  forbidinput = 0;
#ifdef USE_XIM
  inputcontext = NULL;
#endif
  req = NULL;
  mapNr = 0;
  container = NULL;
  contAutoResize = false;
  msgW = msgH = -1;
}

int AWindow::create(AWindow *parent,int tx,int ty,int width,int height,int tbg,const char *ttitle)
{
  Display *dsp=aguix->getDisplay();
  Window ParentWnd;
  XTextProperty windowname,ticonname;
  int scr=aguix->getScreen();

  if ( width < 1 ) width = 1;
  if ( height < 1 ) height = 1;

  parentawindow=parent;
  if(parent==NULL) {
    ParentWnd=RootWindow(dsp,scr);
  } else {
    ParentWnd=parent->win;
  }
  Visual *vis=DefaultVisual(dsp,scr);
  unsigned long mask=CWEventMask|CWBackPixel|CWColormap;
  XSetWindowAttributes attr;
  attr.event_mask = ExposureMask | ButtonPressMask | ButtonReleaseMask | StructureNotifyMask | KeyPressMask | KeyReleaseMask |
    ButtonMotionMask | PointerMotionHintMask | FocusChangeMask;
  attr.background_pixel=aguix->getPixel(tbg);
  attr.colormap=aguix->getColormap();
  win=XCreateWindow(dsp,ParentWnd,tx,ty,width,height,0,aguix->getDepth(),InputOutput,vis,mask,&attr);
  if(!win) {
    return 1;
  }

  this->name=dupstring(ttitle);
  this->title=dupstring(ttitle);
  this->iconname=dupstring(ttitle);

  XClassHint classhint;
  classhint.res_name=dupstring(ttitle);
  classhint.res_class=aguix->getClassname();
  XSetClassHint(dsp,win,&classhint);
  _freesafe(classhint.res_class);
  _freesafe(classhint.res_name);
  XSetWMProtocols(dsp,win,aguix->getCloseAtom(),1);
  XSizeHints *SizeHints;
  XWMHints *WMHints;
  SizeHints=XAllocSizeHints();
  WMHints=XAllocWMHints();
  SizeHints->flags=PSize;
  SizeHints->width=width;
  SizeHints->height=height;
  WMHints->input=True;

  WMHints->window_group=aguix->getGroupWin();

  WMHints->flags=InputHint|WindowGroupHint;
  XStringListToTextProperty(&title,1,&windowname);
  XStringListToTextProperty(&title,1,&ticonname);
  XSetWMName(dsp,win,&windowname);
  XSetWMIconName(dsp,win,&ticonname);
  XSetWMHints(dsp,win,WMHints);
  XSetWMNormalHints(dsp,win,SizeHints);
  XStoreName(dsp,win,ttitle);
  XFree(SizeHints);
  XFree(WMHints);
  XFree( windowname.value );
  XFree( ticonname.value );
  w=width;
  h=height;
  this->x=tx;
  this->y=ty;
  maxw=-1;
  maxh=-1;
  minw=-1;
  minh=-1;
  XClearWindow(dsp,win);
  XSetCommand(dsp,win,aguix->getargv(),aguix->getargc());
  /* jetzt this noch in aguix anhngen */
  aguix->insertWindow(this);
  if(parent!=NULL) {
    parent->addSubAWindow(this);
  }
  XFlush(dsp);
  this->bg=tbg;

  created = true;

#ifdef USE_XIM
  createXIC();
#endif
  return 0;
}

AGUIX *AWindow::getAGUIX() const
{
  return aguix;
}

void AWindow::sizeChanged(int width,int height)
{
#if 0
  if((width!=w)||(height!=h)) {
#endif
    AGMessage *agmsg = AGUIX_allocAGMessage();
    agmsg->type=AG_SIZECHANGED;
    agmsg->size.window=win;
    agmsg->size.neww=width;
    agmsg->size.newh=height;
    aguix->putAGMsg(agmsg);
    w=width;
    h=height;
    if ( ( container != NULL ) && ( contAutoResize == true ) ) {
      container->resize( w, h );
      container->rearrange();
    }
#if 0
  }
#endif
}

void AWindow::redraw()
{
  int id;

  if ( created == true ) {
    id = gui->initEnum();
    GUIElement *telem = (GUIElement*)gui->getFirstElement( id );
    while ( telem != NULL ) {
      telem->redraw();
      telem = (GUIElement*)gui->getNextElement( id );
    }
    gui->closeEnum( id );
    
    id = subawins->initEnum();
    AWindow *tawin = (AWindow*)subawins->getFirstElement( id );
    while ( tawin != NULL ) {
      tawin->redraw();
      tawin = (AWindow*)subawins->getNextElement( id );
    }
    subawins->closeEnum( id );
  }
}

Window AWindow::getWindow() const
{
  return win;
}

void AWindow::close()
{
  int id;

  if ( created == true ) {
    id = gui->initEnum();
    GUIElement *telem=(GUIElement*)gui->getFirstElement(id);
    while(telem!=NULL) {
      delete telem;  // destructor will remove this element from the list through remove()
      telem=(GUIElement*)gui->getFirstElement(id);
    }
    gui->closeEnum(id);
    
    // delete all subawindows
    id = subawins->initEnum();
    AWindow *tawin = (AWindow*)subawins->getFirstElement( id );
    while ( tawin != NULL ) {
      delete tawin;  /* notice that the destructor will call removeSubAWindow
		      * so it's not okay to remove this entry here */
      tawin = (AWindow*)subawins->getFirstElement( id );
    }
    subawins->closeEnum( id );

    // check for remaining subwins (not awins!)
    // this should never happen and isn't a real problem
    // but give a warning
    if ( subwins->size() > 0 ) {
      fprintf( stderr, "Worker Warning: there are still %d subwindows present for window %s!\n  Please inform the author!\n", subwins->size(), title );
    }

    Display *dsp=aguix->getDisplay();
    aguix->removeWindow( this );

#if WMDESTROYBUGFIX
    if ( parentawindow == NULL ) {
      aguix->Flush();
      XSync( dsp, False );
      waittime( 10 );
    }
#endif
#ifdef USE_XIM
    closeXIC();
#endif    
    if ( win != 0 ) {
      XDestroyWindow(dsp,win);
      win = 0;
    }
    aguix->Flush();
    if ( parentawindow != NULL ) {
      parentawindow->removeSubAWindow( this );
      parentawindow = NULL;
    }
  }

  onScreen = false;
  created = false;
  if ( title != NULL ) {
    _freesafe( title );
    title = NULL;
  }
  if ( name != NULL ) {
    _freesafe( name );
    name = NULL;
  }
  if ( iconname != NULL ) {
    _freesafe( iconname );
    iconname = NULL;
  }
  if ( container != NULL ) {
    delete container;
    container = NULL;
  }
}

AWindow::~AWindow()
{
  if ( req != NULL ) {
    delete req;
    req = NULL;
  }
  close();
  delete gui;
  delete subwins;
  delete subawins;
  delete focusElements;
}

int AWindow::getBG() const
{
  return bg;
}

void AWindow::setBG(int tbg)
{
  this->bg=tbg;
}

void AWindow::ReactMessage(Message *msg)
{
  if((msg->window)!=win) return;
  // diese Message interressiert uns
}

GUIElement *AWindow::add(GUIElement *new_element)
{
  if ( gui->getIndex( new_element ) < 0 ) {
    // only add and call setParent when not already in list
    gui->addElement(new_element);
    new_element->setParent(this);
  }
  return new_element;
}

void AWindow::setMaxSize(int mwidth,int mheight)
{
  XSizeHints *SizeHints;
  Display *dsp=aguix->getDisplay();
  SizeHints=XAllocSizeHints();
  if(minw>=0) {
    SizeHints->flags=PMaxSize|PMinSize;
    SizeHints->min_width=minw;
    SizeHints->min_height=minh;
  } else {
    SizeHints->flags=PMaxSize;
  }
  SizeHints->max_width=mwidth;
  SizeHints->max_height=mheight;
  maxw=mwidth;
  maxh=mheight;
  XSetWMNormalHints(dsp,win,SizeHints);
  XFree(SizeHints);
}

void AWindow::setMinSize(int mwidth,int mheight)
{
  XSizeHints *SizeHints;
  Display *dsp=aguix->getDisplay();
  SizeHints=XAllocSizeHints();
  if(maxw>=0) {
    SizeHints->flags=PMaxSize|PMinSize;
    SizeHints->max_width=maxw;
    SizeHints->max_height=maxh;
  } else {
    SizeHints->flags=PMinSize;
  }
  SizeHints->min_width=mwidth;
  SizeHints->min_height=mheight;
  minw=mwidth;
  minh=mheight;
  XSetWMNormalHints(dsp,win,SizeHints);
  XFree(SizeHints);
}

void AWindow::show()
{
  int cnr;

  if(onScreen==true) return;
  Display *dsp=aguix->getDisplay();

  cnr = getMapNr();
  XMapWindow(dsp,win);
  onScreen=true;
  while ( getMapNr() == cnr ) aguix->doXMsgs( MES_WAIT, NULL, false );
}

void AWindow::hide()
{
  if(onScreen==false) return;
  Display *dsp=aguix->getDisplay();
  XUnmapWindow(dsp,win);
  onScreen=false;
}

void AWindow::hide(Window subwin)
{
  if(hasSubWin(subwin)==true) {
    Display *dsp=aguix->getDisplay();
    XUnmapWindow(dsp,subwin);
  }
}

void AWindow::show(Window subwin)
{
  if(hasSubWin(subwin)==true) {
    Display *dsp=aguix->getDisplay();
    XMapWindow(dsp,subwin);
  }
}

const char *AWindow::getTitle() const
{
  return title;
}

void AWindow::setTitle(const char *newtitle)
{
  if(title!=NULL) _freesafe(title);
  title=dupstring(newtitle);
  Display *dsp=aguix->getDisplay();
  XClassHint classhint;
  classhint.res_name=dupstring(title);
  classhint.res_class=aguix->getClassname();
  XSetClassHint(dsp,win,&classhint);
  _freesafe(classhint.res_class);
  _freesafe(classhint.res_name);
  XTextProperty windowname,ticonname;
  XStringListToTextProperty(&title,1,&windowname);
  XStringListToTextProperty(&title,1,&ticonname);
  XSetWMName(dsp,win,&windowname);
  XSetWMIconName(dsp,win,&ticonname);
  XStoreName(dsp,win,title);
  XFree( windowname.value );
  XFree( ticonname.value );
}

void AWindow::resizeSubWin(Window twin,int tw,int th)
{
  Display *dsp=aguix->getDisplay();

  if ( tw < 1 ) tw = 1;
  if ( th < 1 ) th = 1;

  XResizeWindow(dsp,twin,tw,th);
}

void AWindow::moveSubWin(Window twin,int tx,int ty)
{
  Display *dsp=aguix->getDisplay();
  XMoveWindow(dsp,twin,tx,ty);
}

bool AWindow::handleMessage(XEvent *E,Message *msg)
{
  bool returnvalue=false;
  int res;
  bool conLock;

  // continue when we own lockElement
  conLock = false;
  if ( msg->lockElement != NULL ) {
    if ( contains( msg->lockElement ) == true ) conLock = true;
  }
  
  //check for some input unrelated events
  if ( ( ( msg->type == MapNotify ) &&
         ( msg->window == win ) ) ||
       ( ( msg->type == UnmapNotify ) &&
         ( msg->window == win ) ) ) {
    mapNr++;
  }

  // return if we don't handle messages
  if ( ( getForbidInput() == true ) &&
       ( msg->type != Expose ) &&
       ( msg->type != FocusIn ) &&
       ( msg->type != FocusOut ) ) {
    // but only without lockElement or we don't own it
    if ( conLock == false ) return returnvalue;
  }

  if ( msg->loop == 0 ) {
    // the following code has to be executed no matter
    // in which state the gui is so does it always but only the first loop
    switch ( msg->type ) {
      case FocusIn:
#ifdef USE_XIM
	if ( inputcontext != NULL ) {
	  if ( msg->window == win ) {
	    XSetICFocus( inputcontext );
	  }
	}
#endif      
	if ( ( getForbidInput() == true ) && ( conLock == false ) ) return returnvalue;
	break;
      case FocusOut:
#ifdef USE_XIM
	if ( inputcontext != NULL ) {
	  if ( msg->window == win ) {
	    XUnsetICFocus( inputcontext );
	  }
	}
#endif      
	if ( ( getForbidInput() == true ) && ( conLock == false ) ) return returnvalue;
	break;
    }
    
    if(msg->window==win) {
      if(msg->type==ConfigureNotify) {
        if ( ( msg->width != msgW ) || ( msg->height != msgH ) ) {
          msgW = msg->width;
          msgH = msg->height;
          sizeChanged( msg->width, msg->height );
	}
      }
    }
  }
  // Alle Element bekommen Nachricht
  int id=gui->initEnum();
  GUIElement *guie=(GUIElement*)gui->getFirstElement(id);
  res = 0;
  while((guie!=NULL)&&(returnvalue==false)) {
    res = guie->handleMessageLock( E, msg );
    if ( ( res == 1 ) && ( conLock == false ) ) returnvalue = true;
    guie=(GUIElement*)gui->getNextElement(id);
  }
  gui->closeEnum(id);

  if ( msg->lockElement == NULL ) {
    if ( doTabCycling == true ) {
      if ( isParent( msg->window, false ) == true ) {
	if ( msg->type == KeyPress ) {
	  int keystate = KEYSTATEMASK( msg->keystate );
	  if ( ( msg->key == XK_Tab ) && ( keystate == 0 ) ) {
	    nextFocus();
	  } else if ( ( ( msg->key == XK_ISO_Left_Tab ) ||
			( msg->key == XK_Tab ) ) && ( keystate == ShiftMask ) ) {
	    // my system sends ISO_Left_Tab when tab is pressed together with ShiftMask
	    // so I allow normal tab key and this variant
	    prevFocus();
	  }
	}
      }
    }
  }

  return returnvalue;
}

bool AWindow::hasSubWin(Window twin)
{
  int pos=subwins->getIndex((void*)twin);
  if(pos==-1) return false;
  else return true;
}

Window AWindow::getSubWindow(Window parent,int tx,int ty,int tw,int th)
{
  Window newwin,pwin;
  Display *dsp=aguix->getDisplay();
  int scr=aguix->getScreen();
  Visual *vis=DefaultVisual(dsp,scr);
  unsigned long mask=CWEventMask|CWBackPixel|CWColormap;

  if ( tw < 1 ) tw = 1;
  if ( th < 1 ) th = 1;
  
  XSetWindowAttributes attr;
  attr.event_mask=ExposureMask|ButtonPressMask|ButtonReleaseMask|StructureNotifyMask|KeyPressMask|KeyReleaseMask|ButtonMotionMask|EnterWindowMask|LeaveWindowMask|PointerMotionHintMask;
  attr.background_pixel=aguix->getPixel(bg);
  attr.colormap=aguix->getColormap();
  if(parent!=0) pwin=parent; else pwin=win;
  newwin=XCreateWindow(dsp,pwin,tx,ty,tw,th,0,aguix->getDepth(),InputOutput,vis,mask,&attr);
  if(!newwin) {
    return newwin;
  }
  XMapRaised(dsp,newwin);
  XClearWindow(dsp,newwin);
  subwins->addElement((void*)newwin);
  return newwin;
}

void AWindow::removeSubWin(Window subwin)
{
  if(hasSubWin(subwin)==true) {
    Display *dsp=aguix->getDisplay();
    XDestroyWindow(dsp,subwin);
    subwins->removeElement((void*)subwin);
  }
}

void AWindow::addSubAWindow(AWindow *child)
{
  if ( child == NULL ) return;
  subawins->addElement(child);
}

void AWindow::removeSubAWindow(AWindow *child)
{
  GUIElement *te;

  if ( child == NULL ) return;
  te = getFocusOwner();
  if ( te != NULL ) {
    // check if the child has the focusowner
    if ( child->contains( te ) == true ) {
      invalidFocus();
    }
  }
  // now remove all elements in child which accept a focus
  removeFocusElements( child );
  subawins->removeElement(child);
  if ( container != NULL ) removeFromContainer( child );
  if ( parentawindow != NULL ) parentawindow->removeFromContainer( child );
}

bool AWindow::isParent(Window childwin,bool direct)
{
  int id=subawins->initEnum();
  bool found=false;
  if(direct==false) {
    AWindow *tawin=(AWindow*)subawins->getFirstElement(id);
    while(tawin!=NULL) {
      //TODO: bis jetzt war das zweite Argument hier "true"
      // aber dies hat Probleme bereitet und macht eigentlich
      // auch keinen Sinn, deshalb jetzt false
      if ( tawin->isParent( childwin, false ) == true ) found = true;
      tawin=(AWindow*)subawins->getNextElement(id);
    }
  }
  if(hasSubWin(childwin)==true) found=true;
  if(childwin==win) found=true;
  subawins->closeEnum(id);
  return found;
}

bool AWindow::remove(GUIElement *elem)
{
  bool returnvalue=false;
  int pos=gui->getIndex(elem);
  if(pos>=0) {
    returnvalue=true;
    if ( isOwner( elem ) == true ) {
      invalidFocus();
    }
    gui->removeElementAt(pos);
    elem->setParent( NULL );
    if ( container != NULL ) container->remove( elem );
  }
  return returnvalue;
}

void AWindow::move(int nx,int ny)
{
  Display *dsp=aguix->getDisplay();
  XMoveWindow(dsp,win,nx,ny);
  x=nx;
  y=ny;
}

void AWindow::resize(int nw,int nh)
{
  int newminw, newminh, newmaxw, newmaxh;

  if ( ( nw > 0 ) && ( nh > 0 ) ) {
    if ( ( nw != w ) || ( nh != h ) ) {
      Display *dsp=aguix->getDisplay();

      // first check if we doesn't crash with the min/maxwidths
      // some WMs doesn't allow resize in this case (atleast KWM)
      if ( minw > 0 ) {
	newminw = ( nw < minw ) ? nw : minw;
	newminh = ( nh < minh ) ? nh : minh;
	if ( ( newminw != minw ) || ( newminh != minh ) ) {
	  setMinSize( newminw, newminh );
	}
      }
      if ( maxw > 0 ) {
	newmaxw = ( nw > maxw ) ? nw : maxw;
	newmaxh = ( nh > maxh ) ? nh : maxh;
	if ( ( newmaxw != maxw ) || ( newmaxh != maxh ) ) {
	  setMaxSize( newmaxw, newmaxh );
	}
      }

      XResizeWindow(dsp,win,nw,nh);
#if 1
      // it's possible to not call sizeChanged here
      // but the size has to be stored so if one use
      // this case, the test in sizeChanged for different size
      // has to be removed because it would be always true in the
      // manual resize case
      w=nw;
      h=nh;
#else
      sizeChanged( nw, nh );
#endif
    }
  }
}

GUIElement *AWindow::findGUIElement(Window child)
{
  int id;
  id=gui->initEnum();
  GUIElement *guiel=(GUIElement*)gui->getFirstElement(id);
  while(guiel!=NULL) {
    if(guiel->isParent(child)==true) break;
    guiel=(GUIElement*)gui->getNextElement(id);
  }
  gui->closeEnum(id);
  return guiel;
}

int AWindow::getWidth() const
{
  return w;
}

int AWindow::getHeight() const
{
  return h;
}

void AWindow::setCursor(int type)
{
  aguix->setCursor(win,type);
}

void AWindow::unsetCursor()
{
  aguix->unsetCursor(win);
}

void AWindow::invalidFocus()
{
  // we could had a focus owner before getting a parent awindow
  // so first check if we have not one
  if(parentawindow!=NULL) {
    // parent awindow so he should store the focus owner
    parentawindow->invalidFocus();
  }
  // if we have a focus owner, release it
  if(focusOwner!=NULL) {
    GUIElement *oldf=focusOwner;
    focusOwner=NULL; // to prevent leaveFokus() to recall this method
    oldf->leaveFocus();
  }
}

void AWindow::applyFocus(GUIElement *newfocus)
{
  //TODO: auf isOwner prfen damit nicht erst invalid gemacht wird?
  invalidFocus();
  if(newfocus!=NULL) {
    if(parentawindow!=NULL) {
      parentawindow->applyFocus(newfocus);
    } else {
      focusOwner=newfocus;
      newfocus->takeFocus();
    }
  }
}

bool AWindow::isOwner(GUIElement *f) const
{
  if(parentawindow!=NULL) {
    return parentawindow->isOwner(f);
  } else {
    if(focusOwner==f) return true;
  }
  return false;
}

void AWindow::maximizeX()
{
  int id;
  int mw,tw;

  mw = 0;
  id = gui->initEnum();
  GUIElement *guiel = (GUIElement*)gui->getFirstElement( id );
  while ( guiel != NULL ) {
    tw = guiel->getX() + guiel->getWidth() + border;
    if ( tw > mw )
      mw = tw;
    guiel = (GUIElement*)gui->getNextElement( id );
  }
  gui->closeEnum( id );

  id = subawins->initEnum();
  AWindow *swin = (AWindow*)subawins->getFirstElement( id );
  while ( swin != NULL ) {
    tw = swin->getX() + swin->getWidth() + border;
    if ( tw > mw )
      mw = tw;
    swin = (AWindow*)subawins->getNextElement( id );
  }
  subawins->closeEnum( id );
  
  if ( mw > w ) {
    resize( mw, h );
  }
}

void AWindow::maximizeY()
{
  int id;
  int mh,th;

  mh = 0;
  id = gui->initEnum();
  GUIElement *guiel = (GUIElement*)gui->getFirstElement( id );
  while ( guiel != NULL ) {
    th = guiel->getY() + guiel->getHeight() + border;
    if ( th > mh )
      mh = th;
    guiel = (GUIElement*)gui->getNextElement( id );
  }
  gui->closeEnum( id );
  
  id = subawins->initEnum();
  AWindow *swin = (AWindow*)subawins->getFirstElement( id );
  while ( swin != NULL ) {
    th = swin->getY() + swin->getHeight() + border;
    if ( th > mh )
      mh = th;
    swin = (AWindow*)subawins->getNextElement( id );
  }
  subawins->closeEnum( id );

  if ( mh > h ) {
    resize( w, mh );
  }
}

int AWindow::getBorderWidth() const
{
  return border;
}

void AWindow::setBorderWidth( int nv )
{
  if ( ( nv < 0 ) ||
       ( nv >= w/2 ) ) return;
  border = nv;
}

int AWindow::addTextFromString( const char *text,
				int tx,
				int ty,
				int vspace,
				Text ***return_texts,
				int *return_count,
				int *return_y )
{
  int i;
  int lines;
  char **liness;
  Text **text_elems;

  if ( text == NULL ) return 1;

  if ( tx < 0 ) tx = 0;
  if ( ty < 0 ) ty = 0;
  if ( vspace < 0 ) vspace = 0;

  lines = createLines( text, &liness );

  text_elems = (Text**)_allocsafe( sizeof( Text* ) * lines );

  for ( i = 0; i < lines; i++ ) {
    text_elems[i] = (Text*)add( new Text( aguix, tx, ty, liness[i], 1 ) );
    ty += text_elems[i]->getHeight() + vspace;
  }

  if ( return_texts != NULL ) {
    // caller want the Text list
    *return_texts = text_elems;
  } else _freesafe( text_elems );

  if ( return_count != NULL ) {
    *return_count = lines;
  }
  
  if ( return_y != NULL ) {
    // ty is not the real y value because vspace is always added
    *return_y = ty - vspace;
  }

  for ( i = 0; i < lines; i++ ) _freesafe( liness[i] );
  _freesafe( liness );

  return 0;
}

int AWindow::getX() const
{
  return x;
}

int AWindow::getY() const
{
  return y;
}

void AWindow::centerScreen()
{
  int rw, rh, nx, ny;

  rw = aguix->getRootWindowWidth();
  rh = aguix->getRootWindowHeight();
  if ( ( rw < 1 ) || ( rh < 1 ) ) return;
  
  nx = ( rw / 2 ) - ( w / 2 );
  ny = ( rh / 2 ) - ( h / 2 );
  move( nx, ny );
}

bool AWindow::contains( GUIElement *elem )
{
  bool returnvalue = false;
  int id;

  id = gui->initEnum();
  GUIElement *guie = (GUIElement*)gui->getFirstElement( id );
  while ( guie != NULL ) {
    if ( elem == guie ) {
      returnvalue = true;
      break;
    }
    guie = (GUIElement*)gui->getNextElement( id );
  }
  gui->closeEnum( id );
  
  if ( returnvalue == false ) {
    // check all subawindows
    id = subawins->initEnum();
    AWindow *tawin = (AWindow*)subawins->getFirstElement( id );
    while ( tawin != NULL ) {
      if ( tawin->contains( elem ) == true ) {
	returnvalue = true;
	break;
      }
      tawin = (AWindow*)subawins->getNextElement( id );
    }
    subawins->closeEnum( id );
  }

  return returnvalue;
}

GUIElement *AWindow::getFocusOwner() const
{
  if ( parentawindow != NULL )
    return parentawindow->getFocusOwner();
  else
    return focusOwner;
}

void AWindow::registerFocusElement( GUIElement *elem )
{
  if ( elem != NULL ) {
    if ( parentawindow != NULL ) {
      parentawindow->registerFocusElement( elem );
    } else {
      int id;
      GUIElement *te;
      
      id = focusElements->initEnum();
      te = (GUIElement*)focusElements->getFirstElement( id );
      while ( te != NULL ) {
	if ( te == elem ) break;
	te = (GUIElement*)focusElements->getNextElement( id );
      }
      focusElements->closeEnum( id );
      if ( te == NULL ) {
	// not already in list
	focusElements->addElement( elem );
      }
    }
  }
}

void AWindow::unregisterFocusElement( GUIElement *elem )
{
  if ( elem != NULL ) {
    if ( parentawindow != NULL ) {
      parentawindow->unregisterFocusElement( elem );
    } else {
      int id;
      GUIElement *te;
      
      id = focusElements->initEnum();
      te = (GUIElement*)focusElements->getFirstElement( id );
      while ( te != NULL ) {
	if ( te == elem ) break;
	te = (GUIElement*)focusElements->getNextElement( id );
      }
      focusElements->closeEnum( id );
      if ( te != NULL ) {
	// found
	focusElements->removeElement( elem );
      }
    }
  }
}

void AWindow::removeFocusElements( AWindow *child )
{
  if ( child != NULL ) {
    if ( parentawindow != NULL ) {
      parentawindow->removeFocusElements( child );
    } else {
      // check all elements in list whether they are in child
      int id, pos;
      GUIElement *te;
      
      id = focusElements->initEnum();
      te = (GUIElement*)focusElements->getFirstElement( id );
      pos = 0;
      while ( te != NULL ) {
	if ( child->contains( te ) == true ) {
	  focusElements->removeElementAt( pos );
	} else pos++;
	te = (GUIElement*)focusElements->getElementAt( pos );
      }
      focusElements->closeEnum( id );
    }
  }
}

void AWindow::nextFocus()
{
  int pos, opos;
  GUIElement *te;

  if ( parentawindow != NULL ) {
    parentawindow->nextFocus();
  } else {
    if ( focusElements->size() > 0 ) {
      if ( focusOwner != NULL ) {
	pos = focusElements->getIndex( focusOwner );
	if ( pos >= 0 ) {
	  pos++;
	  if ( pos >= focusElements->size() )
	    pos = 0;
	} else pos = 0;
      } else pos = 0;
      opos = pos;
      // now search for the next visible element starting at pos
      do {
	te = (GUIElement*)focusElements->getElementAt( pos );
	if ( te != NULL ) {
	  if ( te->isVisible() == true ) break;
	  else {
	    pos++;
	    if ( pos >= focusElements->size() )
	      pos = 0;
	  }
	} else break;
      } while ( pos != opos );
      if ( te != NULL ) {
	applyFocus( te );
      }
    }
  }
}

void AWindow::prevFocus()
{
  int pos, opos;
  GUIElement *te;

  if ( parentawindow != NULL ) {
    parentawindow->prevFocus();
  } else {
    if ( focusElements->size() > 0 ) {
      if ( focusOwner != NULL ) {
	pos = focusElements->getIndex( focusOwner );
	if ( pos >= 0 ) {
	  pos--;
	  if ( pos < 0 )
	    pos = focusElements->size() - 1;
	} else pos = 0;
      } else pos = 0;
      opos = pos;
      // now search for the prev visible element starting at pos
      do {
	te = (GUIElement*)focusElements->getElementAt( pos );
	if ( te != NULL ) {
	  if ( te->isVisible() == true ) break;
	  else {
	    pos--;
	    if ( pos < 0 )
	      pos = focusElements->size() - 1;
	  }
	} else break;
      } while ( pos != opos );
      if ( te != NULL ) {
	applyFocus( te );
      }
    }
  }
}

void AWindow::setDoTabCycling( bool nv )
{
  doTabCycling = nv;
}

bool AWindow::getDoTabCycling() const
{
  return doTabCycling;
}

bool AWindow::isTopParent( Window childwin )
{
  if ( parentawindow != NULL )
    return parentawindow->isTopParent( childwin );
  else
    return isParent( childwin, false );
}

bool AWindow::isVisible() const
{
  if ( onScreen == false ) return false;
  else {
    if ( parentawindow != NULL ) {
      if ( parentawindow->isVisible() == true ) return true;
      else return false;
    } else {
      return true;
    }
  }
}

void AWindow::useStippleBackground()
{
  aguix->setWindowBackgroundPixmap( win );
}

void AWindow::forbidUserInput()
{
  forbidinput++;
}

void AWindow::permitUserInput()
{
  forbidinput--;
  if ( forbidinput < 0 ) forbidinput = 0;
}

bool AWindow::getForbidInput() const
{
  if ( parentawindow != NULL )
    return parentawindow->getForbidInput();
  else
    return ( forbidinput == 0 ) ? false : true;
}

void AWindow::setTransientForAWindow( const AWindow *twin )
{
  if ( win == 0 ) return;
  if ( twin == NULL ) {
    XSetTransientForHint( aguix->getDisplay(), getWindow(), None );
  } else {
    XSetTransientForHint( aguix->getDisplay(), getWindow(), twin->getWindow() );
  }
}

#ifdef USE_XIM
XIC AWindow::getXIC() const
{
  return inputcontext;
}

int AWindow::createXIC()
{
  long im_event_mask;
  XWindowAttributes attr;

  if ( parentawindow != NULL ) {
    inputcontext = NULL;
    return 0;
  }

  if ( ( aguix->getXIM() != NULL ) && ( aguix->getXIMStyle() != 0 ) ) {
    aguix->Flush();
    inputcontext = XCreateIC( aguix->getXIM(),
			      XNInputStyle, aguix->getXIMStyle(),
			      XNClientWindow, win,
			      NULL );
    if ( inputcontext == NULL ) {
      fprintf( stderr, "Worker Warning: Cannot create input context\n");
    } else {
      XGetICValues( inputcontext, XNFilterEvents, &im_event_mask, NULL );
      XGetWindowAttributes( aguix->getDisplay(), win, &attr );
      XSelectInput( aguix->getDisplay(), win, attr.your_event_mask | im_event_mask );
    }
  } else {
    inputcontext = NULL;
  }
  return ( ( inputcontext != NULL ) ? 0 : 1 );
}

void AWindow::closeXIC()
{
  if ( inputcontext != NULL ) {
    XDestroyIC( inputcontext );
    inputcontext = NULL;
  }
}

void AWindow::XICdestroyed()
{
  inputcontext = NULL;
}

#endif

bool AWindow::isTopLevel() const
{
  return ( ( parentawindow == NULL ) ? true : false );
}

int AWindow::request( const char *reqtitle, const char *text, const char *buttons, Requester::request_flags_t flags )
{
  if ( req == NULL ) {
    req = new Requester( aguix, this );
  }
  return req->request( reqtitle, text, buttons, flags );
}

int AWindow::string_request( const char *reqtitle, const char *lines, const char *default_str, const char *buttons, char **return_str, Requester::request_flags_t flags )
{
  if ( req == NULL ) {
    req = new Requester( aguix, this );
  }
  return req->string_request( reqtitle, lines, default_str, buttons, return_str, flags );
}

int AWindow::getMapNr() const
{
  return mapNr;
}

AContainer *AWindow::setContainer( AContainer *newcont, bool autoResize )
{
  container = newcont;
  contAutoResize = autoResize;
  return newcont;
}

void AWindow::removeContainer( AContainer *cont )
{
  if ( container == cont ) {
    container = NULL;
  }
}

void AWindow::removeFromContainer( AWindow *twin )
{
  if ( container != NULL ) {
    container->remove( twin );
  }
}

void AWindow::contMaximize( bool applyMinSize, bool applyMaxSize )
{
  int tw, th;

  if ( container != NULL ) {
    // to maximize the window get the min width/height for
    // the container by resizing to 0,0 and finally add position
    container->resize( 0, 0 );
    container->rearrange();
    tw = container->getWidth();
    th = container->getHeight();
    tw += container->getX();
    th += container->getY();
    if ( applyMinSize == true ) setMinSize( tw, th );
    if ( applyMaxSize == true ) setMaxSize( tw, th );
    resize( tw, th );
  }
}
