/* WordPerfectCollector: Collects sections and runs of text from a 
 * wordperfect file (and styles to go along with them) and writes them 
 * to a target file
 *
 * Copyright (C) 2002-2003 William Lachance (william.lachance@sympatico.ca)
 * Copyright (C) 2003 Net Integration Technologies (http://www.net-itech.com)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * For further information visit http://libwpd.sourceforge.net
 *
 */

/* "This product is not manufactured, approved, or supported by 
 * Corel Corporation or Corel Corporation Limited."
 */

#ifndef _WORDPERFECTCOLLECTOR_H
#define _WORDPERFECTCOLLECTOR_H
//#include "TextRunStyle.h"
#include "SectionStyle.h"

#include <libwpd/libwpd.h>
#include <glib.h>
#include <gsf/gsf-utils.h>
#include <gsf/gsf-output-stdio.h>
#include <gsf/gsf-outfile.h>
#include <gsf/gsf-input.h>
#include <vector>
#include <map>
#include <stack>
using namespace std;

class DocumentElement;
class TagOpenElement;
class FontStyle;
class ListStyle;

class ParagraphStyle;
class TableStyle;
class TextRunStyle;
class PageSpan;

// the state we use for writing the final document
typedef struct _WriterDocumentState WriterDocumentState;
struct _WriterDocumentState
{
	bool mbFirstElement;
	
	stack<int> mFakeSectionStack;
	bool mbListElementOpenedAtCurrentLevel;

	bool mbTableCellOpened;
};

enum WriterListType { unordered, ordered };

struct ltstr
{
  bool operator()(const UTF8String & s1, const UTF8String & s2) const
  {
    return strcmp(s1.getUTF8(), s2.getUTF8()) < 0;
  }
};

class WordPerfectCollector : public WPXHLListenerImpl
{
public:
	WordPerfectCollector();
	virtual ~WordPerfectCollector() {}
	bool filter(GsfInput *pInput, GsfOutfile *pOutfile);

	virtual void setDocumentMetaData(const UCSString &author, const UCSString &subject,
					 const UCSString &publisher, const UCSString &category,
					 const UCSString &keywords, const UCSString &language,
					 const UCSString &abstract, const UCSString &descriptiveName,
					 const UCSString &descriptiveType) {}

	virtual void startDocument() {}
	virtual void endDocument() {}

	virtual void openPageSpan(const int span, const bool isLastPageSpan,
				  const float marginLeft, const float marginRight,
				  const float marginTop, const float marginBottom);
	virtual void closePageSpan() {}
	virtual void openHeaderFooter(const WPXHeaderFooterType headerFooterType, const WPXHeaderFooterOccurence headerFooterOccurence);
	virtual void closeHeaderFooter(const WPXHeaderFooterType headerFooterType, const WPXHeaderFooterOccurence headerFooterOccurence);

	virtual void openSection(const unsigned int numColumns, const float spaceAfter);
	virtual void closeSection();
	virtual void openParagraph(const guint8 paragraphJustification, const guint32 textAttributeBits,
				   const float marginLeftOffset, const float marginRightOffset,
				   const gchar *fontName, const float fontSize, 
				   const float lineSpacing,
				   const bool isColumnBreak, const bool isPageBreak);
	virtual void closeParagraph();
	virtual void openSpan(const guint32 textAttributeBits, const gchar *fontName, const float fontSize);
	virtual void closeSpan();

	virtual void insertTab();
	virtual void insertText(const UCSString &text);
	virtual void insertLineBreak();

	virtual void defineOrderedListLevel(const int listID, const guint16 listLevel, const WPXNumberingType listType, 
					    const UCSString &textBeforeNumber, const UCSString &textAfterNumber,
					    const int startingNumber);
	virtual void defineUnorderedListLevel(const int listID, const guint16 listLevel, const UCSString &bullet);
	virtual void openOrderedListLevel(const int listID);
	virtual void openUnorderedListLevel(const int listID);
	virtual void closeOrderedListLevel();
	virtual void closeUnorderedListLevel();
	virtual void openListElement(const guint8 paragraphJustification, const guint32 textAttributeBits,
				     const float marginLeftOffset, const float marginRightOffset,
				     const gchar *fontName, const float fontSize, 
				     const float lineSpacing);
	virtual void closeListElement();

	virtual void openFootnote(int number);
	virtual void closeFootnote();
	virtual void openEndnote(int number);
	virtual void closeEndnote();

 	virtual void openTable(const guint8 tablePositionBits, const float marginLeftOffset, const float marginRightOffset,
			       const float leftOffset, const vector < WPXColumnDefinition > &columns);
	virtual void openTableRow(); 
	virtual void closeTableRow();
 	virtual void openTableCell(const guint32 col, const guint32 row, const guint32 colSpan, const guint32 rowSpan, 
				   const guint8 borderBits, const RGBSColor * cellFgColor, const RGBSColor * cellBgColor);
	virtual void closeTableCell();
	virtual void insertCoveredTableCell(const guint32 col, const guint32 row);
	virtual void closeTable();

protected:	
	void _resetDocumentState();
	bool _parseSourceDocument(GsfInput *pInput);
	bool _writeTargetDocument(GsfOutfile *pOutfile);
	// _requestParagraphStyle: returns a text run style, if it already exists. creates it, adds it 
	// to the list of defined styles, and returns it otherwise.
	ParagraphStyle * _requestParagraphStyle(const guint8 iParagraphJustification, const guint32 iTextAttributeBits,
						const float fMarginLeftOffset, const float fMarginRightOffset,
						const gchar *pFontName, const float fFontSize, 
						const float fLineSpacing,
						const bool bColumnBreak, const bool bPageBreak,
						const gchar *pParentName = NULL, const gchar *pName = NULL);
	ParagraphStyle * _requestListParagraphStyle(const ListStyle * pListStyle, const guint8 iParagraphJustification, 
						    const guint32 iTextAttributeBits,
						    const float fMarginLeftOffset, const float fMarginRightOffset,
						    const gchar *pFontName, const float fFontSize, 
						    const float fLineSpacing);
	void _writeContentPreamble(GsfOutput *pOutput);
	void _writeDefaultStyles(GsfOutput *pOutput);
	void _writeMasterPages(GsfOutput *pOutput);
	void _writePageMasters(GsfOutput *pOutput);
	void _allocateFontName(const UTF8String &);
private:
	void _openListLevel(TagOpenElement *pListLevelOpenElement);
	void _closeListLevel(const char *szListType);
	
	WriterDocumentState mWriterDocumentState;

	// paragraph + span styles
	map<UTF8String, TextRunStyle *, ltstr> mParagraphStyleHash;
	guint miNumStyles;
	
	// font styles
	map<UTF8String, FontStyle *, ltstr> mFontHash;

	// section styles
	guint miNumSections;
	vector<SectionStyle *> mSectionStyles;

	// table styles
	guint miNumTables;
	vector<TableStyle *> mTableStyles;

	// list styles
	guint miNumListStyles;

	// style elements
	vector<DocumentElement *> mStylesElements;
	// content elements
	vector<DocumentElement *> mBodyElements;
	// the current set of elements that we're writing to
	vector<DocumentElement *> * mpCurrentContentElements;

	// page state
	vector<PageSpan *> mPageSpans;
	PageSpan *mpCurrentPageSpan;
	int miNumPageStyles;

	// section state
	int miCurrentNumColumns;
	float mfSectionSpaceAfter;

	// list styles / state
	ListStyle *mpCurrentListStyle;
	guint miCurrentListLevel;
	guint miLastListLevel;
	guint miLastListNumber;
	vector<ListStyle *> mListStyles;
	bool mbListContinueNumbering;
	bool mbListElementOpened;
	bool mbListElementParagraphOpened;

	// table state
	TableStyle *mpCurrentTableStyle; 
	bool mbTableFirstRow;       	
};
#endif
