/* Copyright 2001 Matt Flax <flatmax@ieee.org>
   This file is part of MFFM Time Scale Modification for Audio.

   MFFM Time Scale Modification for Audio is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   MFFM Time Scale Modification for Audio is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You have received a copy of the GNU General Public License
   along with MFFM Time Scale Modification for Audio
 */

/* This test file checks the embedded operation of WSOLA.
   If you are only interested in working with files then it is simpler to use
   the standard WSOLA test file as a guide.
   If on the other hand you are interested in streaming to or from network,
   audio hardware or anything else which is not a file, then this is the start
   point for you !
 */

#include <stdlib.h>

#include <mffm/libSndFileWrapper.H>

#include "WSOLA.H"

void exitRoutine(void){std::cout<<std::endl;}

int main(int argc, char *argv[]){

  atexit(exitRoutine);

  std::cout<<"WSOLA version "<<VERSION_NUM<<"\nAuthor : mffm <flatmax@ieee.org>"<<std::endl;

  if (argc<4){
    std::cerr<<"Useage :\n"<<argv[0]<<" inputFile outputFile factor"<<std::endl;
    exit(-1);
  }

  double tau=atof(argv[3]);
  std::cout<<"Factor="<<tau<<std::endl;

  int cnt, fs, channels;


  // These are the input and output buffers to use
  ATYPE *input, *output;

  //open the input file
  SF_INFO sfInfoIn;
  sfInfoIn.format=0;
  SNDFILE* inputFile=sf_open(argv[1], SFM_READ,&sfInfoIn);
  if (inputFile==NULL){
    perror("couldn't open input file\n");
    sf_perror(inputFile);
    exit(-1);
  }

  channels=sfInfoIn.channels;
  fs=sfInfoIn.samplerate;
  if (channels<1){
    std::cerr<<argv[0]<<":: input channels must be more then 1"<<std::endl;
    closeSndFile(inputFile); //Make sure you close the input file
    exit(-3);
  }

  //open the output file
  SF_INFO sfInfoOut;
  sfInfoOut.format=0;
  SNDFILE* outputFile=openWavWrite(&sfInfoOut, argv[2], fs, channels, 16);
  if (outputFile==NULL){
    closeSndFile(inputFile); //Make sure you close the input file
    cout<<"Couldn't open the output file "<<argv[2]<<" aborting"<<endl;
    return -2;
  }

  std::cout<<"Processing for "<<channels<<" channels at "<<fs<<" Hz"<<std::endl;

  int windowSize=channels*HANNING_LENGTH(fs);

  cout<<"windowSize="<<windowSize<<endl;

  // Make sure that the half window size is an integer number of channels
  while (remainder((double)windowSize/2.0/(double)channels, floor((double)windowSize/2.0/(double)channels)) != 0.0){
  cout<<"windowSize/2/channels "<<(double)windowSize/2.0/(double)channels<<endl;
  cout<<"remainder "<<remainder((double)windowSize/2.0/(double)channels, floor((double)windowSize/2.0/(double)channels))<<endl;
    windowSize++;
  }
  cout<<"windowSize="<<windowSize<<endl;

  // Embedded version should have only memory addressing between itself and the
  // WSOLA algorithm ... there should be input and output streams (memory,
  // network, or device addressing) ... for testing we will simply read and
  // spit to and from file.

  WSOLA wsola(windowSize, fs, channels);//Version 4 instantiation
  
  //Reserve local memory stores
  input = new ATYPE[wsola.getSourceLength()]; //The input store
  if (input==NULL){
    cout<<"error defining input memory"<<endl;
    closeSndFile(inputFile); //Make sure you close the input/output devices
    closeSndFile(outputFile);
    return -1;
  }
  output = new ATYPE[windowSize]; //The output store
  if (output==NULL){
    cout<<"error defining output memory"<<endl;
    delete [] input;
    closeSndFile(inputFile); //Make sure you close the input/output devices
    closeSndFile(outputFile);
    return -2;
  }
  //Load the required audio from your device (for now I use a file)
  int ret=1; //return variable >=0 for ok operation
  
  
  //Init with the tau value for the first window
  //Load required input to the local memory buffer from device
  if (readNWav(wsola.getSourceIndex()/channels, wsola.getSourceLength()/channels, channels, inputFile, input)!=wsola.getSourceLength()/channels){
    ret=WSOLA::INPUT_READ2SF_ERR;
    cout<<'\n';
    cout<<"error reading from input file to source frame.."<<endl;
    sf_perror(inputFile);
    ret=WSOLA::INPUT_READ2SF_ERR;
  }
  wsola.initProcess(input, tau);
  cout<<"about to process"<<endl;

  while (ret>=0){
    //Load required input to the local memory buffer from device
    if (readNWav(wsola.getSourceIndex()/channels, wsola.getSourceLength()/channels, channels, inputFile, input)!=wsola.getSourceLength()/channels){
      ret=WSOLA::INPUT_READ2SF_ERR;
      cout<<'\n';
      cout<<"error reading from input file to source frame.."<<endl;
      sf_perror(inputFile);
      break;
    }
    //Shift memory from local buffer to WSOL
    wsola.loadSourceInput(input);

    if (readNWav(wsola.getDesiredIndex()/channels, wsola.getDesiredLength()/channels, channels, inputFile, input)!=wsola.getDesiredLength()/channels){
      ret=WSOLA::INPUT_READ2SF_ERR;
      cout<<'\n';
      cout<<"error reading from input file to source frame.."<<endl;
      sf_perror(inputFile);
      break;
    }
    //Shift memory from local buffer to WSOLA
    wsola.loadDesiredInput(input);

    //Process outputting to the local memory buffer 
    ret=wsola.processFrame(output, tau); //process halfWindow
    if (ret<0)
      break;

    //Pipe the output out to device
    int written;
    if ((written=writeNSndFile(outputFile, (int)((double)windowSize/2.0), output))!=(int)((double)windowSize/2.0)){
      cout<<"WSOLATest.embedded:: error writing to output file. Wanted to write "<<(int)((double)windowSize/2.0)<<" but wrote "<<written <<" instead"<<endl;
      sf_perror(outputFile);
      ret=WSOLA::WRITE_ERR;
      break;
    }
  }

  if (ret==WSOLA::INPUT_READ2DF_ERR)
    cout<<"Destination frame read error, most probably using a factor >=1.0 and at the end of the input file"<<endl;
  else if (ret==WSOLA::INPUT_READ2SF_ERR)
    cout<<"Source frame read error, most probably using a factor <1.0, and at the end of the input file"<<endl;
  else if (ret==WSOLA::WRITE_ERR)
    cout<<"Is your disk full, we encountered a write error"<<endl;
  else if (ret==WSOLA::PROCFFT_ERR)
    cout<<"Please contact the author - this error shouldn't occur."<<endl;
  else if (ret==WSOLA::DEFAULT_ERR)
    cout<<"Un-known default error encountered"<<endl;
  else //Handle other errors here
    cout<<"Unknown error - unreliable failure"<<endl;

  closeSndFile(inputFile); //Make sure you close the files
  closeSndFile(outputFile);

  delete [] input;
  delete [] output;

  return 1;
}
