/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "IFFT.h"
#include "Exception.h"

#include <wx/object.h>		// Not sure why !
#include <wx/debug.h>

CIFFT::CIFFT(int len) :
m_fftLen(len),
m_plan(NULL),
m_in(NULL),
m_out(NULL)
{
	wxASSERT(m_fftLen > 0);

	m_in   = (fftw_complex*)::fftw_malloc(m_fftLen * sizeof(fftw_complex));
	m_out  = (fftw_complex*)::fftw_malloc(m_fftLen * sizeof(fftw_complex));

	if (m_in == NULL || m_out == NULL)
		throw CException(wxT("Cannot allocate IFFT arrays"));

#if defined(HAVE_FFTW3)
	m_plan = ::fftw_plan_dft_1d(m_fftLen, m_in, m_out, FFTW_BACKWARD, FFTW_ESTIMATE);
#else
	m_plan = ::fftw_create_plan(m_fftLen, FFTW_BACKWARD, FFTW_ESTIMATE | FFTW_OUT_OF_PLACE | FFTW_USE_WISDOM);
#endif

	if (m_plan == NULL)
		throw CException(wxT("Cannot create IFFT plan"));
}

CIFFT::~CIFFT()
{
	wxASSERT(m_plan != NULL);
	wxASSERT(m_in   != NULL);

	::fftw_destroy_plan(m_plan);

	::fftw_free(m_in);
	::fftw_free(m_out);
}

void CIFFT::process(double* in, double* bins)
{
	wxASSERT(m_plan != NULL);
	wxASSERT(m_in   != NULL);
	wxASSERT(m_out  != NULL);
	wxASSERT(in     != NULL);
	wxASSERT(bins   != NULL);

#if defined(HAVE_FFTW3)
	for (int i = 0; i < m_fftLen; i++) {
		m_in[i][0] = in[i];
		m_in[i][1] = 0.0;
	}

	::fftw_execute(m_plan);

	for (int i = 0; i < m_fftLen; i++)
		bins[i] = m_out[i][0];
#else
	for (int i = 0; i < m_fftLen; i++) {
		c_re(m_in[i]) = in[i];
		c_im(m_in[i]) = 0.0;
	}

	::fftw_one(m_plan, m_in, m_out);

	for (int i = 0; i < m_fftLen; i++)
		bins[i] = c_re(m_out[i]);
#endif
}
