/*
 * file com_dg_server.c - send ingame datagrams to server
 *
 * $Id: com_dg_server.c,v 1.3 2004/05/14 10:00:33 alfie Exp $
 *
 * Program XBLAST 
 * (C) by Oliver Vogel (e-mail: m.vogel@ndh.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will be entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "com_dg_server.h"

#include "client.h"
#include "com_dgram.h"

/*
 * locals typedefs
 */
typedef struct {
  XBCommDgram dgram;
} XBCommDgramServer;

/*
 *
 */
static void
ReceivePing (XBCommDgram *dComm, unsigned clientID, unsigned short pingTime)
{
  if (0 == clientID) {
    /* server wants a reply */
    Dgram_SendPing (dComm); 
  } else {
    /* we have received the ping time of a client */
    if (pingTime < 0xFFFF) {
      Client_ReceivePingTime (clientID, (int) ((unsigned) pingTime) );
    } else {
      Client_ReceivePingTime (clientID, -1);
    }
  }
  dComm->connected = XBTrue;
} /* ReceivePing */

/*
 *
 */
static void
ReceiveFinish (XBCommDgram *dComm)
{
  Client_ReceiveFinish ();
} /* ReceiveFinish */

/*
 *
 */
static void
ReceivePlayerAction (XBCommDgram *dComm, int gameTime, const PlayerAction *playerAction)
{
  Client_ReceivePlayerAction (gameTime, playerAction);
} /* ReceivePlayerAction */

/*
 * create datagramm connection server
 */
XBComm *
D2S_CreateComm (const char *localname, const char *hostname, unsigned short port)
{
  XBCommDgramServer *dComm;
  XBSocket          *pSocket;

  /* create socket */
  pSocket = Net_BindUdp (localname, 0);
  if (NULL == pSocket) {
    return NULL;
  }
  /* connect it */
  if (! Net_ConnectUdp (pSocket, hostname, port)) {
    Net_Close (pSocket);
    return NULL;
  }
  /* create communication data structure */
  dComm = calloc (1, sizeof (*dComm) );
  assert (NULL != dComm);
  /* set values */
  Dgram_CommInit (&dComm->dgram, COMM_DgServer, pSocket, XBFalse, ReceivePing, ReceiveFinish, ReceivePlayerAction);
  /* that's all */
  return &dComm->dgram.comm;
} /* D2C_CreateComm */

/*
 * get port for client
 */
unsigned short
D2S_Port (const XBComm *comm)
{
  return Dgram_Port ((XBCommDgram *) comm);
} /* D2C_Port */

/*
 * reset datagram connection
 */
void
D2S_Reset (XBComm *comm)
{
  Dgram_Reset((XBCommDgram *) comm);
} /* D2S_Reset */

/*
 * send player action to client
 */
void
D2S_SendPlayerAction (XBComm  *comm, int gameTime, const PlayerAction *playerAction)
{
  Dgram_SendPlayerAction ((XBCommDgram *) comm, gameTime, playerAction);
} /* D2C_SendPlayerAction */

/*
 * acknowledge level finish 
 */
void
D2S_SendFinish (XBComm *comm, int gameTime)
{
  Dgram_SendFinish ((XBCommDgram *) comm, gameTime);
} /* D2S_SendFinish */


/*
 * flush remaing data
 */
XBBool
D2S_Flush (XBComm *comm)
{
  return Dgram_Flush ((XBCommDgram *) comm);
} /* D2S_Flush */

/*
 * check for datagram timeout
 */
XBBool
D2S_Timeout (const XBComm *comm, const struct timeval *tv)
{
  XBCommDgram *dgram = (XBCommDgram *) comm;

  assert (NULL != dgram);
  assert (NULL != tv);
  
  return (0 != dgram->lastSnd.tv_sec && 
	  tv->tv_sec - dgram->lastSnd.tv_sec > LINK_LOST);
} /* D2S_Timeout */

/*
 * check if server is already connected (i.e. has send a datagram)
 */
XBBool
D2S_Connected (const XBComm *comm)
{
  XBCommDgram *dgram = (XBCommDgram *) comm;

  assert (NULL != dgram);
  return dgram->connected;
} /* D2S_Connected */

/*
 * send connect datagram to server
 */
void
D2S_SendConnect (XBComm *comm)
{
  Dgram_SendPing ((XBCommDgram *) comm);
} /* D2S_SendConnect */

/*
 * end of file com_dg_server.c
 */
