/*
 * file level.c - Setting up level given in database
 *
 * $Id: level.c,v 1.3 2004/05/14 10:00:35 alfie Exp $
 *
 * Program XBLAST 
 * (C) by Oliver Vogel (e-mail: m.vogel@ndh.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will be entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "level.h"

#include "cfg_level.h" // 0307

#include "atom.h"
#include "bomb.h"
#include "func.h"
#include "info.h"
#include "map.h"
#include "scramble.h"
#include "shrink.h"

static const char    *gameModeString  = "R23456STDL"; //0307

/*
 * Configure new level
 */
XBBool
ConfigLevel (const DBRoot *level)
{
  const char      *s;
  unsigned         i,gameMode = 0;

#ifdef DEBUG
  Dbg_StartClock ();
#endif
  /* sanity check */
  assert (level != NULL);
  /* reset level info data */
  ResetInfo ();
  /* TODO: disallow random positions in some levels */
  /* setup players */

  if (! DB_GetEntryString (DB_GetSection (level, atomInfo), atomGameMode, &s) ) {
    return XBFalse;
  }

  for (i = 0; s[i] != 0 && gameModeString[i] != 0; i ++) {
    if (s[i] == gameModeString[i]) {
      gameMode |= (1 << i);
    } 
  }
  
  if (! ConfigLevelPlayers (DB_GetSection (level, atomPlayer), XBTrue, gameMode) ) { // 0307
    fprintf (stderr, "error in player");
    return XBFalse;
  }
  /* setup shrink pattern */
  if (! ConfigLevelShrink (DB_GetSection (level, atomShrink) ) ) {
    fprintf (stderr, "error in shrink");
    return XBFalse;
  }
  /* setup scrambling blocks */
  if (! ConfigLevelScramble (DB_GetSection (level, atomScrambleDraw),
			     DB_GetSection (level, atomScrambleDel) ) ) {
    fprintf (stderr, "error in scramble");
    return XBFalse;
  }
  /* setup function pointers */
  if (! ConfigLevelFunc (DB_GetSection (level, atomFunc) ) ) {
    fprintf (stderr, "error in func");
    return XBFalse;
  }
  /* setup bombs */
  if (! ConfigLevelBombs (DB_GetSection (level, atomBombs) ) ) {
    fprintf (stderr, "error in bombs");
    return XBFalse;
  }
  /* setup graphics */
  if (! ConfigLevelGraphics (DB_GetSection (level, atomGraphics) ) ) {
    fprintf (stderr, "error in graphics");
    return XBFalse;
  }
  /* setup map layout */
  if (! ConfigLevelMap (DB_GetSection (level, atomMap) ) ) {
    fprintf (stderr, "error in map");
    return XBFalse;
  }
#ifdef DEBUG
  fprintf (stderr, "parse level: %lu msec\n", Dbg_FinishClock ());
#endif
  return XBTrue;
} /* ConfigLevel */

/*
 * clean up after level
 */
void
FinishLevel (void)
{
  /* shriking */
  FinishLevelShrink ();
  /* graphics */
  FinishLevelGraphics ();
} /* FinishLevel */

/*
 * get string of level
 */
static const char *
GetLevelString (const DBRoot *level, XBAtom atom)
{
  const char      *s;
  const DBSection *section;
  
  assert (level != NULL);
  section = DB_GetSection (level, atomInfo);
  if (NULL == section) {
    return NULL;
  }
  if (! DB_GetEntryString (section, atom, &s) ) {
    return NULL;
  }
  return s;
} /* GetLevelString */

/*
 * get name of level
 */
const char *
GetLevelName (const DBRoot *level)
{
  return GetLevelString (level, atomName);
} /* GetLevelName */

/*
 * get author of level
 */
const char *
GetLevelAuthor (const DBRoot *level)
{
  return GetLevelString (level, atomAuthor);
} /* GetLevelAuthor */

/*
 * get hint for level
 */
const char *
GetLevelHint (const DBRoot *level)
{
  return GetLevelString (level, atomHint);
} /* GetLevelAuthor */

/*
 * end of file level.c
 */
