/*
 * net_socket.c - low level functions to network communication
 *
 * $Id: net_socket.c,v 1.4 2004/06/26 03:23:18 iskywalker Exp $
 *
 * Program XBLAST 
 * (C) by Oliver Vogel (e-mail: m.vogel@ndh.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will be entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef WIN32
#include <windows.h>
#endif

#include "net_socket.h"

/*
 * delete socket stucture
 */
void
Net_Close (XBSocket *pSocket)
{
  assert (NULL != pSocket);

  Socket_Close (pSocket);
  Socket_Free (pSocket);
} /* Net_Close */

/*
 * try to connect to server (via TCP/IP)
 */
XBSocket *
Net_ConnectInet (const char *hostName, unsigned short port)
{
  XBSocket *pSocket = NULL;
  
  /* create socket structure */
  pSocket = Socket_Alloc (AF_INET);
  /* create socket */
  if (! Socket_Open (pSocket, SOCK_STREAM) ) {
    goto Error;
  }
  if (! Socket_SetAddressInet (pSocket, XBTrue, hostName, port)) {
    goto Error;
  }
  /* now try to connect */
  if (! Socket_Connect (pSocket) ) {
    goto Error;
  }
  /* that's all */
  return pSocket;
  /*
   * Error handling
   */
 Error:
  Net_Close (pSocket);
  return NULL;
} /* Net_ConnectInet */

/*
 * listen for client to connect
 */
XBSocket *
Net_ListenInet (unsigned short port)
{
  XBSocket *pSocket = NULL;

  /* alloc socket data structure */
  pSocket = Socket_Alloc (AF_INET);
  /* create socket */
  if (! Socket_Open (pSocket, SOCK_STREAM) ) {
    goto Error;
  }
  /* set socket address structure */
  if (! Socket_SetAddressInet (pSocket, XBFalse, NULL, port)) {
    goto Error;
  }
  if (! Socket_Bind (pSocket) ) {
    goto Error;
  }
  /* now activate listen socket */
  if (! Socket_Listen (pSocket) ) {
    goto Error;
  }
  /* that's all */
  return pSocket;
  /*
   * Error handling
   */
 Error:
  Net_Close (pSocket);
  return NULL;
} /* Net_ListenInet */


/*
 * accept client connection
 */
XBSocket *
Net_Accept (const XBSocket *pListen)
{
  XBSocket *pSocket;

  /* alloc Socket data structure */
  pSocket = Socket_Alloc (Socket_Family (pListen));
  assert (pSocket != NULL);
  /* try to accept client */
  if (! Socket_Accept (pSocket, pListen)) {
    goto Error;
  }
  /* that's all */
  return pSocket;
  /*
   * Error handling
   */
 Error:
  Net_Close (pSocket);
  return NULL;
} /* Net_AcceptInet */

/*
 * create datgram socket for host
 */
XBSocket *
Net_BindUdp (const char *localname, unsigned port)
{
  XBSocket      *pSocket;

  /* create socket structure */
  pSocket = Socket_Alloc (AF_INET);
  assert (pSocket != NULL);
  /* create socket */
  if (! Socket_Open (pSocket, SOCK_DGRAM) ) {
    goto Error;
  }
  /* we need only epheremal prot, but on a given device */
  if (! Socket_SetAddressInet (pSocket, XBFalse, localname, port)) {
    goto Error;
  }
  /* now bind socket */
  if (! Socket_Bind (pSocket) ) {
    goto Error;
  }
  /* that's all */
  return pSocket;
  /*
   * Error handling
   */
 Error:
  Net_Close (pSocket);
  return NULL;
} /* Net_BindInet */

/*
 * connect datagram socket to port
 */
XBBool
Net_ConnectUdp (XBSocket *pSocket, const char *hostName, unsigned short port)
{
  /* set it */
  if (! Socket_SetAddressInet (pSocket, XBTrue, hostName, port)) {
    return XBFalse;
  }
  /* now try to connect */
  if (! Socket_Connect (pSocket) ) {
    return XBFalse;
  }
  return XBTrue;
} /* Net_ConnectUdp */

/*
 * get host name of client 
 */
const char *
Net_LocalName (const XBSocket *pSocket)
{
  const char *name;
  static char hostName[32];

  assert (pSocket != NULL);
  if (NULL == (name = Socket_HostName (pSocket, XBFalse) ) ) {
    return NULL;
  }
  assert (sizeof (hostName) > strlen (name));
  strcpy (hostName, name);
  return hostName;
} /* Net_Hostname */

/*
 * get host name of client 
 */
const char *
Net_RemoteName (const XBSocket *pSocket)
{
  const char *name;
  static char hostName[32];

  assert (pSocket != NULL);
  if (NULL == (name = Socket_HostName (pSocket, XBTrue) ) ) {
    return NULL;
  }
  assert (sizeof (hostName) > strlen (name));
  strcpy (hostName, name);
  return hostName;
} /* Net_RemoteName */

/*
 * 
 */
unsigned
Net_LocalPort (const XBSocket *pSocket)
{
  assert (pSocket != NULL);
  return Socket_HostPort (pSocket, XBFalse);
} /* Net_LocalPort */

/*
 * 
 */
unsigned
Net_RemotePort (const XBSocket *pSocket)
{
  assert (pSocket != NULL);
  return Socket_HostPort (pSocket, XBTrue);
} /* Net_LocalPort */

/*
 * end of file net_socket.c
 */
