#!/usr/bin/python

# Copyright (c) 2009 The Chromium OS Authors. All rights reserved.
# Copyright (c) 2010 Canonical Ltd.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
# Written by Colin Watson for Canonical Ltd.

# TODO(cjwatson): Not all the necessary metadata is available in packages,
# although we might be able to do better in future about detecting whether a
# package can usefully be converted using dpkg-cross and installed on the build
# system. For the time being, given the target package set, the most economical
# approach is to hardcode various special cases in a configuration file.

from cStringIO import StringIO

import ConfigParser
import os
import pprint
import sys

# The name of the config option that designates a parent section to inherit
# values from.  Used by TargetConfig.
_PARENT_OPTION = 'parent'


# The format of the section name for a target config.
_TARGET_SECTION_FORMAT = 'target-%s-%s'


# Constants for the config file option names of interest.
_BLACKLIST_NAME = 'blacklist'
_WHITELIST_NAME = 'whitelist'
_CROSS_BLACKLIST_NAME = 'cross_blacklist'
_PARALLEL_BLACKLIST_NAME = 'parallel_blacklist'
_BUILDDEP_WHITELIST_NAME = 'builddep_whitelist'
_NATIVE_IMPORT_SOURCE_NAME = 'native_import_source'
_NATIVE_IMPORT_NAME = 'native_import'
_OPTIONS_NAME = 'options'


# Default configuration files to search.
_DEFAULT_CONFIGS = ['/etc/xdeb/xdeb.cfg',
                    os.path.join(sys.path[0], 'xdeb.cfg'),
                    '.xdeb',
                    'xdeb.cfg']


class Error(Exception):
    pass


class ConfigFileParseException(Error):
    """Error parsing the config files."""
    pass


class TargetConfig(object):
    """Represents the parsed configuration for a particular build target.

    Currently, this is just an abstraction of the lists of packages to build
    for each target.  In the future, this may be extended to include other
    target specific build configurations.

    The configuration file format is the standard Python ConfigParser
    format.  Each config option is considered to be a white-space separated
    set of values.  The configuration for a particular architecture and
    variant should be given in a section named:

      [target-$arch-$variant]

    So, for example, for an "armel", with a "generic" variant, the section
    name should be:

      [target-armel-generic]

    Each section has a special option "parent."  The "parent" attribute is
    treated as a single string instead of a list, and specifies another
    config section to inherit values from.  A child section can override, or
    modify the options in the parent section by specifying its own version
    of the options as follows.

      [my-parent]
        foo: yay wee
        bar: hello bonjour
        baz: bye

      [target-armel-generic]
        parent: my-parent
        foo: override the value
        +bar: hello wazzzzzup
        -baz: bye

    In this example, the resulting configuration for target-armel-generic
    is:

      foo: set(['override', 'the', 'value'])
      bar: set(['hello', 'bonjour', 'wazzzzzup'])
      baz: set([])

    Basically, if you respecify the option in a child, it overrides the
    parent's value unless you specify it with a '+' or '-' prefix.  If you
    give it a '+' or '-' prefix, it will respectively do set union, or set
    subtraction to obtain the final value for target-armel-generic.
    """

    def __init__(self, arch, variant):
        """Creates a TargetConfig to contain options for arch and variant.

        The InitializeFromConfigs() must be called to actually load the data
        for this object.

        Args:
          arch: A string with the the architecture for this target.
          variant: A string with the architecture variant for this target.
        """
        self._architecture = arch
        self._variant = variant
        self._value_dict = {
            _BLACKLIST_NAME: set(),
            _WHITELIST_NAME: set(),
            _CROSS_BLACKLIST_NAME: set(),
            _PARALLEL_BLACKLIST_NAME: set(),
            _BUILDDEP_WHITELIST_NAME: set(),
            _NATIVE_IMPORT_NAME: set(),
            _OPTIONS_NAME: {},
            }

    def InitializeFromConfigs(self, config_paths):
        """Parses the config paths, and extract options for our arch/variant.

        The config files are parsed in the order they are passed in.  If
        there are duplicate entries, the last one wins.

        TODO(ajwong): There should be a factory method to call this.

        Args:
          config_paths: A sequence of path strings for config files to load.
        """
        config = ConfigParser.SafeConfigParser(
            {_PARENT_OPTION: '',
             'architecture': self._architecture,
             'variant': self._variant})

        self._default_configs_read = config.read(_DEFAULT_CONFIGS)

        self._configs_read = None
        if config_paths:
            self._configs_read = config.read(config_paths)
            if self._configs_read != config_paths:
                raise ConfigFileParseException(
                    'Only read [%s] but expected to read [%s]' %
                    (', '.join(self._configs_read), ', '.join(config_paths)))
        self._ProcessTargetConfigs(config)

        if config.has_section('Options'):
            for option in config.options('Options'):
                self._value_dict['options'][option] = config.get('Options',
                                                                 option)

    def _ProcessTargetConfigs(self, config):
        """Extracts config settings for the given arch and variant.

        Finds the configuration section "target-$arch-$variant", resolves
        the parent references, and populates self._value_dict with the final
        value.

        Args:
          config: A ConfigParser object with a "target-$arch-$variant" section.
        """
        # Find the parents.
        parent_chain = []
        current_frame = _TARGET_SECTION_FORMAT % (self.architecture,
                                                  self.variant)
        while current_frame:
            parent_chain.append(current_frame)
            current_frame = config.get(current_frame, _PARENT_OPTION)
        parent_chain.reverse()

        # Merge all the configs into one dictionary.
        for section in parent_chain:
            for option in config.options(section):
                raw_value = config.get(section, option)
                if raw_value:
                    if option in self._value_dict:
                        # Merged option.
                        values = raw_value.split()
                        self.MergeConfig(self._value_dict, option, values)
                    else:
                        # Simple option. No merging required.
                        self._value_dict[option] = raw_value

    def MergeConfig(self, merged_config, option_name, values):
        """Merges a new value into the configuration dictionary.

        Given a set of configuration values in merged_config, either add or
        remove the values from the set of values in
        merged_config[option_name].

        Args:
          merged_config: A dictionary mapping an option name to a set of
            values.
          option_name: The name of the option to modify.
          values: A string of white-space separated values to add or remove
            from merged_config[option_name].
        """
        if option_name[0] == '+':
            real_name = option_name[1:]
            if merged_config.has_key(real_name):
                merged_config[real_name].update(values)
            else:
                merged_config[real_name] = set(values)
        elif option_name[0] == '-':
            real_name = option_name[1:]
            if merged_config.has_key(real_name):
                merged_config[real_name].difference_update(values)
        else:
            merged_config[option_name] = set(values)

    @property
    def architecture(self):
        """A string with the architecture of this target config."""
        return self._architecture

    @property
    def variant(self):
        """A string with the architecture variant of this target config."""
        return self._variant

    @property
    def blacklist(self):
        """Set of packages that packages that get in the way if cross-built.

        These packages will not be be cross-built.
        """
        return self._value_dict[_BLACKLIST_NAME]

    @property
    def cross_blacklist(self):
        """Set of packages needed for deps, but cannot be cross-converted.

        We should follow the dependencies for these packages, but these
        packages themselves should not be cross-converted."""
        return self._value_dict[_CROSS_BLACKLIST_NAME]

    @property
    def parallel_blacklist(self):
        """Set of packages that fail if built with dpkg-buildpackage -jjobs."""
        return self._value_dict[_PARALLEL_BLACKLIST_NAME]

    @property
    def whitelist(self):
        """Set of packages that need to be explicitly built for this target.

        Note that anything with a pkg-config file needs to be cross-built.
        """
        return self._value_dict[_WHITELIST_NAME]

    @property
    def builddep_whitelist(self):
        """Set of packages that are not crossable, but need to be installed.

        These packages are needed to satisfy build-dependencies despite not
        being in the usual crossable list."""
        return self._value_dict[_BUILDDEP_WHITELIST_NAME]

    @property
    def native_import_source(self):
        """APT repository for native imports."""
        return self._value_dict.get(_NATIVE_IMPORT_SOURCE_NAME, '')

    @property
    def native_import(self):
        """Set of packages that need to be imported from native builds."""
        return self._value_dict[_NATIVE_IMPORT_NAME]

    @property
    def options(self):
        """Default values of command-line options in the config file."""
        return self._value_dict[_OPTIONS_NAME]

    def __str__(self):
        """Dump the TargetConfig values for debugging."""
        output = StringIO()
        pp = pprint.PrettyPrinter(indent=4)
        output.write('Default configs parsed: %s\n' %
                     pp.pformat(self._default_configs_read))
        if self._configs_read:
            output.write('Additional configs parsed: %s\n' %
                         pp.pformat(self._configs_read))
        output.write('\n')

        # Print out each property.
        class_members = self.__class__.__dict__.keys()
        class_members.sort()
        for prop in class_members:
            if isinstance(self.__class__.__dict__[prop], property):
                output.write('%s: %s\n' %
                             (prop, pp.pformat(getattr(self, prop))))

        return output.getvalue()
