/* -*-Mode: C++;-*-
 * $Id: async.cc 1.4 Wed, 16 May 2001 03:33:56 +0400 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

static pthread_mutex_t ASYNC_STARTED_MUTEX = PTHREAD_MUTEX_INITIALIZER;
static bool            ASYNC_STARTED       = false;

#if 0
static pthread_mutex_t ASYNC_QUEUE_MUTEX;

static guint           ASYNC_QUEUE_SIZE  = 1024;
static guint           ASYNC_FORCE_USECS    = 10000;
static guint           ASYNC_SLEEP_USECS    = 1000 * 1000;
static guint           ASYNC_SLEEP_MAX_SECS = 10 * 1000 * 1000;

class ASYNC_FD
{
public:

    ASYNC_FD ()
    {
	pthread_mutex_init (& mutex, & ERRORCHECK_MUTEX_ATTR);
    }

    int             fd;
    pthread_mutex_t mutex;

    struct Hash
    {
	uint operator () (const ASYNC_FD *a) const { return a->fd; }
	uint operator () (const int      &k) const { return k; }
    };

    struct Equals
    {
	bool operator () (const int &k, ASYNC_FD *a) { return k == a->fd; }
    };

    elink<ASYNC_FD> list_link;
    elink<ASYNC_FD> map_link;
};

typedef ELIST     (ASYNC_FD,list_link) ASYNC_FD_LIST;
typedef EHASH_MAP (ASYNC_FD,map_link,ASYNC_FD::Hash,ASYNC_FD::Equals) ASYNC_FD_MAP;

class ASYNC_QUEUE
{
public:

    ASYNC_QUEUE ()
	: _array (new ASYNC_FD [ASYNC_QUEUE_SIZE])
    {
	for (uint i = 0; i < ASYNC_QUEUE_SIZE; i += 1) {
	    _free.push_back (& _array[i]);
	}
    }

    int sync1 ()
    {
	MONITOR   mon;
	MONITOR   afd_mon;
	ASYNC_FD *afd;
	int       ret;
	guint     sleep_usecs = ASYNC_SLEEP_USECS;

	if (0) {
	  again:
	    mon.release     ();
	    afd_mon.release ();

	    usleep (ASYNC_FORCE_USECS);
	}

	if (0) {
	  retry:
	    mon.release     ();
	    afd_mon.release ();

	    DEBUG_SYNC ("sync1 retry");

	    usleep (ASYNC_FORCE_USECS);
	}

	if (0) {
	  empty:
	    mon.release     ();
	    afd_mon.release ();

	    DEBUG_SYNC ("fsync write-behind empty: sleep %d", sleep_usecs);

	    usleep (sleep_usecs);

	    sleep_usecs *= 2;

	    sleep_usecs = MIN (sleep_usecs, ASYNC_SLEEP_MAX_SECS);
	}

	mon.lock (& ASYNC_QUEUE_MUTEX);

	if (_queue.empty ()) {
	    goto empty;
	}

	afd = & _queue.front ();

	if ((ret = afd_mon.trylock (& afd->mutex))) {
	    goto retry;
	}

	mon.release ();

	DEBUG_SYNC ("fsync write-behind %d", afd->fd);

	sleep_usecs = ASYNC_SLEEP_USECS;

	if ((ret = fsync (afd->fd))) {
	    ret = errno;
	    SYS_ERROR (ret) ("fsync_fd: %d", afd->fd);
	    return ret;
	}

	mon.lock (& ASYNC_QUEUE_MUTEX);

	_map.erase       (afd);
	_queue.erase     (afd);
	_free.push_front (afd);

	goto again;
    }

    int force (int fd)
    {
	MONITOR   mon;
	MONITOR   afd_mon;
	ASYNC_FD *afd;
	int       ret;

	if (0) {
	  retry:
	    mon.release     ();
	    afd_mon.release ();

	    DEBUG_SYNC ("force retry");

	    usleep (ASYNC_FORCE_USECS);
	}

	mon.lock (& ASYNC_QUEUE_MUTEX);

	if ((afd = _map.find (fd)) == NULL) {
	    return 0;
	}

	if ((ret = afd_mon.trylock (& afd->mutex))) {
	    goto retry;
	}

	mon.release ();

	DEBUG_SYNC ("fsync force %d", afd->fd);

	if ((ret = fsync (afd->fd))) {
	    ret = errno;
	    SYS_ERROR (ret) ("fsync_fd: %d", afd->fd);
	    return ret;
	}

	mon.lock (& ASYNC_QUEUE_MUTEX);

	_map.erase       (afd);
	_queue.erase     (afd);
	_free.push_front (afd);

	return 0;
    }

    int async_fd (int fd)
    {
	MONITOR mon (& ASYNC_QUEUE_MUTEX);

	ASYNC_FD *afd;

	if ((afd = _map.find (fd))) {
	    DEBUG_SYNC ("fsync duplicate %d", fd);
	    return 0;
	}

	if (_free.empty ()) {
	    DBFS_ERROR ("async queue full");
	    g_assert_not_reached ();
	} else {

	    // Enqueue
	    afd = & _free.pop_front ();

	    DEBUG_SYNC ("fsync enqueue %d", fd);

	    afd->fd = fd;

	    _queue.push_back (afd);
	    _map.insert      (afd);
	}

	return 0;
    }

    ASYNC_FD      *_array;
    ASYNC_FD_MAP   _map;
    ASYNC_FD_LIST  _queue;
    ASYNC_FD_LIST  _free;
};

static ASYNC_QUEUE  *async_queue;
#endif

//static void*   dbfs_async_thread (void *null);
//static int     dbfs_async_close  (int fd);
//static ssize_t dbfs_async_write  (int fd, const void *buffer, size_t nbytes);
static int     dbfs_async_fsync  (int fd);

int
dbfs_async_txn_chkflags (int beginflags)
{
    if ((beginflags & DBFS_TXN_ASYNC) && ! ASYNC_STARTED) {
	DBFS_ERROR ("async transaction with no async thread");
	return DBFS_INVAL;
    }

    return 0;
}

int
dbfs_async_start (int oflags)
{
    MONITOR mon (& ASYNC_STARTED_MUTEX);

    if (! (oflags & DBFS_ASYNC_INIT)) {
	return 0;
    }

    if (ASYNC_STARTED) {
	return 0;
    }

    ASYNC_STARTED = true;

    // @@@ This code is pretty experimental

//      async_queue = new ASYNC_QUEUE;
//      pthread_mutex_init (& ASYNC_QUEUE_MUTEX, & ERRORCHECK_MUTEX_ATTR);

    db_env_set_func_fsync (dbfs_async_fsync);
//      db_env_set_func_close (dbfs_async_close);
//      db_env_set_func_write (dbfs_async_write);
//      db_env_set_func_read  (dbfs_async_read);
//      db_env_set_func_seek  (dbfs_async_seek);
//      db_env_set_func_open  ((int (*)(const char *, int, ...)) dbfs_async_open);

//      pthread_t tid;
    INFO_DBFS ("async write-behind started -- UNSAFE");
//      pthread_create (& tid, NULL, & dbfs_async_thread, NULL);
//      pthread_detach (tid);

    return 0;
}

int
dbfs_async_fsync (int fd)
{
    // @@@ IGNORE FOR NOW
    return 0;
}

#if 0
void*
dbfs_async_thread (void *null)
{
    int ret;

    ret = async_queue->sync1 ();

    DBFS_ERROR (ret) ("async thread failed");

    return NULL;
}

int
dbfs_async_close (int fd)
{
    int ret;

    if ((ret = async_queue->force (fd))) {
	return ret;
    }

    return close (fd);
}

ssize_t
dbfs_async_write (int fd, const void *buffer, size_t nbytes)
{
//      int ret;

//      if ((ret = async_queue->force (fd))) {
//  	return ret;
//      }

    return write (fd, buffer, nbytes);
}

int
dbfs_async_fsync (int fd)
{
    int ret;

    if ((ret = async_queue->async_fd (fd))) {
	PROP_ERROR (ret) ("async_fsync");
    }

    return ret;
}

static int     dbfs_async_open   (const char *path, int flags, int mode);
static int     dbfs_async_seek   (int fd, size_t pgsize, db_pgno_t pageno, u_int32_t relative, int rewind, int whence);
static ssize_t dbfs_async_read   (int fd, void *buf, size_t nbytes);

int
dbfs_async_open (const char *path, int flags, int mode)
{
    return open (path, flags, mode);
}

ssize_t
dbfs_async_read (int fd, void *buf, size_t nbytes)
{
    return read (fd, buf, nbytes);
}

int
dbfs_async_seek (int fd, size_t pgsize,
		db_pgno_t pageno, u_int32_t relative,
		int rewind, int whence)
{
    off_t offset = (off_t) pgsize * (off_t) pageno + (off_t) relative;
    int   ret;

    if (rewind) {
	offset = -offset;
    }

    if ((ret = lseek (fd, offset, whence)) < 0) {
	ret = errno;
 	SYS_ERROR (ret) ("lseek: %d:%qu:%d", fd, (guint64) offset, whence);
	return ret;
    }

    return 0;
}
#endif
