#ifndef request_h
#define request_h

#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <time.h>
#include <unistd.h>
#include <stdio.h>

#include <xdfs_cpp.h>

#include "globals.h"
#include "socket.h"

#define MAX_FILE_SIZE 100000
#define MAX_HEADER_SIZE 1024
#define MAX_NUM_HEADERS 10

class HTTPState {

 public:
  HTTPState();
  ~HTTPState();
  void reset();

  char method[16];
  char url[MAX_HEADER_SIZE];
  bool HTTP11; // true if HTTP 1.1; otherwise, false (HTTP 1.0)
  char server[MAX_HEADER_SIZE]; // dual purpose: store server to which request is sent; store "Server" response header
  int port;
  char resource[MAX_HEADER_SIZE];
  char reasonPhrase[MAX_HEADER_SIZE];

  // START: relevant HTTP headers that are parsed out from the request/response and do not appear in additionalHeaders.

  int HTTPCode; // HTTP response code
  time_t date; // Date
  time_t expires; // Expires
  time_t ifModifiedSince; // If-Modified-Since
  time_t originalIfModifiedSince;
  time_t lastModified; // Last-Modified
  int contentLength; // Content-Length
  char contentMD5[17]; // Content-MD5
  bool haveContentMD5;
  char deltaBase[17]; // Delta-Base
  bool haveDeltaBase;
  char contentType[MAX_HEADER_SIZE]; // Content-Type
  bool IMXDelta; // true if "IM: xdelta"; otherwise, false
  bool aIMXDelta; // true if "A-IM: xdelta" (other delta algorithms might be specified); otherwise, false
  bool connectionClose; // true if "Connection: close" or "Proxy-Connection: close"; otherwise, false
  bool transferEncodingChunked; // true if "Transfer-Encoding: chunked"; otherwise, false
  bool contentEncodingGZIP; // true if "Content-Encoding: gzip" (no other content encodings allowed); otherwise, false
  bool acceptEncodingGZIP; // true if "Accept-Encoding: gzip" (other encodings might be specified); otherwise, false
  char eTag[MAX_HEADER_SIZE]; // dual purpose: store "If-Match" request header; store "ETag" response header
  char originalETag[MAX_HEADER_SIZE];
  bool hasCookie; // true if reques has "Set-Cookie" header or response has "Cookie" header; cookie appears in additionalHeaders
  int maxAge; // Cache-Control: max-age=xxx; this header doesn't appear in additionalHeaders, it's used only to determine expiration date

  // END: relevant HTTP headers.

  char additionalHeaders[(MAX_NUM_HEADERS/2)*MAX_HEADER_SIZE];
  bool doneReadingChunks; // true if we're done reading chunked data; otherwise, false
};

class Request {

 public:
  Request();
  ~Request();
  void reset();

  Socket *in; // socket to read data from
  Socket *out; // socket to write data to

  RequestState state; // state of the request

  HTTPState requestFromClient; // state for HTTP request from client
  HTTPState responseFromServer; // state for HTTP response from server

  char *bodyFromServer; // buffer for data from server
  char *bodyFromServerStart; // pointer to the start of the data from server
  int bodyFromServerLength; // total length of data from server
  int bodyFromServerAllocatedLength; // total buffer length allocated for bodyFromServer

  char *bodyToClient; // buffer for data to client
  char *bodyToClientStart; // pointer to the start of the data to client
  int bodyToClientLength; // total length of data to client
  int bodyToClientAllocatedLength; // total buffer length allocated for bodyToClient

  char tempBuffer[MAX_NUM_HEADERS*MAX_HEADER_SIZE]; // buffer used for preparing request and response headers; reserve half for already parsed known headers and half for additional headers including entity body for POST requests.
  char *tempBufferStart;
  int tempBufferLength;

  char localMD5[17]; // MD5 of local version
  bool haveLocalMD5; //
  //Inode ino; // local version inode (if any)

  // Various times in microseconds.
  GTimer *processClientRequestTimer;
  int processClientRequestTime;
  GTimer *processServerResponseTimer;
  int processServerResponseTime;
  GTimer *parseResponseHeaderTimer;
  int parseResponseHeaderTime;

  int originalSize; // size of response sent by the source web server/server proxy
  int currentSize; // size of response sent by the server proxy to the client proxy (or by the client proxy to the client); need this because bodyToClientLength gets decremented
};

#endif request_h
