# parse.tcl
# $Id: parse.tcl,v 1.9 1999/07/28 07:51:56 chris Exp $
# Parses command lines.  Yes, in Tcl.  *duck*
#
#
# XED --- an X-enabled implementation of ed, the standard editor.
# Copyright (C) 1999  Chris Laas
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# Chris Laas
# golem@mit.edu
# 22 Pratt Court
# Cohasset, MA 02025 USA


array set handlers {
	a command-append
	c command-change
	d command-delete
	e command-edit-or-read-file E command-edit-or-read-file
	f command-default-filename
	g command-match
	G command-match
	H command-toggle-errors
	h command-last-error
	i command-insert
	j command-join
	k command-mark
	m command-move
	l command-print n command-print p command-print
	P command-toggle-prompt
	q command-quit Q command-quit
	r command-edit-or-read-file
	s command-substitute
	t command-transfer
	u command-undo
	v command-match
	V command-match
	w command-write-file W command-write-file
	x command-put-cut-buffer y command-yank-cut-buffer
	z command-scroll
	! command-shell
	# command-comment
	= command-print-line-number
	{} command-print-line

	~ command-eval-tcl

	C command-change-interactive
	| command-pipe
	I command-indent
}
array set default_ranges {
	a . c . d . e {} E {} f {} g 1,$ G 1,$ h {} H {}
	i . j .,.+1 k . l . m . n . p . P {} q {} Q {}
	r $ s . t . u {} v 1,$ V 1,$ w 1,$ W 1,$ x . y . z .+1
	! {} . {} = $ {} .+1

	~ {}

	C . | 1,$ I .
}


set command_list {}
set this_undo_commands {}
set last_undo_commands {}

proc process-line {line} {
	global prompt verbose default_filename current_addr last_addr dirty \
			debug input_mode handlers default_ranges ttymode confirm \
			command_list last_command_list continuation \
			this_undo_commands last_undo_commands
	switch -exact -- [lindex $input_mode 0] {
		command - subcommand - batchcommand {
			set script {
				if {[lindex $input_mode 0] == "command"} {
					set last_undo_commands $this_undo_commands
					set this_undo_commands [list \
							"~set dirty $dirty" \
							"~set prompt {$prompt}" \
							"~set default_filename {$default_filename}" \
							"~set current_addr $current_addr"]
				} elseif {[lindex $input_mode 0] == "subcommand"} {
					if {$command_list == {} && $line == {}} {
						set input_mode [lrange $input_mode 1 end]
						return
					}
					if {$command_list == {} && $line == "&"} {
						if {![info exists last_command_list]} {
							error "No previous command list"
						}
						foreach line $last_command_list { process-line $line }
						return
					}
					if ![regsub {\\$} $line {} line] {
						set input_mode [lrange $input_mode 1 end]
						set last_command_list [concat $command_list $line]
						set command_list {}
					} else {
						lappend command_list $line
					}
				}
				# Note that batchcommand has no special processing.

				read-addr-range $line line addr1 addr2
				set command [string index $line 0]
				set rest [string trim [string range $line 1 end]]
				if {![info exists handlers($command)]} {
					error "Invalid command $line"
				}
				if {![string match {[eEqQ]} $command]} { set confirm {} }
				if {$addr1 == {} && $default_ranges($command) != {}} {
					read-addr-range $default_ranges($command) junk \
							addr1 addr2
				}
				$handlers($command) $addr1 $addr2 $command $rest
			}
			if $debug {
				eval $script
			} else {
				switch [catch $script error] {
					1 {
						# TCL_ERROR
						if $verbose {
							puts-error $error
						} else {
							puts-error ?
						}
						set last_error $error
					}
					2 { return }
				}
			}
		}
		continuation {
			set continuation $line
			set input_mode [lrange $input_mode 1 end]
		}
		text {
			if {$line == "."} {
				set input_mode [lrange $input_mode 1 end]
			} else {
				buffer-put $current_addr $line
				incr current_addr
				incr last_addr
				set dirty 1
			}
		}
	}
}

proc read-addr-range {linein linevar addr1var addr2var} {
	upvar 1 $linevar line $addr1var addr1 $addr2var addr2
	global current_addr last_addr
	set line [string trimleft $linein]
	set addrlist {}
	while 1 {
		set match [read-addr $line line addr]
		switch -exact -- [string index $line 0] {
			, - % {
				# Effectively, % is the same as , .  Read the GNU ed source.
				if $match {
					lappend addrlist $addr
				} else {
					if {$addrlist != {}} {
						lappend addrlist $last_addr
					} else {
						set addrlist [list 1 $last_addr]
					}
				}
				set line [string range $line 1 end]
			}
			; {
				if $match {
					set current_addr $addr
					lappend addrlist $addr
				} else {
					if {$addrlist != {}} {
						lappend addrlist $last_addr
					} else {
						set addrlist [list $current_addr $last_addr]
					}
				}
				set line [string range $line 1 end]
			}
			default { if $match { lappend addrlist $addr } ; break }
		}
	}
	switch [llength $addrlist] {
		0 { set addr1 {} ; set addr2 {} }
		1 { set addr1 [lindex $addrlist 0] ; set addr2 $addr1 }
		default {
			set addr1 [lindex $addrlist [expr [llength $addrlist] - 2]]
			set addr2 [lindex $addrlist [expr [llength $addrlist] - 1]]
			if {$addr1 > $addr2} {
				error "Invalid address range, must be in ascending order"
			}
		}
	}
}

proc read-addr {linein linevar addrvar} {
	upvar 1 $addrvar addr $linevar line
	global current_addr last_addr last_regexp
	set line [string trimleft $linein]
	set matched 1
	switch -exact -- [string index $line 0] {
		0 - 1 - 2 - 3 - 4 - 5 - 6 - 7 - 8 - 9 {
			regexp {^([0-9]+)(.*)$} $line garbage addr line
		}
		. { set addr $current_addr ; set line [string range $line 1 end] }
		$ { set addr $last_addr    ; set line [string range $line 1 end] }
		/ {
			set regexp [regexp-compile [parse-pattern line delimiter]]
			set addr [buffer-search-forwards  $regexp [expr $current_addr+1]]
		}
		? {
			set regexp [regexp-compile [parse-pattern line delimiter]]
			set addr [buffer-search-backwards $regexp [expr $current_addr-1]]
		}
		' {
			set c [string index $line 1]
			set addr [buffer-mark-get $c]
			if {$addr == {}} { error "No such mark \"$c\"" }
			set line [string range $line 2 end]
		}
		default { set matched 0 }
	}
	while 1 {
		switch -exact -- [string index $line 0] {
			" " - \t - 0 - 1 - 2 - 3 - 4 - 5 - 6 - 7 - 8 - 9 {
				if {[regexp {^[ \t]*([0-9]+)(.*)$} $line junk i line]} {
					if {!$matched} { set addr 0 ; set matched 1 }
					incr addr $i
				} else {
					set line [string trimleft $line]
				}
			}
			+ {
				if {!$matched} { set addr $current_addr ; set matched 1 }
				if {[regexp {^\+[ \t]*([0-9]+)(.*)$} $line junk i line]} {
					incr addr $i
				} else {
					incr addr
					set line [string range $line 1 end]
				}
			}
			- - ^ {
				if {!$matched} { set addr $current_addr ; set matched 1 }
				if {[regexp {^[-^][ \t]*([0-9]+)(.*)$} $line junk i line]} {
					incr addr -$i
				} else {
					incr addr -1
					set line [string range $line 1 end]
				}
			}
			default { break }
		}
	}
	if {$matched && ($addr < 0 || $last_addr < $addr)} {
		error "Address out of range: [string range $linein 0 [expr \
				[string length $linein] - [string length $line] - 1]] == $addr"
	}
	return $matched
}


######################################################################
# batch-process-lines

set batching 0
proc batch-process-lines {list} {
	global batching batch_lines
	if {$batching} {
		error "Cannot nest batched functions such as g, v, and u"
	}
	set batching 1
	set batch_lines $list
	after idle batch-process-line
	vwait batching
}
proc batch-process-line {} {
	global batching batch_lines
	if {[llength $batch_lines] == 0} {
		set batching 0
	} else {
		set id [after idle batch-process-line]
		set line [lindex $batch_lines 0]
		set batch_lines [lrange $batch_lines 1 end]
		process-line $line
	}
}
