;;; ede-proj-info.el --- EDE Generic Project texinfo support

;;;  Copyright (C) 1998, 1999, 2000, 2001  Eric M. Ludlam

;; Author: Eric M. Ludlam <zappo@gnu.org>
;; Keywords: project, make
;; RCS: $Id: ede-proj-info.el,v 1.12 2001/04/27 00:18:05 zappo Exp $

;; This software is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This software is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:
;;
;; Handle texinfo in and EDE Project file.

(require 'ede-pmake)

;;; Code:
(defclass ede-proj-target-makefile-info (ede-proj-target-makefile)
  ((menu :initform nil)
   (keybindings :initform nil)
   (availablecompilers :initform (ede-makeinfo-compiler
				  ede-texi2html-compiler))
   (sourcetype :initform (ede-makeinfo-source))
   (mainmenu :initarg :mainmenu
	     :initform ""
	     :type string
	     :custom string
	     :documentation "The main menu resides in this file.
All other sources should be included independently."))
  "Target for a single info file.")

(defvar ede-makeinfo-source
  (ede-sourcecode "ede-makeinfo-source"
		  :name "Texinfo"
		  :sourcepattern "\\.texi?$"
		  :garbagepattern '("*.info" "*.html"))
  "Texinfo source code definition.")

(defvar ede-makeinfo-compiler
  (ede-compiler
   "ede-makeinfo-compiler"
   :name "makeinfo"
   :variables '(("MAKEINFO" . "makeinfo"))
   :commands '("$(MAKEINFO) -o $@ $<")
   :autoconf '(("AC_CHECK_PROG" . "MAKEINFO, makeinfo"))
   :sourcetype '(ede-makeinfo-source)
   )
  "Compile texinfo files into info files.")

(defvar ede-texi2html-compiler
  (ede-compiler
   "ede-texi2html-compiler"
   :name "texi2html"
   :variables '(("TEXI2HTML" . "makeinfo -html"))
   :commands '("makeinfo -o $@ $<")
   :sourcetype '(ede-makeinfo-source)
   )
  "Compile texinfo files into html files.")

;;; Makefile generation
;;
(defmethod ede-proj-configure-add-missing
  ((this ede-proj-target-makefile-info))
  "Query if any files needed by THIS provided by automake are missing.
Results in --add-missing being passed to automake."
  (not (ede-expand-filename (ede-toplevel) "texinfo.tex")))

(defmethod ede-proj-makefile-sourcevar ((this ede-proj-target-makefile-info))
  "Return the variable name for THIS's sources."
  (concat (ede-pmake-varname this) "_TEXINFOS"))

(defmethod ede-proj-makefile-insert-source-variables
  ((this ede-proj-target-makefile-info) &optional moresource)
  "Insert the source variables needed by THIS info target.
Optional argument MORESOURCE is a list of additional sources to add to the
sources variable.
Does the usual for Makefile mode, but splits source into two variables
when working in Automake mode."
  (if (not (ede-proj-automake-p))
      (call-next-method)
    (let* ((sv (ede-proj-makefile-sourcevar this))
	   (src (copy-sequence (oref this source)))
	   (menu (or (oref this menu) (car src))))
      (setq src (delq menu src))
      ;; the info_TEXINFOS variable is probably shared
      (ede-pmake-insert-variable-shared "info_TEXINFOS"
	(insert menu))
      ;; Now insert the rest of the source elsewhere
      (ede-pmake-insert-variable-shared sv
	(insert (mapconcat 'identity src " ")))
      (if moresource
	  (error "Texinfo files should not have moresource")))))

(defmethod ede-proj-makefile-target-name ((this ede-proj-target-makefile-info))
  "Return the name of the main target for THIS target."
  ;; The target should be the main-menu file name translated to .info.
  (let ((mm (if (not (string= (oref this mainmenu) ""))
		(oref this mainmenu)
	      (car (oref this source)))))
    (concat (file-name-sans-extension mm) ".info")))

(defmethod ede-proj-makefile-insert-dist-dependencies ((this ede-proj-target-makefile-info))
  "Insert any symbols that the DIST rule should depend on.
Texinfo files want to insert generated `.info' files.
Argument THIS is the target which needs to insert an info file."
  ;; In some cases, this is ONLY the index file.  That should generally
  ;; be ok.
  (insert " " (ede-proj-makefile-target-name this))
  )

(defmethod ede-proj-makefile-insert-dist-filepatterns ((this ede-proj-target-makefile-info))
  "Insert any symbols that the DIST rule should depend on.
Texinfo files want to insert generated `.info' files.
Argument THIS is the target which needs to insert an info file."
  ;; In some cases, this is ONLY the index file.  That should generally
  ;; be ok.
  (insert " " (ede-proj-makefile-target-name this) "*")
  )

;  (let ((n (ede-name this)))
;    (if (string-match "\\.info$" n)
;	n
;      (concat n ".info"))))

(defmethod object-write ((this ede-proj-target-makefile-info))
  "Before commiting any change to THIS, make sure the mainmenu is first."
   (let ((mm (oref this mainmenu))
	 (s (oref this source))
	 (nl nil))
     (if (or (string= mm "") (not mm) (string= mm (car s)))
	 nil
       ;; Make sure that MM is first in the list of items.
       (setq nl (cons mm (delq mm s)))
       (oset this source nl)))
   (call-next-method))

(defmethod ede-documentation ((this ede-proj-target-makefile-info))
  "Return a list of files that provides documentation.
Documentation is not for object THIS, but is provided by THIS for other
files in the project."
  (oref this source))

(provide 'ede-proj-info)

;;; ede-proj-info.el ends here
