/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt2x00
	Abstract: rt2x00 global information.
	Supported chipsets: rt2570 & rt2573.
 */

#ifndef RT2X00USB_H
#define RT2X00USB_H

/*
 * Register defines.
 * When register access attempts should be repeated
 * only REGISTER_BUSY_COUNT attempts with a delay
 * of REGISTER_BUSY_DELAY us should be taken.
 * For USB vendor requests we need to pass a timeout
 * time in ms, for this we use the REGISTER_TIMEOUT,
 * however when loading firmware a higher value is
 * required. For that we use the REGISTER_TIMEOUT_FIRMWARE.
 */
#define REGISTER_BUSY_COUNT		5
#define REGISTER_BUSY_DELAY		100
#define REGISTER_TIMEOUT		20
#define REGISTER_TIMEOUT_FIRMWARE	1000

/*
 * Device specific structure.
 */
struct rt2x00_usb{
	/*
	 * Device flags.
	 */
	unsigned int				flags;

	/*
	 * usb device structure.
	 */
	struct usb_interface			*usb_intf;

	/*
	 * IEEE80211 Configuration block.
	 */
	struct ieee80211_hw			hw;

	/*
	 * Chipset identification.
	 */
	struct rt2x00_chip			chip;

	/*
	 * RF values for channel switching.
	 */
	u32					rf1;
	u32					rf2;
	u32					rf3;
	u32					rf4;

	u32					tx_power;

	/*
	 * Queue for deferred interrupt work.
	 */
	struct workqueue_struct		*workqueue;

	/*
	 * Interface identication required for requesting beacons
	 * from dscape stack.
	 */
	int					interface_id;

	/*
	 * Interface configuration.
	 */
	u8					interface_bssid[ETH_ALEN];
	short					interface_promisc;

	/*
	 * Current working mode.
	 */
	int					interface_type;

	/*
	 * Monitor mode count, the number of interfaces
	 * in monitor mode that that have been added.
	 */
	int					monitor_count;

	/*
	 * Frequency offset (for rt73usb).
	 */
	u8					freq_offset;

	/*
	 * Led status
	 */
	u8					led_mode;

	/*
	 * LED register (for rt73usb).
	 */
	u16					led_reg;

	/*
	 * EEPROM BBP data.
	 */
	u16					*eeprom;

	/*
	 * Low level statistics which will have
	 * to be kept up to date while device is running.
	 */
	struct ieee80211_low_level_stats	low_level_stats;

	/*
	 * RX configuration information.
	 */
	struct ieee80211_rx_status		rx_params;

	/*
	 * work structure for configuration changes.
	 */
	struct work_struct			config_work;

	/*
	 * work structure for interface configuration.
	 */
	struct work_struct			interface_work;

	/*
	 * Scanning structure and workqueue
	 * required for scheduling scanning requests.
	 */
	struct work_struct			scan_work;
	struct scanning			*scan;

	/*
	 * Data rings for both RX and TX.
	 * The first entries must be the normal TX
	 * rings, followed by a possible ATIM ring
	 * (when atim is used atim_available must be set)
	 * after that the beacon and RX ring follow.
	 */
	struct data_ring			*ring;
};

static inline struct data_ring* rt2x00usb_get_ring(
	struct rt2x00_usb *rt2x00usb, const int ring)
{
	int atim;

	atim = GET_FLAG(rt2x00usb, DEVICE_SUPPORT_ATIM);

	/*
	 * Check if the rings have been allocated.
	 */
	if (!rt2x00usb->ring)
		return NULL;

	/*
	 * Check for beacon ring, the beacon ring
	 * is located behing the normal TX and, when available
	 * the atim ring.
	 */
	if (ring == IEEE80211_TX_QUEUE_BEACON)
		return &rt2x00usb->ring[rt2x00usb->hw.queues + atim];

	if (ring == IEEE80211_TX_QUEUE_AFTER_BEACON && atim)
		return &rt2x00usb->ring[rt2x00usb->hw.queues];

	/*
	 * Make sure the requested ring does not exceed
	 * the maximum number of rings.
	 */
	if (ring < rt2x00usb->hw.queues)
		return &rt2x00usb->ring[ring];

	return NULL;
}

/*
 * USB request types.
 */
#define USB_VENDOR_REQUEST		( USB_TYPE_VENDOR | USB_RECIP_DEVICE )
#define USB_VENDOR_REQUEST_IN		( USB_DIR_IN | USB_VENDOR_REQUEST )
#define USB_VENDOR_REQUEST_OUT		( USB_DIR_OUT | USB_VENDOR_REQUEST )

/*
 * USB vendor commands.
 */
#define USB_DEVICE_MODE			0x01
#define USB_SINGLE_WRITE		0x02
#define USB_SINGLE_READ			0x03
#define USB_MULTI_WRITE			0x06
#define USB_MULTI_READ			0x07
#define USB_EEPROM_WRITE		0x08
#define USB_EEPROM_READ			0x09
#define USB_LED_CONTROL			0x0a	/* RT73USB */
#define USB_RX_CONTROL			0x0c

/*
 * Device modes offset
 */
#define USB_MODE_RESET			0x01
#define USB_MODE_UNPLUG			0x02
#define USB_MODE_FUNCTION		0x03
#define USB_MODE_TEST			0x04
#define USB_MODE_SLEEP			0x07	/* RT73USB */
#define USB_MODE_FIRMWARE		0x08	/* RT73USB */
#define USB_MODE_WAKEUP			0x09	/* RT73USB */

/*
 * Vendor request command for sending
 * vendor specific instructions to the device.
 */
static inline int rt2x00usb_vendor_request(
	const struct rt2x00_usb *rt2x00usb,
	const u8 request,
	const u8 type,
	const u16 offset,
	u32 value,
	void *buffer,
	const u16 buffer_length,
	const u16 timeout)
{
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00usb->usb_intf);
	int status;
	u8 i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		status = usb_control_msg(
			usb_dev,
			(type == USB_VENDOR_REQUEST_IN) ?
				usb_rcvctrlpipe(usb_dev, 0) :
				usb_sndctrlpipe(usb_dev, 0),
			request, type, value, offset, buffer, buffer_length,
			timeout);
		if (status >= 0)
			return 0;
	}

	return status;
}

#endif /* RT2X00USB_H */
