#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <inttypes.h>
#include <fcntl.h>
#include <errno.h>
#include <pthread.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/wait.h>
#include <linux/if.h>
#include <linux/if_tun.h>

#include <xs.h>
#include <xenctrl.h>
#include <xen/io/xenbus.h>
#include <xen/io/netif.h>

#include "shared.h"
#include "list.h"
#include "daemon.h"
#include "xenbackd.h"
#include "checksum.h"

/* ------------------------------------------------------------- */

struct netstats {
    uint64_t              bytes_rx;
    uint64_t              bytes_tx;
    uint64_t              packet_rx;
    uint64_t              packet_tx;
    uint64_t              error_rx;
    uint64_t              error_tx;
    uint64_t              error_drop;
    uint64_t              wakeups;
};

struct netdev {
    struct xendev         xendev;  /* must be first */
    char                  *mac;
    char                  *bridge;
    char                  *interface;
    int                   tx_work;
    int                   tx_ring_ref;
    int                   rx_ring_ref;
    struct netif_tx_sring *txs;
    struct netif_rx_sring *rxs;
    netif_tx_back_ring_t  tx_ring;
    netif_rx_back_ring_t  rx_ring;

    /* worker threads */
    pthread_t             tx_thread;
    pthread_t             rx_thread;

    /* tun device */
    int                   tun;
    int                   nocsum;
    char                  ifname[IFNAMSIZ];

    /* statistics */
    struct netstats       st;
    struct netstats       st_prev;
    struct timeval        tv_prev;
    struct netstats       st_rate;
    struct netstats       st_peak;
    int                   st_count;
};

static char *be_name = "netbackd";
static int stat_secs = 10;

/* prototypes */
static void netdev_nocsum(struct netdev *netdev, int nocsum);

/* ------------------------------------------------------------- */

static void net_tx_thread_gosleep(struct netdev *netdev)
{
    if (!netdev->tx_work)
	if (-1 != wait_for_event(&netdev->xendev))
	    netdev->st.wakeups++;
    netdev->tx_work = 0;
}
 
static void net_tx_response(struct netdev *netdev, netif_tx_request_t *txp, int8_t st)
{
    RING_IDX i = netdev->tx_ring.rsp_prod_pvt;
    netif_tx_response_t *resp;
    int notify;
    
    resp = RING_GET_RESPONSE(&netdev->tx_ring, i);
    resp->id     = txp->id;
    resp->status = st;

    if (NETIF_RSP_OKAY == st) {
	netdev->st.packet_tx += 1;
	netdev->st.bytes_tx  += txp->size;
    } else {
	netdev->st.error_tx  += 1;
    }
    
#if 0
    if (txp->flags & NETTXF_extra_info)
	RING_GET_RESPONSE(&netdev->tx_ring, ++i)->status = NETIF_RSP_NULL;
#endif
    
    netdev->tx_ring.rsp_prod_pvt = ++i;
    RING_PUSH_RESPONSES_AND_CHECK_NOTIFY(&netdev->tx_ring, notify);
    if (notify)
	xc_evtchn_notify(netdev->xendev.evtchnd, netdev->xendev.local_port);
    
    if (i == netdev->tx_ring.req_cons) {
	int more_to_do;
	RING_FINAL_CHECK_FOR_REQUESTS(&netdev->tx_ring, more_to_do);
	if (more_to_do)
	    netdev->tx_work++;
    }
}

static void net_tx_error(struct netdev *netdev, netif_tx_request_t *txp, RING_IDX end)
{
#if 0
        /*
	 * Hmm, why netback fails everything in the ring?
	 * Should we do that even when not supporting SG and TSO?
	 */
	RING_IDX cons = netdev->tx_ring.req_cons;

	do {
		make_tx_response(netif, txp, NETIF_RSP_ERROR);
		if (cons >= end)
			break;
		txp = RING_GET_REQUEST(&netdev->tx_ring, cons++);
	} while (1);
	netdev->tx_ring.req_cons = cons;
	netif_schedule_work(netif);
	netif_put(netif);
#else
	net_tx_response(netdev, txp, NETIF_RSP_ERROR);
#endif
}

static void *net_tx_thread(void *arg)
{
    struct netdev *netdev = arg;
    netif_tx_request_t txreq;
    RING_IDX rc, rp;
    void *page;
    int len, status;
    
    d1printf("%s: start\n", __FUNCTION__);
    for (;;) {
	if (netdev->xendev.state != XENDEV_CONNECTED)
	    break;

	rc = netdev->tx_ring.req_cons;
	rp = netdev->tx_ring.sring->req_prod;
	rmb(); /* Ensure we see queued requests up to 'rp'. */
	
	while ((rc != rp)) {
	    if (RING_REQUEST_CONS_OVERFLOW(&netdev->tx_ring, rc))
		break;
	    memcpy(&txreq, RING_GET_REQUEST(&netdev->tx_ring, rc), sizeof(txreq));
	    netdev->tx_ring.req_cons = ++rc;

#if 1
	    /* should not happen in theory, we don't announce the *
	     * feature-{sg,gso,whatelse} flags in xenstore (yet?) */
	    if (txreq.flags & NETTXF_extra_info) {
		d1printf("%s: FIXME: extra info flag\n", __FUNCTION__);
		net_tx_error(netdev, &txreq, rc);
		continue;
	    }
	    if (txreq.flags & NETTXF_more_data) {
		d1printf("%s: FIXME: more data flag\n", __FUNCTION__);
		net_tx_error(netdev, &txreq, rc);
		continue;
	    }
#endif

	    if (txreq.size < 14) {
		d1printf("%s: bad packet size: %d\n", __FUNCTION__, txreq.size);
		net_tx_error(netdev, &txreq, rc);
		continue;
	    }

	    if ((txreq.offset + txreq.size) > PAGE_SIZE) {
		d1printf("%s: error: page crossing\n", __FUNCTION__);
		net_tx_error(netdev, &txreq, rc);
		continue;
	    }

#if 1
	    d2printf("%s: packet ref %d, off %d, len %d, flags 0x%x%s%s%s%s\n",
		     __FUNCTION__, txreq.gref, txreq.offset, txreq.size, txreq.flags,
		     (txreq.flags & NETTXF_csum_blank)     ? " csum_blank"     : "",
		     (txreq.flags & NETTXF_data_validated) ? " data_validated" : "",
		     (txreq.flags & NETTXF_more_data)      ? " more_data"      : "",
		     (txreq.flags & NETTXF_extra_info)     ? " extra_info"     : "");
#endif

	    page = xc_gnttab_map_grant_ref(netdev->xendev.gnttabdev,
					   netdev->xendev.dom,
					   txreq.gref, PROT_READ);
	    d3printf("%s: map grant ref %d -> %p\n", __FUNCTION__,
		     txreq.gref, page);
	    if (NULL == page) {
		d1printf("%s: error: gref dereference failed\n", __FUNCTION__);
		net_tx_error(netdev, &txreq, rc);
		continue;
	    }
	    if (txreq.flags & NETTXF_csum_blank)
		checksum_calculate(page + txreq.offset, txreq.size);
	    len = write(netdev->tun, page + txreq.offset, txreq.size);
	    xc_gnttab_munmap(netdev->xendev.gnttabdev, page, 1);
	    status = (len == txreq.size) ? NETIF_RSP_OKAY : NETIF_RSP_ERROR;
	    net_tx_response(netdev, &txreq, status);
	}

	net_tx_thread_gosleep(netdev);
    }
    d1printf("%s: exit\n", __FUNCTION__);
    return NULL;
}

/* ------------------------------------------------------------- */

static void net_rx_response(struct netdev *netdev,
			    netif_rx_request_t *req, int8_t st,
			    uint16_t offset, uint16_t size,
			    uint16_t flags)
{
    RING_IDX i = netdev->rx_ring.rsp_prod_pvt;
    netif_rx_response_t *resp;
    int notify;
    
    resp = RING_GET_RESPONSE(&netdev->rx_ring, i);
    resp->offset     = offset;
    resp->flags      = flags;
    resp->id         = req->id;
    resp->status     = (int16_t)size;
    if (st < 0) {
	resp->status = (int16_t)st;
	netdev->st.error_rx  += 1;
    } else {
	netdev->st.packet_rx += 1;
	netdev->st.bytes_rx  += size;
    }
    
#if 1
    d2printf("%s: idx %d, status %d, flags 0x%x\n",
	     __FUNCTION__, i, resp->status, resp->flags);
#endif
    
    netdev->rx_ring.rsp_prod_pvt = ++i;
    RING_PUSH_RESPONSES_AND_CHECK_NOTIFY(&netdev->rx_ring, notify);
    if (notify)
	xc_evtchn_notify(netdev->xendev.evtchnd, netdev->xendev.local_port);
}

#define NET_IP_ALIGN 2

static void *net_rx_thread(void *arg)
{
    netif_rx_request_t rxreq;
    struct netdev *netdev = arg;
    RING_IDX rc, rp;
    int len, status;
    void *page;
    
    d1printf("%s: start\n", __FUNCTION__);
    for (;;) {
	if (netdev->xendev.state != XENDEV_CONNECTED)
	    break;
	if (1 != wait_for_data(netdev->tun, 0))
	    continue;

	rc = netdev->rx_ring.req_cons;
	rp = netdev->rx_ring.sring->req_prod;
	rmb(); /* Ensure we see queued requests up to 'rp'. */
	d2printf("%s: %d %d\n", __FUNCTION__, rc, rp);
	
	if (rc == rp || RING_REQUEST_CONS_OVERFLOW(&netdev->rx_ring, rc)) {
	    char buf[PAGE_SIZE];
	    d2printf("%s: no buffer, drop packet\n", __FUNCTION__);
	    read(netdev->tun, buf, sizeof(buf));
	    netdev->st.error_drop++;
	    continue;
	}

	memcpy(&rxreq, RING_GET_REQUEST(&netdev->rx_ring, rc), sizeof(rxreq));
	netdev->rx_ring.req_cons = ++rc;

	page = xc_gnttab_map_grant_ref(netdev->xendev.gnttabdev,
				       netdev->xendev.dom,
				       rxreq.gref, PROT_WRITE);
	d3printf("%s: map grant ref %d -> %p\n", __FUNCTION__,
		 rxreq.gref, page);
	if (NULL == page) {
	    d1printf("%s: error: gref dereference failed\n", __FUNCTION__);
	    net_rx_response(netdev, &rxreq, NETIF_RSP_ERROR, 0, 0, 0);
	    continue;
	}
	len = read(netdev->tun, page + NET_IP_ALIGN, PAGE_SIZE - NET_IP_ALIGN);
	xc_gnttab_munmap(netdev->xendev.gnttabdev, page, 1);
	status = (len > 0) ? NETIF_RSP_OKAY : NETIF_RSP_ERROR;
	net_rx_response(netdev, &rxreq, status, NET_IP_ALIGN, len, 0);
    }
    d1printf("%s: exit\n", __FUNCTION__);
    return NULL;
}

/* ------------------------------------------------------------- */

static void netdev_bridge_init(struct netdev *netdev)
{
    pid_t pid;
    int status;

    /* FIXME: should run script instead */
    if (0 == (pid = fork())) {
	execlp("brctl", "brctl", "addif", netdev->bridge, netdev->ifname, NULL);
	d1printf("%s: exec brctl: %s\n", __FUNCTION__, strerror(errno));
	exit(1);
    }
    waitpid(pid, &status, 0);

    if (0 == (pid = fork())) {
	execlp("ip", "ip", "link", "set", netdev->ifname, "up", NULL);
	d1printf("%s: exec ip: %s\n", __FUNCTION__, strerror(errno));
	exit(1);
    }
    waitpid(pid, &status, 0);
}

static void netdev_nocsum(struct netdev *netdev, int nocsum)
{
    if (netdev->nocsum == nocsum)
	return;
    netdev->nocsum = nocsum;
    if (-1 == ioctl(netdev->tun, TUNSETNOCSUM, netdev->nocsum))
	d1printf("ioctl TUNSETNOCSUM %d: %s\n", netdev->nocsum, strerror(errno));
    else
	d1printf("ioctl TUNSETNOCSUM %d: ok\n", netdev->nocsum);
}
	    
/* ------------------------------------------------------------- */

static void net_setup_backend(struct netdev *netdev)
{
    struct ifreq ifreq;
    
    if (netdev->xendev.state != XENDEV_PROBED)
	return;
    if (!netdev->xendev.online)
	return;
    
    /* read xenstore entries */
    if (NULL == netdev->mac)
	netdev->mac = read_be_str(&netdev->xendev, "mac");
    if (NULL == netdev->bridge)
	netdev->bridge = read_be_str(&netdev->xendev, "bridge");
    if (NULL == netdev->interface)
	netdev->interface = read_be_str(&netdev->xendev, "ifname");

    /* do we have all we need? */
    if (NULL == netdev->mac)
	return;
    if (NULL == netdev->interface && NULL == netdev->bridge)
	return;

    /* setup tun device */
    if (-1 == netdev->tun) {
	d1printf("%s: xenstore: mac=\"%s\" if=\"%s\" br=\"%s\"\n", __FUNCTION__,
		 netdev->mac, netdev->interface, netdev->bridge);
	netdev->tun = open("/dev/net/tun", O_RDWR);
	if (-1 == netdev->tun) {
	    d1printf("can't open %s: %s\n", "/dev/net/tun", strerror(errno));
	    return;
	}
	memset(&ifreq, 0, sizeof(ifreq));
	ifreq.ifr_flags = IFF_TAP | IFF_NO_PI;
	if (netdev->interface) {
	    strncpy(ifreq.ifr_name, netdev->interface, IFNAMSIZ);
	} else {
	    strncpy(ifreq.ifr_name, "xenner%d", IFNAMSIZ);
	}
	if (ioctl(netdev->tun, TUNSETIFF, &ifreq) < 0) {
	    d1printf("ioctl TUNSETIFF(%s): %s\n", ifreq.ifr_name, strerror(errno));
	    goto err_tun;
	}
	strncpy(netdev->ifname, ifreq.ifr_name, IFNAMSIZ);
	d1printf("%s: created interface %s\n", __FUNCTION__, netdev->ifname);

	netdev->nocsum = -1;
	netdev_nocsum(netdev, 1);
	if (netdev->bridge)
	    netdev_bridge_init(netdev);
	if (-1 == ioctl(netdev->tun, TUNSETPERSIST, 0))
	    d1printf("ioctl TUNSETPERSIST off: %s\n", strerror(errno));
	    
	write_be_str(&netdev->xendev, "hotplug-status", "connected");
    }

    /* fill info */
    write_be_int(&netdev->xendev, "feature-rx-copy", 1);
    write_be_int(&netdev->xendev, "feature-rx-flip", 0);

    change_state_xendev(&netdev->xendev, XenbusStateInitWait);
    netdev->xendev.state = XENDEV_INITIALISED;
    return;

err_tun:
    close(netdev->tun);
    netdev->tun = -1;
    return;
}

static void net_connect_frontend(struct netdev *netdev)
{
    int rx_copy;
    
    if (netdev->xendev.state != XENDEV_INITIALISED)
	return;

    netdev->tx_ring_ref = read_fe_int(&netdev->xendev, "tx-ring-ref");
    netdev->rx_ring_ref = read_fe_int(&netdev->xendev, "rx-ring-ref");
    netdev->xendev.remote_port = read_fe_int(&netdev->xendev, "event-channel");
    rx_copy = read_fe_int(&netdev->xendev, "request-rx-copy");
    if (0 == rx_copy ||
	0 == netdev->tx_ring_ref ||
	0 == netdev->rx_ring_ref ||
	0 == netdev->xendev.remote_port)
	return;

    netdev->txs = xc_gnttab_map_grant_ref(netdev->xendev.gnttabdev,
					  netdev->xendev.dom,
					  netdev->tx_ring_ref,
					  PROT_READ | PROT_WRITE);
    netdev->rxs = xc_gnttab_map_grant_ref(netdev->xendev.gnttabdev,
					  netdev->xendev.dom,
					  netdev->rx_ring_ref,
					  PROT_READ | PROT_WRITE);
    if (!netdev->txs || !netdev->rxs)
	return;
    BACK_RING_INIT(&netdev->tx_ring, netdev->txs, PAGE_SIZE);
    BACK_RING_INIT(&netdev->rx_ring, netdev->rxs, PAGE_SIZE);

    netdev->xendev.local_port = xc_evtchn_bind_interdomain
	(netdev->xendev.evtchnd, netdev->xendev.dom, netdev->xendev.remote_port);
    xc_evtchn_unmask(netdev->xendev.evtchnd, netdev->xendev.local_port);

    d1printf("%s: ok: tx-ring-ref %d, rx-ring-ref %d, "
	     "remote port %d, local port %d\n", __FUNCTION__,
	     netdev->tx_ring_ref, netdev->rx_ring_ref,
	     netdev->xendev.remote_port, netdev->xendev.local_port);

    change_state_xendev(&netdev->xendev, XenbusStateConnected);
    netdev->xendev.state = XENDEV_CONNECTED;

    pthread_create(&netdev->tx_thread, NULL, net_tx_thread, netdev);
    pthread_create(&netdev->rx_thread, NULL, net_rx_thread, netdev);
}

static void net_check_ring(struct netdev *netdev)
{
    if (netdev->tx_thread)
	pthread_kill(netdev->tx_thread, SIGUSR2);
}

static void net_disconnect(struct netdev *netdev, enum xenbus_state state)
{
    void *dummy;

    if (netdev->xendev.state == XENDEV_DISCONNECTED)
	return;
    netdev->xendev.state = XENDEV_DISCONNECTED;

    if (netdev->tx_thread) {
	pthread_kill(netdev->tx_thread, SIGUSR2);
	pthread_join(netdev->tx_thread, &dummy);
    }
    if (netdev->rx_thread) {
	pthread_kill(netdev->rx_thread, SIGUSR2);
	pthread_join(netdev->rx_thread, &dummy);
    }

    if (-1 != netdev->tun) {
	close(netdev->tun);
	netdev->tun = -1;
    }
    if (netdev->xendev.local_port) {
	xc_evtchn_unbind(netdev->xendev.evtchnd, netdev->xendev.local_port);
	netdev->xendev.local_port = 0;
    }

    if (netdev->txs) {
	xc_gnttab_munmap(netdev->xendev.gnttabdev, netdev->txs, 1);
	netdev->txs = NULL;
    }
    if (netdev->rxs) {
	xc_gnttab_munmap(netdev->xendev.gnttabdev, netdev->rxs, 1);
	netdev->rxs = NULL;
    }

    if (state)
	change_state_xendev(&netdev->xendev, state);
}

/* ------------------------------------------------------------- */

static int net_alloc(struct xendev *xendev)
{
    struct netdev *netdev = container_of(xendev, struct netdev, xendev);

    d1printf("%s: %p\n", __FUNCTION__, netdev);
    netdev->xendev.state = XENDEV_PROBED;
    netdev->tun = -1;
    return 0;
}

static int net_setup(struct xendev *xendev)
{
    struct netdev *netdev = container_of(xendev, struct netdev, xendev);

    d1printf("%s: %p\n", __FUNCTION__, netdev);
    net_setup_backend(netdev);
    net_connect_frontend(netdev);
    net_check_ring(netdev);
    return 0;
}

static int net_backend(struct xendev *xendev, char *node, char *val)
{
    struct netdev *netdev = container_of(xendev, struct netdev, xendev);

    d1printf("%s: %p \"%s\" = \"%s\"\n", __FUNCTION__, netdev, node,
	     val ? val : "<deleted>");
    net_setup_backend(netdev);
    return 0;
}

static int net_frontend(struct xendev *xendev, char *node, char *val)
{
    struct netdev *netdev = container_of(xendev, struct netdev, xendev);

    d1printf("%s: %p \"%s\" = \"%s\"\n", __FUNCTION__, netdev, node,
	     val ? val : "<deleted>");
    switch (xendev->fe_state) {
    case XenbusStateInitialised:
    case XenbusStateConnected:
	net_connect_frontend(netdev);
	net_check_ring(netdev);
	break;

    case XenbusStateClosing:
    case XenbusStateClosed:
	net_disconnect(netdev, xendev->fe_state);
	break;

    case XenbusStateInitialising:
    case XenbusStateInitWait:
    case XenbusStateUnknown:
	/* make gcc happy */
	break;
    }
    return 0;
}

static void net_rates(struct netstats *now, struct netstats *last,
		      struct netstats *rate, struct netstats *peak,
		      int msecs)
{
    uint64_t *n = (uint64_t*) now;
    uint64_t *l = (uint64_t*) last;
    uint64_t *r = (uint64_t*) rate;
    uint64_t *p = (uint64_t*) peak;
    int i, count = sizeof(struct netstats) / sizeof(uint64_t);

    for (i = 0; i < count; i++) {
	r[i] = (n[i] - l[i]) * 1000 / msecs;
	if (p[i] < r[i])
	    p[i] = r[i];
    }
}

static void net_print_rate(char *type, struct netstats *st, int msecs)
{
    d1printf("  i/o %s, MBit/s:"
	     "  rx %5" PRId64 ","
	     "  tx %5" PRId64 "  |"
	     "  wake %4" PRId64 ","
	     "  drop %3" PRId64 "  |"
	     "  %2d.%03d sec\n",
	     type,
	     st->bytes_rx * 8 / 1024 / 1024,
	     st->bytes_tx * 8 / 1024 / 1024,
	     st->wakeups,
	     st->error_drop,
	     msecs / 1000, msecs % 1000);
}

static void net_stats(struct xendev *xendev, FILE *fp)
{
    struct netdev *netdev = container_of(xendev, struct netdev, xendev);
    struct netstats *st = &netdev->st;
    struct timeval tv;

    fprintf(fp,
	    "    tx: bytes %" PRId64 " packets %" PRId64 "\n"
	    "    rx: bytes %" PRId64 " packets %" PRId64 "\n"
	    "    error: tx %" PRId64 " rx %" PRId64 " drop %" PRId64 "\n",
	    st->bytes_tx, st->packet_tx,
	    st->bytes_rx, st->packet_rx,
	    st->error_tx, st->error_rx, st->error_drop);

    gettimeofday(&tv, NULL);
    if (netdev->tv_prev.tv_sec) {
	uint32_t msecs = timediff_msecs(&tv, &netdev->tv_prev);
	net_rates(st, &netdev->st_prev, &netdev->st_rate, &netdev->st_peak, msecs);
	fprintf(fp,
		"    rate : rx %" PRId64 " tx %" PRId64 " wake %" PRId64 "\n"
		"    peak : rx %" PRId64 " tx %" PRId64 " wake %" PRId64 "\n",
		netdev->st_rate.bytes_rx,
		netdev->st_rate.bytes_tx,
		netdev->st_rate.wakeups,
		netdev->st_peak.bytes_rx,
		netdev->st_peak.bytes_tx,
		netdev->st_peak.wakeups);
	net_print_rate("rate", &netdev->st_rate, msecs);
	if (!(++netdev->st_count % 10))
	    net_print_rate("PEAK", &netdev->st_peak, 0);
    }
    netdev->tv_prev = tv;
    netdev->st_prev = *st;
}

static int net_free(struct xendev *xendev)
{
    struct netdev *netdev = container_of(xendev, struct netdev, xendev);
 
    d1printf("%s: %p\n", __FUNCTION__, netdev);
    net_disconnect(netdev, 0);
    return 0;
}

static struct devops netdev_ops = {
    .size  = sizeof(struct netdev),
    .alloc = net_alloc,
    .setup = net_setup,
    .xs_be = net_backend,
    .xs_fe = net_frontend,
    .stats = net_stats,
    .free  = net_free,
};

/* ------------------------------------------------------------- */

static void usage(FILE *fp)
{
    fprintf(fp,
	    "\n"
	    "netbackd --  xenner network backend daemon\n"
	    "\n"
	    "usage: netbackd [options]\n"
	    "options:\n"
	    "   -h            print this text\n"
	    "   -d            increase debuglevel\n"
	    "   -p <file>     specify pidfile\n"
	    "   -l <file>     specify logfile\n"
	    "   -n <name>     backend name                  [%s]\n"
	    "   -i <secs>     stats interval                [%d]\n"
	    "\n"
	    "-- \n"
	    "(c) 2007 Gerd Hoffmann <kraxel@redhat.com>\n"
	    "\n",
	    be_name, stat_secs);
}

int main(int argc, char *argv[])
{
    int c;

    for (;;) {
        if (-1 == (c = getopt(argc, argv, "hdp:l:i:")))
            break;
        switch (c) {
        case 'd':
	    debug++;
	    break;

	case 'i':
	    stat_secs = atoi(optarg);
	    break;
	case 'p':
	    pidfile = optarg;
	    break;
	case 'l':
	    log_setfile(optarg);
	    break;

        case 'h':
            usage(stdout);
            exit(0);
        default:
            usage(stderr);
            exit(1);
        }
    }

    return mainloop(&netdev_ops, be_name, stat_secs);
}
