#undef PAGE_SIZE
#undef PAGE_MASK

#include <linux/kvm.h>
#include <linux/kvm_para.h>
#include "libkvm.h"
#include "kvm-x86.h"

#include "processor.h"

/* vm core file */
#define VMCORE_HEADER  (2 * 1024 * 1024)   /* 2MB page */
#define VMCORE_PATH    "/var/run/xenner"

#define GRANT_FRAMES_MAX (16)
#define GRANT_ENTRIES (GRANT_FRAMES_MAX * PAGE_SIZE / sizeof(struct grant_entry))

#define VCPUS_MAX        (4)

/* useful helper macros */
#define GETNAME(a, i) ( ((i) < sizeof(a)/sizeof(a[0]) && a[i]) ? a[i] : "UNKNOWN")
#define SETBIT(a, n)  ( (a)[(n)/(sizeof(a[0])*8)] |= (1<< ((n)%(sizeof(a[0])*8)) ))
#define TESTBIT(a, n) ( (a)[(n)/(sizeof(a[0])*8)]  & (1<< ((n)%(sizeof(a[0])*8)) ))


/* common flags */
#define ALL_PGFLAGS    (_PAGE_PRESENT | _PAGE_ACCESSED | _PAGE_DIRTY)

/* emulator code+data */
#define EMU_PGFLAGS    (ALL_PGFLAGS | _PAGE_GLOBAL | _PAGE_RW)

/* machphys table */
#define M2P_PGFLAGS_32 (ALL_PGFLAGS | _PAGE_GLOBAL | _PAGE_RW)
#define M2P_PGFLAGS_64 (ALL_PGFLAGS | _PAGE_GLOBAL | _PAGE_RW | _PAGE_USER)

/* linear page tables */
#define LPT_PGFLAGS    (ALL_PGFLAGS | _PAGE_RW)

/* pmd/pgt: pte pointer (map cache) */
#define PGT_PGFLAGS_32 (ALL_PGFLAGS | _PAGE_GLOBAL | _PAGE_RW)
#define PGT_PGFLAGS_64 (ALL_PGFLAGS | _PAGE_GLOBAL | _PAGE_RW | _PAGE_USER)


/* misc xen defines */
#define XEN_HCALL_MAX      64
#define XEN_DEFAULT_PERIOD (10000000ll) /* 10 ms aka 100 Hz */

/* misc xen addresses */
#define XEN_M2P_32         0xfc000000
#define XEN_LPT_32         0xfe000000
#define XEN_MAP_32         0xfe800000
#define XEN_TXT_32         0xff000000

#define XEN_M2P_PAE        0xf5800000
#define XEN_LPT_PAE        0xfd800000
#define XEN_MAP_PAE        0xfe800000
#define XEN_TXT_PAE        0xff000000

#define XEN_M2P_64         0xffff800000000000  //  256 GB, pgd:256
#define XEN_LPT_64         0xffff808000000000  //  512 GB, pgd:257
#define XEN_MAP_64         0xffff820000000000  //  512 GB, pgd:260
#define XEN_RAM_64         0xffff830000000000  //    1 TB, pgd:262,263
#define XEN_DOM_64         0xffff880000000000  //  120 TB, pgd:272+

#define INVALID_M2P_ENTRY  (~0UL)

/* ------------------------------------------------------------------ */
/* statistics                                                         */

#define XEN_FAULT_ILLEGAL_INSTRUCTION          0
#define XEN_FAULT_GENERAL_PROTECTION          10
#define XEN_FAULT_GENERAL_PROTECTION_GUEST    11
#define XEN_FAULT_GENERAL_PROTECTION_EMUINS   12
#define XEN_FAULT_PAGE_FAULT                  20
#define XEN_FAULT_PAGE_FAULT_GUEST            21
#define XEN_FAULT_PAGE_FAULT_FIX_RO           22
#define XEN_FAULT_PAGE_FAULT_FIX_USER         23
#define XEN_FAULT_PAGE_FAULT_FIX_EXTAB        24
#define XEN_FAULT_UPDATE_VA_FIX_RO            25
#define XEN_FAULT_UPDATE_VA_FIX_USER          26
#define XEN_FAULT_SYSCALL                     30
#define XEN_FAULT_INT_80                      31
#define XEN_FAULT_EVENT_CALLBACK              32
#define XEN_FAULT_LAZY_FPU                    33
#define XEN_FAULT_BOUNCE_TRAP                 34
#define XEN_FAULT_MAPS_MAPIT                  40
#define XEN_FAULT_MAPS_REUSE                  41
#define XEN_FAULT_OTHER_CR3_LOAD              50
#define XEN_FAULT_OTHER_SWITCH_MODE           51
#define XEN_FAULT_OTHER_CR3_CACHE_HIT         52
#define XEN_FAULT_OTHER_FLUSH_TLB_ALL         53
#define XEN_FAULT_OTHER_FLUSH_TLB_PAGE        54
#define XEN_FAULT_OTHER_FLUSH_TLB_NONE        55

#define XEN_FAULT_TMP_1                      240
#define XEN_FAULT_TMP_2                      241
#define XEN_FAULT_TMP_3                      242
#define XEN_FAULT_TMP_4                      243
#define XEN_FAULT_TMP_5                      244
#define XEN_FAULT_TMP_6                      245
#define XEN_FAULT_TMP_7                      246
#define XEN_FAULT_TMP_8                      247

#define XEN_FAULT_MAX                        256

#define XEN_EVENT_MAX                         64
#define XEN_ENAME_LEN                         20

/* ------------------------------------------------------------------ */
/* state info                                                         */

#define XENNER_ABI_VERSION 40

struct xenner_info {
    /* state bits info */
    uint32_t      abi_version;
    uint32_t      dying:1;

    uint64_t      vcpus_online;
    uint64_t      vcpus_running;
    uint64_t      vcpus;

    /* statistics */
    uint64_t      hcalls[XEN_HCALL_MAX];
    uint64_t      faults[XEN_FAULT_MAX];
    uint64_t      events[XEN_EVENT_MAX];
    char          enames[XEN_EVENT_MAX * XEN_ENAME_LEN];
};

/* ------------------------------------------------------------------ */

static inline uint32_t fix_sel32(uint32_t sel)
{
    /* fixup DPL: 0 -> 1 */
    if (sel && 0 == (sel & 0x03))
	sel |= 0x01;
    return sel;
}

static inline uint32_t unfix_sel32(uint32_t sel)
{
    /* reverse DPL fixup: 1 -> 0 */
    if (0x01 == (sel & 0x03))
	sel &= ~0x03;
    return sel;
}

static inline void fix_desc32(struct descriptor_32 *desc)
{
    if (desc->b & (1<<15)) {             /* present ?        */
	if (0 == (desc->b & (3 << 13)))  /*   dpl == 0 ?     */
	    desc->b |= 1 << 13;          /*     fix: dpl = 1 */
    }
}    

static inline uint32_t fix_sel64(uint32_t sel)
{
    /* fixup DPL: 0 -> 3 */
    if (sel && 0 == (sel & 0x03))
	sel |= 0x03;
    return sel;
}    

static inline void fix_desc64(struct descriptor_32 *desc)
{
    if (desc->b & (1<<15)) {             /* present ?        */
	if (0 == (desc->b & (3 << 13)))  /*   dpl == 0 ?     */
	    desc->b |= 3 << 13;          /*     fix: dpl = 3 */
    }
}    

