/* $Id: desktop-menu-parser.c,v 1.6 2005/01/03 18:58:55 bmeurer Exp $ */
/*-
 * Copyright (c) 2004 os-cillation
 * All rights reserved.
 *
 * Written by Benedikt Meurer <bm@os-cillation.de>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <libxfcegui4/libxfcegui4.h>

#include "desktop-menu-parser.h"
#include "xfce-menu.h"
#include "xfce-menu-app-item.h"



typedef enum   _ParserState ParserState;
typedef struct _Parser      Parser;



static void start_element_handler (GMarkupParseContext *context,
                                   const gchar         *element_name,
                                   const gchar        **attribute_names,
                                   const gchar        **attribute_values,
                                   gpointer             user_data,
                                   GError             **error);
static void end_element_handler   (GMarkupParseContext  *context,
                                   const gchar          *element_name,
                                   gpointer              user_data,
                                   GError              **error);
static void
include_system_menu (XdgMenu         *system_menu,
                     XdgDesktopCache *cache,
                     GtkWidget       *parent);
static gboolean eval (const gchar *condition);



enum _ParserState
{
  PARSER_START,
  PARSER_START_INCLUDED,
  PARSER_XFDESKTOPMENU,
  PARSER_APP,
  PARSER_MENU,
  PARSER_SEPARATOR,
  PARSER_INCLUDE,
  PARSER_BUILTIN,
  PARSER_TITLE,
};



typedef XFCE_GENERIC_STACK(GtkWidget*)  MenuStack;
typedef XFCE_GENERIC_STACK(ParserState) ParserStack;


struct _Parser
{
  /* file */
  const gchar     *filename;

  /* state */
  MenuStack       *menus;
  ParserStack     *states;

  /* system menu support */
  XdgMenu         *system_menu;
  XdgDesktopCache *cache;
};



static GMarkupParser markup_parser =
{
  start_element_handler,
  end_element_handler,
  NULL,
  NULL,
  NULL,
};



gboolean
desktop_menu_parse (const gchar     *filename,
                    XdgMenu         *system_menu,
                    XdgDesktopCache *cache,
                    GtkWidget       *parent,
                    gboolean         included,
                    GError         **error)
{
  GMarkupParseContext *context;
  Parser               parser;
  gchar               *content;
  gsize                content_len;
  gboolean             failed = TRUE;

  g_return_val_if_fail (filename != NULL, FALSE);
  g_return_val_if_fail (system_menu != NULL, FALSE);
  g_return_val_if_fail (XFCE_IS_MENU (parent), FALSE);

  if (!g_file_get_contents (filename, &content, &content_len, error))
    return FALSE;

  /* initialize the parser */
  parser.filename = filename;
  parser.cache = cache;
  parser.system_menu = system_menu;

  parser.menus = xfce_stack_new (MenuStack);
  xfce_stack_push (parser.menus, parent);

  parser.states = xfce_stack_new (ParserStack);
  xfce_stack_push (parser.states, included ? PARSER_START_INCLUDED : PARSER_START);

  context = g_markup_parse_context_new (&markup_parser, 0, &parser, NULL);

  if (!g_markup_parse_context_parse (context, content, content_len, error))
    goto end;

  if (!g_markup_parse_context_end_parse (context, error))
    goto end;

  failed = FALSE;

end:
  g_markup_parse_context_free (context);
  xfce_stack_free (parser.states);
  xfce_stack_free (parser.menus);
  g_free (content);
  return !failed;
}



static void
start_element_handler (GMarkupParseContext *context,
                       const gchar         *element_name,
                       const gchar        **attribute_names,
                       const gchar        **attribute_values,
                       gpointer             user_data,
                       GError             **error)
{
  const gchar *attr_name = NULL;
  const gchar *attr_icon = NULL;
  const gchar *attr_cmd = NULL;
  const gchar *attr_term = NULL;
  const gchar *attr_snotify = NULL;
  const gchar *attr_visible = NULL;
  const gchar *attr_type = NULL;
  const gchar *attr_tip = NULL;
  const gchar *attr_src = NULL;
  GtkWidget   *submenu;
  GtkWidget   *menu;
  GtkWidget   *item;
  Parser      *parser = (Parser *) user_data;
  gchar       *dir;
  gchar       *file;
  gchar       *spec;
  gint         n;

  menu = xfce_stack_top (parser->menus);

  /* lookup common attributes */
  for (n = 0; attribute_names[n] != NULL; ++n)
    {
      if (strcmp (attribute_names[n], "name") == 0)
        attr_name = attribute_values[n];
      else if (strcmp (attribute_names[n], "icon") == 0)
        attr_icon = attribute_values[n];
      else if (strcmp (attribute_names[n], "cmd") == 0)
        attr_cmd = attribute_values[n];
      else if (strcmp (attribute_names[n], "term") == 0)
        attr_term = attribute_values[n];
      else if (strcmp (attribute_names[n], "snotify") == 0)
        attr_snotify = attribute_values[n];
      else if (strcmp (attribute_names[n], "visible") == 0)
        attr_visible = attribute_values[n];
      else if (strcmp (attribute_names[n], "type") == 0)
        attr_type = attribute_values[n];
      else if (strcmp (attribute_names[n], "tip") == 0)
        attr_tip = attribute_values[n];
      else if (strcmp (attribute_names[n], "src") == 0)
        attr_src = attribute_values[n];
    }

  switch (xfce_stack_top (parser->states))
    {
    case PARSER_START:
      if (strcmp (element_name, "xfdesktop-menu") == 0)
        xfce_stack_push (parser->states, PARSER_XFDESKTOPMENU);
      else
        goto unknown_element;
      break;

    case PARSER_START_INCLUDED:
    case PARSER_XFDESKTOPMENU:
    case PARSER_MENU:
      if (strcmp (element_name, "app") == 0)
        {
          if (G_UNLIKELY (attr_name == NULL || attr_cmd == NULL))
            {
              g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                           "<app> requires attributes name and cmd");
              return;
            }

          item = xfce_menu_app_item_new (attr_name, attr_icon, attr_cmd, attr_tip,
                                         eval (attr_snotify), eval (attr_term));
          gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

          if (attr_visible == NULL || strcmp (attr_visible, "false") != 0)
            gtk_widget_show (item);

          xfce_stack_push (parser->states, PARSER_APP);
        }
      else if (strcmp (element_name, "menu") == 0)
        {
          if (G_UNLIKELY (attr_name == NULL))
            {
              g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                           "<menu> requires attribute name");
              return;
            }

          submenu = xfce_menu_new (attr_name, attr_icon);
          xfce_menu_append_submenu (XFCE_MENU (menu), submenu);
          xfce_stack_push (parser->menus, submenu);

          xfce_stack_push (parser->states, PARSER_MENU);
        }
      else if (strcmp (element_name, "separator") == 0)
        {
          item = gtk_separator_menu_item_new ();
          gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

          if (attr_visible == NULL || strcmp (attr_visible, "false") != 0)
            gtk_widget_show (item);

          xfce_stack_push (parser->states, PARSER_SEPARATOR);
        }
      else if (strcmp (element_name, "include") == 0)
        {
          if (G_UNLIKELY (attr_type == NULL))
            {
              g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                           "<include> requires attribute type");
              return;
            }

          if (strcmp (attr_type, "system") == 0)
            {
              include_system_menu (parser->system_menu,
                                   parser->cache,
                                   menu);
            }
          else if (strcmp (attr_type, "file") == 0)
            {
              if (G_UNLIKELY (attr_src == NULL))
                {
                  g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                               "<include type=\"file\"> requires attribute src");
                  return;
                }

              if (*attr_src == '/')
                {
                  file = g_strdup (attr_src);
                }
              else
                {
                  dir = g_path_get_dirname (parser->filename);
                  file = g_build_filename (dir, attr_src, NULL);
                  g_free (dir);

                  /* if included file is not present in the same directory
                   * as the file which includes it, try to lookup the file
                   * using the XDG_CONFIG_DIRS resource.
                   */
                  if (!g_file_test (file, G_FILE_TEST_IS_REGULAR))
                    {
                      spec = g_build_filename ("xfce4", "desktop", attr_src, NULL);
                      g_free (file);
                      file = xfce_resource_lookup (XFCE_RESOURCE_CONFIG, spec);
                      g_free (spec);
                    }
                }

              if (G_LIKELY (file != NULL))
                {
                  if (g_file_test (file, G_FILE_TEST_IS_REGULAR))
                    {
                      GError *narf = NULL;

                      if (!desktop_menu_parse (file,
                                          parser->system_menu,
                                          parser->cache,
                                          menu,
                                          TRUE,
                                          &narf))
                        {
                          g_warning ("Unable to parse %s: %s", file, narf->message);
                          g_error_free (narf);
                        }
                    }

                  g_free (file);
                }
            }
          else
            {
              g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                           "Unknown <include> type \"%s\"", attr_type);
              return;
            }

          xfce_stack_push (parser->states, PARSER_INCLUDE);
        }
      else if (strcmp (element_name, "builtin") == 0)
        {
          if (G_UNLIKELY (attr_name == NULL || attr_cmd == NULL))
            {
              g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                           "<builtin> requires attributes name and cmd");
              return;
            }

          if (strcmp (attr_cmd, "quit") != 0)
            {
              g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                           "Unknown builtin command \"%s\"", attr_cmd);
              return;
            }

          item = xfce_menu_item_new (attr_name, attr_icon, attr_tip);
          gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

          if (attr_visible == NULL || strcmp (attr_visible, "false") != 0)
            gtk_widget_show (item);

          // FIXME: why not XfceMenuBuiltinItem ?
          extern void quit (gboolean);
          g_signal_connect_swapped (G_OBJECT (item), "activate",
                                    G_CALLBACK (quit), FALSE);

          xfce_stack_push (parser->states, PARSER_BUILTIN);
        }
      else if (strcmp (element_name, "title") == 0)
        {
          xfce_stack_push (parser->states, PARSER_TITLE);
        }
      else
        goto unknown_element;
      break;

    default:
      goto unknown_element;
    }

  return;

unknown_element:
  g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
               "Unknown element <%s>", element_name);
  return;
}



static void
end_element_handler (GMarkupParseContext  *context,
                     const gchar          *element_name,
                     gpointer              user_data,
                     GError              **error)
{
  Parser *parser = (Parser *) user_data;

  switch (xfce_stack_top (parser->states))
    {
    case PARSER_START:
      g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                   "End element handler called while in root context");
      return;

    case PARSER_XFDESKTOPMENU:
      if (strcmp (element_name, "xfdesktop-menu") != 0)
        goto unknown_element;
      break;

    case PARSER_APP:
      if (strcmp (element_name, "app") != 0)
        goto unknown_element;
      break;

    case PARSER_MENU:
      if (strcmp (element_name, "menu") == 0)
        xfce_stack_pop (parser->menus);
      else
        goto unknown_element;
      break;

    case PARSER_SEPARATOR:
      if (strcmp (element_name, "separator") != 0)
        goto unknown_element;
      break;

    case PARSER_INCLUDE:
      if (strcmp (element_name, "include") != 0)
        goto unknown_element;
      break;

    case PARSER_BUILTIN:
      if (strcmp (element_name, "builtin") != 0)
        goto unknown_element;
      break;

    case PARSER_TITLE:
      if (strcmp (element_name, "title") != 0)
        goto unknown_element;
      break;

    default:
      goto unknown_element;
    }

  xfce_stack_pop (parser->states);
  return;

unknown_element:
  g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
               "Unknown closing element <%s>", element_name);
  return;
}



static void
include_system_menu (XdgMenu         *system_menu,
                     XdgDesktopCache *cache,
                     GtkWidget       *parent)
{
  GtkWidget *submenu;
  XdgMenu   *child;

  for (child = system_menu->cfirst; child != NULL; child = child->next)
    {
      submenu = xfce_menu_new_from_parsed (child, cache);
      xfce_menu_append_submenu (XFCE_MENU (parent), submenu);
      include_system_menu (child, cache, submenu);
    }
}



static gboolean
eval (const gchar *condition)
{
  return condition != NULL
    && (strcmp (condition, "true") == 0
        || strcmp (condition, "yes") == 0);
}




