/* $Id: xdg-pattern.c,v 1.4 2004/11/04 16:02:52 bmeurer Exp $ */
/*-
 * Copyright (c) 2004 os-cillation
 * All rights reserved.
 *
 * Written by Benedikt Meurer <bm@os-cillation.de>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <string.h>

#include "xdg-pattern.h"



XdgPattern*
xdg_pattern_new (XdgPatternType type,
                 const gchar   *name)
{
  XdgPattern *pattern;

  pattern         = g_new (XdgPattern, 1);
  pattern->type   = type;
  pattern->parent = NULL;

  if (type == XDG_PATTERN_FILENAME || type == XDG_PATTERN_CATEGORY)
    pattern->name = g_strdup (name);
  else
    pattern->children = NULL;

  return pattern;
}



XdgPattern*
xdg_pattern_copy (const XdgPattern *source)
{
  XdgPattern *pattern;
  XdgPattern *child;
  GList      *lp;

  if (source == NULL)
    return NULL;

  pattern = g_memdup (source, sizeof (*source));

  switch (pattern->type)
    {
    case XDG_PATTERN_CATEGORY:
    case XDG_PATTERN_FILENAME:
      pattern->name = g_strdup (pattern->name);
      break;

    case XDG_PATTERN_ALL:
      break;

    case XDG_PATTERN_OR:
    case XDG_PATTERN_AND:
    case XDG_PATTERN_NOT:
      pattern->children = NULL;
      for (lp = source->children; lp != NULL; lp = lp->next)
        {
          child = xdg_pattern_copy (XDG_PATTERN (lp->data));
          pattern->children = g_list_append (pattern->children, child);
        }
      break;

    default:
      g_assert_not_reached ();
    }

  return pattern;
}



XdgPattern*
xdg_pattern_merge (XdgPattern *first,
                   XdgPattern *second)
{
  XdgPattern *pattern;

  if (first == NULL && second == NULL)
    return NULL;
  else if (first == NULL && second != NULL)
    return second;
  else if (first != NULL && second == NULL)
    return first;

  pattern = xdg_pattern_new (XDG_PATTERN_OR, NULL);
  xdg_pattern_append (pattern, first);
  xdg_pattern_append (pattern, second);

  return pattern;
}



gboolean
xdg_pattern_match (const XdgPattern      *pattern,
                   const XdgDesktopEntry *entry)
{
  gboolean result;
  GList   *lp;
  gsize    n;

  switch (pattern->type)
    {
    case XDG_PATTERN_OR:
      result = FALSE;
      for (lp = pattern->children; lp != NULL; lp = lp->next)
        if (xdg_pattern_match (XDG_PATTERN (lp->data), entry))
          {
            result = TRUE;
            break;
          }
      break;

    case XDG_PATTERN_AND:
      result = TRUE;
      for (lp = pattern->children; lp != NULL; lp = lp->next)
        if (!xdg_pattern_match (XDG_PATTERN (lp->data), entry))
          {
            result = FALSE;
            break;
          }
      break;

    case XDG_PATTERN_NOT:
      result = TRUE;
      for (lp = pattern->children; lp != NULL; lp = lp->next)
        if (xdg_pattern_match (XDG_PATTERN (lp->data), entry))
          {
            result = FALSE;
            break;
          }
      break;

    case XDG_PATTERN_ALL:
      result = TRUE;
      break;

    case XDG_PATTERN_CATEGORY:
      result = FALSE;
      for (n = 0; entry->categories[n] != NULL; ++n)
        if (strcmp (entry->categories[n], pattern->name) == 0)
          {
            result = TRUE;
            break;
          }
      break;

    case XDG_PATTERN_FILENAME:
      if (strcmp (entry->id, pattern->name) == 0)
        result = TRUE;
      else
        result = FALSE;
      break;

    default:
      g_assert_not_reached ();
      result = FALSE;
    }

  return result;
}


void
xdg_pattern_free (XdgPattern *pattern)
{
  GList *lp;

  switch (pattern->type)
    {
    case XDG_PATTERN_OR:
    case XDG_PATTERN_AND:
    case XDG_PATTERN_NOT:
      for (lp = pattern->children; lp != NULL; lp = lp->next)
        xdg_pattern_free (XDG_PATTERN (lp->data));
      g_list_free (pattern->children);
      break;

    case XDG_PATTERN_ALL:
      break;

    case XDG_PATTERN_CATEGORY:
    case XDG_PATTERN_FILENAME:
      g_free (pattern->name);
      break;

    default:
      g_assert_not_reached ();
    }

  g_free (pattern);
}


void
xdg_pattern_append (XdgPattern *parent,
                    XdgPattern *pattern)
{
  switch (parent->type)
    {
    case XDG_PATTERN_OR:
    case XDG_PATTERN_AND:
    case XDG_PATTERN_NOT:
      parent->children = g_list_append (parent->children, pattern);
      pattern->parent = parent;
      break;

    default:
      g_assert_not_reached ();
    }
}


