// File dialog. Taken from the FOX library and slightly modified.
#include "config.h"
#include "i18n.h"

#include <fox/fx.h>
#include <fox/fxkeys.h>
#include <fox/FXPNGIcon.h>

#include "File.h"
#include "FileList.h"
#include "DirBox.h"
#include "FileDialog.h"
#include "icons.h"

#define FILELISTMASK (ICONLIST_EXTENDEDSELECT|ICONLIST_SINGLESELECT|ICONLIST_BROWSESELECT|ICONLIST_MULTIPLESELECT)



// Map
FXDEFMAP(FileSelector) FileSelectorMap[]={
            FXMAPFUNC(SEL_COMMAND,FileSelector::ID_ACCEPT,FileSelector::onCmdAccept),
            FXMAPFUNC(SEL_COMMAND,FileSelector::ID_FILEFILTER,FileSelector::onCmdFilter),
            FXMAPFUNC(SEL_DOUBLECLICKED,FileSelector::ID_FILELIST,FileSelector::onCmdItemDblClicked),
            FXMAPFUNC(SEL_SELECTED,FileSelector::ID_FILELIST,FileSelector::onCmdItemSelected),
            FXMAPFUNC(SEL_DESELECTED,FileSelector::ID_FILELIST,FileSelector::onCmdItemDeselected),
            FXMAPFUNC(SEL_COMMAND,FileSelector::ID_DIRECTORY_UP,FileSelector::onCmdDirectoryUp),
            FXMAPFUNC(SEL_UPDATE,FileSelector::ID_DIRECTORY_UP,FileSelector::onUpdDirectoryUp),
            FXMAPFUNC(SEL_COMMAND,FileSelector::ID_DIRTREE,FileSelector::onCmdDirTree),
            FXMAPFUNC(SEL_COMMAND,FileSelector::ID_HOME,FileSelector::onCmdHome),
            FXMAPFUNC(SEL_COMMAND,FileSelector::ID_WORK,FileSelector::onCmdWork),
        };


// Implementation
FXIMPLEMENT(FileSelector,FXPacker,FileSelectorMap,ARRAYNUMBER(FileSelectorMap))


// Default pattern
static const FXchar allfiles[]="All Files (*)";

// File selector object
FileSelector::FileSelector(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
        FXPacker(p,opts,x,y,w,h)
{
    target=tgt;
    message=sel;
    FXHorizontalFrame *buttons=new FXHorizontalFrame(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X,0,0,0,0, DEFAULT_SPACING,DEFAULT_SPACING,DEFAULT_SPACING,DEFAULT_SPACING, 0,0);
    FXMatrix *fields=new FXMatrix(this,3,MATRIX_BY_COLUMNS|LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X);
    new FXLabel(fields,_("&File Name:"),NULL,JUSTIFY_LEFT|LAYOUT_CENTER_Y);
    filename=new FXTextField(fields,25,this,ID_ACCEPT,TEXTFIELD_ENTER_ONLY|LAYOUT_FILL_COLUMN|LAYOUT_FILL_X|FRAME_SUNKEN|FRAME_THICK);
    new FXButton(fields,_("&OK"),NULL,this,ID_ACCEPT,BUTTON_INITIAL|BUTTON_DEFAULT|FRAME_RAISED|FRAME_THICK|LAYOUT_FILL_X,0,0,0,0,20,20);
    accept=new FXButton(buttons,NULL,NULL,NULL,0,LAYOUT_FIX_X|LAYOUT_FIX_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,0,0, 0,0,0,0);
    new FXLabel(fields,_("File F&ilter:"),NULL,JUSTIFY_LEFT|LAYOUT_CENTER_Y);
    FXHorizontalFrame *filterframe=new FXHorizontalFrame(fields,LAYOUT_FILL_COLUMN|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);
    filefilter=new FXComboBox(filterframe,10,4,this,ID_FILEFILTER,COMBOBOX_STATIC|LAYOUT_FILL_X|FRAME_SUNKEN|FRAME_THICK);
    readonly=new FXCheckButton(filterframe,_("Read Only"),NULL,0,ICON_BEFORE_TEXT|JUSTIFY_LEFT|LAYOUT_CENTER_Y);
    cancel=new FXButton(fields,_("&Cancel"),NULL,NULL,0,BUTTON_DEFAULT|FRAME_RAISED|FRAME_THICK|LAYOUT_FILL_X,0,0,0,0,20,20);
    FXHorizontalFrame *frame=new FXHorizontalFrame(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y|FRAME_SUNKEN|FRAME_THICK,0,0,0,0,0,0,0,0);
    filebox=new FileList(frame,this,ID_FILELIST,ICONLIST_MINI_ICONS|ICONLIST_BROWSESELECT|ICONLIST_AUTOSIZE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    new FXLabel(buttons,_("Folder:"),NULL,LAYOUT_CENTER_Y);
    updiricon=new FXPNGIcon(getApp(),dirup);
    listicon=new FXPNGIcon(getApp(),smallicons);
    iconsicon=new FXPNGIcon(getApp(),bigicons);
    detailicon=new FXPNGIcon(getApp(),details);
    homeicon=new FXPNGIcon(getApp(),home);
    workicon=new FXPNGIcon(getApp(),work);
    shownicon=new FXPNGIcon(getApp(),hidehidden);
    hiddenicon=new FXPNGIcon(getApp(),showhidden);
    dirbox=new DirBox(buttons,5,this,ID_DIRTREE,FRAME_SUNKEN|FRAME_THICK|LAYOUT_FIX_WIDTH|LAYOUT_CENTER_Y,0,0,160,0);
    new FXFrame(buttons,LAYOUT_FIX_WIDTH,0,0,4,1);
    new FXButton(buttons,_("\tGo up one folder\tMove up to higher folder."),updiricon,this,ID_DIRECTORY_UP,BUTTON_TOOLBAR|FRAME_RAISED,0,0,0,0, 3,3,3,3);
    new FXButton(buttons,_("\tGo to home folder\tBack to home folder."),homeicon,this,ID_HOME,BUTTON_TOOLBAR|FRAME_RAISED,0,0,0,0, 3,3,3,3);
    new FXButton(buttons,_("\tGo to working folder\tBack to working folder."),workicon,this,ID_WORK,BUTTON_TOOLBAR|FRAME_RAISED,0,0,0,0, 3,3,3,3);
    new FXButton(buttons,_("\tShow list\tDisplay folder with small icons."),listicon,filebox,FileList::ID_SHOW_MINI_ICONS,BUTTON_TOOLBAR|FRAME_RAISED,0,0,0,0, 3,3,3,3);
    new FXButton(buttons,_("\tShow icons\tDisplay folder with big icons."),iconsicon,filebox,FileList::ID_SHOW_BIG_ICONS,BUTTON_TOOLBAR|FRAME_RAISED,0,0,0,0, 3,3,3,3);
    new FXButton(buttons,_("\tShow details\tDisplay detailed folder listing."),detailicon,filebox,FileList::ID_SHOW_DETAILS,BUTTON_TOOLBAR|FRAME_RAISED,0,0,0,0, 3,3,3,3);
    new FXToggleButton(buttons,_("\tShow hidden files\tShow hidden files and folders."),_("\tHide hidden Files\tHide hidden files and folders."),hiddenicon,shownicon,filebox,FileList::ID_TOGGLE_HIDDEN,TOGGLEBUTTON_TOOLBAR|FRAME_RAISED,0,0,0,0, 3,3,3,3);
    readonly->hide();
    setSelectMode(SELECTFILE_ANY);    // For backward compatibility, this HAS to be the default!
    setPatternList(allfiles);
    setDirectory(FXFile::getCurrentDirectory());
    filebox->setFocus();
    accept->hide();
}


// Double-clicked item in file list
long FileSelector::onCmdItemDblClicked(FXObject*,FXSelector,void* ptr)
{
    FXint index=(FXint)(long)ptr;
    if(index<0)
        return 1;

    // If directory, open the directory
    if(filebox->isItemDirectory(index))
    {
        setDirectory(filebox->getItemPathname(index));
        return 1;
    }

    // Only return if we wanted a file
    if(selectmode!=SELECTFILE_DIRECTORY)
    {
        if(filebox->isItemFile(index))
        {
            FXObject *tgt=accept->getTarget();
            FXSelector sel=accept->getSelector();
            if(tgt)
                tgt->handle(accept,MKUINT(sel,SEL_COMMAND),(void*)1);
        }
    }
    return 1;
}


// Change in items which are selected
long FileSelector::onCmdItemSelected(FXObject*,FXSelector,void* ptr)
{
    FXint index=(FXint)(long)ptr;
    FXString text,file;
    if(selectmode==SELECTFILE_MULTIPLE)
    {
        for(FXint i=0; i<filebox->getNumItems(); i++)
        {
            if(filebox->isItemFile(i) && filebox->isItemSelected(i))
            {
                if(!text.empty())
                    text+=' ';
                text+="\""+filebox->getItemFilename(i)+"\"";
            }
        }
        filename->setText(text);
    }
    else if(selectmode==SELECTFILE_MULTIPLE_ALL)
    {
        for(FXint i=0; i<filebox->getNumItems(); i++)
        {
            if(filebox->isItemSelected(i) && filebox->getItemFilename(i)!="..")
            {
                if(!text.empty())
                    text+=' ';
                text+="\""+filebox->getItemFilename(i)+"\"";
            }
        }
        filename->setText(text);
    }
    else if(selectmode==SELECTFILE_DIRECTORY)
    {
        if(filebox->isItemDirectory(index))
        {
            text=filebox->getItemFilename(index);
            filename->setText(text);
        }
    }
    else
    {
        if(filebox->isItemFile(index))
        {
            text=filebox->getItemFilename(index);
            filename->setText(text);
        }
    }
    return 1;
}


// Change in items which are deselected
long FileSelector::onCmdItemDeselected(FXObject*,FXSelector,void*)
{
    FXString text,file;
    if(selectmode==SELECTFILE_MULTIPLE)
    {
        for(FXint i=0; i<filebox->getNumItems(); i++)
        {
            if(filebox->isItemFile(i) && filebox->isItemSelected(i))
            {
                if(!text.empty())
                    text+=' ';
                text+="\""+filebox->getItemFilename(i)+"\"";
            }
        }
        filename->setText(text);
    }
    else if(selectmode==SELECTFILE_MULTIPLE_ALL)
    {
        for(FXint i=0; i<filebox->getNumItems(); i++)
        {
            if(filebox->isItemSelected(i) && filebox->getItemFilename(i)!="..")
            {
                if(!text.empty())
                    text+=' ';
                text+="\""+filebox->getItemFilename(i)+"\"";
            }
        }
        filename->setText(text);
    }
    return 1;
}


// Hit the accept button or enter in text field
long FileSelector::onCmdAccept(FXObject*,FXSelector,void*)
{
    FXSelector sel=accept->getSelector();
    FXObject *tgt=accept->getTarget();

    // Get (first) filename
    FXString path=getFilename();

    // Only do something if a selection was made
    if(!path.empty())
    {

        // Is directory?
        if(isDirectory(path))
        {

            // In directory mode:- we got our answer!
            if(selectmode==SELECTFILE_DIRECTORY || selectmode==SELECTFILE_MULTIPLE_ALL)
            {
                if(tgt)
                    tgt->handle(accept,MKUINT(sel,SEL_COMMAND),(void*)1);
                return 1;
            }

            // Hop over to that directory
            dirbox->setDirectory(path);
            filebox->setDirectory(path);
            filename->setText(FXString::null);
            return 1;
        }

        // Get directory part of path
        FXString dir=FXFile::directory(path);

        // In file mode, directory part of path should exist
        if(isDirectory(dir))
        {

            // In any mode, existing directory part is good enough
            if(selectmode==SELECTFILE_ANY)
            {
                if(tgt)
                    tgt->handle(accept,MKUINT(sel,SEL_COMMAND),(void*)1);
                return 1;
            }

            // In existing mode, the whole filename must exist and be a file
            else if(selectmode==SELECTFILE_EXISTING)
            {
                if(isFile(path))
                {
                    if(tgt)
                        tgt->handle(accept,MKUINT(sel,SEL_COMMAND),(void*)1);
                    return 1;
                }
            }

            // In multiple mode, return if all selected files exist
            else if(selectmode==SELECTFILE_MULTIPLE)
            {
                for(FXint i=0; i<filebox->getNumItems(); i++)
                {
                    if(filebox->isItemSelected(i) && filebox->isItemFile(i))
                    {
                        if(tgt)
                            tgt->handle(accept,MKUINT(sel,SEL_COMMAND),(void*)1);
                        return 1;
                    }
                }
            }

            // Multiple files and/or directories
            else
            {
                for(FXint i=0; i<filebox->getNumItems(); i++)
                {
                    if(filebox->isItemSelected(i) && filebox->getItemFilename(i)!="..")
                    {
                        if(tgt)
                            tgt->handle(accept,MKUINT(sel,SEL_COMMAND),(void*)1);
                        return 1;
                    }
                }
            }
        }

        // Go up to the lowest directory which still exists
        while(!FXFile::isTopDirectory(dir) && !isDirectory(dir))
            dir=FXFile::upLevel(dir);

        // Switch as far as we could go
        dirbox->setDirectory(dir);
        filebox->setDirectory(dir);

        // Put the tail end back for further editing
        FXASSERT(dir.length()<=path.length());
        if(ISPATHSEP(path[dir.length()]))
            path.remove(0,dir.length()+1);
        else
            path.remove(0,dir.length());

        // Replace text box with new stuff
        filename->setText(path);
        filename->selectAll();
    }

    // Beep
    getApp()->beep();
    return 1;
}


// User clicked up directory button
long FileSelector::onCmdDirectoryUp(FXObject*,FXSelector,void*)
{
    setDirectory(FXFile::upLevel(filebox->getDirectory()));
    return 1;
}


// Can we still go up
long FileSelector::onUpdDirectoryUp(FXObject* sender,FXSelector,void*)
{
    if(FXFile::isTopDirectory(filebox->getDirectory()))
        sender->handle(this,MKUINT(ID_DISABLE,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
    return 1;
}


// Back to home directory
long FileSelector::onCmdHome(FXObject*,FXSelector,void*)
{
    setDirectory(FXFile::getHomeDirectory());
    return 1;
}


// Back to current working directory
long FileSelector::onCmdWork(FXObject*,FXSelector,void*)
{
    setDirectory(FXFile::getCurrentDirectory());
    return 1;
}



// Switched directories using directory tree
long FileSelector::onCmdDirTree(FXObject*,FXSelector,void* ptr)
{
    filebox->setDirectory((FXchar*)ptr);
    if(selectmode==SELECTFILE_DIRECTORY)
        filename->setText(FXString::null);
    return 1;
}



// Strip pattern from text if present
FXString FileSelector::patternFromText(const FXString& pattern)
{
    FXint beg,end;
    end=pattern.findb(')');         // Search from the end so we can allow ( ) in the pattern name itself
    beg=pattern.findb('(',end-1);
    if(0<=beg && beg<end)
        return pattern.mid(beg+1,end-beg-1);
    return pattern;
}


// Return the first extension "ext1" found in the pattern if the
// pattern is of the form "*.ext1,*.ext2,..." or the empty string
// if the pattern contains other wildcard combinations.
FXString FileSelector::extensionFromPattern(const FXString& pattern)
{
    FXint beg,end,c;
    beg=0;
    if(pattern[beg]=='*')
    {
        beg++;
        if(pattern[beg]=='.')
        {
            beg++;
            end=beg;
            while((c=pattern[end])!='\0' && c!=',' && c!='|')
            {
                if(c=='*' || c=='?' || c=='[' || c==']' || c=='^' || c=='!')
                    return FXString::null;
                end++;
            }
            return pattern.mid(beg,end-beg);
        }
    }
    return FXString::null;
}


// Change the pattern; change the filename to the suggested extension
long FileSelector::onCmdFilter(FXObject*,FXSelector,void* ptr)
{
    FXString pat=patternFromText((FXchar*)ptr);
    filebox->setPattern(pat);
    if(selectmode==SELECTFILE_ANY)
    {
        FXString ext=extensionFromPattern(pat);
        if(!ext.empty())
        {
            FXString name=FXFile::stripExtension(filename->getText());
            if(!name.empty())
                filename->setText(name+"."+ext);
        }
    }
    return 1;
}


// Set directory
void FileSelector::setDirectory(const FXString& path)
{
    FXString abspath=FXFile::absolute(path);
    filebox->setDirectory(abspath);
    dirbox->setDirectory(abspath);
    if(selectmode!=SELECTFILE_ANY)
        filename->setText(FXString::null);
}


// Get directory
FXString FileSelector::getDirectory() const
{
    return filebox->getDirectory();
}


// Set file name
void FileSelector::setFilename(const FXString& path)
{
    FXString abspath=FXFile::absolute(path);
    filebox->setCurrentFile(abspath);
    dirbox->setDirectory(FXFile::directory(abspath));
    filename->setText(FXFile::name(abspath));
}


// Get complete path + filename
FXString FileSelector::getFilename() const
{
    register FXint i;
    if(selectmode==SELECTFILE_MULTIPLE_ALL)
    {
        for(i=0; i<filebox->getNumItems(); i++)
        {
            if(filebox->isItemSelected(i) && filebox->getItemFilename(i)!="..")
                return FXFile::absolute(filebox->getDirectory(),filebox->getItemFilename(i));
        }
    }
    else if(selectmode==SELECTFILE_MULTIPLE)
    {
        for(i=0; i<filebox->getNumItems(); i++)
        {
            if(filebox->isItemSelected(i) && filebox->isItemFile(i))
                return FXFile::absolute(filebox->getDirectory(),filebox->getItemFilename(i));
        }
    }
    else
    {
        if(!filename->getText().empty())
            return FXFile::absolute(filebox->getDirectory(),filename->getText());
    }
    return FXString::null;
}


// Return empty-string terminated list of selected file names, or NULL
FXString* FileSelector::getFilenames() const
{
    register FXString *files=NULL;
    register FXint i,n;
    if(filebox->getNumItems())
    {
        if(selectmode==SELECTFILE_MULTIPLE_ALL)
        {
            for(i=n=0; i<filebox->getNumItems(); i++)
            {
                if(filebox->isItemSelected(i) && filebox->getItemFilename(i)!="..")
                    n++;
            }
            if(n)
            {
                files=new FXString [n+1];
                for(i=n=0; i<filebox->getNumItems(); i++)
                {
                    if(filebox->isItemSelected(i) && filebox->getItemFilename(i)!="..")
                        files[n++]=filebox->getItemPathname(i);
                }
                files[n]=FXString::null;
            }
        }
        else
        {
            for(i=n=0; i<filebox->getNumItems(); i++)
            {
                if(filebox->isItemSelected(i) && filebox->isItemFile(i))
                    n++;
            }
            if(n)
            {
                files=new FXString [n+1];
                for(i=n=0; i<filebox->getNumItems(); i++)
                {
                    if(filebox->isItemSelected(i) && filebox->isItemFile(i))
                        files[n++]=filebox->getItemPathname(i);
                }
                files[n]=FXString::null;
            }
        }
    }
    return files;
}


// Set bunch of patterns
void FileSelector::setPatternList(const FXchar **ptrns)
{
    filefilter->clearItems();
    if(ptrns)
    {
        while(ptrns[0] && ptrns[1])
        {
            filefilter->appendItem(FXStringFormat("%s (%s)",ptrns[0],ptrns[1]));
            ptrns+=2;
        }
    }
    if(!filefilter->getNumItems())
        filefilter->appendItem(allfiles);
    setCurrentPattern(0);
}


// Change patterns, each pattern separated by newline
void FileSelector::setPatternList(const FXString& patterns)
{
    FXString pat;
    FXint i;
    filefilter->clearItems();
    for(i=0; !(pat=patterns.extract(i,'\n')).empty(); i++)
        filefilter->appendItem(pat);
    if(!filefilter->getNumItems())
        filefilter->appendItem(allfiles);
    setCurrentPattern(0);
}


// Return list of patterns
FXString FileSelector::getPatternList() const
{
    FXString pat;
    FXint i;
    for(i=0; i<filefilter->getNumItems(); i++)
    {
        if(!pat.empty())
            pat+='\n';
        pat+=filefilter->getItemText(i);
    }
    return pat;
}


// Set current filter pattern
void FileSelector::setPattern(const FXString& ptrn)
{
    filefilter->setText(ptrn);
    filebox->setPattern(ptrn);
}


// Get current filter pattern
FXString FileSelector::getPattern() const
{
    return filebox->getPattern();
}


// Set current file pattern from the list
void FileSelector::setCurrentPattern(FXint patno)
{
    if(patno<0 || patno>=filefilter->getNumItems())
        fxerror("%s::setCurrentPattern: index out of range.\n",getClassName());
    filefilter->setCurrentItem(patno);
    filebox->setPattern(patternFromText(filefilter->getItemText(patno)));
}


// Return current pattern
FXint FileSelector::getCurrentPattern() const
{
    return filefilter->getCurrentItem();
}


// Change pattern for pattern number patno
void FileSelector::setPatternText(FXint patno,const FXString& text)
{
    if(patno<0 || patno>=filefilter->getNumItems())
        fxerror("%s::setPatternText: index out of range.\n",getClassName());
    filefilter->setItemText(patno,text);
    if(patno==filefilter->getCurrentItem())
        setPattern(patternFromText(text));
}


// Return pattern text of pattern patno
FXString FileSelector::getPatternText(FXint patno) const
{
    if(patno<0 || patno>=filefilter->getNumItems())
        fxerror("%s::getPatternText: index out of range.\n",getClassName());
    return filefilter->getItemText(patno);
}


// Change space for item
void FileSelector::setItemSpace(FXint s)
{
    filebox->setItemSpace(s);
}


// Get space for item
FXint FileSelector::getItemSpace() const
{
    return filebox->getItemSpace();
}


// Change File List style
void FileSelector::setFileBoxStyle(FXuint style)
{
    filebox->setListStyle(style);
}


// Return File List style
FXuint FileSelector::getFileBoxStyle() const
{
    return filebox->getListStyle();
}


// Change file selection mode
void FileSelector::setSelectMode(FXuint mode)
{
    switch(mode)
    {
    case SELECTFILE_EXISTING:
        filebox->showOnlyDirectories(FALSE);
        filebox->setListStyle((filebox->getListStyle()&~FILELISTMASK)|ICONLIST_BROWSESELECT);
        break;
    case SELECTFILE_MULTIPLE:
    case SELECTFILE_MULTIPLE_ALL:
        filebox->showOnlyDirectories(FALSE);
        filebox->setListStyle((filebox->getListStyle()&~FILELISTMASK)|ICONLIST_EXTENDEDSELECT);
        break;
    case SELECTFILE_DIRECTORY:
        filebox->showOnlyDirectories(TRUE);
        filebox->setListStyle((filebox->getListStyle()&~FILELISTMASK)|ICONLIST_BROWSESELECT);
        break;
    default:
        filebox->showOnlyDirectories(FALSE);
        filebox->setListStyle((filebox->getListStyle()&~FILELISTMASK)|ICONLIST_BROWSESELECT);
        break;
    }
    selectmode=mode;
}


// Show readonly button
void FileSelector::showReadOnly(FXbool show)
{
    show ? readonly->show() : readonly->hide();
}


// Return TRUE if readonly is shown
FXbool FileSelector::shownReadOnly() const
{
    return readonly->shown();
}



// Set initial state of readonly button
void FileSelector::setReadOnly(FXbool state)
{
    readonly->setCheck(state);
}


// Get readonly state
FXbool FileSelector::getReadOnly() const
{
    return readonly->getCheck();
}


// Save data
void FileSelector::save(FXStream& store) const
{
    FXPacker::save(store);
    store << filebox;
    store << filename;
    store << filefilter;
    store << readonly;
    store << dirbox;
    store << accept;
    store << cancel;
    store << updiricon;
    store << listicon;
    store << detailicon;
    store << iconsicon;
    store << homeicon;
    store << workicon;
    store << shownicon;
    store << hiddenicon;
    store << selectmode;
}


// Load data
void FileSelector::load(FXStream& store)
{
    FXPacker::load(store);
    store >> filebox;
    store >> filename;
    store >> filefilter;
    store >> readonly;
    store >> dirbox;
    store >> accept;
    store >> cancel;
    store >> updiricon;
    store >> listicon;
    store >> detailicon;
    store >> iconsicon;
    store >> homeicon;
    store >> workicon;
    store >> shownicon;
    store >> hiddenicon;
    store >> selectmode;
}


// Cleanup; icons must be explicitly deleted
FileSelector::~FileSelector()
{
    delete updiricon;
    delete listicon;
    delete detailicon;
    delete iconsicon;
    delete homeicon;
    delete workicon;
    delete shownicon;
    delete hiddenicon;
    filebox=(FileList*)-1;
    filename=(FXTextField*)-1;
    filefilter=(FXComboBox*)-1;
    readonly=(FXCheckButton*)-1;
    dirbox=(DirBox*)-1;
    accept=(FXButton*)-1;
    cancel=(FXButton*)-1;
    updiricon=(FXIcon*)-1;
    listicon=(FXIcon*)-1;
    detailicon=(FXIcon*)-1;
    iconsicon=(FXIcon*)-1;
    homeicon=(FXIcon*)-1;
    workicon=(FXIcon*)-1;
    shownicon=(FXIcon*)-1;
    hiddenicon=(FXIcon*)-1;
}




// Object implementation
FXIMPLEMENT(FileDialog,FXDialogBox,NULL,0)


// File Dialog object
FileDialog::FileDialog(FXWindow* owner,const FXString& name,FXuint opts,FXint x,FXint y,FXint w,FXint h):
        FXDialogBox(owner,name,opts|DECOR_TITLE|DECOR_BORDER|DECOR_RESIZE,x,y,w,h,0,0,0,0,4,4)
{
    filebox=new FileSelector(this,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y);
    filebox->acceptButton()->setTarget(this);
    filebox->acceptButton()->setSelector(FXDialogBox::ID_ACCEPT);
    filebox->cancelButton()->setTarget(this);
    filebox->cancelButton()->setSelector(FXDialogBox::ID_CANCEL);
}


// Set file name
void FileDialog::setFilename(const FXString& path)
{
    filebox->setFilename(path);
}


// Get filename, if any
FXString FileDialog::getFilename() const
{
    return filebox->getFilename();
}


// Return empty-string terminated list of selected file names,
FXString* FileDialog::getFilenames() const
{
    return filebox->getFilenames();
}


// Set pattern
void FileDialog::setPattern(const FXString& ptrn)
{
    filebox->setPattern(ptrn);
}


// Get pattern
FXString FileDialog::getPattern() const
{
    return filebox->getPattern();
}


// Change patterns, each pattern separated by newline
void FileDialog::setPatternList(const FXString& patterns)
{
    filebox->setPatternList(patterns);
}


// Return list of patterns
FXString FileDialog::getPatternList() const
{
    return filebox->getPatternList();
}


// Set directory
void FileDialog::setDirectory(const FXString& path)
{
    filebox->setDirectory(path);
}


// Get directory
FXString FileDialog::getDirectory() const
{
    return filebox->getDirectory();
}


// Set current file pattern from the list
void FileDialog::setCurrentPattern(FXint n)
{
    filebox->setCurrentPattern(n);
}


// Return current pattern
FXint FileDialog::getCurrentPattern() const
{
    return filebox->getCurrentPattern();
}

FXString FileDialog::getPatternText(FXint patno) const
{
    return filebox->getPatternText(patno);
}


void FileDialog::setPatternText(FXint patno,const FXString& text)
{
    filebox->setPatternText(patno,text);
}


// Set list of patterns (DEPRECATED)
void FileDialog::setPatternList(const FXchar **ptrns)
{
    filebox->setPatternList(ptrns);
}


// Change space for item
void FileDialog::setItemSpace(FXint s)
{
    filebox->setItemSpace(s);
}


// Get space for item
FXint FileDialog::getItemSpace() const
{
    return filebox->getItemSpace();
}


// Change File List style
void FileDialog::setFileBoxStyle(FXuint style)
{
    filebox->setFileBoxStyle(style);
}


// Return File List style
FXuint FileDialog::getFileBoxStyle() const
{
    return filebox->getFileBoxStyle();
}


// Change file selection mode
void FileDialog::setSelectMode(FXuint mode)
{
    filebox->setSelectMode(mode);
}


// Return file selection mode
FXuint FileDialog::getSelectMode() const
{
    return filebox->getSelectMode();
}


// Show readonly button
void FileDialog::showReadOnly(FXbool show)
{
    filebox->showReadOnly(show);
}


// Return TRUE if readonly is shown
FXbool FileDialog::shownReadOnly() const
{
    return filebox->shownReadOnly();
}



// Set initial state of readonly button
void FileDialog::setReadOnly(FXbool state)
{
    filebox->setReadOnly(state);
}


// Get readonly state
FXbool FileDialog::getReadOnly() const
{
    return filebox->getReadOnly();
}


// Save data
void FileDialog::save(FXStream& store) const
{
    FXDialogBox::save(store);
    store << filebox;
}


// Load data
void FileDialog::load(FXStream& store)
{
    FXDialogBox::load(store);
    store >> filebox;
}


// Cleanup
FileDialog::~FileDialog()
{
    filebox=(FileSelector*)-1;
}


// Open existing filename
FXString FileDialog::getOpenFilename(FXWindow* owner,const FXString& caption,const FXString& path,const FXString& patterns,FXint initial)
{
    FileDialog opendialog(owner,caption);
    FXString filename;
    opendialog.setSelectMode(SELECTFILE_EXISTING);
    opendialog.setPatternList(patterns);
    opendialog.setCurrentPattern(initial);
    opendialog.setFilename(path);
    if(opendialog.execute())
    {
        filename=opendialog.getFilename();
        if(isFile(filename))
            return filename;
    }
    return FXString::null;
}


// Save to filename
FXString FileDialog::getSaveFilename(FXWindow* owner,const FXString& caption,const FXString& path,const FXString& patterns,FXint initial)
{
    FileDialog savedialog(owner,caption);
    savedialog.setSelectMode(SELECTFILE_ANY);
    savedialog.setPatternList(patterns);
    savedialog.setCurrentPattern(initial);
    savedialog.setFilename(path);
    if(savedialog.execute())
        return savedialog.getFilename();
    return FXString::null;
}


// Open multiple existing files
FXString* FileDialog::getOpenFilenames(FXWindow* owner,const FXString& caption,const FXString& path,const FXString& patterns,FXint initial)
{
    FileDialog opendialog(owner,caption);
    opendialog.setSelectMode(SELECTFILE_MULTIPLE);
    opendialog.setPatternList(patterns);
    opendialog.setCurrentPattern(initial);
    opendialog.setFilename(path);
    if(opendialog.execute())
        return opendialog.getFilenames();
    return NULL;
}


// Open existing directory name
FXString FileDialog::getOpenDirectory(FXWindow* owner,const FXString& caption,const FXString& path)
{
    FileDialog dirdialog(owner,caption);
    FXString dirname;
    dirdialog.setSelectMode(SELECTFILE_DIRECTORY);
    dirdialog.setFilename(path);
    if(dirdialog.execute())
    {
        dirname=dirdialog.getFilename();
        if(isDirectory(dirname))
            return dirname;
    }
    return FXString::null;
}

