// File list. Taken from the FOX library and slightly modified.
#include "config.h"
#include "i18n.h"

#include <time.h>
#if defined(linux)
#include <mntent.h>
#endif

#include <fox/fx.h>
#include <fox/fxkeys.h>
#include <fox/FXPNGIcon.h>

#include "icons.h"
#include "FileList.h"
#include "InputDialog.h"
#include "File.h"
#include "FileDict.h"
#include "MessageBox.h"


// Root directory string
#ifndef ROOTDIR
#define ROOTDIR "/"
#endif

// Maximum length of a file name
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

// Minimum file name size in detailed view
#ifndef MIN_NAME_SIZE
#define MIN_NAME_SIZE 75
#endif

// Time interval before refreshing the view
#define REFRESHINTERVAL     1000
#define REFRESHINTERVALLONG 15000

// Time interval before opening a folder
#define OPENINTERVAL		1000

#define HASH1(x,n) (((unsigned int)(x)*13)%(n))           // Probe Position [0..n-1]
#define HASH2(x,n) (1|(((unsigned int)(x)*17)%((n)-1)))   // Probe Distance [1..n-1]


// Format used for displaying the modification time in the file list
// Either use a fixed format such as :
// #define TIMEFORMAT "%m/%d/%Y %H:%M:%S"
// or a locale dependent format such as :
// #define TIMEFORMAT "%x %X"
#ifndef TIMEFORMAT
#define TIMEFORMAT "%x %X"
#endif

// To prevent a misleading warning in GCC (or a bug?) about locale and date format
size_t my_strftime(char *s, size_t max, const char  *fmt,  const struct tm *tm)
{
	return strftime(s, max, fmt, tm);
}


FXPNGIcon  *mini_folderopen;
FXPNGIcon  *mini_folderup;
FXPNGIcon  *mini_lockfolder;
FXPNGIcon  *mini_cdev;
FXPNGIcon  *mini_bdev;
FXPNGIcon  *mini_sock;
FXPNGIcon  *mini_pipe;
FXPNGIcon  *hard_disk;
FXPNGIcon  *mini_link;

FXPNGIcon  *big_folderopen;
FXPNGIcon  *big_folderup;
FXPNGIcon  *big_lockfolder;
FXPNGIcon  *big_cdev;
FXPNGIcon  *big_bdev;
FXPNGIcon  *big_sock;
FXPNGIcon  *big_pipe;
FXPNGIcon  *big_harddisk;
FXPNGIcon  *big_link;

FXint 		prevIndex=-1;

extern FXbool ask_before_copy;

#if defined(linux)
FXStringDict* devices;
#endif


// The two functions below are taken from the FXIconList class
// and hacked to right justify the second column of the file list (file size)
// They replace the normal functions...

#define SIDE_SPACING             4    // Left or right spacing between items
#define DETAIL_TEXT_SPACING      2    // Spacing between text and icon in detail icon mode

// Helper function
static inline FXint count(const FXchar* label)
{
    register FXint c=0;
    while(label[c] && label[c]!='\t')
        c++;
    return c;
}

// Draw detail
void FXIconItem::drawDetails(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint,FXint h) const
{
    register const FXchar *text=label.text();
    register FXFont *font=list->getFont();
    FXHeader *header=list->getHeader();
    register FXint iw=0,ih=0,tw=0,th=0;
    register FXint tlen,tdrw,ddw,dw,s,hi,space,tail;
    if(header->getNumItems()==0)
        return;
    if(isSelected())
    {
        dc.setForeground(list->getSelBackColor());
        dc.fillRectangle(x,y,header->getWidth(),h);
    }
    if(hasFocus())
    {
        dc.drawFocusRectangle(x+1,y+1,header->getWidth()-2,h-2);
    }
    x+=SIDE_SPACING/2;
    if(miniIcon)
    {
        iw=miniIcon->getWidth();
        ih=miniIcon->getHeight();
        dc.drawIcon(miniIcon,x,y+(h-ih)/2);
        x+=iw+DETAIL_TEXT_SPACING;
    }
    if(text)
    {
        th=font->getFontHeight();
        ddw=font->getTextWidth("...",3);
        y+=(h-th-4)/2;
        if(isSelected())
            dc.setForeground(list->getSelTextColor());
        else
            dc.setForeground(list->getTextColor());
        tail=iw+DETAIL_TEXT_SPACING+SIDE_SPACING/2;
        for(hi=0; hi<header->getNumItems(); hi++)
        {
            space=header->getItemSize(hi)-tail;
            tlen=count(text);
            if(tlen>0)
            {
				tw=font->getTextWidth(text,tlen);
                tdrw=tlen;
                dw=0;
                if(tw>space-4)
                {
                    dw=ddw;
                    s=space-4-dw;
                    while((tw=font->getTextWidth(text,tdrw))>s && tdrw>1)
                        --tdrw;
                    if(tw>space-4)
                        dw=0;
                }
                if(tw<=(space-4))
                {
                	// !!!! Hack to right justify the second column (file size) !!!!
					if (hi==1 && dw==0)
                    	dc.drawText(x+space-tw-20,y+font->getFontAscent()+2,text,tdrw);
					else
                    	dc.drawText(x+2,y+font->getFontAscent()+2,text,tdrw);
					if(dw)
					{
                        dc.drawText(x+tw+2,y+font->getFontAscent()+2,"...",3);
					}
                }
            }
            if(!text[tlen])
                break;
            x+=space;
			text=text+tlen+1;
            tail=0;
        }
    }
}
// End of hack!!!



// Object implementation
FXIMPLEMENT(FileItem,FXIconItem,NULL,0)


// Map
FXDEFMAP(FileList) FileListMap[]={
                                     FXMAPFUNC(SEL_DRAGGED,0,FileList::onDragged),
                                     FXMAPFUNC(SEL_TIMEOUT,FileList::ID_REFRESHTIMER,FileList::onRefreshTimer),
                                     FXMAPFUNC(SEL_TIMEOUT,FileList::ID_OPENTIMER,FileList::onOpenTimer),
                                     FXMAPFUNC(SEL_DND_ENTER,0,FileList::onDNDEnter),
                                     FXMAPFUNC(SEL_DND_LEAVE,0,FileList::onDNDLeave),
                                     FXMAPFUNC(SEL_DND_DROP,0,FileList::onDNDDrop),
                                     FXMAPFUNC(SEL_DND_MOTION,0,FileList::onDNDMotion),
                                     FXMAPFUNC(SEL_DND_REQUEST,0,FileList::onDNDRequest),
                                     FXMAPFUNC(SEL_BEGINDRAG,0,FileList::onBeginDrag),
                                     FXMAPFUNC(SEL_ENDDRAG,0,FileList::onEndDrag),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_DIRECTORY_UP,FileList::onCmdDirectoryUp),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_NAME,FileList::onCmdSortByName),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_TYPE,FileList::onCmdSortByType),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_SIZE,FileList::onCmdSortBySize),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_TIME,FileList::onCmdSortByTime),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_USER,FileList::onCmdSortByUser),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_GROUP,FileList::onCmdSortByGroup),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_REVERSE,FileList::onCmdSortReverse),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SET_PATTERN,FileList::onCmdSetPattern),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SET_DIRECTORY,FileList::onCmdSetDirectory),
                                     FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,FileList::onCmdSetValue),
                                     FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE,FileList::onCmdSetStringValue),
                                     FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETSTRINGVALUE,FileList::onCmdGetStringValue),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SHOW_HIDDEN,FileList::onCmdShowHidden),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_HIDE_HIDDEN,FileList::onCmdHideHidden),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_TOGGLE_HIDDEN,FileList::onCmdToggleHidden),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_HEADER_CHANGE,FileList::onCmdHeader),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_HEADER_CHANGE,FileList::onUpdHeader),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_DIRECTORY_UP,FileList::onUpdDirectoryUp),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_NAME,FileList::onUpdSortByName),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_TYPE,FileList::onUpdSortByType),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_SIZE,FileList::onUpdSortBySize),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_TIME,FileList::onUpdSortByTime),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_USER,FileList::onUpdSortByUser),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_GROUP,FileList::onUpdSortByGroup),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_REVERSE,FileList::onUpdSortReverse),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SET_PATTERN,FileList::onUpdSetPattern),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SET_DIRECTORY,FileList::onUpdSetDirectory),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SHOW_HIDDEN,FileList::onUpdShowHidden),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_HIDE_HIDDEN,FileList::onUpdHideHidden),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_TOGGLE_HIDDEN,FileList::onUpdToggleHidden),
                                 };


// Object implementation
FXIMPLEMENT(FileList,FXIconList,FileListMap,ARRAYNUMBER(FileListMap))



// File List
FileList::FileList(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
        FXIconList(p,tgt,sel,opts,x,y,w,h),directory(ROOTDIR),orgdirectory(ROOTDIR),pattern("*")
{
    flags|=FLAG_ENABLED|FLAG_DROPTARGET;
    associations=NULL;
    appendHeader(_("Name"),NULL,200);
    appendHeader(_("Size"),NULL,60);
    appendHeader(_("Type"),NULL,100);
    appendHeader(_("Modified Date"),NULL,150);
    appendHeader(_("User"),NULL,50);
    appendHeader(_("Group"),NULL,50);
    appendHeader(_("Permissions"),NULL,100);

    big_folder=new FXPNGIcon(getApp(),bigfolder);
    mini_folder=new FXPNGIcon(getApp(),minifolder);
    big_doc=new FXPNGIcon(getApp(),bigdoc);
    mini_doc=new FXPNGIcon(getApp(),minidoc);
    big_app=new FXPNGIcon(getApp(),bigapp);
    mini_app=new FXPNGIcon(getApp(),miniapp);

    mini_folderopen=new FXPNGIcon(getApp(),minifolderopen);
    mini_folderup=new FXPNGIcon(getApp(),minifolderup);
    mini_lockfolder=new FXPNGIcon(getApp(),minifolderlocked);
    mini_cdev=new FXPNGIcon(getApp(),minichardev);
    mini_bdev=new FXPNGIcon(getApp(),miniblockdev);
    mini_sock=new FXPNGIcon(getApp(),minisocket);
    mini_pipe=new FXPNGIcon(getApp(),minipipe);
    hard_disk=new FXPNGIcon(getApp(),harddisk);
    mini_link=new FXPNGIcon(getApp(),minilink);

    big_folderopen=new FXPNGIcon(getApp(),bigfolderopen);
    big_folderup=new FXPNGIcon(getApp(),bigfolderup);
    big_lockfolder=new FXPNGIcon(getApp(),bigfolderlocked);
    big_cdev=new FXPNGIcon(getApp(),bigchardev);
    big_bdev=new FXPNGIcon(getApp(),bigblockdev);
    big_sock=new FXPNGIcon(getApp(),bigsocket);
    big_pipe=new FXPNGIcon(getApp(),bigpipe);
    big_harddisk=new FXPNGIcon(getApp(),bigharddisk);
    big_link=new FXPNGIcon(getApp(),biglink);

    matchmode=FILEMATCH_FILE_NAME|FILEMATCH_NOESCAPE;
    if(!(options&FILELIST_NO_OWN_ASSOC))
        associations=new FileDict(getApp());
    dropaction=DRAG_MOVE;
    sortfunc=cmpFName;
    refreshtimer=NULL;
    opentimer=NULL;
    timestamp=0;

#if defined(linux)
    devices = new FXStringDict();
#endif
}


// Create the file list
void FileList::create()
{
#if defined(linux)
    FILE *fstab=setmntent("/etc/fstab","r");
    struct mntent *mnt;
    if(fstab)
    {
        while((mnt=getmntent(fstab)))
        {
            if(strcmp(mnt->mnt_type,MNTTYPE_IGNORE) && strcmp(mnt->mnt_type,MNTTYPE_SWAP) && strcmp(mnt->mnt_type,"proc") && strcmp(mnt->mnt_dir,"/"))
            {
                if(!strncmp(mnt->mnt_dir,"/dev/fd",7))
                    devices->insert(mnt->mnt_dir,"floppy");
                else if (!strncmp(mnt->mnt_type,"iso",3))
                    devices->insert(mnt->mnt_dir,"cdrom");
                else
                    devices->insert(mnt->mnt_dir,"harddisk");
            }
        }
        endmntent(fstab);
    }
#endif

    FXIconList::create();
    if(!deleteType)
        deleteType=getApp()->registerDragType(deleteTypeName);
    if(!urilistType)
        urilistType=getApp()->registerDragType(urilistTypeName);
    if(!refreshtimer)
        refreshtimer=getApp()->addTimeout(REFRESHINTERVAL,this,ID_REFRESHTIMER);
    big_folder->create();
    mini_folder->create();
    big_doc->create();
    mini_doc->create();
    big_app->create();
    mini_app->create();

    mini_folderopen->create();
    mini_lockfolder->create();
    mini_cdev->create();
    mini_bdev->create();
    mini_sock->create();
    mini_pipe->create();

    big_folderopen->create();
    big_lockfolder->create();
    big_cdev->create();
    big_bdev->create();
    big_sock->create();
    big_pipe->create();

    listDirectory();    // Do we have to do this here?
    sortItems();
}


// Detach disconnects the icons
void FileList::detach()
{
    FXIconList::detach();
    if(refreshtimer)
        refreshtimer=getApp()->removeTimeout(refreshtimer);
    if(opentimer)
        opentimer=getApp()->removeTimeout(opentimer);
    big_folder->detach();
    mini_folder->detach();
    big_doc->detach();
    mini_doc->detach();
    big_app->detach();
    mini_app->detach();

    mini_lockfolder->detach();
    mini_cdev->detach();
    mini_bdev->detach();
    mini_sock->detach();
    mini_pipe->detach();

    mini_folderopen->detach();
    big_folderopen->detach();
    big_lockfolder->detach();
    big_cdev->detach();
    big_bdev->detach();
    big_sock->detach();
    big_pipe->detach();

    deleteType=0;
    urilistType=0;
}


// Destroy zaps the icons
void FileList::destroy()
{
    FXIconList::destroy();
    if(refreshtimer)
        refreshtimer=getApp()->removeTimeout(refreshtimer);
    if(opentimer)
        opentimer=getApp()->removeTimeout(opentimer);
    big_folder->destroy();
    mini_folder->destroy();
    big_doc->destroy();
    mini_doc->destroy();
    big_app->destroy();
    mini_app->destroy();
    mini_folderopen->destroy();
    mini_lockfolder->destroy();
    mini_cdev->destroy();
    mini_bdev->destroy();
    mini_sock->destroy();
    mini_pipe->destroy();

    big_folderopen->destroy();
    big_lockfolder->destroy();
    big_cdev->destroy();
    big_bdev->destroy();
    big_sock->destroy();
    big_pipe->destroy();
}

// Open up folder when hovering long over a folder
long FileList::onOpenTimer(FXObject*,FXSelector,void*)
{
    FXint xx,yy,index;
    FXuint state;
    opentimer=NULL;
    getCursorPosition(xx,yy,state);
    index=getItemAt(xx,yy);
    if(0<=index && isItemDirectory(index))
    {
        dropdirectory=getItemPathname(index);
        setDirectory(dropdirectory);
        opentimer=getApp()->addTimeout(OPENINTERVAL,this,ID_OPENTIMER);
		prevIndex=-1;
    }
    return 1;
}

// Handle drag-and-drop enter
long FileList::onDNDEnter(FXObject* sender,FXSelector sel,void* ptr)
{
    FXIconList::onDNDEnter(sender,sel,ptr);

    // Keep original directory
    orgdirectory=getDirectory();

    return 1;
}


// Handle drag-and-drop leave
long FileList::onDNDLeave(FXObject* sender,FXSelector sel,void* ptr)
{
    FXIconList::onDNDLeave(sender,sel,ptr);

	if(prevIndex!=-1)
    {
        setItemMiniIcon(prevIndex,mini_folder);
        setItemBigIcon(prevIndex,big_folder);
        prevIndex=-1;
    }

    // Cancel open up timer
    if(opentimer)
        opentimer=getApp()->removeTimeout(opentimer);

    // Stop scrolling
    stopAutoScroll();

    // Restore original directory
    setDirectory(orgdirectory);
    return 1;
}


// Handle drag-and-drop motion
long FileList::onDNDMotion(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXint index=-1;

    // Cancel open up timer
    if(opentimer)
        opentimer=getApp()->removeTimeout(opentimer);

    // Start autoscrolling
    if(startAutoScroll(event->win_x,event->win_y,FALSE))
        return 1;

    // Give base class a shot
    if(FXIconList::onDNDMotion(sender,sel,ptr))
        return 1;

    // Dropping list of filenames
    if(offeredDNDType(FROM_DRAGNDROP,urilistType))
    {
        index=getItemAt(event->win_x,event->win_y);
        if(prevIndex!=-1 && prevIndex != index)
        {
            setItemMiniIcon(prevIndex,mini_folder);
            setItemBigIcon(prevIndex,big_folder);
            prevIndex=-1;
        }

        // Drop in the background
        dropdirectory=getDirectory();

        // What is being done (move,copy,link)
        dropaction=inquireDNDAction();

        // We will open up a folder if we can hover over it for a while
        if(0<=index && isItemDirectory(index))
        {
            // Set open up timer
            opentimer=getApp()->addTimeout(OPENINTERVAL,this,ID_OPENTIMER);

            prevIndex=index;
            setItemMiniIcon(index,mini_folderopen);
            setItemBigIcon(index,big_folderopen);

            // Directory where to drop, or directory to open up
            dropdirectory=getItemPathname(index);
        }

        // See if dropdirectory is writable
        if(FXFile::isWritable(dropdirectory))
            acceptDrop(DRAG_ACCEPT);
        return 1;
    }
    return 0;
}


// Handle drag-and-drop drop
long FileList::onDNDDrop(FXObject* sender,FXSelector sel,void* ptr)
{
    FXuchar *data;
    FXuint len;
    FXbool showdialog=TRUE;
    File *f=NULL;

	if(prevIndex!=-1)
    {
        setItemMiniIcon(prevIndex,mini_folder);
        setItemBigIcon(prevIndex,big_folder);
        prevIndex=-1;
    }

    // Cancel open up timer
    if(opentimer)
        opentimer=getApp()->removeTimeout(opentimer);

    // Stop scrolling
    stopAutoScroll();

    // Restore original directory
    setDirectory(orgdirectory);

    // Perhaps target wants to deal with it
    if(FXIconList::onDNDDrop(sender,sel,ptr))
        return 1;

    // Get uri-list of files being dropped
    if(getDNDData(FROM_DRAGNDROP,urilistType,data,len))
    {
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
        FXchar *p,*q;
        p=q=(FXchar*)data;
        FXString buf=p;
        int num=buf.count('\n')+1; // number of selected items

        // File object
        if (dropaction==DRAG_COPY)
            f=new File(getApp(),_("File copy"),COPY);
        else if (dropaction==DRAG_MOVE)
            f=new File(getApp(),_("File move"),MOVE);
        else
            return 0;

        while(*p)
        {
            while(*q && *q!='\r')
                q++;
            FXString url(p,q-p);
            FXString source(FXURL::fileFromURL(url));
            FXString target(dropdirectory+PATHSEPSTRING+FXFile::name(source));
            FXString sourcedir=FXFile::directory(source);
            FXString targetdir=FXFile::directory(target);

            // File operation dialog, if needed
            if(ask_before_copy & showdialog)
            {
                FXIcon *icon=NULL;
                FXString title,message;
                if (dropaction==DRAG_COPY)
                {
                    title=_("Copy ");
                    icon = new FXPNGIcon(getApp(),copy_big);
                    if (num==1)
                        message=title+source;
                    else
                        message="Copy "+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }
                else if (dropaction==DRAG_MOVE)
                {
                    title=_("Move ");
                    icon = new FXPNGIcon(getApp(),move_big);
                    if (num==1)
                        message=title+source;
                    else
                        message=_("Move ")+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }

                InputDialog* dialog = new InputDialog(this,targetdir,message,title,_("To:"),icon);
                dialog->CursorEnd();
                int rc=1;
                rc=dialog->execute();
                target=dialog->getText();
                target=::filePath(target);
                if (num>1)
                    showdialog=FALSE;
                delete dialog;
                delete icon;
                if (!rc)
                    return 0;
            }

            // Move or copy the source file
            if(dropaction==DRAG_MOVE)
            {
                // Move file
                f->create();
                f->move(source,target);

                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,MBOX_OK,_("Error"),_("Move file operation cancelled!"));
                }
            }
            else if(dropaction==DRAG_COPY)
            {
                // Copy file
                f->create();
                f->copy(source,target);

                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,MBOX_OK,_("Error"),_("Copy file operation cancelled!"));
                }
            }
            if(*q=='\r')
                q+=2;
            p=q;
        }
        delete f;
        FXFREE(&data);
        return 1;
    }

    return 0;
}


// Somebody wants our dragged data
long FileList::onDNDRequest(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXuchar *data;
    FXuint len;

    // Perhaps the target wants to supply its own data
    if(FXIconList::onDNDRequest(sender,sel,ptr))
        return 1;

    // Return list of filenames as a uri-list
    if(event->target==urilistType)
    {
        if(!dragfiles.empty())
        {
            len=dragfiles.length();
            FXMEMDUP(&data,FXuchar,dragfiles.text(),len);
            setDNDData(FROM_DRAGNDROP,event->target,data,len);
        }
        return 1;
    }

    // Delete selected files
    if(event->target==deleteType)
        return 1;

    return 0;
}



// Start a drag operation
long FileList::onBeginDrag(FXObject* sender,FXSelector sel,void* ptr)
{
    register FXint i;
    if(FXIconList::onBeginDrag(sender,sel,ptr))
        return 1;
    if(beginDrag(&urilistType,1))
    {
        dragfiles=FXString::null;
        for(i=0; i<getNumItems(); i++)
        {
            if(isItemSelected(i))
            {
                if(!dragfiles.empty())
                    dragfiles+="\r\n";
                dragfiles+=FXURL::fileToURL(getItemPathname(i));
                //dragfiles+="\r\n";
            }
        }
        return 1;
    }
    return 0;
}


// End drag operation
long FileList::onEndDrag(FXObject* sender,FXSelector sel,void* ptr)
{
    if(FXIconList::onEndDrag(sender,sel,ptr))
        return 1;
    endDrag((didAccept()!=DRAG_REJECT));
    setDragCursor(getDefaultCursor());
    dragfiles=FXString::null;
    return 1;
}


// Dragged stuff around
long FileList::onDragged(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent* event=(FXEvent*)ptr;
    FXDragAction action;
    if(FXIconList::onDragged(sender,sel,ptr))
        return 1;
    action=DRAG_MOVE;
    if(event->state&CONTROLMASK)
        action=DRAG_COPY;
    if(event->state&SHIFTMASK)
        action=DRAG_MOVE;
    handleDrag(event->root_x,event->root_y,action);
    if(didAccept()!=DRAG_REJECT)
    {
        if(action==DRAG_MOVE)
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDMOVE_CURSOR));
        else
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDCOPY_CURSOR));
    }
    else
        setDragCursor(getApp()->getDefaultCursor(DEF_DNDSTOP_CURSOR));
    return 1;
}


// Set value from a message
long FileList::onCmdSetValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr)
        setCurrentFile((const FXchar*)ptr);
    return 1;
}


// Set current directory from dir part of filename
long FileList::onCmdSetStringValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr==NULL)
        fxerror("%s::onCmdSetStringValue: NULL pointer.\n",getClassName());
    setCurrentFile(*((FXString*)ptr));
    return 1;
}


// Get current file name (NULL if no current file)
long FileList::onCmdGetStringValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr==NULL)
        fxerror("%s::onCmdGetStringValue: NULL pointer.\n",getClassName());
    *((FXString*)ptr)=getCurrentFile();
    return 1;
}


// Toggle hidden files display
long FileList::onCmdToggleHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(!showHiddenFiles());
    return 1;
}


// Update toggle hidden files widget
long FileList::onUpdToggleHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


// Show hidden files
long FileList::onCmdShowHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(TRUE);
    return 1;
}


// Update show hidden files widget
long FileList::onUpdShowHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


// Hide hidden files
long FileList::onCmdHideHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(FALSE);
    return 1;
}


// Update hide hidden files widget
long FileList::onUpdHideHidden(FXObject* sender,FXSelector,void*)
{
    if(!showHiddenFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


// Move up one level
long FileList::onCmdDirectoryUp(FXObject*,FXSelector,void*)
{
    setDirectory(FXFile::upLevel(directory));
    return 1;
}


// Determine if we can still go up more
long FileList::onUpdDirectoryUp(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg = FXFile::isTopDirectory(directory) ? ID_DISABLE : ID_ENABLE;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    return 1;
}


// Change pattern
long FileList::onCmdSetPattern(FXObject*,FXSelector,void* ptr)
{
    if(!ptr)
        return 0;
    setPattern((const char*)ptr);
    return 1;
}


// Update pattern
long FileList::onUpdSetPattern(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,MKUINT(FXWindow::ID_SETVALUE,SEL_COMMAND),(void*)pattern.text());
    return 1;
}


// Change directory
long FileList::onCmdSetDirectory(FXObject*,FXSelector,void* ptr)
{
    if(!ptr)
        return 0;
    setDirectory((const char*)ptr);
    return 1;
}


// Update directory
long FileList::onUpdSetDirectory(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,MKUINT(FXWindow::ID_SETVALUE,SEL_COMMAND),(void*)directory.text());
    return 1;
}


// Sort by name
long FileList::onCmdSortByName(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==cmpFName) ? cmpRName : cmpFName;
    sortItems();
    return 1;
}


// Update sender
long FileList::onUpdSortByName(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=(sortfunc==cmpFName || sortfunc==cmpRName) ? ID_CHECK : ID_UNCHECK;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    return 1;
}


// Sort by type
long FileList::onCmdSortByType(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==cmpFType) ? cmpRType : cmpFType;
    sortItems();
    return 1;
}


// Update sender
long FileList::onUpdSortByType(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=(sortfunc==cmpFType || sortfunc==cmpRType) ? ID_CHECK : ID_UNCHECK;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    return 1;
}


// Sort by size
long FileList::onCmdSortBySize(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==cmpFSize) ? cmpRSize : cmpFSize;
    sortItems();
    return 1;
}


// Update sender
long FileList::onUpdSortBySize(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=(sortfunc==cmpFSize || sortfunc==cmpRSize) ? ID_CHECK : ID_UNCHECK;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    return 1;
}


// Sort by time
long FileList::onCmdSortByTime(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==cmpFTime) ? cmpRTime : cmpFTime;
    sortItems();
    return 1;
}


// Update sender
long FileList::onUpdSortByTime(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=(sortfunc==cmpFTime || sortfunc==cmpRTime) ? ID_CHECK : ID_UNCHECK;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    return 1;
}


// Sort by user
long FileList::onCmdSortByUser(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==cmpFUser) ? cmpRUser : cmpFUser;
    sortItems();
    return 1;
}


// Update sender
long FileList::onUpdSortByUser(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=(sortfunc==cmpFUser || sortfunc==cmpRUser) ? ID_CHECK : ID_UNCHECK;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    return 1;
}


// Sort by group
long FileList::onCmdSortByGroup(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==cmpFGroup) ? cmpRGroup : cmpFGroup;
    sortItems();
    return 1;
}


// Update sender
long FileList::onUpdSortByGroup(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=(sortfunc==cmpFGroup || sortfunc==cmpRGroup) ? ID_CHECK : ID_UNCHECK;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    return 1;
}


// Reverse sort order
long FileList::onCmdSortReverse(FXObject*,FXSelector,void*)
{
    if(sortfunc==cmpFName)
        sortfunc=cmpRName;
    else if(sortfunc==cmpRName)
        sortfunc=cmpFName;
    else if(sortfunc==cmpFType)
        sortfunc=cmpRType;
    else if(sortfunc==cmpRType)
        sortfunc=cmpFType;
    else if(sortfunc==cmpFSize)
        sortfunc=cmpRSize;
    else if(sortfunc==cmpRSize)
        sortfunc=cmpFSize;
    else if(sortfunc==cmpFTime)
        sortfunc=cmpRTime;
    else if(sortfunc==cmpRTime)
        sortfunc=cmpFTime;
    else if(sortfunc==cmpFUser)
        sortfunc=cmpRUser;
    else if(sortfunc==cmpRUser)
        sortfunc=cmpFUser;
    else if(sortfunc==cmpFGroup)
        sortfunc=cmpRGroup;
    else if(sortfunc==cmpRGroup)
        sortfunc=cmpFGroup;
    sortItems();
    return 1;
}


// Update sender
long FileList::onUpdSortReverse(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=ID_UNCHECK;
    if(sortfunc==cmpRName)
        msg=ID_CHECK;
    if(sortfunc==cmpRType)
        msg=ID_CHECK;
    if(sortfunc==cmpRSize)
        msg=ID_CHECK;
    if(sortfunc==cmpRTime)
        msg=ID_CHECK;
    if(sortfunc==cmpRUser)
        msg=ID_CHECK;
    if(sortfunc==cmpRGroup)
        msg=ID_CHECK;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    return 1;
}


// Clicked header button
long FileList::onCmdHeader(FXObject*,FXSelector,void* ptr)
{
    if(((FXuint)(long)ptr)<6)
        handle(this,MKUINT((ID_SORT_BY_NAME+(FXuint)(long)ptr),SEL_COMMAND),NULL);
    if (getHeaderSize(0)<MIN_NAME_SIZE)
    	setHeaderSize(0,MIN_NAME_SIZE);	
    return 1;
}


// Clicked header button
long FileList::onUpdHeader(FXObject*,FXSelector,void*)
{
    header->setArrowDir(0,(sortfunc==cmpFName)  ? FALSE : (sortfunc==cmpRName) ? TRUE:MAYBE);   // Name
    header->setArrowDir(1,(sortfunc==cmpFSize)  ? FALSE : (sortfunc==cmpRSize) ? TRUE:MAYBE);   // Size
    header->setArrowDir(2,(sortfunc==cmpFType)  ? FALSE : (sortfunc==cmpRType) ? TRUE:MAYBE);   // Type
    header->setArrowDir(3,(sortfunc==cmpFTime)  ? FALSE : (sortfunc==cmpRTime) ? TRUE:MAYBE);   // Date
    header->setArrowDir(4,(sortfunc==cmpFUser)  ? FALSE : (sortfunc==cmpRUser) ? TRUE:MAYBE);   // User
    header->setArrowDir(5,(sortfunc==cmpFGroup) ? FALSE : (sortfunc==cmpRGroup)? TRUE:MAYBE);   // Group
    if (getHeaderSize(0)<MIN_NAME_SIZE)
    	setHeaderSize(0,MIN_NAME_SIZE);	
    return 1;
}



// Compare file names
FXint FileList::cmpFName(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            return 0;
        p++;
        q++;
    }
    return 0;
}


// Compare file types
FXint FileList::cmpFType(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    register const unsigned char *p=(const unsigned char*)strchr(a->label.text(),'\t')+1;
    register const unsigned char *q=(const unsigned char*)strchr(b->label.text(),'\t')+1;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            return 0;
        p++;
        q++;
    }
    return 0;
}


// Compare file size
FXint FileList::cmpFSize(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    return (long)a->size - (long)b->size;
}


// Compare file time
FXint FileList::cmpFTime(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    return (long)a->date - (long)b->date;
}


// Compare file user
FXint FileList::cmpFUser(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register const unsigned char *p,*q;
    register int i;
    FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    for(p=(const unsigned char*)a->label.text(),i=4; *p && i; i-=(*p++=='\t'))
        ;
    for(q=(const unsigned char*)b->label.text(),i=4; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            return 0;
        p++;
        q++;
    }
    return 0;
}


// Compare file group
FXint FileList::cmpFGroup(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register const unsigned char *p,*q;
    register int i;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    for(p=(const unsigned char*)a->label.text(),i=5; *p && i; i-=(*p++=='\t'))
        ;
    for(q=(const unsigned char*)b->label.text(),i=5; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            return 0;
        p++;
        q++;
    }
    return 0;
}

// Reversed compare file name
FXint FileList::cmpRName(const FXIconItem* pa,const FXIconItem* pb)
{
    return -FileList::cmpFName(pa,pb);
}


// Reversed compare file type
FXint FileList::cmpRType(const FXIconItem* pa,const FXIconItem* pb)
{
    return -FileList::cmpFType(pa,pb);
}


// Reversed compare file size
FXint FileList::cmpRSize(const FXIconItem* pa,const FXIconItem* pb)
{
    return -FileList::cmpFSize(pa,pb);
}


// Reversed compare file time
FXint FileList::cmpRTime(const FXIconItem* pa,const FXIconItem* pb)
{
    return -FileList::cmpFTime(pa,pb);
}


// Reversed compare file user
FXint FileList::cmpRUser(const FXIconItem* pa,const FXIconItem* pb)
{
    return -FileList::cmpFUser(pa,pb);
}


// Reversed compare file group
FXint FileList::cmpRGroup(const FXIconItem* pa,const FXIconItem* pb)
{
    return -FileList::cmpFGroup(pa,pb);
}


// Got refresh timeout
long FileList::onRefreshTimer(FXObject*,FXSelector,void*)
{
    FXuint interval=REFRESHINTERVAL;
    struct stat inf;
    FXTime changetime;

    // Only update if user is not interacting with the file list
    if(flags&FLAG_UPDATE)
    {
        // Test if a change has occurred
        if(info(directory,inf))
        {
            changetime=(FXTime)FXMAX(inf.st_mtime,inf.st_ctime);
            if((timestamp!=changetime) || (changetime==0))
            {
                // File system does not support mod-time on directory:- we schedule
                // next refresh in REFRESHINTERVALLONG ms instead of REFRESHINTERVAL ms
                if(changetime==0)
                    interval=REFRESHINTERVALLONG;

                // Refresh contents
                listDirectory();
                sortItems();

                // Last time checked
                timestamp=changetime;
            }
        }

        // Move to higher directory
        else
            setDirectory(FXFile::upLevel(directory));
    }

    // Reset timer again
    refreshtimer=getApp()->addTimeout(interval,this,ID_REFRESHTIMER);

    return 0;
}


// Set current filename
void FileList::setCurrentFile(const FXString& pathname)
{    // FIXME notify argument?
    if(!pathname.empty())
    {
        setDirectory(FXFile::directory(pathname));
        setCurrentItem(findItem(FXFile::name(pathname)));
    }
}


// Get pathname to current file, if any
FXString FileList::getCurrentFile() const
{
    if(current<0)
        return FXString::null;
    return getItemPathname(current);
}


// Set directory being displayed
void FileList::setDirectory(const FXString& pathname)
{    
	// FIXME notify argument?
    if(!pathname.empty())
    {
        FXString path=FXFile::absolute(directory,pathname);
        while(!FXFile::isTopDirectory(path) && !isDirectory(path))
            path=FXFile::upLevel(path);
        if(directory!=path)
        {
            directory=path;
            clearItems();
            listDirectory();
            sortItems();
        }
    }
}


// Set the pattern to filter
void FileList::setPattern(const FXString& ptrn)
{
    if(ptrn.empty())
        return;
    if(pattern!=ptrn)
    {
        pattern=ptrn;
        listDirectory();
        sortItems();
    }
}


// Change file match mode
void FileList::setMatchMode(FXuint mode)
{
    if(matchmode!=mode)
    {
        matchmode=mode;
        listDirectory();
        sortItems();
    }
}


// Return TRUE if showing hidden files
FXbool FileList::showHiddenFiles() const
{
    return (options&FILELIST_SHOWHIDDEN)!=0;
}


// Change show hidden files mode
void FileList::showHiddenFiles(FXbool shown)
{
    FXuint opts=shown ? (options|FILELIST_SHOWHIDDEN) : (options&~FILELIST_SHOWHIDDEN);
    if(opts!=options)
    {
        options=opts;
        listDirectory();
        sortItems();
    }
}


// Return TRUE if showing directories only
FXbool FileList::showOnlyDirectories() const
{
    return (options&FILELIST_SHOWDIRS)!=0;
}


// Change show directories only mode
void FileList::showOnlyDirectories(FXbool shown)
{
    FXuint opts=shown ? (options|FILELIST_SHOWDIRS) : (options&~FILELIST_SHOWDIRS);
    if(opts!=options)
    {
        options=opts;
        listDirectory();
        sortItems();
    }
}


// Compare till '\t' or '\0'
static FXbool fileequal(const FXchar* a,const FXchar* b)
{
    register FXchar c1,c2;
    do
    {
        c1=*a++;
        c2=*b++;
    }
    while(c1!='\0' && c1!='\t' && c1==c2);
    return (c1=='\0' || c1=='\t') && ((c2=='\0' || c2=='\t'));
}


// Hash till '\t' or '\0'
static FXint hash(const FXchar* str)
{
    register FXint h=0;
    register FXint g;
    while(*str!='\0' && *str!='\t')
    {
        h=(h<<4)+*str++;
        g=h&0xF0000000;
        if(g)
            h^=g>>24;
        h&=0x0fffffff;
    }
    return h;
}


// Create custom item
FXIconItem *FileList::createItem(const FXString& text,FXIcon *big,FXIcon* mini,void* ptr)
{
    return new FileItem(text,big,mini,ptr);
}


// List directory (from Fox library)
void FileList::listDirectory()
{
    FXchar grpid[64],usrid[64],atts[16],mod[32];
    FXchar pathname[MAXPATHLEN];
    FileItem *curitem=NULL;
    FileItem *item;
    FXchar *pathtail,*name;
    FileAssoc *fileassoc;
    FXString extension;
    FXIcon *big,*mini;
    FileItem **hashlist;
    int nhashlist,i,x,p,h;
    time_t filetime;
    struct stat info;
    struct dirent *dp;
    DIR *dirp;
    int islink;

    // Remember current item
    if(0<=current)
        curitem=(FileItem*)items[current];

    // Place existing items into a hash table
    for(nhashlist=32; nhashlist<=nitems; nhashlist<<=1)
        ;
    FXCALLOC(&hashlist,FileItem**,nhashlist);
    for(i=0; i<nitems; i++)
    {
        item=(FileItem*)items[i];
        h=hash(item->label.text());
        for(p=HASH1(h,nhashlist),x=HASH2(h,nhashlist); hashlist[p]; p=(p+x)%nhashlist)
            ;
        hashlist[p]=item;
    }

    // Start inserting
    nitems=0;

    // Get info about directory
    if(stat(directory.text(),&info)==0)
    {

        // Need latest change no matter what actually changed!
        timestamp=FXMAX(info.st_mtime,info.st_ctime);

        // Set path to stat with
        strcpy(pathname,directory.text());
        pathtail=pathname+strlen(pathname)-1;
        if(!ISPATHSEP(*pathtail))
            *++pathtail=PATHSEP;
        ++pathtail;

        // Get directory stream pointer
        dirp=opendir(directory.text());

        // Managed to open directory
        if(dirp)
        {

            // Loop over directory entries
            while((dp=readdir(dirp))!=NULL)
            {
                name=dp->d_name;

                // Hidden file (.xxx) or directory (. or .yyy) normally not shown,
                // but directory .. is always shown so we can navigate up or down
                if(name[0]=='.' && (name[1]==0 || (!(name[1]=='.'&& name[2]==0) && !(options&FILELIST_SHOWHIDDEN))))
                    continue;

                // Build full pathname
                strcpy(pathtail,name);

                // Get file/link info
                if(lstat(pathname,&info)!=0)
                    continue;

                // If its a link, get the info on file itself
                islink=S_ISLNK(info.st_mode);

                // If it is a link, modify the extension
                if(islink)
                    // If it is a broken link
                    if (stat(pathname,&info)!=0)
                        extension=_("Broken link to ");
                    else
                        extension=_("Link to ");
                else
                    extension="";

                // If not a directory and we want only directories, skip it
                if(!S_ISDIR(info.st_mode) && (options&FILELIST_SHOWDIRS))
                    continue;

                // Is it a directory or does it match the pattern?
                if(!S_ISDIR(info.st_mode) && !fxfilematch(pattern.text(),name,matchmode))
                    continue;

                // Anything about file has changed
                filetime=info.st_mtime;
                if(filetime<0)
                    filetime=0;

                // Take item from hash table if found
                h=hash(name);
                for(p=HASH1(h,nhashlist),x=HASH2(h,nhashlist); hashlist[p]; p=(p+x)%nhashlist)
                {
                    item=hashlist[p];
                    if((item!=(FileItem*)-1) && fileequal(name,item->label.text()))
                    {
                        hashlist[p]=(FileItem*)-1;
                        goto fnd;
                    }
                }

                // Make new item if we have to
                item=(FileItem*)createItem(NULL,NULL,NULL,NULL);

                // Append item in list
fnd:
                FXRESIZE(&items,FXIconItem*,nitems+1);
                if(item==curitem)
                    current=nitems;
                items[nitems]=item;
                nitems++;

                // Obtain user name
                fxgetusername(usrid,info.st_uid);

                // Obtain group name
                fxgetgroupname(grpid,info.st_gid);

                // Permissions
                fxgetpermissions(atts,info.st_mode);

                // Mod time
                my_strftime(mod,sizeof(mod),TIMEFORMAT,localtime(&filetime));

                // Item flags
                if(S_ISDIR(info.st_mode))
                    item->state|=FileItem::FOLDER;
                else
                    item->state&=~FileItem::FOLDER;
                if(S_ISLNK(info.st_mode))
                    item->state|=FileItem::SYMLINK;
                else
                    item->state&=~FileItem::SYMLINK;
                if(S_ISCHR(info.st_mode))
                    item->state|=FileItem::CHARDEV;
                else
                    item->state&=~FileItem::CHARDEV;
                if(S_ISBLK(info.st_mode))
                    item->state|=FileItem::BLOCKDEV;
                else
                    item->state&=~FileItem::BLOCKDEV;
                if(S_ISFIFO(info.st_mode))
                    item->state|=FileItem::FIFO;
                else
                    item->state&=~FileItem::FIFO;
                if(S_ISSOCK(info.st_mode))
                    item->state|=FileItem::SOCK;
                else
                    item->state&=~FileItem::SOCK;
                if((info.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH)) && !(S_ISDIR(info.st_mode)||S_ISCHR(info.st_mode)||S_ISBLK(info.st_mode)||S_ISFIFO(info.st_mode)||S_ISSOCK(info.st_mode)))
                    item->state|=FileItem::EXECUTABLE;
                else
                    item->state&=~FileItem::EXECUTABLE;

                // We can drag items
                item->state|=FileItem::DRAGGABLE;

                // Assume no associations
                fileassoc=NULL;

                // Determine icons and type
                if(item->state&FileItem::FOLDER)
                {
                    if(access(pathname,R_OK|X_OK))
                    {
                        big=big_lockfolder;
                        mini=mini_lockfolder;
                        extension+=_("Folder");
                    }
                    else
                    {
                        big=big_folder;
                        mini=mini_folder;
                        extension+=_("Folder");
                    }
                    if(associations)
                        fileassoc=associations->findDirBinding(pathname);
                }
                else if(item->state&FileItem::CHARDEV)
                {
                    big=big_cdev;
                    mini=mini_cdev;
                    extension+=_("Character Device");
                }
                else if(item->state&FileItem::BLOCKDEV)
                {
                    big=big_bdev;
                    mini=mini_bdev;
                    extension+=_("Block Device");
                }
                else if(item->state&FileItem::FIFO)
                {
                    big=big_pipe;
                    mini=mini_pipe;
                    extension+=_("Named Pipe");
                }
                else if(item->state&FileItem::SOCK)
                {
                    big=big_sock;
                    mini=mini_sock;
                    extension+=_("Socket");
                }
                else if(item->state&FileItem::EXECUTABLE)
                {
                    big=big_app;
                    mini=mini_app;
                    extension+=_("Application");
                    if(associations)
                        fileassoc=associations->findFileBinding(pathname);
                }
                else
                {
                    big=big_doc;
                    mini=mini_doc;
                    extension+=_("Document");
                    if(associations)
                        fileassoc=associations->findFileBinding(pathname);
                }

                // If association is found, use it
                if(fileassoc)
                {
                    // If it is a link, modify the extension
                    if(islink)
                    {
                        // If it is a broken link
                        if (stat(pathname,&info)!=0)
                            extension=_("Broken link to ");
                        else
                            extension=_("Link to ");

                        // Complete the extension
                        extension+=fileassoc->extension.text();
                    }
                    else
                        extension=fileassoc->extension.text();

                    if(fileassoc->bigicon)
                        big=fileassoc->bigicon;
                    if(fileassoc->miniicon)
                        mini=fileassoc->miniicon;
                }

                // Update item information
                FXchar size[64], hsize[64];
				sprintf(size,"%lu",(unsigned long)info.st_size);
				::Size(size,hsize);
				item->label.format("%s\t%s\t%s\t%s\t%s\t%s\t%s",name,hsize,extension.text(),mod,usrid,grpid,atts);
                item->bigIcon=big;
                item->miniIcon=mini;
                item->size=(unsigned long)info.st_size;
                item->assoc=fileassoc;
                item->date=filetime;

#if defined(linux)
    			// Devices have a specific icon
				if(devices->find(pathname))
				{
					item->bigIcon=big_harddisk;
					item->miniIcon=hard_disk;
 				}
#endif
    			// Dotdot folders have a specific icon
				if(name[0]=='.' && name[1]=='.' && name[2]==0)
				{
					item->bigIcon=big_folderup;
					item->miniIcon=mini_folderup;
 				}

    			// Symbolic links have a specific icon				
				if (islink)
				{
					item->bigIcon=big_link;
					item->miniIcon=mini_link;
				}
					
                // Create item
                if(id())
                    item->create();
            }
            closedir(dirp);
        }
    }

    // Wipe remaining items from list
    for(i=0; i<nhashlist; i++)
    {
        if((hashlist[i]!=NULL) && (hashlist[i]!=(FileItem*)-1))
            delete hashlist[i];
    }
    FXFREE(&hashlist);

    // Validate
    if(current>=nitems)
        current=-1;
    if(anchor>=nitems)
        anchor=-1;
    if(extent>=nitems)
        extent=-1;

    // Gotta recalc size of content
    recalc();
}


// Is directory
FXbool FileList::isItemDirectory(FXint index) const
{
    if(index<0 || nitems<=index)
        fxerror("%s::isItemDirectory: index out of range.\n",getClassName());
    return (((FileItem*)items[index])->state&FileItem::FOLDER)!=0;
}


// Is file
FXbool FileList::isItemFile(FXint index) const
{
    if(index<0 || nitems<=index)
        fxerror("%s::isItemFile: index out of range.\n",getClassName());
    return (((FileItem*)items[index])->state&(FileItem::FOLDER|FileItem::CHARDEV|FileItem::BLOCKDEV|FileItem::FIFO|FileItem::SOCK))==0;
}


// Is executable
FXbool FileList::isItemExecutable(FXint index) const
{
    if(index<0 || nitems<=index)
        fxerror("%s::isItemExecutable: index out of range.\n",getClassName());
    return (((FileItem*)items[index])->state&FileItem::EXECUTABLE)!=0;
}


// Get number of selected items 
FXint FileList::getNumSelectedItems(void) const
{
    FXint num=0;
    for (int u=0; u<getNumItems(); u++)
        if (isItemSelected(u))
            num++;
	return num;
}


// Get number of selected items and index of first selected item
FXint FileList::getNumSelectedItems(FXint *index) const
{
    FXint num=0, itm=-1;
    for (int u=0; u<getNumItems(); u++)
        if (isItemSelected(u))
		{
			if (itm== -1)
				itm= u;
            num++;
		}
	(*index)=itm;
	return num;
}


// Get file name from item
FXString FileList::getItemFilename(FXint index) const
{
    if(index<0 || nitems<=index)
        fxerror("%s::getItemFilename: index out of range.\n",getClassName());
    FXString label=items[index]->getText();
    return label.extract(0,'\t');
}


// Get full pathname to item
FXString FileList::getItemPathname(FXint index) const
{
    if(index<0 || nitems<=index)
        fxerror("%s::getItemPathname: index out of range.\n",getClassName());
    FXString label=items[index]->getText();
    return FXFile::absolute(directory,label.extract(0,'\t'));
}


// Get associations (if any) from the file
FileAssoc* FileList::getItemAssoc(FXint index) const
{
    if(index<0 || nitems<=index)
        fxerror("%s::getItemAssoc: index out of range.\n",getClassName());
    return ((FileItem*)items[index])->assoc;
}

// Return file size of the item
unsigned long FileList::getItemFileSize(FXint index) const
{
    if(index<0 || nitems<=index)
        fxerror("%s::getItemFileSize: index out of range.\n",getClassName());
    return ((FileItem*)items[index])->size;
}


// Change associations table
void FileList::setAssociations(FileDict* assocs)
{
    if(associations!=assocs)
    {
        associations=assocs;
        clearItems();
        listDirectory();
        sortItems();
    }
}


// Save data
void FileList::save(FXStream& store) const
{
    FXIconList::save(store);
    store << directory;
    store << associations;
    store << pattern;
    store << matchmode;
    store << big_folder;
    store << mini_folder;
    store << big_doc;
    store << mini_doc;
    store << big_app;
    store << mini_app;
}


// Load data
void FileList::load(FXStream& store)
{
    FXIconList::load(store);
    store >> directory;
    store >> associations;
    store >> pattern;
    store >> matchmode;
    store >> big_folder;
    store >> mini_folder;
    store >> big_doc;
    store >> mini_doc;
    store >> big_app;
    store >> mini_app;
}


// Cleanup
FileList::~FileList()
{
    if(refreshtimer)
        getApp()->removeTimeout(refreshtimer);
    if(opentimer)
        getApp()->removeTimeout(opentimer);
    if(!(options&FILELIST_NO_OWN_ASSOC))
        delete associations;
    delete big_folder;
    delete mini_folder;
    delete big_doc;
    delete mini_doc;
    delete big_app;
    delete mini_app;
    associations=(FileDict*)-1;
    big_folder=(FXIcon*)-1;
    mini_folder=(FXIcon*)-1;
    big_doc=(FXIcon*)-1;
    mini_doc=(FXIcon*)-1;
    big_app=(FXIcon*)-1;
    mini_app=(FXIcon*)-1;
    refreshtimer=(FXTimer*)-1;
    opentimer=(FXTimer*)-1;
}


