/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include <sys/select.h>

#include <signal.h>

#include <glib.h>
#include <gmodule.h>
#include <gdk/gdk.h>

#include <libxfce4util/libxfce4util.h>

#include <xfmedia/xfmedia-plugin.h>
#include <xfmedia/xfmedia-interface.h>

typedef struct
{
    GThread *pollth;
    GAsyncQueue *q;
    gchar *pipefile;
} InfopipeData;

static GQuark errq = 0;

static void
infopipe_unloading_cb(XfMediaPlugin *plugin)
{
    InfopipeData *ipdata = g_object_get_data(G_OBJECT(plugin),
            "xfmedia-infopipe-data");
    
    g_async_queue_push(ipdata->q, GUINT_TO_POINTER(0xdeadbeef));
    g_thread_join(ipdata->pollth);
    g_async_queue_unref(ipdata->q);
    
    unlink(ipdata->pipefile);
    g_free(ipdata->pipefile);
    g_free(ipdata);
}

static void
pipe_write_data(XfMediaPlugin *plugin, gint fd)
{
    gchar buf[4096], *tmp, *codec = NULL;
    gint bitrate, sample_rate, channels, bits_per_sample, width, height;
    gdouble fps, aspect;
    XfMediaStatus status = xfmedia_engine_get_status(plugin);
    XfMediaSpeed speed = xfmedia_engine_get_speed(plugin);
    XfMediaPlaylist *plist = xfmedia_plugin_get_playlist(plugin);
    
    if(status == XFMEDIA_STATUS_PLAYING) {
        if(speed == XFMEDIA_SPEED_PAUSED)
            tmp = "Paused";
        else
            tmp = "Playing";
    } else
        tmp = "Stopped";
    
    g_snprintf(buf, 4096, _("Status:                  %s\n"), tmp);
    write(fd, buf, strlen(buf));
    
    tmp = xfmedia_info_get_name(plugin);
    g_snprintf(buf, 4096, _("Now Playing:             %s\n"), tmp);
    write(fd, buf, strlen(buf));
    g_free(tmp);
    
    tmp = (gchar *)xfmedia_info_get_location(plugin);
    g_snprintf(buf, 4096, _("Filename:                %s\n"),
            tmp?tmp:_("(none)"));
    write(fd, buf, strlen(buf));
    
    g_snprintf(buf, 4096, _("Track Position:          %d\n"),
            xfmedia_engine_get_current_time(plugin) / 1000);
    write(fd, buf, strlen(buf));
    
    g_snprintf(buf, 4096, _("Track Length:            %d\n"),
            xfmedia_engine_get_total_time(plugin) / 1000);
    write(fd, buf, strlen(buf));
    
    if(xfmedia_info_get_audio_info(plugin, &codec, &bitrate, &sample_rate,
            &channels, &bits_per_sample))
    {
        g_snprintf(buf, 4096, _("Audio Codec:             %s\nAudio Bitrate:           %d\nAudio Sample Rate:       %d\nAudio Channels:          %d\nAudio Bits Per Sample:   %d\n"),
                codec?codec:_("(none)"), bitrate, sample_rate, channels,
                bits_per_sample);
        write(fd, buf, strlen(buf));
        g_free(codec);
        codec = NULL;
    }
    
    if(xfmedia_info_get_video_info(plugin, &codec, &bitrate, &fps,
            &width, &height, &aspect))
    {
        g_snprintf(buf, 4096, _("Video Codec:            %s\nVideo Bitrate:          %d\nVideo Frames Per Second: %.02f\nVideo Frame Size:        %dx%d\nVideo Aspect Ratio:      %.02f\n"),
                codec?codec:_("(none)"), bitrate, fps, width, height, aspect);
        write(fd, buf, strlen(buf));
        g_free(codec);
        codec = NULL;
    }
    
    g_snprintf(buf, 4096, _("Playlist Length:         %d\n"),
            xfmedia_playlist_get_n_entries(plist));
    write(fd, buf, strlen(buf));
    
    g_snprintf(buf, 4096, _("Current Playlist Index:  %d\n"),
            xfmedia_playlist_get_selected(plist));
    write(fd, buf, strlen(buf));
}

static gpointer
pipe_thread(gpointer data)
{
    XfMediaPlugin *plugin = data;
    InfopipeData *ipdata = g_object_get_data(G_OBJECT(plugin),
            "xfmedia-infopipe-data");
    gboolean done = FALSE;
    gint fd;
    fd_set wfd;
    struct timeval tv;
    
    g_async_queue_ref(ipdata->q);
    
    while(!done) {
        /* O_WRONLY seems to cause a hang on shutdown, and O_NONBLOCK here
         * (as opposed to fcntl() below) causes a segfault */
        fd = open(ipdata->pipefile, O_RDWR);
        if(fd < 0) {
            g_warning("Xfmedia-Infopipe: Failed to open pipe.");
            break;
        }
        
        fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) | O_NONBLOCK);
        
        for(;;) {
            if(GPOINTER_TO_UINT(g_async_queue_try_pop(ipdata->q)) == 0xdeadbeef) {
                done = TRUE;
                break;
            }
            
            FD_ZERO(&wfd);
            FD_SET(fd, &wfd);
            tv.tv_sec = 0;
            tv.tv_usec = 500000;
            if(select(fd+1, NULL, &wfd, NULL, &tv) > 0 && FD_ISSET(fd, &wfd)) {
                gdk_threads_enter();
                pipe_write_data(plugin, fd);
                gdk_threads_leave();
                close(fd);
                /* this is kinda lame, but... */
                sleep(1);
                break;
            }
        }
    }
    
    g_async_queue_unref(ipdata->q);
    
    return NULL;
}

static gboolean
start_pipe_thread(XfMediaPlugin *plugin, GError **err)
{
    InfopipeData *ipdata = g_object_get_data(G_OBJECT(plugin),
            "xfmedia-infopipe-data");
    
    ipdata->q = g_async_queue_new();
    
    ipdata->pollth = g_thread_create(pipe_thread, plugin, TRUE, err);
    if(!ipdata->pollth)
        return FALSE;
    return TRUE;
}

static gboolean
setup_pipe(XfMediaPlugin *plugin, GError **err)
{
    InfopipeData *ipdata = g_object_get_data(G_OBJECT(plugin),
            "xfmedia-infopipe-data");
    gchar *pipefile;
    
    pipefile = g_strdup_printf("%s/xfmedia-infopipe.%d.%d", g_get_tmp_dir(),
            (gint)getuid(), xfmedia_interface_get_session_number(plugin));
    
    DBG("opening pipe file %s", pipefile);
    
    if(g_file_test(pipefile, G_FILE_TEST_EXISTS)) {
        if(unlink(pipefile)) {
            if(err) {
                g_set_error(err, errq, 0, 
                        _("Xfmedia-Infopipe: Unable to delete file '%s'."),
                        pipefile);
            }
            g_free(pipefile);
            return FALSE;
        }
    }
    
    if(mkfifo(pipefile, 0600)) {
        if(err) {
            g_set_error(err, errq, 0, 
                    _("Xfmedia-Infopipe: Unable to create pipe '%s'."),
                    pipefile);
        }
        g_free(pipefile);
        return FALSE;
    }
    
    ipdata->pipefile = pipefile;
    
    return TRUE;    
}

G_MODULE_EXPORT gboolean
xfmedia_plugin_get(XfMediaPlugin *plugin, GError **error)
{
    InfopipeData *ipdata;
    
    xfmedia_plugin_set_name(plugin, _("Xfmedia Infopipe"));
    xfmedia_plugin_set_description(plugin,
            _("The Infopipe plugin creates a pipe in /tmp that you can use to retrieve information about the currently playing track."));
    xfmedia_plugin_set_version(plugin, VERSION);
    xfmedia_plugin_set_author(plugin, "Brian Tarricone");
    xfmedia_plugin_set_website(plugin, WEBSITE);
    xfmedia_plugin_set_license(plugin, "GPL");
    xfmedia_plugin_set_date(plugin, "2004-2005");
    
    if(!errq)
        errq = g_quark_from_static_string("xfmedia-infopipe-error");
    
    signal(SIGPIPE, SIG_IGN);
    
    ipdata = g_new0(InfopipeData, 1);
    g_object_set_data(G_OBJECT(plugin), "xfmedia-infopipe-data", ipdata);
    
    if(!setup_pipe(plugin, error) || !start_pipe_thread(plugin, error)) {
        g_free(ipdata);
        return FALSE;
    }
    
    g_signal_connect(G_OBJECT(plugin), "unloading",
            G_CALLBACK(infopipe_unloading_cb), NULL);
    
    return TRUE;        
}

XFMEDIA_PLUGIN_DECL
