/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>
#ifndef MAP_FILE
#define MAP_FILE 0
#endif
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <glib.h>
#include <libxfce4util/libxfce4util.h>

#ifdef GTK_DISABLE_DEPRECATED
#undef GTK_DISABLE_DEPRECATED
#endif
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <libxfcegui4/libxfcegui4.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include "mediamarks.h"
#include <xfmedia/xfmedia-settings.h>
#include "mainwin.h"
#include <xfmedia/xfmedia-playlist.h>

#define MEDIAMARKS_FILE  "xfmedia/mediamarks.xml"
#define MM_PATH_MAX      4096

#ifndef GTK_STOCK_EDIT
#define GTK_STOCK_EDIT GTK_STOCK_PREFERENCES
#endif
#ifndef GTK_STOCK_FILE
#define GTK_STOCK_FILE GTK_STOCK_NEW
#endif

/* NOTE: keep in sync with playlist.c.  though i don't remember why. */
enum {
	DROP_TGT_REORDER = 0,
	DROP_TGT_URILIST,
	DROP_TGT_STRING
};

typedef struct
{
    GNode *cur_folder;
    gint next_index;
    gboolean started;
} XfMediaMMState;

static GNode *mediamarks_root = NULL;
static GList *mm_changed_cb = NULL;
static GList *mm_changed_data = NULL;

static void
mm_changed_notify()
{
    GList *callback, *data;
    
    for(callback = mm_changed_cb, data = mm_changed_data;
        callback && data;
        callback = callback->next, data = data->next)
    {
        MMChangedCallback cb = callback->data;
        if(cb)
            cb(data->data);
    }
}

static void
mm_write_all()
{
    gchar *mmfile;
    
    g_return_if_fail(mediamarks_root);
    
    mmfile = xfce_resource_save_location(XFCE_RESOURCE_CONFIG, MEDIAMARKS_FILE,
            TRUE);
    if(mmfile) {
        if(!xfmedia_mediamarks_save(mmfile))
            g_critical("Unable to save mediamarks to file %s", mmfile);
        g_free(mmfile);
    }
}

static gboolean
mediamarks_free_all(GNode *node, gpointer data)
{
    XfMediaMediamark *mmark = node->data;
    
    if(!mmark)
        return FALSE;
    
    if(mmark->name)
        g_free(mmark->name);
    if(mmark->uri)
        g_free(mmark->uri);
    
    g_free(mmark);
    node->data = NULL;
    
    return FALSE;
}

static void
mediamarks_xml_start(GMarkupParseContext *context, const gchar *element_name,
        const gchar **attribute_names, const gchar **attribute_values,
        gpointer user_data, GError **error)
{
    XfMediaMMState *state = user_data;
    XfMediaMediamark *mmark;
    GNode *node;
    
    if(!state->started) {
        if(!strcmp(element_name, "xfmedia-mediamarks"))
            state->started = TRUE;
        return;
    }
    
    if(!strcmp(element_name, "folder")) {
        if(attribute_names[0] && !strcmp(attribute_names[0], "name")) {
            mmark = g_new0(XfMediaMediamark, 1);
            mmark->name = g_strdup(attribute_values[0]);
            mmark->index = state->next_index;
            node = g_node_new(mmark);
            g_node_append(state->cur_folder, node);
            
            state->cur_folder = node;
            state->next_index = 0;
        }
    } else if(!strcmp(element_name, "mediamark")) {
        const gchar *name = NULL, *uri = NULL;
        gint i;
        
        for(i = 0; attribute_names[i]; i++) {
            if(!strcmp(attribute_names[i], "name"))
                name = attribute_values[i];
            else if(!strcmp(attribute_names[i], "uri"))
                uri = attribute_values[i];
        }
        
        if(name && uri) {
            mmark = g_new0(XfMediaMediamark, 1);
            mmark->name = g_strdup(name);
            mmark->uri = g_strdup(uri);
            mmark->index = state->next_index++;
            node = g_node_new(mmark);
            g_node_append(state->cur_folder, node);
        }
    }
}

static void
mediamarks_xml_end(GMarkupParseContext *context, const gchar *element_name,
        gpointer user_data, GError **error)
{
    XfMediaMMState *state = user_data;
    
    if(!strcmp(element_name, "xfmedia-mediamarks"))
        state->started = FALSE;
    else if(!strcmp(element_name, "folder")) {
        state->next_index = ((XfMediaMediamark *)state->cur_folder->data)->index + 1;
        if(!G_NODE_IS_ROOT(state->cur_folder))
            state->cur_folder = state->cur_folder->parent;
    }
}

static void
mediamarks_write_xml_tree(FILE *fp, GNode *node, gint depth)
{
    gchar tabs[128], *name_esc, *uri_esc = NULL;
    XfMediaMediamark *mmark;
    GNode *child;
    
    if(depth > 127)
        depth = 127;
    memset(tabs, '\t', depth);
    tabs[depth] = 0;
    
    for(child = node->children; child; child = child->next) {
        mmark = child->data;
        if(!mmark || !mmark->name)
            continue;
        
        name_esc = g_markup_escape_text(mmark->name, strlen(mmark->name));
        if(mmark->uri)
            uri_esc = g_markup_escape_text(mmark->uri, strlen(mmark->uri));
        
        if(mmark->uri)
            fprintf(fp, "%s<mediamark name='%s' uri='%s' />\n", tabs, name_esc, uri_esc);
        else {
            fprintf(fp, "%s<folder name='%s'>\n", tabs, name_esc);
            mediamarks_write_xml_tree(fp, child, depth+1);
            fprintf(fp, "%s</folder>\n", tabs);
        }
        
        g_free(name_esc);
        if(mmark->uri)
            g_free(uri_esc);
    }
}

G_GNUC_UNUSED static gboolean
mediamarks_node_is_in_tree(GNode *node)
{
    /* FIXME: implement this */
    return TRUE;
}

static GNode *
mediamarks_insert_node(GNode *parent, const gchar *name, const gchar *uri,
        gint index)
{
    XfMediaMediamark *mmark;
    GNode *node = NULL;
    
    mmark = g_new0(XfMediaMediamark, 1);
    mmark->name = g_strdup(name);
    if(uri)
        mmark->uri = g_strdup(uri);
    if(index < 0)
        mmark->index = g_node_n_children(parent);
    
    node = g_node_new(mmark);
    if(index < 0)
        g_node_append(parent, node);
    else {
        gint i;
        
        g_node_insert(parent, index, node);
        
        for(i = index + 1; ; i++) {
            GNode *node = g_node_nth_child(parent, i);
            if(!node)
                break;
            ((XfMediaMediamark *)node->data)->index = i;
        }
    }
    
    return node;
}

void
xfmedia_mediamarks_init()
{
    gchar *mmfile;
    XfMediaMediamark *mmark;
    
    g_return_if_fail(!mediamarks_root);
    
    mmark = g_new0(XfMediaMediamark, 1);
    mmark->name = g_strdup("/");
    mmark->index = -1;
    mediamarks_root = g_node_new(mmark);
    
    mmfile = xfce_resource_save_location(XFCE_RESOURCE_CONFIG, MEDIAMARKS_FILE,
            FALSE);
    if(mmfile) {
        if(!xfmedia_mediamarks_load(mmfile)) {
            xfmedia_mediamarks_cleanup();
            mmark = g_new0(XfMediaMediamark, 1);
            mmark->name = g_strdup("/");
            mmark->index = -1;
            mediamarks_root = g_node_new(mmark);
        }
        g_free(mmfile);
    }
}

void
xfmedia_mediamarks_cleanup()
{
    g_return_if_fail(mediamarks_root);
    
    mm_write_all();
    
    g_node_traverse(mediamarks_root, G_IN_ORDER, G_TRAVERSE_ALL, -1,
            (GNodeTraverseFunc)mediamarks_free_all, NULL);
    
    g_node_destroy(mediamarks_root);
    mediamarks_root = NULL;
}

gboolean
xfmedia_mediamarks_load(const gchar *filename)
{
    gboolean ret = FALSE;
    struct stat st;
#ifdef HAVE_SYS_MMAN_H
    void *addr = NULL;
    gint fd;
#endif
    gchar *file_contents = NULL;
    gint file_length;
    GMarkupParser parser = {
        mediamarks_xml_start, mediamarks_xml_end, NULL, NULL, NULL
    };
    GMarkupParseContext *pctx;
    XfMediaMMState state;
    GError *err = NULL;
    
    if(stat(filename, &st))
        return FALSE;
    
#ifdef HAVE_SYS_MMAN_H
    fd = open(filename, O_RDONLY);
    if(fd < 0)
        return FALSE;
    addr = mmap(NULL, st.st_size, PROT_READ, MAP_SHARED|MAP_FILE, fd, 0);
    if(MAP_FAILED != addr)
        file_contents = addr;
#endif
    
    if(!file_contents)
        g_file_get_contents(filename, &file_contents, &file_length, NULL);
    
    state.cur_folder = mediamarks_root;
    state.next_index = 0;
    state.started = FALSE;
    pctx = g_markup_parse_context_new(&parser, 0, &state, NULL);    
    if(!g_markup_parse_context_parse(pctx, file_contents, st.st_size, &err)) {
        if(err) {
            g_critical("Unable to parse mediamarks file (%d): %s",
                    err->code, err->message);
            g_error_free(err);
        }
    } else {
        if(!g_markup_parse_context_end_parse(pctx, &err)) {
            g_critical("Unable to finish parsing mediamarks file (%d): %s",
                    err->code, err->message);
            g_error_free(err);
        } else
            ret = TRUE;
    }
    g_markup_parse_context_free(pctx);

#ifdef HAVE_SYS_MMAN_H
    if(MAP_FAILED != addr) {
        munmap(addr, st.st_size);
        file_contents = NULL;
    }
    if(fd >= 0)
        close(fd);
#endif
    
    if(file_contents)
        g_free(file_contents);
    
    return ret;
}
    
gboolean
xfmedia_mediamarks_save(const gchar *filename)
{
    gchar *mmfile_new = NULL;
    FILE *fp = NULL;
    gboolean ret = FALSE;
    
    g_return_val_if_fail(mediamarks_root && filename, FALSE);
    
    mmfile_new = g_strdup_printf("%s.new", filename);
    
    fp = fopen(mmfile_new, "w");
    if(!fp)
        goto cleanup;
    
    fputs("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n\n<xfmedia-mediamarks>\n", fp);
    mediamarks_write_xml_tree(fp, mediamarks_root, 1);
    fputs("</xfmedia-mediamarks>\n", fp);
    
    ret = TRUE;
    fclose(fp);
    
    if(g_file_test(filename, G_FILE_TEST_EXISTS)) {
        if(!unlink(filename)) {
            if(!link(mmfile_new, filename)) {
                if(unlink(mmfile_new))
                    g_warning("Unable to delete temp file when writing mediamarks");
            } else {
                g_warning("Unable to copy temp file to mediamarks file");
                ret = FALSE;
            }
        } else {
            g_warning("Unable to remove old mediamarks file");
            ret = FALSE;
        }
    } else {
        if(!link(mmfile_new, filename)) {
            if(unlink(mmfile_new))
                g_warning("Unable to delete temp file when writing mediamarks");
        } else {
            g_warning("Unable to copy temp file to mediamarks file");
            ret = FALSE;
        }
    }
    
    cleanup:
    
    if(mmfile_new)
        g_free(mmfile_new);
    
    return ret;
}

GNode *
xfmedia_mediamarks_insert_folder(GNode *parent, const gchar *name, gint index)
{
    GNode *node;
    
    g_return_val_if_fail(parent && name
            && mediamarks_node_is_in_tree(parent), NULL);
    
    node = mediamarks_insert_node(parent, name, NULL, index);
    
    mm_changed_notify();
    mm_write_all();
    
    return node;
}

gboolean
xfmedia_mediamarks_remove_folder(GNode *folder_node)
{
    g_return_val_if_fail(folder_node
            && !((XfMediaMediamark *)folder_node->data)->uri
            && mediamarks_node_is_in_tree(folder_node), FALSE);
    
    g_node_traverse(folder_node, G_IN_ORDER, G_TRAVERSE_ALL, -1,
            (GNodeTraverseFunc)mediamarks_free_all, NULL);
    g_node_unlink(folder_node);
    g_node_destroy(folder_node);
    
    mm_changed_notify();
    mm_write_all();
    
    return TRUE;
}

GNode *
xfmedia_mediamarks_insert_mark(GNode *parent, const gchar *name,
        const gchar *uri, gint index)
{
    GNode *node;
    
    g_return_val_if_fail(parent && name && uri
            && mediamarks_node_is_in_tree(parent), FALSE);
    
    node = mediamarks_insert_node(parent, name, uri, index);
    
    mm_changed_notify();
    mm_write_all();
    
    return node;
}

gboolean
xfmedia_mediamarks_remove_mark(GNode *mmark_node)
{
    g_return_val_if_fail(mmark_node
            && ((XfMediaMediamark *)mmark_node->data)->uri
            && mediamarks_node_is_in_tree(mmark_node), FALSE);
    
    g_node_unlink(mmark_node);
    g_node_destroy(mmark_node);
    
    mm_changed_notify();
    mm_write_all();
    
    return TRUE;
}

GNode *
xfmedia_mediamarks_get_tree()
{
    return mediamarks_root;
}

/***************************************/

enum {
    MMADD_COL_ICON = 0,
    MMADD_COL_NAME,
    MMADD_COL_NODE,
    MMADD_N_COLS
};

enum {
    MMEDIT_COL_ICON = 0,
    MMEDIT_COL_NAME,
    MMEDIT_COL_URI,
    MMEDIT_COL_NODE,
    MMEDIT_N_COLS
};

static GdkPixbuf *folder_pix = NULL;
static GdkPixbuf *mmark_pix = NULL;

static void
mm_ditch_stock_pixbufs()
{
    if(folder_pix)
        g_object_unref(G_OBJECT(folder_pix));
    if(mmark_pix)
        g_object_unref(G_OBJECT(mmark_pix));
}

static void
mm_load_stock_pixbufs()
{
    GtkWidget *dummy;
    
    /* lame */
    dummy = gtk_invisible_new();
    gtk_widget_realize(dummy);
    
    folder_pix = gtk_widget_render_icon(dummy, GTK_STOCK_DIRECTORY,
            GTK_ICON_SIZE_MENU, NULL);
    mmark_pix = gtk_widget_render_icon(dummy, GTK_STOCK_FILE,
            GTK_ICON_SIZE_MENU, NULL);
    
    gtk_widget_destroy(dummy);
    
    g_atexit(mm_ditch_stock_pixbufs);
}

static void
mm_add_folder_cb(GtkWidget *w, gpointer user_data)
{
    GtkWidget *entry = user_data;
    GtkWidget *treeview = g_object_get_data(G_OBJECT(w), "xfmedia-addmm-treeview");
    GtkTreeStore *ts = GTK_TREE_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(treeview)));
    GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    GtkTreeIter itr, pitr;
    gchar *name = NULL;
    GNode *parent = NULL, *node;
    
    if(!gtk_tree_selection_get_selected(sel, NULL, &pitr))
        gtk_tree_model_get_iter_first(GTK_TREE_MODEL(ts), &pitr);
    
    name = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);
    if(!name)
        return;
    gtk_tree_model_get(GTK_TREE_MODEL(ts), &pitr, MMADD_COL_NODE, &parent, -1);
    node = xfmedia_mediamarks_insert_folder(parent, name, -1);
    
    if(!node) {
        g_free(name);
        return;
    }
    
    gtk_tree_store_append(ts, &itr, &pitr);
    DBG("adding name %s", name);
    gtk_tree_store_set(ts, &itr, MMADD_COL_ICON, folder_pix,
            MMADD_COL_NAME, name, MMADD_COL_NODE, node, -1);
    
    g_free(name);
    gtk_tree_view_expand_all(GTK_TREE_VIEW(treeview));
    gtk_entry_set_text(GTK_ENTRY(entry), "");
}

static gboolean
mm_add_folder_entry_kp_cb(GtkWidget *w, GdkEventKey *evt, gpointer user_data)
{
    if(evt->keyval == GDK_Return || evt->keyval == GDK_KP_Enter)
        mm_add_folder_cb(GTK_WIDGET(user_data), w);
    
    return FALSE;
}

static gboolean
mmtree_add_folders(GNode *node, gpointer data)
{
    GtkTreeStore *ts = data;
    XfMediaMediamark *mmark = node->data;
    static gint parent_depth = 1;
    static GtkTreeIter itr, pitr;
    
    if(mmark->uri)
        return FALSE;
    
    if(!folder_pix && !mmark_pix)
        mm_load_stock_pixbufs();
    
    if(g_node_depth(node) == 1) {
        DBG("adding root node");
        gtk_tree_store_append(ts, &pitr, NULL);
        gtk_tree_store_set(ts, &pitr, MMADD_COL_ICON, folder_pix,
                MMADD_COL_NAME, _("Mediamarks"),
                MMADD_COL_NODE, node, -1);
    } else {
        if(g_node_depth(node) == parent_depth+1) {
            DBG("item is at same level");
            gtk_tree_store_append(GTK_TREE_STORE(ts), &itr, &pitr);
        } else if(g_node_depth(node) == parent_depth+2) {
            DBG("item is child of current level");
            memcpy(&pitr, &itr, sizeof(GtkTreeIter));
            gtk_tree_store_append(GTK_TREE_STORE(ts), &itr, &pitr);
            parent_depth++;
        } else if(g_node_depth(node) < parent_depth+1) {
            GtkTreePath *path = gtk_tree_model_get_path(GTK_TREE_MODEL(ts),
                    &pitr);
            DBG("item is ancestor of current level");
            while(g_node_depth(node) < parent_depth+1) {
                gtk_tree_path_up(path);
                parent_depth--;
            }
            gtk_tree_model_get_iter(GTK_TREE_MODEL(ts), &pitr, path);
            gtk_tree_path_free(path);
            gtk_tree_store_append(GTK_TREE_STORE(ts), &itr, &pitr);
        }
        
        gtk_tree_store_set(GTK_TREE_STORE(ts), &itr, MMADD_COL_ICON, folder_pix,
                MMADD_COL_NAME, mmark->name, MMADD_COL_NODE, node, -1);
    }
    
    return FALSE;
}

static GtkWidget *
mm_create_folder_treeview(XfMediaMainwin *mwin, GtkBox *box)
{
    GtkWidget *treeview, *sw;
    GtkTreeStore *ts;
    GtkCellRenderer *render;
    GtkTreeViewColumn *col;
    
    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
            GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
            GTK_SHADOW_ETCHED_IN);
    gtk_widget_show(sw);
    gtk_box_pack_start(box, sw, TRUE, TRUE, 0);
    
    ts = gtk_tree_store_new(MMADD_N_COLS, GDK_TYPE_PIXBUF, G_TYPE_STRING,
            G_TYPE_POINTER);
    treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(ts));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
    
    col = gtk_tree_view_column_new();
    gtk_tree_view_column_set_title(col, "folder");
    gtk_tree_view_column_set_expand(col, TRUE);
    
    render = gtk_cell_renderer_pixbuf_new();
    gtk_tree_view_column_pack_start(col, render, FALSE);
    gtk_tree_view_column_set_attributes(col, render, "pixbuf", MMADD_COL_ICON, NULL);
    
    render = exo_cell_renderer_ellipsized_text_new();
    if(gtk_major_version == 2 && gtk_minor_version >= 6)
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END, NULL);
	else {
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END,
				"ellipsize-set", TRUE, NULL);
	}
    gtk_tree_view_column_pack_start(col, render, TRUE);
    gtk_tree_view_column_set_attributes(col, render, "text", MMADD_COL_NAME, NULL);
    
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    g_node_traverse(xfmedia_mediamarks_get_tree(), G_IN_ORDER,
            G_TRAVERSE_ALL, -1,
            (GNodeTraverseFunc)mmtree_add_folders, ts);
    
    gtk_widget_show(treeview);
    gtk_container_add(GTK_CONTAINER(sw), treeview);
    
    return treeview;
}

GNode *
xfmedia_mediamarks_add(XfMediaMainwin *mwin, GtkWindow *parent,
        const gchar *title, const gchar *uri)
{
    GtkTreePath *path;
    GtkTreeIter itr;
    GtkWidget *dlg, *lbl, *name_entry, *uri_entry, *vbox, *hbox, *treeview,
            *entry, *btn;
    GtkSizeGroup *sg;
    GtkTreeSelection *sel;
    GNode *parent_node = NULL, *new_node = NULL;
    
    g_return_val_if_fail(mwin, NULL);
    
    dlg = gtk_dialog_new_with_buttons(_("Add Mediamark"), parent,
            GTK_DIALOG_DESTROY_WITH_PARENT|GTK_DIALOG_NO_SEPARATOR,
            GTK_BUTTONS_NONE);
    gtk_dialog_add_button(GTK_DIALOG(dlg), GTK_STOCK_CANCEL,
            GTK_RESPONSE_CANCEL);
    btn = xfmedia_custom_button_new(_("_Add Mediamark"), GTK_STOCK_ADD);
    GTK_WIDGET_SET_FLAGS(btn, GTK_CAN_DEFAULT);
    gtk_widget_show(btn);
    gtk_dialog_add_action_widget(GTK_DIALOG(dlg), btn, GTK_RESPONSE_ACCEPT);
    gtk_dialog_set_default_response(GTK_DIALOG(dlg), GTK_RESPONSE_ACCEPT);
    gtk_widget_grab_default(btn);
    gtk_window_set_default_size(GTK_WINDOW(dlg), 500, 350);
    gtk_container_set_border_width(GTK_CONTAINER(dlg), BORDER);
    
    vbox = GTK_DIALOG(dlg)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), BORDER/2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), BORDER/4);
    
    sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_Name:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sg, lbl);
    
    name_entry = gtk_entry_new();
    if(title) {
        gint len = strlen(title);
        gtk_entry_set_text(GTK_ENTRY(name_entry), title);
        gtk_entry_set_width_chars(GTK_ENTRY(name_entry), len > 35 ? 35 : len);
    }
    gtk_entry_set_activates_default(GTK_ENTRY(name_entry), TRUE);
    gtk_widget_show(name_entry);
    gtk_box_pack_start(GTK_BOX(hbox), name_entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), name_entry);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_Location:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sg, lbl);
    
    uri_entry = gtk_entry_new();
    if(uri)
        gtk_entry_set_text(GTK_ENTRY(uri_entry), uri);
    gtk_entry_set_activates_default(GTK_ENTRY(name_entry), TRUE);
    gtk_widget_show(uri_entry);
    gtk_box_pack_start(GTK_BOX(hbox), uri_entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), uri_entry);
    
    
    lbl = gtk_label_new_with_mnemonic(_("File _in:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(vbox), lbl, FALSE, FALSE, 0);
    gtk_size_group_add_widget(sg, lbl);
    
    treeview = mm_create_folder_treeview(mwin, GTK_BOX(vbox));
    gtk_tree_view_expand_all(GTK_TREE_VIEW(treeview));
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), treeview);
    
    sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    gtk_tree_selection_set_mode(sel, GTK_SELECTION_SINGLE);
    path = gtk_tree_path_new_first();
    gtk_tree_selection_select_path(sel, path);
    gtk_tree_path_free(path);
    
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), _("New Folder"));
    gtk_entry_set_activates_default(GTK_ENTRY(name_entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    btn = xfmedia_custom_button_new(_("Add _Folder"), GTK_STOCK_ADD);
    g_object_set_data(G_OBJECT(btn), "xfmedia-addmm-treeview", treeview);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(mm_add_folder_cb), entry);
    
    /* fire off the clicked signal if the user presses enter in the GtkEntry */
    gtk_widget_add_events(entry, GDK_BUTTON_PRESS_MASK);
    g_signal_connect(G_OBJECT(entry), "key-press-event",
            G_CALLBACK(mm_add_folder_entry_kp_cb), btn);
    
    if(gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_ACCEPT) {
        gchar *new_name, *new_uri;
        GtkTreeModel *ts = gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));
        
        if(!gtk_tree_selection_get_selected(sel, NULL, &itr))
            gtk_tree_model_get_iter_first(ts, &itr);
        gtk_tree_model_get(ts, &itr, MMADD_COL_NODE, &parent_node, -1);
        if(parent) {
            new_name = gtk_editable_get_chars(GTK_EDITABLE(name_entry), 0, -1);
            new_uri = gtk_editable_get_chars(GTK_EDITABLE(uri_entry), 0, -1);
            if(new_name && new_uri) {
                new_node = xfmedia_mediamarks_insert_mark(parent_node, new_name,
                        new_uri, -1);
            }
            if(new_name)
                g_free(new_name);
            if(new_uri)
                g_free(new_uri);
        }
    }
    gtk_widget_destroy(dlg);
    
    return new_node;
}

static void
xfmedia_mediamarks_add_index(XfMediaMainwin *mwin, GtkWindow *parent,
        gint index)
{
    gchar *title = NULL, *uri = NULL;
    
    if(xfmedia_playlist_get(mwin->plist, index, &title, NULL, &uri)) {
        xfmedia_mediamarks_add(mwin, parent, title, uri);
        g_free(title);
        g_free(uri);
    }
}

static void
mm_add_cur_stream_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint index;
    
    if(!mwin->cur_playing)
        return;
    
    index = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
    if(index != -1)
        xfmedia_mediamarks_add_index(mwin, GTK_WINDOW(mwin->window), index);
}

static gboolean
mmtree_add_all(GNode *node, gpointer data)
{
    GtkTreeStore *ts = data;
    XfMediaMediamark *mmark = node->data;
    static gint parent_depth = 1;
    static GtkTreeIter itr, pitr;
    
    if(!folder_pix && !mmark_pix)
        mm_load_stock_pixbufs();
	
	DBG("item has depth of %d", g_node_depth(node));
    
    if(g_node_depth(node) == 1) {
        DBG("  adding root node");
        gtk_tree_store_append(ts, &pitr, NULL);
        gtk_tree_store_set(ts, &pitr, MMEDIT_COL_ICON, folder_pix,
                MMEDIT_COL_NAME, _("Mediamarks"),
                MMEDIT_COL_NODE, node, -1);
		parent_depth = 1;
    } else {
        if(g_node_depth(node) == parent_depth+1) {
            DBG("  item is at same level");
            gtk_tree_store_append(GTK_TREE_STORE(ts), &itr, &pitr);
        } else if(g_node_depth(node) == parent_depth+2) {
            DBG("  item is child of current level");
            memcpy(&pitr, &itr, sizeof(GtkTreeIter));
            gtk_tree_store_append(GTK_TREE_STORE(ts), &itr, &pitr);
            parent_depth++;
        } else if(g_node_depth(node) < parent_depth+1) {
            GtkTreePath *path = gtk_tree_model_get_path(GTK_TREE_MODEL(ts),
                    &pitr);
            DBG("  item is ancestor of current level");
            while(g_node_depth(node) < parent_depth+1) {
                gtk_tree_path_up(path);
                parent_depth--;
            }
            gtk_tree_model_get_iter(GTK_TREE_MODEL(ts), &pitr, path);
            gtk_tree_path_free(path);
            gtk_tree_store_append(GTK_TREE_STORE(ts), &itr, &pitr);
        }
        
        if(mmark->uri) {
            gtk_tree_store_set(GTK_TREE_STORE(ts), &itr,
                    MMEDIT_COL_ICON, mmark_pix,
                    MMEDIT_COL_NAME, mmark->name,
                    MMEDIT_COL_URI, mmark->uri,
                    MMEDIT_COL_NODE, node, -1);
        } else {
            gtk_tree_store_set(GTK_TREE_STORE(ts), &itr,
                    MMEDIT_COL_ICON, folder_pix,
                    MMEDIT_COL_NAME, mmark->name,
                    MMEDIT_COL_NODE, node, -1);
        }
    }
    
    return FALSE;
}

typedef struct
{
    XfMediaMainwin *mwin;
    GtkWidget *treeview;
} MMManagerData;

static void mmedit_refresh_cb(GtkWidget *w, gpointer user_data);

static void
mmedit_addmm_cb(GtkWidget *w, gpointer user_data)
{
    MMManagerData *mdata = user_data;
    GtkWindow *toplevel = GTK_WINDOW(gtk_widget_get_toplevel(mdata->treeview));
    
    xfmedia_mediamarks_add(mdata->mwin, toplevel, NULL, NULL);
    mmedit_refresh_cb(w, user_data);
}

static void
mmedit_addfolder_cb(GtkWidget *w, gpointer user_data)
{
    MMManagerData *mdata = user_data;
    GtkWidget *dlg, *vbox, *hbox, *treeview, *lbl, *entry, *btn;
    GtkTreePath *path;
    
    dlg = gtk_dialog_new_with_buttons(_("Add Mediamark Folder"),
            GTK_WINDOW(gtk_widget_get_toplevel(mdata->treeview)),
            GTK_DIALOG_DESTROY_WITH_PARENT|GTK_DIALOG_NO_SEPARATOR,
            GTK_BUTTONS_NONE);
    gtk_dialog_add_button(GTK_DIALOG(dlg), GTK_STOCK_CANCEL,
            GTK_RESPONSE_CANCEL);
    btn = xfmedia_custom_button_new(_("_Add Folder"), GTK_STOCK_ADD);
    GTK_WIDGET_SET_FLAGS(btn, GTK_CAN_DEFAULT);
    gtk_widget_show(btn);
    gtk_dialog_add_action_widget(GTK_DIALOG(dlg), btn, GTK_RESPONSE_ACCEPT);
    gtk_dialog_set_default_response(GTK_DIALOG(dlg), GTK_RESPONSE_ACCEPT);
    gtk_widget_grab_default(btn);
    gtk_container_set_border_width(GTK_CONTAINER(dlg), BORDER);
    
    vbox = GTK_DIALOG(dlg)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), BORDER/2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), BORDER/4);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_Folder Name:"));
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);
    
    lbl = gtk_label_new_with_mnemonic(_("File _in:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(vbox), lbl, FALSE, FALSE, 0);
    
    treeview = mm_create_folder_treeview(mdata->mwin, GTK_BOX(vbox));
    path = gtk_tree_path_new_first();
    gtk_tree_view_expand_row(GTK_TREE_VIEW(treeview), path, FALSE);
    gtk_tree_path_free(path);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), treeview);
    
    if(gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_ACCEPT) {
        gchar *name;
        GtkTreeSelection *sel;
        GtkTreeModel *ts;
        GtkTreeIter itr;
        GNode *parent = NULL;
        
        if(!(name = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1))) {
            gtk_widget_destroy(dlg);
            return;
        }
        
        sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
        ts = gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));
        
        if(!gtk_tree_selection_get_selected(sel, NULL, &itr))
            gtk_tree_model_get_iter_first(ts, &itr);
        
        gtk_tree_model_get(ts, &itr, MMADD_COL_NODE, &parent, -1);
        if(parent) {
            xfmedia_mediamarks_insert_folder(parent, name, -1);
            mmedit_refresh_cb(w, mdata);
        }
        g_free(name);
    }
    gtk_widget_destroy(dlg);
}

static void
mmedit_rmitem_cb(GtkWidget *w, gpointer user_data)
{
    MMManagerData *mdata = user_data;
    GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(mdata->treeview));
    GtkTreeModel *ts = NULL;
    GtkTreeIter itr;
    
    if(gtk_tree_selection_get_selected(sel, &ts, &itr)
            && gtk_tree_store_iter_depth(GTK_TREE_STORE(ts), &itr) != 0)
    {
        GtkWidget *toplevel = gtk_widget_get_toplevel(mdata->treeview);
        GNode *item_node = NULL;
        XfMediaMediamark *mmark;
        gint resp;
        
        gtk_tree_model_get(ts, &itr, MMEDIT_COL_NODE, &item_node, -1);
        mmark = item_node->data;
        
        if(mmark->uri) {
            resp = xfce_message_dialog(GTK_WINDOW(toplevel),
                    _("Remove Mediamark"), GTK_STOCK_DIALOG_QUESTION,
                    _("Are you sure?"), _("You're about to remove a Mediamark, which can't be undone."),
                    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                    GTK_STOCK_REMOVE, GTK_RESPONSE_ACCEPT, NULL);
        } else {
            resp = xfce_message_dialog(GTK_WINDOW(toplevel), _("Remove Folder"),
                    GTK_STOCK_DIALOG_QUESTION, _("Are you sure?"),
                    _("Removing a folder also removes all items and subfolders contained in it, and can't be undone."),
                    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                    GTK_STOCK_REMOVE, GTK_RESPONSE_ACCEPT, NULL);
        }
        
        if(resp == GTK_RESPONSE_ACCEPT) {
            if(mmark->uri)
                xfmedia_mediamarks_remove_mark(item_node);
            else
                xfmedia_mediamarks_remove_folder(item_node);
            
            mmedit_refresh_cb(w, mdata);
        }
    }
}

static void
mmedit_mvup_cb(GtkWidget *w, gpointer user_data)
{
    MMManagerData *mdata = user_data;
    GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(mdata->treeview));
    GtkTreeModel *ts = NULL;
    GtkTreeIter itr, target_itr;
    
    if(gtk_tree_selection_get_selected(sel, &ts, &itr)
            && gtk_tree_store_iter_depth(GTK_TREE_STORE(ts), &itr) != 0)
    {
        GtkTreePath *path = gtk_tree_model_get_path(ts, &itr);
        if(gtk_tree_path_prev(path)
                && gtk_tree_model_get_iter(ts, &target_itr, path))
        {
            GNode *node = NULL, *target_node = NULL;
            
            gtk_tree_model_get(ts, &itr, MMEDIT_COL_NODE, &node, -1);
            gtk_tree_model_get(ts, &target_itr, MMEDIT_COL_NODE, &target_node, -1);
            
            g_node_unlink(node);
            g_node_insert_before(target_node->parent, target_node, node);
            
            gtk_tree_store_move_before(GTK_TREE_STORE(ts), &itr, &target_itr);
            
            mm_changed_notify();
            gtk_tree_selection_select_path(sel, path);
            mm_write_all();
        }
        gtk_tree_path_free(path);
    }
}

static void
mmedit_mvdown_cb(GtkWidget *w, gpointer user_data)
{
    MMManagerData *mdata = user_data;
    GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(mdata->treeview));
    GtkTreeModel *ts = NULL;
    GtkTreeIter itr, target_itr;
    
    if(gtk_tree_selection_get_selected(sel, &ts, &itr)
            && gtk_tree_store_iter_depth(GTK_TREE_STORE(ts), &itr) != 0)
    {
        GtkTreePath *path = gtk_tree_model_get_path(ts, &itr);
        gtk_tree_path_next(path);
        if(gtk_tree_model_get_iter(ts, &target_itr, path)) {
            GNode *node = NULL, *target_node = NULL;
            
            gtk_tree_model_get(ts, &itr, MMEDIT_COL_NODE, &node, -1);
            gtk_tree_model_get(ts, &target_itr, MMEDIT_COL_NODE, &target_node, -1);
            
            g_node_unlink(node);
            g_node_insert_after(target_node->parent, target_node, node);
            
            gtk_tree_store_move_after(GTK_TREE_STORE(ts), &itr, &target_itr);
            
            mm_changed_notify();
            gtk_tree_selection_select_path(sel, path);
            mm_write_all();
        }
        gtk_tree_path_free(path);
    }
}

static void
mmedit_refresh_cb(GtkWidget *w, gpointer user_data)
{
    MMManagerData *mdata = user_data;
    GtkTreeStore *ts = GTK_TREE_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(mdata->treeview)));
    GtkTreePath *path;
    
    gtk_tree_store_clear(ts);
    
    g_node_traverse(xfmedia_mediamarks_get_tree(), G_PRE_ORDER,
            G_TRAVERSE_ALL, -1,
            (GNodeTraverseFunc)mmtree_add_all, ts);
    
    path = gtk_tree_path_new_first();
    gtk_tree_view_expand_row(GTK_TREE_VIEW(mdata->treeview), path, FALSE);
    gtk_tree_path_free(path);
}

static void
mmedit_props_cb(GtkWidget *w, gpointer user_data)
{
    MMManagerData *mdata = user_data;
    GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(mdata->treeview));
    GtkTreeModel *ts = NULL;
    GtkTreeIter itr;
    
    if(gtk_tree_selection_get_selected(sel, &ts, &itr)
            && gtk_tree_store_iter_depth(GTK_TREE_STORE(ts), &itr) != 0)
    {
        GtkWidget *dlg, *vbox, *hbox, *lbl, *name_entry, *uri_entry = NULL;
        GtkWindow *parent = GTK_WINDOW(gtk_widget_get_toplevel(mdata->treeview));
        GtkSizeGroup *sg;
        GNode *node = NULL;
        XfMediaMediamark *mmark;
        
        gtk_tree_model_get(ts, &itr, MMEDIT_COL_NODE, &node, -1);
        mmark = node->data;
        
        dlg = gtk_dialog_new_with_buttons(
                mmark->uri ? _("Mediamark Properties") : _("Folder Properties"),
                parent,
                GTK_DIALOG_DESTROY_WITH_PARENT|GTK_DIALOG_NO_SEPARATOR,
                GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL);
        gtk_container_set_border_width(GTK_CONTAINER(dlg), BORDER);
        
        vbox = GTK_DIALOG(dlg)->vbox;
        gtk_box_set_spacing(GTK_BOX(vbox), BORDER/2);
        gtk_container_set_border_width(GTK_CONTAINER(vbox), BORDER/4);
        
        sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
        
        hbox = gtk_hbox_new(FALSE, BORDER);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        
        lbl = gtk_label_new_with_mnemonic(_("_Name:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        gtk_size_group_add_widget(sg, lbl);
        
        name_entry = gtk_entry_new();
        if(mmark->name) {
            gint len = strlen(mmark->name);
            gtk_entry_set_text(GTK_ENTRY(name_entry), mmark->name);
            gtk_entry_set_width_chars(GTK_ENTRY(name_entry), len > 35 ? 35 : len);
        }
        gtk_entry_set_activates_default(GTK_ENTRY(name_entry), TRUE);
        gtk_widget_show(name_entry);
        gtk_box_pack_start(GTK_BOX(hbox), name_entry, TRUE, TRUE, 0);
        gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), name_entry);
        
        if(mmark->uri) {
            hbox = gtk_hbox_new(FALSE, BORDER);
            gtk_widget_show(hbox);
            gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
            
            lbl = gtk_label_new_with_mnemonic(_("_Location:"));
            gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
            gtk_widget_show(lbl);
            gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
            gtk_size_group_add_widget(sg, lbl);
            
            uri_entry = gtk_entry_new();
            if(mmark->uri)
                gtk_entry_set_text(GTK_ENTRY(uri_entry), mmark->uri);
            gtk_entry_set_activates_default(GTK_ENTRY(name_entry), TRUE);
            gtk_widget_show(uri_entry);
            gtk_box_pack_start(GTK_BOX(hbox), uri_entry, TRUE, TRUE, 0);
            gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), uri_entry);
        }
        
        if(gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_ACCEPT) {
            gchar *new_name = NULL, *new_uri = NULL;
            
            new_name = gtk_editable_get_chars(GTK_EDITABLE(name_entry), 0, -1);
            if(mmark->uri)
                new_uri = gtk_editable_get_chars(GTK_EDITABLE(uri_entry), 0, -1);
            
            if(new_name && (!mmark->uri || new_uri)) {
                g_free(mmark->name);
                g_free(mmark->uri);
                
                mmark->name = new_name;
                mmark->uri = new_uri;
                
                mm_changed_notify();
                mm_write_all();
                mmedit_refresh_cb(w, mdata);
            } else if(new_name)
                g_free(new_name);
            else if(new_uri)
                g_free(new_uri);
        }
        gtk_widget_destroy(dlg);
    }
    
}

static gboolean
mmedit_ditch_rtclick_menu_idled(gpointer user_data)
{
    gtk_widget_destroy(GTK_WIDGET(user_data));
    return FALSE;
}

static gboolean
mmedit_btnpress_cb(GtkWidget *w, GdkEventButton *evt, gpointer user_data)
{
    MMManagerData *mdata = user_data;
    
    if(evt->button == 1 && evt->type == GDK_2BUTTON_PRESS) {
        GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(mdata->treeview));
        GtkTreeModel *ts = NULL;
        GtkTreeIter itr;
        
        if(gtk_tree_selection_get_selected(sel, &ts, &itr)) {
            GNode *node = NULL;
            XfMediaMediamark *mmark;
            
            gtk_tree_model_get(ts, &itr, MMEDIT_COL_NODE, &node, -1);
            mmark = node->data;
            if(!mmark->uri) {
                GtkTreePath *path = gtk_tree_model_get_path(ts, &itr);
                if(!gtk_tree_view_row_expanded(GTK_TREE_VIEW(mdata->treeview), path))
                    gtk_tree_view_expand_row(GTK_TREE_VIEW(mdata->treeview), path, FALSE);
                else
                    gtk_tree_view_collapse_row(GTK_TREE_VIEW(mdata->treeview), path);
                gtk_tree_path_free(path);
            } else
                xfmedia_mainwin_play_uri(mdata->mwin, mmark->uri);
        }
    } else if(evt->button == 3) {
        GtkWidget *menu, *mi;
        GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(mdata->treeview));
        GtkTreeIter itr;
        GtkTreeModel *ts = NULL;
        GNode *node = NULL;
        GtkTreePath *path = NULL;
        
        if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(mdata->treeview),
                evt->x, evt->y, &path, NULL, NULL, NULL))
        {
            gtk_tree_selection_select_path(sel, path);
            gtk_tree_path_free(path);
        } else
            return FALSE;
        
        if(gtk_tree_selection_get_selected(sel, &ts, &itr)) {
            gtk_tree_model_get(ts, &itr, MMEDIT_COL_NODE, &node, -1);
            if(!node || node == xfmedia_mediamarks_get_tree())
                return FALSE;
        } else
            return FALSE;
        
        menu = gtk_menu_new();
        gtk_widget_show(menu);
        g_signal_connect_swapped(G_OBJECT(menu), "deactivate",
                G_CALLBACK(g_idle_add),
                (gpointer)mmedit_ditch_rtclick_menu_idled);
        
        mi = gtk_image_menu_item_new_from_stock(GTK_STOCK_PROPERTIES, NULL);
        gtk_widget_show(mi);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
        g_signal_connect(G_OBJECT(mi), "activate",
                G_CALLBACK(mmedit_props_cb), mdata);
        
        mi = gtk_separator_menu_item_new();
        gtk_widget_show(mi);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
        
        mi = gtk_image_menu_item_new_from_stock(GTK_STOCK_REMOVE, NULL);
        gtk_widget_show(mi);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
        g_signal_connect(G_OBJECT(mi), "activate",
                G_CALLBACK(mmedit_rmitem_cb), mdata);
        
        mi = gtk_image_menu_item_new_from_stock(GTK_STOCK_GO_UP, NULL);
        gtk_widget_show(mi);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
        g_signal_connect(G_OBJECT(mi), "activate",
                G_CALLBACK(mmedit_mvup_cb), mdata);
        
        mi = gtk_image_menu_item_new_from_stock(GTK_STOCK_GO_DOWN, NULL);
        gtk_widget_show(mi);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
        g_signal_connect(G_OBJECT(mi), "activate",
                G_CALLBACK(mmedit_mvdown_cb), mdata);
        
        gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, evt->button, evt->time);
        
        /* eat event so DnD doesn't kick in */
        return TRUE;
    }
    
    return FALSE;
}

void
mmedit_treeview_drag_get_cb(GtkWidget *widget, GdkDragContext *drag_context,
        GtkSelectionData *data, guint info, guint time, gpointer user_data)
{
    MMManagerData *mdata = user_data;
    GdkAtom _MEDIAMARK_ITEM = gdk_atom_intern("MEDIAMARK_ITEM", FALSE);
    
    if(data->target == _MEDIAMARK_ITEM) {
        GtkTreeRowReference *srcrref = g_object_get_data(G_OBJECT(drag_context),
                "gtk-tree-view-source-row");
        GtkTreePath *sourcerow = gtk_tree_row_reference_get_path(srcrref);
        GtkTreeIter *itr;
        GtkTreeModel *ts = gtk_tree_view_get_model(GTK_TREE_VIEW(mdata->treeview));
        
        if(!sourcerow)
            return;
        
        itr  = g_new0(GtkTreeIter, 1);
        gtk_tree_model_get_iter(ts, itr, sourcerow);
        
        gtk_selection_data_set(data, _MEDIAMARK_ITEM, 8,
                (gpointer)&itr, sizeof(itr));
        
        gtk_tree_path_free(sourcerow);
    }
}

void
mmedit_treeview_drag_rcv_cb(GtkWidget *widget, GdkDragContext *drag_context,
        gint x, gint y,  GtkSelectionData *data, guint info, guint time,
        gpointer user_data)
{
    MMManagerData *mdata = user_data;
    GdkAtom _MEDIAMARK_ITEM = gdk_atom_intern("MEDIAMARK_ITEM", FALSE);
#if 0
    GdkAtom _URI_LIST = gdk_atom_intern("text/uri-list", FALSE);
    GdkAtom _STRING = gdk_atom_intern("STRING", FALSE);
#endif
    GtkTreePath *dest_path = NULL;
    GtkTreeViewDropPosition drop_pos;
    GtkTreeIter *src_itr, dest_itr;
    gboolean drag_success = FALSE;
    GtkTreeModel *ts = gtk_tree_view_get_model(GTK_TREE_VIEW(mdata->treeview));
    
    DBG("got DnD target: %s", gdk_atom_name(data->target));
    
    if(data->target == _MEDIAMARK_ITEM) {
        src_itr = *(GtkTreeIter **)data->data;
        if(!src_itr) {
            gtk_drag_finish(drag_context, FALSE, FALSE, time);
            return;
        }
        
        if(gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(mdata->treeview),
                    x, y, &dest_path, &drop_pos))
        {
            if(gtk_tree_model_get_iter(ts, &dest_itr, dest_path)
                    && gtk_tree_store_iter_is_valid(GTK_TREE_STORE(ts), src_itr))
            {
                GNode *src_node = NULL, *dest_node = NULL;
                GtkTreeSelection *sel;
				gboolean dest_is_folder;
				XfMediaMediamark *mmark;
				
                gtk_tree_model_get(ts, &dest_itr, MMEDIT_COL_NODE, &dest_node, -1);
				
				if(dest_node && src_node && dest_node->parent && src_node->parent) {
					mmark = src_node->data;
					dest_is_folder = !((XfMediaMediamark *)dest_node->data)->uri;
					
					g_node_unlink(src_node);
					
					if(dest_is_folder && (drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE
							|| drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_AFTER))
					{
						DBG("drop into folder");
						g_node_insert_before(dest_node, dest_node->children, src_node);
					} else if(drop_pos == GTK_TREE_VIEW_DROP_BEFORE) {
						DBG("drop before item");
						g_node_insert_before(dest_node->parent, dest_node, src_node);
					} else if(drop_pos == GTK_TREE_VIEW_DROP_AFTER) {
						DBG("drop after item");
						g_node_insert_after(dest_node->parent, dest_node, src_node);
					}
					
					drag_success = TRUE;
					
					mm_changed_notify();
					mmedit_refresh_cb(widget, mdata);
					gtk_tree_view_expand_to_path(GTK_TREE_VIEW(mdata->treeview), dest_path);
					sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(mdata->treeview));
					gtk_tree_selection_select_path(sel, dest_path);
					mm_write_all();
				}
            } else {
                DBG("one of the itrs is invalid");
            }
            
            gtk_tree_path_free(dest_path);
        } else {
            DBG("gtk_tree_view_get_dest_row_at_pos() failed");
        }
        
        g_free(src_itr);
    }
#if 0  /* perhaps i'll eventually implement dropping files from a FM */
    else if(data->target == _URI_LIST) {
        GList *uris, *l;
        gint from_index = -1, index, num_added = 0;
        
        DBG("we have an uri list: '%s'", (gchar *)data->data);
        
        if(gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(mwin->plist->treeview),
                    x, y, &dest_path, &drop_pos))
        {
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(mwin->plist->file_list),
                    &dest_itr, dest_path))
            {
                gtk_tree_model_get(GTK_TREE_MODEL(mwin->plist->file_list),
                        &dest_itr, PLAYLIST_COL_NUM, &from_index, -1);
                if(drop_pos == GTK_TREE_VIEW_DROP_BEFORE
                        || drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)
                {
                    from_index--;
                }
            }
            gtk_tree_path_free(dest_path);
        }
        
        index = from_index;
        uris = xfmedia_uri_list_extract_uris((const gchar *)data->data);
        for(l = uris; l; l = l->next) {
            gchar *filename = l->data;
            
            if(!g_ascii_strncasecmp(filename, "file://", 7))
                filename += 7;
            DBG("adding '%s'", filename);
            xfmedia_mainwin_add_file(mwin, filename, index, TRUE);
            g_free(l->data);
            if(index != -1)
                index++;
            num_added++;
        }
        if(uris) {
            if(from_index == -1) {
                from_index = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(mwin->plist->file_list), NULL);
                from_index -= num_added - 1;
            }
            xfmedia_playlist_renumber_file_list(mwin->plist->file_list,
                    from_index, -1);
            g_list_free(uris);
            mwin->plist->save_on_exit = TRUE;
            
            xfmedia_playlist_shuffle_add_entries(mwin->plist, from_index, num_added);
        }
        
        drag_success = TRUE;
    } else if(data->target == _STRING) {
        gchar *filename = data->data;
        gint from_index = -1;
        
        if(gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(mwin->plist->treeview),
                    x, y, &dest_path, &drop_pos))
        {
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(mwin->plist->file_list),
                    &dest_itr, dest_path))
            {
                gtk_tree_model_get(GTK_TREE_MODEL(mwin->plist->file_list),
                        &dest_itr, PLAYLIST_COL_NUM, &from_index, -1);
                if(drop_pos == GTK_TREE_VIEW_DROP_BEFORE
                        || drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)
                {
                    from_index--;
                }
            }
            gtk_tree_path_free(dest_path);
        }
        
        if(filename) {
            if(!g_ascii_strncasecmp(filename, "file://", 7))
                filename += 7;
            xfmedia_mainwin_add_file(mwin, filename, from_index, TRUE);
            xfmedia_playlist_renumber_file_list(mwin->plist->file_list,
                    from_index, -1);
            xfmedia_playlist_shuffle_add_entries(mwin->plist, from_index, 1);
            mwin->plist->save_on_exit = TRUE;
            drag_success = TRUE;
        }
    }
#endif
    
    gtk_drag_finish(drag_context, drag_success, FALSE, time);
}

static gboolean
mm_manager_configure_cb(GtkWidget *w, GdkEventConfigure *evt,
        gpointer user_data)
{
    xfmedia_settings_set_int("/xfmedia/mediamarks/manager_x", evt->x);
    xfmedia_settings_set_int("/xfmedia/mediamarks/manager_y", evt->y);
    xfmedia_settings_set_int("/xfmedia/mediamarks/manager_width", evt->width);
    xfmedia_settings_set_int("/xfmedia/mediamarks/manager_height", evt->height);
    
    return FALSE;
}

static void
mm_manager_zero_pointer(GtkWidget **w)
{
    if(w)
        *w = NULL;
}

static void
manage_mediamarks_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    static GtkWidget *win = NULL;
    GtkWidget *topvbox, *vbox, *hbox, *toolbar, *img, *sw, *treeview, *btn;
    GtkTreeStore *ts;
    GtkCellRenderer *render;
    GtkTreeViewColumn *col;
    GtkTreePath *path;
    MMManagerData *mdata;
    const GtkTargetEntry tv_targets[] = {
        { "MEDIAMARK_ITEM", GTK_TARGET_SAME_WIDGET, DROP_TGT_REORDER },
        { "text/uri-list", 0, DROP_TGT_URILIST },
        { "STRING", 0, DROP_TGT_STRING },
    };
    
    if(win) {
        gtk_window_present(GTK_WINDOW(win));
        return;
    }
    
    mdata = g_new0(MMManagerData, 1);
    mdata->mwin = mwin;
    
    win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(win), _("Manage Mediamarks"));
    gtk_container_set_border_width(GTK_CONTAINER(win), 0);
    gtk_widget_add_events(win, GDK_CONFIGURE);
    g_signal_connect(G_OBJECT(win), "configure-event",
            G_CALLBACK(mm_manager_configure_cb), NULL);
    g_signal_connect_swapped(G_OBJECT(win), "destroy",
            G_CALLBACK(mm_manager_zero_pointer), &win);
    g_signal_connect_swapped(G_OBJECT(win), "destroy",
            G_CALLBACK(g_free), mdata);
    
    topvbox = gtk_vbox_new(FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(topvbox), 0);
    gtk_widget_show(topvbox);
    gtk_container_add(GTK_CONTAINER(win), topvbox);
    
    toolbar = gtk_toolbar_new();
    gtk_toolbar_set_icon_size(GTK_TOOLBAR(toolbar), GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(toolbar);
    gtk_box_pack_start(GTK_BOX(topvbox), toolbar, FALSE, FALSE, 0);
    
    img = gtk_image_new_from_stock(GTK_STOCK_ADD, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Add Mediamark..."),
            _("Add a new mediamark"), NULL, img,
            G_CALLBACK(mmedit_addmm_cb), mdata);
    
    img = gtk_image_new_from_stock(GTK_STOCK_DIRECTORY, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Add Folder..."),
            _("Add a new folder"), NULL, img,
            G_CALLBACK(mmedit_addfolder_cb), mdata);
    
    img = gtk_image_new_from_stock(GTK_STOCK_REMOVE, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Remove Item"),
            _("Remove the selected item"), NULL, img,
            G_CALLBACK(mmedit_rmitem_cb), mdata);
    
    gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_UP, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Move Item Up"),
            _("Move the selected item up"), NULL, img,
            G_CALLBACK(mmedit_mvup_cb), mdata);
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Move Item Down"),
            _("Move the selected item down"), NULL, img,
            G_CALLBACK(mmedit_mvdown_cb), mdata);
    
    gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
    
    img = gtk_image_new_from_stock(GTK_STOCK_REFRESH, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Refresh"),
            _("Refresh the list"), NULL, img,
            G_CALLBACK(mmedit_refresh_cb), mdata);
    
    img = gtk_image_new_from_stock(GTK_STOCK_PROPERTIES, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), _("Properties..."),
            _("Edit the selected item's properties"), NULL, img,
            G_CALLBACK(mmedit_props_cb), mdata);
            
    vbox = gtk_vbox_new(FALSE, BORDER);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), BORDER);
    gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(topvbox), vbox, TRUE, TRUE, 0);
    
    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER,
            GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
            GTK_SHADOW_ETCHED_IN);
    gtk_widget_show(sw);
    gtk_box_pack_start(GTK_BOX(vbox), sw, TRUE, TRUE, 0);
    
    ts = gtk_tree_store_new(MMEDIT_N_COLS, GDK_TYPE_PIXBUF, G_TYPE_STRING,
            G_TYPE_STRING, G_TYPE_POINTER);
    mdata->treeview = treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(ts));
    gtk_tree_view_set_reorderable(GTK_TREE_VIEW(treeview), TRUE);
    gtk_tree_view_enable_model_drag_source(GTK_TREE_VIEW(treeview),
            GDK_BUTTON1_MASK, tv_targets, 3, GDK_ACTION_MOVE);
    gtk_tree_view_enable_model_drag_dest(GTK_TREE_VIEW(treeview), tv_targets,
            3, GDK_ACTION_COPY);
    g_signal_connect(G_OBJECT(treeview), "drag-data-received",
            G_CALLBACK(mmedit_treeview_drag_rcv_cb), mdata);
    g_signal_connect(G_OBJECT(treeview), "drag-data-get",
            G_CALLBACK(mmedit_treeview_drag_get_cb), mdata);
    
    col = gtk_tree_view_column_new();
    gtk_tree_view_column_set_title(col, _("Name"));
    gtk_tree_view_column_set_expand(col, TRUE);
    gtk_tree_view_column_set_resizable(col, TRUE);
    
    render = gtk_cell_renderer_pixbuf_new();
    gtk_tree_view_column_pack_start(col, render, FALSE);
    gtk_tree_view_column_set_attributes(col, render, "pixbuf", MMEDIT_COL_ICON, NULL);
    
    render = exo_cell_renderer_ellipsized_text_new();
    if(gtk_major_version == 2 && gtk_minor_version >= 6)
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END, NULL);
	else {
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END,
				"ellipsize-set", TRUE, NULL);
	}
    gtk_tree_view_column_pack_start(col, render, TRUE);
    gtk_tree_view_column_set_attributes(col, render, "text", MMEDIT_COL_NAME, NULL);
    
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    render = exo_cell_renderer_ellipsized_text_new();
    if(gtk_major_version == 2 && gtk_minor_version >= 6)
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END, NULL);
	else {
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END,
				"ellipsize-set", TRUE, NULL);
	}
    col = gtk_tree_view_column_new_with_attributes(_("Location"), render,
            "text", MMEDIT_COL_URI, NULL);
    gtk_tree_view_column_set_resizable(col, TRUE);
    gtk_tree_view_column_set_expand(col, TRUE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    g_node_traverse(xfmedia_mediamarks_get_tree(), G_PRE_ORDER,
            G_TRAVERSE_ALL, -1,
            (GNodeTraverseFunc)mmtree_add_all, ts);
    
    gtk_widget_show(treeview);
    gtk_container_add(GTK_CONTAINER(sw), treeview);
    gtk_widget_add_events(treeview, GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(treeview), "button-press-event",
            G_CALLBACK(mmedit_btnpress_cb), mdata);
    
    path = gtk_tree_path_new_first();
    gtk_tree_view_expand_row(GTK_TREE_VIEW(treeview), path, FALSE);
    gtk_tree_path_free(path);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_end(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    btn = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect_swapped(G_OBJECT(btn), "clicked",
            G_CALLBACK(gtk_widget_destroy), win);
    
    gtk_widget_realize(win);
    gtk_window_resize(GTK_WINDOW(win),
            xfmedia_settings_get_int("/xfmedia/mediamarks/manager_width"),
            xfmedia_settings_get_int("/xfmedia/mediamarks/manager_height"));
    gtk_window_move(GTK_WINDOW(win),
            xfmedia_settings_get_int("/xfmedia/mediamarks/manager_x"),
            xfmedia_settings_get_int("/xfmedia/mediamarks/manager_y"));
    
    gtk_widget_show(win);
}

static void
mm_item_activate_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    XfMediaMediamark *mmark;
    
    mmark = g_object_get_data(G_OBJECT(w), "xfmedia-mmark");
    if(!mmark || !mmark->uri)
        return;
    
    xfmedia_mainwin_play_uri(mwin, mmark->uri);
}

static void
mm_generate_menu(XfMediaMainwin *mwin, GtkMenuShell *menu, GNode *root)
{
    GNode *child;
    XfMediaMediamark *mmark;
    GtkWidget *mi, *img, *menu2;
    
    mmark = root->data;
    mi = gtk_image_menu_item_new_with_label(mmark->name);
    img = gtk_image_new_from_stock(mmark->uri?GTK_STOCK_FILE:GTK_STOCK_DIRECTORY,
            GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_widget_show(mi);
    gtk_menu_shell_append(menu, mi);
    
    if(mmark->uri) {
        g_object_set_data(G_OBJECT(mi), "xfmedia-mmark", mmark);
        g_signal_connect(G_OBJECT(mi), "activate",
                G_CALLBACK(mm_item_activate_cb), mwin);
    } else {
        menu2 = gtk_menu_new();
        gtk_widget_show(menu2);
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), menu2);
        
        for(child = root->children; child; child = child->next)
            mm_generate_menu(mwin, GTK_MENU_SHELL(menu2), child);
    }
}

GtkWidget *
xfmedia_mediamarks_create_menu(XfMediaMainwin *mwin)
{
    GtkWidget *menu, *mi, *img;
    GNode *root, *child;
    
    menu = gtk_menu_new();
    gtk_widget_show(menu);
    
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Add Current Stream..."));
    img = gtk_image_new_from_stock(GTK_STOCK_ADD, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(mm_add_cur_stream_cb), mwin);
    
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Manage Mediamarks..."));
    img = gtk_image_new_from_stock(GTK_STOCK_EDIT, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(manage_mediamarks_cb), mwin);
    
    mi = gtk_separator_menu_item_new();
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    
    root = xfmedia_mediamarks_get_tree();
    for(child = root->children; child; child = child->next)
        mm_generate_menu(mwin, GTK_MENU_SHELL(menu), child);
    
    return menu;
}

void
xfmedia_mediamarks_hook_changed(MMChangedCallback callback, gpointer data)
{
    mm_changed_cb = g_list_append(mm_changed_cb, callback);
    mm_changed_data = g_list_append(mm_changed_data, data);
}
