/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "remote.h"

#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#ifdef HAVE_SYS_UN_H
#include <sys/un.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#if defined(HAVE_STDARG_H)
#include <stdarg.h>
#elif defined(HAVE_VARARGS_H)
#include <varargs.h>
#define va_start(v, l) va_start(v)
#else
#error Your system does not support variable-argument functions.
#endif

#ifndef PATH_MAX
#define PATH_MAX 4096
#endif

#ifndef UNIX_PATH_MAX
#define UNIX_PATH_MAX 108
#endif

#ifdef HAVE_DBUS

#ifndef DBUS_API_SUBJECT_TO_CHANGE  /* exo 0.2 sets this, but 0.3 doesn't */
#define DBUS_API_SUBJECT_TO_CHANGE
#endif
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef DBUS_USE_OLD_API
#define dbus_bus_request_name dbus_bus_acquire_service
#define dbus_bus_start_service_by_name dbus_bus_activate_service
#define dbus_bus_get_unique_name dbus_bus_get_base_service
#define dbus_bus_name_has_owner dbus_bus_service_exists
#define DBUS_NAME_FLAG_PROHIBIT_REPLACEMENT DBUS_SERVICE_FLAG_PROHIBIT_REPLACEMENT
#define DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER DBUS_SERVICE_REPLY_PRIMARY_OWNER
#endif

#endif  /* HAVE_DBUS */


#include <libxfce4util/libxfce4util.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include "remote.h"
#include <xfmedia/xfmedia-playlist.h>
#include "mainwin.h"
#include "main.h"
#include "xfmedia-internal.h"
#include "playlist-files.h"
#include "xfmedia-common.h"

#define MAX_INSTANCES             1024
#define XFMEDIA_DBUS_VERSION      "1"
#define XFMEDIA_DBUS_BASENAME     "org.spuriousinterrupt.Xfmedia"
#define XFMEDIA_DBUS_INTERFACE    XFMEDIA_DBUS_BASENAME XFMEDIA_DBUS_VERSION
#define XFMEDIA_DBUS_SERVICE_FMT  XFMEDIA_DBUS_BASENAME "%d"
#define XFMEDIA_DBUS_ERROR        XFMEDIA_DBUS_BASENAME XFMEDIA_DBUS_VERSION ".Error"
#define XFMEDIA_DBUS_PATH_FMT     "/org/spuriousinterrupt/Xfmedia/%d"
/*#define XFMEDIA_DBUS_PATH_FMT     XFMEDIA_DBUS_PARENT_PATH "/%d"*/

#ifdef HAVE_DBUS

static void xfmedia_dbus_unregister(DBusConnection *connection, void *user_data);
static DBusHandlerResult xfmedia_dbus_message(DBusConnection *connection, DBusMessage *message, void *user_data);

static const struct DBusObjectPathVTable xfmedia_dbus_vtable =
{
    xfmedia_dbus_unregister,
    xfmedia_dbus_message,
    NULL,
};

static DBusGConnection *__dbus_connection = NULL;

static DBusConnection *
xfmedia_dbus_get_bus_connection()
{
    GError *err = NULL;
    
    if(!__dbus_connection) {
        __dbus_connection = dbus_g_bus_get(DBUS_BUS_SESSION, &err);
        if(!__dbus_connection) {
            g_warning("Failed to open a connection to the D-BUS session bus.  "
                    "Please ensure that the session bus is running. (%d: %s)",
                    err->code, err->message);
            g_error_free(err);
        }
    }
    
    return dbus_g_connection_get_connection(__dbus_connection);
}

static void
xfmedia_dbus_unregister(DBusConnection *connection, void *user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    if(mwin->session_id == -1)
        return;
    
    g_warning("Recieved disconnection from D-BUS.  Perhaps the session bus daemon has quit?");
}

static DBusHandlerResult
xfmedia_dbus_message(DBusConnection *connection, DBusMessage *message,
        void *user_data)
{
    XfMediaMainwin *mwin = user_data;
    DBusHandlerResult ret = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    DBusMessage *msg_ret = NULL;
    DBusError derr;
    
    DBG("You've got DBusMessage!");
    
    if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_IS_RUNNING))
    {
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_CLEAR_PLAYLIST))
    {
        xfmedia_playlist_clear(mwin->plist);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_LOAD_PLAYLIST))
    {
        gchar *filename = NULL;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                    DBUS_TYPE_STRING, &filename,
                    DBUS_TYPE_INVALID)
                && filename && *filename)
        {
            if(xfmedia_playlists_load(mwin->plist, filename))
                msg_ret = dbus_message_new_method_return(message);
            g_free(filename);
        } else
            dbus_error_free(&derr);
        
        if(!msg_ret)
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_LOAD_PLAYLIST);
        
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_SAVE_PLAYLIST))
    {
        gchar *filename = NULL;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                    DBUS_TYPE_STRING, &filename,
                    DBUS_TYPE_INVALID)
                && filename && *filename)
        {
            if(xfmedia_playlists_save(mwin->plist, filename))
                msg_ret = dbus_message_new_method_return(message);
            g_free(filename);
        } else
            dbus_error_free(&derr);
        
        if(!msg_ret)
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_LOAD_PLAYLIST);
        
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_ADD_FILE))
    {
        gchar *filename = NULL;
        gint index = -1;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                    DBUS_TYPE_STRING, &filename,
                    DBUS_TYPE_INT32, &index,
                    DBUS_TYPE_INVALID)
                && filename && *filename)
        {
            gchar *name_utf8 = xfmedia_filename_to_name(filename);
            if(index < 0)
                index = xfmedia_playlist_append_entry(mwin->plist, name_utf8, -1, filename, FALSE);
            else
                index = xfmedia_playlist_insert_entry(mwin->plist, index, name_utf8, -1, filename, FALSE);
            g_free(name_utf8);
            
            msg_ret = dbus_message_new_method_return(message);
            dbus_message_append_args(msg_ret,
#ifdef DBUS_USE_OLD_API
                    DBUS_TYPE_INT32, index,
#else
                    DBUS_TYPE_INT32, &index,
#endif
                    DBUS_TYPE_INVALID);
            dbus_connection_send(connection, msg_ret, NULL);
            dbus_message_unref(msg_ret);
        } else {
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_ADD_FILE);
            dbus_connection_send(connection, msg_ret, NULL);
            dbus_message_unref(msg_ret);
        }
        
        if(filename)
            dbus_free(filename);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_REMOVE_FILE))
    {
        gint index = 0;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                DBUS_TYPE_UINT32, &index,
                DBUS_TYPE_INVALID))
        {
            if(xfmedia_playlist_remove_entry(mwin->plist, index))
                msg_ret = dbus_message_new_method_return(message);
        } else
            dbus_error_free(&derr);
        
        if(!msg_ret)
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_REMOVE_FILE);
        
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_PLAY))
    {
        gint index = -1;
        gboolean done = FALSE;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                DBUS_TYPE_INT32, &index,
                DBUS_TYPE_INVALID))
        {
            if(index == -1) {
                if(gtk_xine_get_status(mwin->gtx) == XINE_STATUS_PLAY
                            && gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE)
                {
                    gtk_xine_set_param(mwin->gtx, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
                    done = TRUE;
                } else
                    index = xfmedia_playlist_get_selected(mwin->plist);
            }
            
            if(done || xfmedia_mainwin_play_file_at_index(mwin, index))
                msg_ret = dbus_message_new_method_return(message);
            else
                msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR, XFMEDIA_REMOTE_PLAY);
            
            dbus_connection_send(connection, msg_ret, NULL);
            dbus_message_unref(msg_ret);
        } else {
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR, XFMEDIA_REMOTE_PLAY);
            dbus_connection_send(connection, msg_ret, NULL);
            dbus_message_unref(msg_ret);
        }
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_PAUSE))
    {
        xfmedia_mainwin_toggle_playpause(mwin);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_STOP))
    {
        xfmedia_mainwin_stop(mwin);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_PREV))
    {
        xfmedia_mainwin_prev(mwin);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_NEXT))
    {
        xfmedia_mainwin_next(mwin);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_QUIT))
    {
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_EXT);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    }
    
    return ret;
}

gint
xfmedia_remote_init(XfMediaMainwin *mwin)
{
    DBusConnection *dbus_conn;
    GError *err = NULL;
    DBusError derr;
    gint i, ret;
    gchar name[64];
    
    g_return_val_if_fail(mwin, -1);
    g_return_val_if_fail(mwin->session_id == -1, mwin->session_id);
    
    dbus_g_thread_init();
    
    dbus_conn = xfmedia_dbus_get_bus_connection();
    if(G_UNLIKELY(!dbus_conn)) {
        if(err) {
            g_warning("Failed to connect to D-BUS session instance (%d): %s." \
                    "  Remote control interface will not be available.",
                    err->code, err->message);
            g_error_free(err);
        }
        return -1;
    }
    
    for(i = 0; i < MAX_INSTANCES; i++) {
        g_snprintf(name, 64, XFMEDIA_DBUS_SERVICE_FMT, i);
        dbus_error_init(&derr);
        ret = dbus_bus_request_name(dbus_conn, name,
                DBUS_NAME_FLAG_PROHIBIT_REPLACEMENT, &derr);
        if(ret < 0) {
            g_warning("Unable to acquire D-BUS service '%s': %s." \
                "  Remote control interface will not be available.",
                name, derr.message);
            dbus_error_free(&derr);
            return -1;
        } else if(ret == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
            DBG("got ownership with id %d", i);
            break;
        } else {
            DBG("looks like someone else owns id %d", i);
        }
    }
    
    if(i >= MAX_INSTANCES) {
        g_warning("Unable to find free slot for remote control.");
        return -1;
    }
    
    g_snprintf(name, 64, XFMEDIA_DBUS_PATH_FMT, i);
    if(dbus_connection_register_object_path(dbus_conn, name,
            &xfmedia_dbus_vtable, mwin))
    {
        DBG("Successfully registered remote with session id %d.", i);
        return i;
    }
    
    g_warning("Unable to register with D-BUS daemon." \
            "  Remote control interface will not be available.");
    
    return -1;
}

void
xfmedia_remote_cleanup(XfMediaMainwin *mwin)
{
    DBusConnection *dbus_conn;
    
    g_return_if_fail(mwin);
    
    dbus_conn = xfmedia_dbus_get_bus_connection();
    
    if(G_UNLIKELY(!dbus_conn || !dbus_connection_get_is_connected(dbus_conn)))
        return;
    
    if(G_LIKELY(mwin->session_id >= 0)) {
        gchar path_name[64];
        g_snprintf(path_name, 64, XFMEDIA_DBUS_PATH_FMT, mwin->session_id);
        mwin->session_id = -1;
        dbus_connection_unregister_object_path(dbus_conn, path_name);
    }
}


/*******************************
 * client interface
 *******************************/

static XfMediaRemoteStatus
remote_do_ping(DBusConnection *dbus_conn, gint session_id)
{
    XfMediaRemoteStatus ret = XFMEDIA_REMOTE_COMMAND_FAILED;
    DBusMessage *msg, *msg_ret;
    DBusError derr;
    gchar service_name[64], path_name[64];
    
    g_snprintf(service_name, 64, XFMEDIA_DBUS_SERVICE_FMT, session_id);
    g_snprintf(path_name, 64, XFMEDIA_DBUS_PATH_FMT, session_id);
    msg = dbus_message_new_method_call(service_name, path_name,
            XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_IS_RUNNING);
    dbus_message_set_auto_activation(msg, FALSE);
    
    dbus_error_init(&derr);
    msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
            msg, 5000, &derr);
    dbus_message_unref(msg);
    
    if(!msg_ret) {
        DBG("Unable to contact Xfmedia instance on id %d", session_id);
        dbus_error_free(&derr);
    } else if(!dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
        ret = session_id;
    
    if(msg_ret)
        dbus_message_unref(msg_ret);
    
    return ret;
}

XfMediaRemoteStatus
#ifdef HAVE_STDARG_H
xfmedia_remote_send_command(gint session_id, const gchar *command, ...)
#else /* ifdef HAVE_VARARGS_H */
xfmedia_remote_send_command(va_alist) va_dcl
#endif
{
    XfMediaRemoteStatus ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
    va_list ap;
    DBusConnection *dbus_conn;
    DBusMessage *msg, *msg_ret;
    DBusError derr;
    gchar service_name[64], path_name[64];
    
    dbus_conn = xfmedia_dbus_get_bus_connection();
    if(!dbus_conn)
        return XFMEDIA_REMOTE_NO_SESSION_BUS;
    
#ifdef HAVE_STDARG_H
    va_start(ap, command);
#else /* ifdef HAVE_VARARGS_H */
    gint session_id;
    const gchar *command;
    
    va_start(ap);
    session_id = va_arg(ap, gint);
    command = va_arg(ap, const gchar *);
#endif
    
    g_snprintf(service_name, 64, XFMEDIA_DBUS_SERVICE_FMT, session_id);
    g_snprintf(path_name, 64, XFMEDIA_DBUS_PATH_FMT, session_id);
    
    if(!strcmp(command, XFMEDIA_REMOTE_IS_RUNNING)) {
        if(session_id >= 0)
            ret = remote_do_ping(dbus_conn, session_id);
        else {
            gint i;
            for(i = 0; i < MAX_INSTANCES; i++) {
                ret = remote_do_ping(dbus_conn, i);
                if(ret != XFMEDIA_REMOTE_COMMAND_FAILED)
                    break;
            }
        }
    } else if(!strcmp(command, XFMEDIA_REMOTE_CLEAR_PLAYLIST)) {
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_CLEAR_PLAYLIST);
        dbus_message_set_auto_activation(msg, FALSE);
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret) {
            ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
            dbus_error_free(&derr);
        } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
            ret = XFMEDIA_REMOTE_COMMAND_FAILED;
        else if(msg_ret)
            ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
        
        if(msg_ret)
            dbus_message_unref(msg_ret);
    } else if(!strcmp(command, XFMEDIA_REMOTE_LOAD_PLAYLIST)
            || !strcmp(command, XFMEDIA_REMOTE_SAVE_PLAYLIST))
    {
        const gchar *filename;
        
        filename = va_arg(ap, const gchar *);
        if(!filename || !*filename)
            ret = XFMEDIA_REMOTE_BAD_ARGUMENTS;
        else {
            msg = dbus_message_new_method_call(service_name, path_name,
                    XFMEDIA_DBUS_INTERFACE, command);
            dbus_message_set_auto_activation(msg, FALSE);
            dbus_message_append_args(msg,
#ifdef DBUS_USE_OLD_API
                    DBUS_TYPE_STRING, filename,
#else
                    DBUS_TYPE_STRING, &filename,
#endif
                    DBUS_TYPE_INVALID);
            
            dbus_error_init(&derr);
            msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                    msg, 5000, &derr);
            dbus_message_unref(msg);
            
            if(!msg_ret) {
                ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
                dbus_error_free(&derr);
            } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
                ret = XFMEDIA_REMOTE_COMMAND_FAILED;
            else if(msg_ret)
                ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
            
            if(msg_ret)
                dbus_message_unref(msg_ret);
        }
    } else if(!strcmp(command, XFMEDIA_REMOTE_ADD_FILE)) {
        const gchar *filename;
        gint index;
        
        filename = va_arg(ap, const gchar *);
        index = va_arg(ap, gint);
        if(index < -1)
            index = -1;
        
        if(!filename || !*filename)
            ret = XFMEDIA_REMOTE_BAD_ARGUMENTS;
        else {
            msg = dbus_message_new_method_call(service_name, path_name,
                    XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_ADD_FILE);
            dbus_message_set_auto_activation(msg, FALSE);
            dbus_message_append_args(msg,
#ifdef DBUS_USE_OLD_API
                    DBUS_TYPE_STRING, filename,
                    DBUS_TYPE_INT32, index,
#else
                    DBUS_TYPE_STRING, &filename,
                    DBUS_TYPE_INT32, &index,
#endif
                    DBUS_TYPE_INVALID);
            
            dbus_error_init(&derr);
            msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                    msg, 5000, &derr);
            dbus_message_unref(msg);
            
            if(!msg_ret) {
                ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
                dbus_error_free(&derr);
            } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
                ret = XFMEDIA_REMOTE_COMMAND_FAILED;
            else if(msg_ret) {
                dbus_error_init(&derr);
                if(!dbus_message_get_args(msg_ret, &derr,
                        DBUS_TYPE_INT32, &ret, DBUS_TYPE_INVALID))
                {
                    ret = XFMEDIA_REMOTE_COMMAND_FAILED;
                    dbus_error_free(&derr);
                } else
                    ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
            }
            
            if(msg_ret)
                dbus_message_unref(msg_ret);
        }
    } else if(!strcmp(command, XFMEDIA_REMOTE_REMOVE_FILE)) {
        guint index;
        
        index = va_arg(ap, guint);
        
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_REMOVE_FILE);
        dbus_message_set_auto_activation(msg, FALSE);
        dbus_message_append_args(msg,
#ifdef DBUS_USE_OLD_API
                DBUS_TYPE_INT32, index,
#else
                DBUS_TYPE_INT32, &index,
#endif
                DBUS_TYPE_INVALID);
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret) {
            ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
            dbus_error_free(&derr);
        } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
            ret = XFMEDIA_REMOTE_COMMAND_FAILED;
        else if(msg_ret)
            ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
        
        if(msg_ret)
            dbus_message_unref(msg_ret);
    } else if(!strcmp(command, XFMEDIA_REMOTE_PLAY)) {
        gint index;
        
        index = va_arg(ap, gint);
        if(index < -1)
            index = -1;
        
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_PLAY);
        dbus_message_set_auto_activation(msg, FALSE);
        dbus_message_append_args(msg,
#ifdef DBUS_USE_OLD_API
                DBUS_TYPE_INT32, index,
#else
                DBUS_TYPE_INT32, &index,
#endif
                DBUS_TYPE_INVALID);
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret) {
            ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
            dbus_error_free(&derr);
        } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
            ret = XFMEDIA_REMOTE_COMMAND_FAILED;
        else if(msg_ret)
            ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
        
        if(msg_ret)
            dbus_message_unref(msg_ret);
    } else if(!strcmp(command, XFMEDIA_REMOTE_PAUSE)
            || !strcmp(command, XFMEDIA_REMOTE_STOP)
            || !strcmp(command, XFMEDIA_REMOTE_PREV)
            || !strcmp(command, XFMEDIA_REMOTE_NEXT)
            || !strcmp(command, XFMEDIA_REMOTE_QUIT))
    {
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, command);
        dbus_message_set_auto_activation(msg, FALSE);
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret) {
            ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
            dbus_error_free(&derr);
        } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
            ret = XFMEDIA_REMOTE_COMMAND_FAILED;
        else if(msg_ret)
            ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
        
        if(msg_ret)
            dbus_message_unref(msg_ret);
    } else {
        ret = XFMEDIA_REMOTE_BAD_COMMAND;
    }
    
    va_end(ap);
    
    return ret;
}

#else /* !defined(HAVE_DBUS) */

gint
xfmedia_remote_init(XfMediaMainwin *mwin)
{
    g_warning("Not compiled with D-BUS support.  Remote control interface is not available");
    
    return FALSE;
}

void
xfmedia_remote_cleanup(XfMediaMainwin *mwin)
{
    /* noop */
}

XfMediaRemoteStatus
xfmedia_remote_send_command(gint session_id, const gchar *command, ...)
{
    g_critical(_("Xfmedia was not compiled with D-BUS support.  Remote control interface is not available."));
    return XFMEDIA_REMOTE_UNKNOWN_ERROR;
}

#endif /* !defined(HAVE_DBUS) */
