/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <X11/Xlib.h>

#include <gdk/gdkx.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#include "xfmedia-common.h"
#include "xfmedia-internal.h"
#include "main.h"
#include "mainwin.h"
#include "gtkxine.h"
#include "jumptofilewin.h"
#include "keybindings.h"
#include <xfmedia/xfmedia-settings.h>

typedef void (*XfMediaKeybindFunc)(XfMediaMainwin *mwin, gpointer user_data);
#define KEYBIND_FUNC(name) static void name(XfMediaMainwin *mwin, gpointer user_data)

typedef struct
{
    gchar *displayname;
    gchar *keybind_id;
    XfMediaKeybindFunc callback;
    gpointer user_data;
} XfMediaKeybindEntry;

KEYBIND_FUNC(kb_play)
{
    xfmedia_playpause_cb(NULL, mwin);
}

KEYBIND_FUNC(kb_pause)
{
    xfmedia_playpause_cb(NULL, mwin);
}

KEYBIND_FUNC(kb_stop)
{
    xfmedia_stop_cb(NULL, mwin);
}

KEYBIND_FUNC(kb_prev_track)
{
    xfmedia_prev_cb(NULL, mwin);
}

KEYBIND_FUNC(kb_next_track)
{
    xfmedia_next_cb(NULL, mwin);
}

KEYBIND_FUNC(kb_quit)
{
    xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_EXT);
}

KEYBIND_FUNC(kb_fullscreen)
{
    gboolean is_fs = gtk_xine_is_fullscreen(mwin->gtx);
    gint x, y;
        
    gtk_xine_set_fullscreen(mwin->gtx, !is_fs);
    if(!is_fs) {
        gtk_window_get_position(GTK_WINDOW(mwin->video_window), &x, &y);
        xfmedia_settings_set_int("/xfmedia/general/vwin_pos_x", x);
        xfmedia_settings_set_int("/xfmedia/general/vwin_pos_y", y);
        gtk_widget_hide(mwin->video_window);
    } else {
        x = xfmedia_settings_get_int("/xfmedia/general/vwin_pos_x");
        y = xfmedia_settings_get_int("/xfmedia/general/vwin_pos_y");
        gtk_window_move(GTK_WINDOW(mwin->video_window), x, y);
        gtk_widget_show(mwin->video_window);
    }
}

KEYBIND_FUNC(kb_repeat)
{
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mwin->repeat_mi),
            !gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mwin->repeat_mi)));
}

KEYBIND_FUNC(kb_shuffle)
{
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mwin->shuffle_mi),
            !gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mwin->shuffle_mi)));
}

KEYBIND_FUNC(kb_audio_vis)
{
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mwin->svis_mi),
            !gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mwin->svis_mi)));
}

KEYBIND_FUNC(kb_faster)
{
    gint cur_speed = gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED);
    gint new_speed;
    
    switch(cur_speed) {
        case XINE_SPEED_PAUSE:
            new_speed = XINE_SPEED_SLOW_4;
            break;
        case XINE_SPEED_SLOW_4:
            new_speed = XINE_SPEED_SLOW_2;
            break;
        case XINE_SPEED_SLOW_2:
            new_speed = XINE_SPEED_NORMAL;
            break;
        case XINE_SPEED_NORMAL:
            new_speed = XINE_SPEED_FAST_2;
            break;
        case XINE_SPEED_FAST_2:
            new_speed = XINE_SPEED_FAST_4;
            break;
        default:
            return;
    }
    
    if(cur_speed == XINE_SPEED_PAUSE) {
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
        xfmedia_mainwin_set_pause_buttons(mwin);
    }
    gtk_xine_set_param(mwin->gtx, XINE_PARAM_SPEED, new_speed);
    xfmedia_plugins_forward_signal_VOID("speed-changed");
}


KEYBIND_FUNC(kb_slower)
{
    gint cur_speed = gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED);
    gint new_speed;
    
    switch(cur_speed) {
        case XINE_SPEED_SLOW_4:
            new_speed = XINE_SPEED_PAUSE;
            break;
        case XINE_SPEED_SLOW_2:
            new_speed = XINE_SPEED_SLOW_4;
            break;
        case XINE_SPEED_NORMAL:
            new_speed = XINE_SPEED_SLOW_2;
            break;
        case XINE_SPEED_FAST_2:
            new_speed = XINE_SPEED_NORMAL;
            break;
        case XINE_SPEED_FAST_4:
            new_speed = XINE_SPEED_FAST_2;
            break;
        default:
            return;
    }
    
    gtk_xine_set_param(mwin->gtx, XINE_PARAM_SPEED, new_speed);
    if(new_speed == XINE_SPEED_PAUSE) {
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 1);
        xfmedia_mainwin_set_play_buttons(mwin);
    }
    xfmedia_plugins_forward_signal_VOID("speed-changed");
}

KEYBIND_FUNC(kb_forward_10s)
{
    gint pos_stream, pos_time, length_time, new_time;
    
    if(gtk_xine_get_status(mwin->gtx) != XINE_STATUS_PLAY)
        return;
    
    if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time)) {
        new_time = pos_time + 10000;
        if(new_time > length_time)
            new_time = length_time - 1000;
        gtk_xine_play(mwin->gtx, 0, new_time);
    }
}

KEYBIND_FUNC(kb_forward_30s)
{
    gint pos_stream, pos_time, length_time, new_time;
    
    if(gtk_xine_get_status(mwin->gtx) != XINE_STATUS_PLAY)
        return;
    
    if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time)) {
        new_time = pos_time + 30000;
        if(new_time > length_time)
            new_time = length_time - 1000;
        gtk_xine_play(mwin->gtx, 0, new_time);
    }
}

KEYBIND_FUNC(kb_forward_60s)
{
    gint pos_stream, pos_time, length_time, new_time;
    
    if(gtk_xine_get_status(mwin->gtx) != XINE_STATUS_PLAY)
        return;
    
    if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time)) {
        new_time = pos_time + 60000;
        if(new_time > length_time)
            new_time = length_time - 1000;
        gtk_xine_play(mwin->gtx, 0, new_time);
    }
}

KEYBIND_FUNC(kb_backward_10s)
{
    gint pos_stream, pos_time, length_time, new_time;
    
    if(gtk_xine_get_status(mwin->gtx) != XINE_STATUS_PLAY)
        return;
    
    if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time)) {
        new_time = pos_time - 10000;
        if(new_time < 0)
            new_time = 0;
        gtk_xine_play(mwin->gtx, 0, new_time);
    }
}

KEYBIND_FUNC(kb_backward_30s)
{
    gint pos_stream, pos_time, length_time, new_time;
    
    if(gtk_xine_get_status(mwin->gtx) != XINE_STATUS_PLAY)
        return;
    
    if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time)) {
        new_time = pos_time - 30000;
        if(new_time < 0)
            new_time = 0;
        gtk_xine_play(mwin->gtx, 0, new_time);
    }
}

KEYBIND_FUNC(kb_backward_60s)
{
    gint pos_stream, pos_time, length_time, new_time;
    
    if(gtk_xine_get_status(mwin->gtx) != XINE_STATUS_PLAY)
        return;
    
    if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time)) {
        new_time = pos_time - 60000;
        if(new_time < 0)
            new_time = 0;
        gtk_xine_play(mwin->gtx, 0, new_time);
    }
}

KEYBIND_FUNC(kb_mute)
{
    gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_MUTE,
            !gtk_xine_get_param(mwin->gtx, XINE_PARAM_AUDIO_MUTE));
}

KEYBIND_FUNC(kb_add_file)
{
    xfmedia_playlist_trigger_add_file(mwin->plist);
}

KEYBIND_FUNC(kb_add_url)
{
    xfmedia_playlist_trigger_add_url(mwin->plist);
}

KEYBIND_FUNC(kb_open_playlist)
{
    xfmedia_playlist_trigger_open_playlist(mwin->plist);
}

KEYBIND_FUNC(kb_save_playlist)
{
    xfmedia_playlist_trigger_save_playlist(mwin->plist);
}

KEYBIND_FUNC(kb_clear_playlist)
{
    xfmedia_playlist_trigger_new_playlist(mwin->plist);
}

KEYBIND_FUNC(kb_av_sync_plus)
{
    gint cur_sync;
    
    cur_sync = gtk_xine_get_param(mwin->gtx, XINE_PARAM_AV_OFFSET);
    gtk_xine_set_param(mwin->gtx, XINE_PARAM_AV_OFFSET, cur_sync+3600);
}

KEYBIND_FUNC(kb_av_sync_minus)
{
    gint cur_sync;
    
    cur_sync = gtk_xine_get_param(mwin->gtx, XINE_PARAM_AV_OFFSET);
    gtk_xine_set_param(mwin->gtx, XINE_PARAM_AV_OFFSET, cur_sync-3600);
}

KEYBIND_FUNC(kb_aspect)
{
    gint aspect;
    
    aspect = gtk_xine_get_param(mwin->gtx, XINE_PARAM_VO_ASPECT_RATIO);
    if(++aspect >= XINE_VO_ASPECT_NUM_RATIOS)
        aspect = 0;
    gtk_xine_set_param(mwin->gtx, XINE_PARAM_VO_ASPECT_RATIO, aspect);
    DBG("set aspect code to %d", aspect);
}

KEYBIND_FUNC(kb_jump_to_file)
{
    xfmedia_playlist_trigger_jump_to_file(mwin->plist);
}


static XfMediaKeybindEntry keybind_entries[] =
{
    { N_("Play"), "play", kb_play, NULL },
    { N_("Pause"), "pause", kb_pause, NULL },
    { N_("Stop"), "stop", kb_stop, NULL },
    { N_("Previous Track"), "prev-track", kb_prev_track, NULL },
    { N_("Next Track"), "next-track", kb_next_track, NULL },
    { N_("Quit"), "quit", kb_quit, NULL },
    { N_("Toggle Fullscreen"), "toggle-fullscreen", kb_fullscreen, NULL },
    { N_("Toggle Shuffle Play"), "toggle-shuffle", kb_shuffle, NULL },
    { N_("Toggle Repeat Play"), "toggle-repeat", kb_repeat, NULL },
    { N_("Toggle Audio Visualization"), "toggle-audio-vis", kb_audio_vis, NULL },
    { N_("Set Speed Faster"), "speed-faster", kb_faster, NULL },
    { N_("Set Speed Slower"), "speed-slower", kb_slower, NULL },
    { N_("Forward 10 Seconds"), "forward-10s", kb_forward_10s, NULL },
    { N_("Forward 30 Seconds"), "forward-30s", kb_forward_30s, NULL },
    { N_("Forward 60 Seconds"), "forward-60s", kb_forward_60s, NULL },
    { N_("Backward 10 Seconds"), "backward-10s", kb_backward_10s, NULL },
    { N_("Backward 30 Seconds"), "backward-30s", kb_backward_30s, NULL },
    { N_("Backward 60 Seconds"), "backward-60s", kb_backward_60s, NULL },
#if 0  /* unimplemented (yet) */
    { N_("Enqueue Track"), "toggle-enqueue-item", kb_enqueue, NULL },
#endif
    { N_("Toggle Audio Mute"), "toggle-mute", kb_mute, NULL },
    { N_("Add File to Playlist"), "add-file", kb_add_file, NULL },
    { N_("Add URL to Playlist"), "add-url", kb_add_url, NULL },
    { N_("Open Saved Playlist"), "open-playlist", kb_open_playlist, NULL },
    { N_("Save Playlist"), "save-playlist", kb_save_playlist, NULL },
    { N_("Clear Playlist"), "clear-playlist", kb_clear_playlist, NULL },
    { N_("A/V Sync Plus"), "av-sync-plus", kb_av_sync_plus, NULL },
    { N_("A/V Sync Minus"), "av-sync-minus", kb_av_sync_minus, NULL },
    { N_("Select Aspect Ratio"), "toggle-aspect", kb_aspect, NULL },
    { N_("Jump to File"), "jump-to-file", kb_jump_to_file, NULL }
};

void
xfmedia_keybindings_activate(XfMediaMainwin *mwin, const gchar *keybind_id)
{
    /* FIXME: put all this stuff in a GHashTable? */
    gint i;
    
    for(i = 0; i < G_N_ELEMENTS(keybind_entries); i++) {
        if(!g_ascii_strcasecmp(keybind_id, keybind_entries[i].keybind_id)) {
            keybind_entries[i].callback(mwin, keybind_entries[i].user_data);
            break;
        }
    }
}

enum {
    KB_LIST_NAME = 0,
    KB_LIST_FUNCTION,
    KB_LIST_KEYBIND,
    KB_N_COLS
};



static gboolean
is_modifier(guint keycode)
{
    XModifierKeymap *keymap;
    gboolean result = FALSE;
    gint i;
    
    keymap = XGetModifierMapping(GDK_DISPLAY());
    for(i = 0; i < keymap->max_keypermod * 8; i++) {
        if(keycode == keymap->modifiermap[i]) {
            result = TRUE;
            break;
        }
    }
    
    XFreeModifiermap(keymap);
    
    return result;
}


typedef struct
{
    guint new_keyval;
    GdkModifierType new_modmask;
} KeybindInfo;

static gboolean
keybind_keypress_cb(GtkWidget *w, GdkEventKey *evt, gpointer user_data)
{
    KeybindInfo *kinfo = user_data;
    GdkModifierType consumed_modmask = 0;
    
    if(is_modifier(evt->hardware_keycode))
        return TRUE;
    
    gdk_keymap_translate_keyboard_state(gdk_keymap_get_default(),
            evt->hardware_keycode, evt->state, evt->group, NULL, NULL, NULL,
            &consumed_modmask);

    kinfo->new_keyval = gdk_keyval_to_lower(evt->keyval);
    kinfo->new_modmask = evt->state;
    
    /*DBG("got key press: key is %d: %s, mod is %d", kinfo->new_keyval,
            gdk_keyval_name(kinfo->new_keyval), (gint)kinfo->new_modmask);*/
    
    gtk_dialog_response(GTK_DIALOG(w), GTK_RESPONSE_ACCEPT);
    
    return FALSE;
}

static void
keybind_show_set_shortcut_dialog(XfMediaMainwin *mwin, const gchar *displayname,
        const gchar *shortcut, GtkListStore *ls, GtkTreeIter *itr,
        GtkWindow *parent)
{
    GtkWidget *dlg, *btn, *lbl;
    gchar *str;
    KeybindInfo kinfo;
    
    dlg = gtk_dialog_new_with_buttons(_("Set Keybinding"), parent,
            GTK_DIALOG_DESTROY_WITH_PARENT|GTK_DIALOG_NO_SEPARATOR|GTK_DIALOG_MODAL,
            GTK_BUTTONS_NONE, NULL);
    btn = xfmedia_custom_button_new(_("_Clear"), GTK_STOCK_CLEAR);
    gtk_widget_show(btn);
    gtk_dialog_add_action_widget(GTK_DIALOG(dlg), btn, GTK_RESPONSE_NO);
    gtk_dialog_add_button(GTK_DIALOG(dlg), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_container_set_border_width(GTK_CONTAINER(dlg), BORDER);
    gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), BORDER/2);
    gtk_widget_add_events(dlg, GDK_KEY_PRESS);
    
    str = g_strdup_printf("<span size='larger' weight='bold'>%s</span>\n\n%s\n",
            _("Set shortcut for:"), displayname);
    lbl = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(lbl), str);
    g_free(str);
    gtk_widget_show(lbl);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), lbl);
    
    g_signal_connect(G_OBJECT(dlg), "key-press-event",
            G_CALLBACK(keybind_keypress_cb), &kinfo);
    
    gtk_widget_show(dlg);
    
    gdk_keyboard_grab(gtk_widget_get_root_window(dlg), TRUE,
            gtk_get_current_event_time());
    
    switch(gtk_dialog_run(GTK_DIALOG(dlg))) {
        case GTK_RESPONSE_ACCEPT:
            if(xfmedia_keybindings_lookup(kinfo.new_keyval, kinfo.new_modmask)) {
                xfce_message_dialog(parent, _("Keybinding Error"),
                        GTK_STOCK_DIALOG_ERROR, _("Keybinding already in use."),
                        _("The key sequence you selected is already in use by another function.  Please set the other function to something else."),
                        GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT, NULL);
            } else {
                str = g_strdup_printf("%s%s%s%s%s",
                        (kinfo.new_modmask & GDK_CONTROL_MASK) ? "<Control>" : "",
                        (kinfo.new_modmask & GDK_MOD1_MASK) ? "<Alt>" : "",
                        (kinfo.new_modmask & GDK_MOD4_MASK) ? "<Super>" : "",
                        (kinfo.new_modmask & GDK_SHIFT_MASK) ? "<Shift>" : "",
                        gdk_keyval_name(kinfo.new_keyval));
                xfmedia_keybindings_modify(shortcut, kinfo.new_keyval,
                        kinfo.new_modmask);
                gtk_list_store_set(ls, itr, KB_LIST_KEYBIND, str, -1);
                xfmedia_keybindings_save(NULL);
                g_free(str);
            }
            break;
        case GTK_RESPONSE_NO:
            xfmedia_keybindings_modify(shortcut, 0, 0);
            gtk_list_store_set(ls, itr, KB_LIST_KEYBIND, _("None"), -1);
            xfmedia_keybindings_save(NULL);
            break;
        default:
            break;
    }
    
    gdk_keyboard_ungrab(gtk_get_current_event_time());
    
    gtk_widget_destroy(dlg);
}    

gboolean
keybind_btnpress_cb(GtkWidget *w, GdkEventButton *evt, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    if(evt->button == 1 && evt->type == GDK_2BUTTON_PRESS) {
        GtkTreePath *path = NULL;
        
        if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(w), evt->x, evt->y,
                                         &path, NULL, NULL, NULL))
        {
            GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(w));
            GtkTreeIter itr;
            
            if(gtk_tree_model_get_iter(model, &itr, path)) {
                gchar *displayname = NULL, *shortcut = NULL;
            
                gtk_tree_model_get(model, &itr,
                        KB_LIST_NAME, &displayname,
                        KB_LIST_FUNCTION, &shortcut, -1);
                if(shortcut) {
                    keybind_show_set_shortcut_dialog(mwin, displayname,
                            shortcut, GTK_LIST_STORE(model), &itr,
                            GTK_WINDOW(gtk_widget_get_toplevel(w)));
                    g_free(shortcut);
                }
            }
            gtk_tree_path_free(path);
        }
    }
    
    return FALSE;
}

/* FIXME: this is slow.  hashtable? */
static G_CONST_RETURN gchar *
kb_lookup_name(XfMediaKeybinding *kb)
{
    gint i;
    
    for(i = 0; i < G_N_ELEMENTS(keybind_entries); i++) {
        if(!strcmp(keybind_entries[i].keybind_id, kb->keybind_id))
            return _(keybind_entries[i].displayname);
    }
    
    return NULL;
}

GtkWidget *
xfmedia_keybindings_create_editor(XfMediaMainwin *mwin, GtkWidget *parent)
{
    GtkWidget *topvbox, *sw, *treeview;
    GtkListStore *ls;
    GList *keybindings, *l;
    GtkTreeIter itr;
    GtkTreeViewColumn *col;
    GtkCellRenderer *render;
    
    topvbox = gtk_vbox_new(FALSE, BORDER);
    gtk_container_set_border_width(GTK_CONTAINER(topvbox), BORDER/2);
    gtk_widget_show(topvbox);
    
    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
            GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_widget_show(sw);
    gtk_box_pack_start(GTK_BOX(topvbox), sw, TRUE, TRUE, 0);
    
    ls = gtk_list_store_new(KB_N_COLS, G_TYPE_STRING, G_TYPE_STRING,
            G_TYPE_STRING);
    keybindings = xfmedia_keybindings_get_list();
    for(l = keybindings; l; l = l->next) {
        XfMediaKeybinding *kb = l->data;
        gchar *kb_str;
        const gchar *kb_displayname;
        
        kb_str = g_strdup_printf("%s%s%s%s%s",
                (kb->modmask & GDK_CONTROL_MASK) ? "<Control>" : "",
                (kb->modmask & GDK_MOD1_MASK) ? "<Alt>" : "",
                (kb->modmask & GDK_MOD4_MASK) ? "<Super>" : "",
                (kb->modmask & GDK_SHIFT_MASK) ? "<Shift>" : "",
                gdk_keyval_name(kb->keyval));
        
        kb_displayname = kb_lookup_name(kb);
        gtk_list_store_append(ls, &itr);
        gtk_list_store_set(ls, &itr,
                KB_LIST_NAME, kb_displayname ? kb_displayname : kb->keybind_id,
                KB_LIST_FUNCTION, kb->keybind_id,
                KB_LIST_KEYBIND, kb_str, -1);
        g_free(kb_str);
    }
    xfmedia_keybindings_free_list(keybindings);
    
    treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(ls));
    gtk_widget_add_events(treeview, GDK_BUTTON_PRESS);
    gtk_widget_show(treeview);
    gtk_container_add(GTK_CONTAINER(sw), treeview);
    
    render = gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new_with_attributes(_("Function"), render,
            "text", KB_LIST_NAME, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    render =gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new_with_attributes(_("Keys"), render,
            "text", KB_LIST_KEYBIND, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    g_signal_connect(G_OBJECT(treeview), "button-press-event",
            G_CALLBACK(keybind_btnpress_cb), mwin);
    
    return topvbox;
}
