/*
 * Copyright (C) 2001-2003 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * the xine engine in a widget - implementation
 *
 * some code originating from totem's xfmedia_xine widget
 * based on GtkXine, taken from gxine
 *
 * Modified: Copyright (c) 2004-2005 Brian Tarricone <bjt23@cornell.edu>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_MATH_H
#include <math.h>
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
       
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xatom.h>
#ifdef HAVE_XTESTEXTENSION
#include <X11/extensions/XTest.h>
#endif /* HAVE_XTESTEXTENSION */
#ifdef HAVE_XSCREENSAVER_EXTENSION
#include <X11/extensions/scrnsaver.h>
#endif

#include <xine.h>

#include <glib-object.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <libxfce4util/libxfce4util.h>

#include "xfmedia-xine.h"

enum {
    SIG_STREAM_ENDED = 0,
    SIG_SET_TITLE,
    SIG_MESSAGE,
    SIG_FORMAT_CHANGE,
    SIG_PROGRESS_MSG,
    SIG_MRL_REFERENCE,
    SIG_SPU_ENTER,
    SIG_SPU_LEAVE,
    N_SIGNALS
};

#define DEFAULT_WIDTH  420
#define DEFAULT_HEIGHT 315
#define DEFAULT_VIS_WIDTH  480
#define DEFAULT_VIS_HEIGHT 360

/* missing stuff from X includes */
#if !defined(XShmGetEventBase) && defined(XINE_GUI_SEND_COMPLETION_EVENT)
extern int XShmGetEventBase(Display *);
#endif

static void xfmedia_xine_class_init(XfmediaXineClass *klass);
static void xfmedia_xine_init(XfmediaXine *gxine);

static void xfmedia_xine_finalize(GObject *object);
static void xfmedia_xine_realize(GtkWidget *widget);
static void xfmedia_xine_unrealize(GtkWidget *widget);

static void xfmedia_xine_size_allocate(GtkWidget *widget, GtkAllocation *allocation);

static void xfmedia_xine_init_driver_options(XfmediaXine *xfx);

static GtkWidgetClass *parent_class = NULL;
static guint xfmedia_xine_sigs[N_SIGNALS] = { 0 };

struct _XfmediaXinePriv
{
    xine_t *xine;
    xine_stream_t *stream;
    gchar *mrl;
    xine_event_queue_t *ev_queue;

    gchar *configfile;

    gchar *video_driver_id;
    gchar *audio_driver_id;

    xine_video_port_t *none_video_port;
    xine_video_port_t *video_port;
    xine_audio_port_t *audio_port;
    x11_visual_t vis;
    gdouble display_ratio;
    Display *display;
    gint screen;
    Window video_window;

    gint xpos, ypos;
    gint oldwidth, oldheight;
    gdouble resize_factor;
    
    gboolean have_xscr;
    gboolean have_xtest;
    gint xtest_keycode;
    guint screensaver_timeout;

    /* visualization */
    xine_post_t *vis_plugin;
    gchar *vis_plugin_id;
    
    guint last_button_press[3];
    guint last_x_coord[3];
    guint last_y_coord[3];
    
    GIOChannel *xine_display_ioc;
    guint xine_display_src;
    
#ifdef XINE_GUI_SEND_COMPLETION_EVENT
    gint completion_event;
#endif
};

/* VOID:INT,INT,INT,BOOLEAN */
static void
xfmedia_xine_marshal_VOID__INT_INT_INT_BOOLEAN(GClosure *closure,
        GValue *return_value, guint n_param_values, const GValue *param_values,
        gpointer invocation_hint, gpointer marshal_data)
{
    typedef void (*GMarshalFunc_VOID__INT_INT_INT_BOOLEAN)(gpointer data1,
            gint arg_1, gint arg_2, gint arg_3, gboolean arg_4, gpointer data2);
    
    register GMarshalFunc_VOID__INT_INT_INT_BOOLEAN callback;
    register GCClosure *cc = (GCClosure*)closure;
    register gpointer data1, data2;
    
    g_return_if_fail (n_param_values == 5);
    
    if(G_CCLOSURE_SWAP_DATA(closure)) {
        data1 = closure->data;
        data2 = g_value_peek_pointer(param_values + 0);
    } else {
        data1 = g_value_peek_pointer(param_values + 0);
        data2 = closure->data;
    }
    
    callback = (GMarshalFunc_VOID__INT_INT_INT_BOOLEAN)(marshal_data ? marshal_data : cc->callback);
    
    callback(data1, g_value_get_int(param_values + 1),
            g_value_get_int(param_values + 2),
            g_value_get_int(param_values + 3),
            g_value_get_boolean(param_values + 4), data2);
}

/* VOID:STRING,INT */
static void
xfmedia_xine_marshal_VOID__STRING_INT(GClosure *closure,
        GValue *return_value, guint n_param_values, const GValue *param_values,
        gpointer invocation_hint, gpointer marshal_data)
{
    typedef void (*GMarshalFunc_VOID__STRING_INT)(gpointer data1, const gchar *arg_1,
            gint arg_2, gpointer data2);
    
    register GMarshalFunc_VOID__STRING_INT callback;
    register GCClosure *cc = (GCClosure*)closure;
    register gpointer data1, data2;
    
    g_return_if_fail (n_param_values == 3);
    
    if(G_CCLOSURE_SWAP_DATA(closure)) {
        data1 = closure->data;
        data2 = g_value_peek_pointer(param_values + 0);
    } else {
        data1 = g_value_peek_pointer(param_values + 0);
        data2 = closure->data;
    }
    
    callback = (GMarshalFunc_VOID__STRING_INT)(marshal_data ? marshal_data : cc->callback);
    
    callback(data1, g_value_get_string(param_values + 1),
            g_value_get_int(param_values + 2), data2);
}

GtkType
xfmedia_xine_get_type(void)
{
    static GtkType xfmedia_xine_type = 0;

    if(!xfmedia_xine_type) {
        static const GTypeInfo xfmedia_xine_info = {
            sizeof(XfmediaXineClass),
            (GBaseInitFunc) NULL,
            (GBaseFinalizeFunc) NULL,
            (GClassInitFunc) xfmedia_xine_class_init,
            (GClassFinalizeFunc) NULL,
            NULL /* class_data */ ,
            sizeof(XfmediaXine),
            0 /* n_preallocs */ ,
            (GInstanceInitFunc) xfmedia_xine_init
        };

        xfmedia_xine_type = g_type_register_static(GTK_TYPE_WIDGET,
                "XfmediaXine", &xfmedia_xine_info, (GTypeFlags)0);
    }

    return xfmedia_xine_type;
}

static void
xfmedia_xine_class_init(XfmediaXineClass *klass)
{
    GObjectClass *object_class;
    GtkWidgetClass *widget_class;
    typedef gboolean (*MyExposeEventFunc)(GtkWidget *w, GdkEventExpose *evt);

    object_class = (GObjectClass *)klass;
    widget_class = (GtkWidgetClass *)klass;

    parent_class = g_type_class_peek_parent(klass);

    /* GtkWidget */
    widget_class->realize = xfmedia_xine_realize;
    widget_class->unrealize = xfmedia_xine_unrealize;
    widget_class->size_allocate = xfmedia_xine_size_allocate;
    /* eat expose events, as xine_display_event_handler() will handle them
     * and pass them to xine itself */
    widget_class->expose_event = (MyExposeEventFunc)gtk_true;

    /* GObject */
    object_class->set_property = NULL;
    object_class->get_property = NULL;
    object_class->finalize = xfmedia_xine_finalize;
    
    /* signals */
    xfmedia_xine_sigs[SIG_STREAM_ENDED] = g_signal_new("stream-ended",
            XFMEDIA_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(XfmediaXineClass, stream_ended), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    xfmedia_xine_sigs[SIG_SET_TITLE] = g_signal_new("set-title",
            XFMEDIA_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(XfmediaXineClass, set_title), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    xfmedia_xine_sigs[SIG_MESSAGE] = g_signal_new("ui-message",
            XFMEDIA_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(XfmediaXineClass, ui_message), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    xfmedia_xine_sigs[SIG_FORMAT_CHANGE] = g_signal_new("format-changed",
            XFMEDIA_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(XfmediaXineClass, format_changed), NULL, NULL,
            xfmedia_xine_marshal_VOID__INT_INT_INT_BOOLEAN, G_TYPE_NONE, 4,
            G_TYPE_INT, G_TYPE_INT, G_TYPE_INT, G_TYPE_BOOLEAN);
    
    xfmedia_xine_sigs[SIG_PROGRESS_MSG] = g_signal_new("progress-message",
            XFMEDIA_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(XfmediaXineClass, progress_message), NULL, NULL,
            xfmedia_xine_marshal_VOID__STRING_INT, G_TYPE_NONE, 2,
            G_TYPE_STRING, G_TYPE_INT);
    
    xfmedia_xine_sigs[SIG_MRL_REFERENCE] = g_signal_new("mrl-reference",
            XFMEDIA_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(XfmediaXineClass, mrl_reference), NULL, NULL,
            xfmedia_xine_marshal_VOID__STRING_INT, G_TYPE_NONE, 4,
            G_TYPE_STRING, G_TYPE_INT, G_TYPE_STRING, G_TYPE_INT);
    
    xfmedia_xine_sigs[SIG_SPU_ENTER] = g_signal_new("spu-button-enter",
            XFMEDIA_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(XfmediaXineClass, spu_button_enter), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    xfmedia_xine_sigs[SIG_SPU_LEAVE] = g_signal_new("spu-button-leave",
            XFMEDIA_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(XfmediaXineClass, spu_button_leave), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}

static void
xfmedia_xine_init(XfmediaXine *xfx)
{
    GTK_WIDGET(xfx)->requisition.width = DEFAULT_WIDTH;
    GTK_WIDGET(xfx)->requisition.height = DEFAULT_HEIGHT;

    /* create a new xine instance, load config values */
    
    xfx->priv = g_new0(XfmediaXinePriv, 1);

    xfx->priv->xine = xine_new();
    
    xfx->priv->resize_factor = 1.0;
    
    xfx->priv->configfile = xfce_get_homefile(".xine", "config", NULL);
    if(g_file_test(xfx->priv->configfile, G_FILE_TEST_EXISTS))
        xine_config_load(xfx->priv->xine, xfx->priv->configfile);
    
    xine_init(xfx->priv->xine);
    
    xfmedia_xine_init_driver_options(xfx);

#ifdef HAVE_XSCREENSAVER_EXTENSION
    {
        int a, b;
        
        xfx->priv->have_xscr = XScreenSaverQueryExtension(GDK_DISPLAY(), &a, &b);
    }
#endif
    
#ifdef HAVE_XTESTEXTENSION
    {
        int a, b, c, d;

        xfx->priv->have_xtest = XTestQueryExtension(gdk_display, &a, &b, &c, &d);
        if(xfx->priv->have_xtest == True)
            xfx->priv->xtest_keycode = XKeysymToKeycode(gdk_display, XK_Shift_L);
    }
#endif

    xfx->priv->vis_plugin = NULL;
    xfx->priv->vis_plugin_id = NULL;
}

static void
xfmedia_xine_finalize(GObject *object)
{
    XfmediaXine *xfx = XFMEDIA_XINE(object);
    
    if(xfx->priv->mrl) {
        g_free(xfx->priv->mrl);
        xfx->priv->mrl = NULL;
    }
    
    /* save configuration */
    xine_config_save(xfx->priv->xine, xfx->priv->configfile);
    g_free(xfx->priv->configfile);

    /* exit xine */
    xine_exit(xfx->priv->xine);
    
    g_free(xfx->priv->video_driver_id);
    g_free(xfx->priv->audio_driver_id);
    
    g_free(xfx->priv);
    xfx->priv = NULL;

    G_OBJECT_CLASS(parent_class)->finalize(object);

    xfx = NULL;
}

#ifdef HAVE_XSCREENSAVER_EXTENSION
static gint
xfmedia_xine_get_x_idle_time(Display *dpy, Window rootwin)
{
    gint ret = -1;
    XScreenSaverInfo *sinfo = XScreenSaverAllocInfo();
    
    XScreenSaverQueryInfo(dpy, rootwin, sinfo);
    ret = sinfo->idle;
    XFree(sinfo);
    
    return ret;
}
#endif

typedef struct
{
    GValue *args;
    guint16 nargs;
    guint16 signal_id;
} XfmediaXineSignalData;

static gboolean
xfmedia_xine_event_emit_idled(gpointer data)
{
    XfmediaXineSignalData *sigdata = data;
    guint i;
    
    gdk_threads_enter();
    g_signal_emitv(sigdata->args, sigdata->signal_id, 0, NULL);
    gdk_threads_leave();
    
    for(i = 0; i < sigdata->nargs; i++)
        g_value_unset(&sigdata->args[i]);
    g_free(sigdata->args);
    g_free(sigdata);
    
    return FALSE;
}

static XfmediaXineSignalData *
xfmedia_xine_signal_data_new(XfmediaXine *xfx, guint signal, guint nargs, ...)
{
    XfmediaXineSignalData *sigdata;
    va_list ap;
    guint i;
    
    g_return_val_if_fail(xfx, NULL);
    
    sigdata = g_new0(XfmediaXineSignalData, 1);
    sigdata->signal_id = xfmedia_xine_sigs[signal];
    sigdata->nargs = nargs;
    
    sigdata->args = g_new0(GValue, nargs+1);
    g_value_init(sigdata->args, G_TYPE_OBJECT);
    g_value_set_object(sigdata->args, xfx);
    
    if(nargs > 0)
        va_start(ap, nargs);
    
    for(i = 1; i <= nargs; i++) {
        GType type = va_arg(ap, GType);
        
        if(type == G_TYPE_INT) {
            gint int_val = va_arg(ap, gint);
            g_value_init(&sigdata->args[i], type);
            g_value_set_int(&sigdata->args[i], int_val);
        } else if(type == G_TYPE_STRING) {
            gchar *str_val = va_arg(ap, gchar *);
            g_value_init(&sigdata->args[i], type);
            g_value_take_string(&sigdata->args[i], str_val);
        } else if(type == G_TYPE_BOOLEAN) {
            gboolean bool_val = va_arg(ap, gboolean);
            g_value_init(&sigdata->args[i], type);
            g_value_set_boolean(&sigdata->args[i], bool_val);
        }
    }
    
    if(nargs > 0)
        va_end(ap);
    
    return sigdata;
}

static void
event_listener_cb(void *user_data, const xine_event_t *event)
{
    XfmediaXine *xfx = user_data;
    XfmediaXineSignalData *sigdata;
    
    switch(event->type) {
        case XINE_EVENT_UI_PLAYBACK_FINISHED:
            sigdata = xfmedia_xine_signal_data_new(xfx, SIG_STREAM_ENDED, 0);
            break;
        
        case XINE_EVENT_UI_SET_TITLE:
            {
                xine_ui_data_t *uidata = (xine_ui_data_t *)event->data;
                
                sigdata = xfmedia_xine_signal_data_new(xfx, SIG_SET_TITLE, 1,
                        G_TYPE_STRING, g_strdup(uidata->str));
            }
            break;
        
        case XINE_EVENT_UI_MESSAGE:
            {
                xine_ui_message_data_t *mdata = (xine_ui_message_data_t *)event->data;
                const gchar *explanation = NULL;
                const gchar *parameter1 = NULL;
                
                if(mdata->explanation)
                    explanation = (gchar *)(((guchar *)mdata) + mdata->explanation);
                
                if(mdata->num_parameters)
                    parameter1 = (gchar *)(((guchar *)mdata) + mdata->parameters);
                
                sigdata = xfmedia_xine_signal_data_new(xfx, SIG_MESSAGE, 1,
                                                       G_TYPE_STRING,
                                                       g_strconcat(explanation,
                                                                   " ",
                                                                   parameter1,
                                                                   NULL));
            }
            break;
        
        case XINE_EVENT_FRAME_FORMAT_CHANGE:
            {
                xine_format_change_data_t *fdata = (xine_format_change_data_t *)event->data;
                
                sigdata = xfmedia_xine_signal_data_new(xfx, SIG_FORMAT_CHANGE, 4,
                        G_TYPE_INT, fdata->width,
                        G_TYPE_INT, fdata->height,
                        G_TYPE_INT, fdata->aspect,
                        G_TYPE_BOOLEAN, fdata->pan_scan ? TRUE : FALSE);
            }
            break;
        
        case XINE_EVENT_PROGRESS:
            {
                xine_progress_data_t* pdata = (xine_progress_data_t *)event->data;
                
                sigdata = xfmedia_xine_signal_data_new(xfx, SIG_PROGRESS_MSG, 2,
                        G_TYPE_STRING, g_strdup(pdata->description),
                        G_TYPE_INT, pdata->percent);
            }
            break;

#ifdef XINE_EVENT_MRL_REFERENCE_EXT
        case XINE_EVENT_MRL_REFERENCE_EXT:
            {
                xine_mrl_reference_data_ext_t *mdata = (xine_mrl_reference_data_ext_t *)event->data;
#else
        case XINE_EVENT_MRL_REFERENCE:
            {
                xine_mrl_reference_data_t *mdata = (xine_mrl_reference_data_t *)event->data;
#endif
                sigdata = xfmedia_xine_signal_data_new(xfx, SIG_MRL_REFERENCE, 4,
                        G_TYPE_STRING, g_strdup((gchar *)mdata->mrl),
                        G_TYPE_INT, mdata->alternative,
#ifdef XINE_EVENT_MRL_REFERENCE_EXT
                        G_TYPE_STRING, g_strdup((gchar *)(mdata->mrl+strlen(mdata->mrl)+1)),
                        G_TYPE_INT, (gint)mdata->duration);
#else
                        G_TYPE_STRING, NULL,
                        G_TYPE_INT, -1);
#endif
            }
            break;
        
        case XINE_EVENT_SPU_BUTTON:
            {
                xine_spu_button_t *sdata = (xine_spu_button_t *)event->data;
                
                sigdata = xfmedia_xine_signal_data_new(xfx,
                                                       sdata->direction == 1
                                                         ? SIG_SPU_ENTER
                                                         : SIG_SPU_LEAVE,
                                                       0);
            }
            break;
        
        default:
            /*g_message("XfmediaXine: Unhandled async event of type %d\n", event->type);*/
            return;
    }
    
    g_idle_add_full(G_PRIORITY_DEFAULT, xfmedia_xine_event_emit_idled,
                    sigdata, NULL);
}

static void
dest_size_cb(gpointer xfx_gen, gint video_width, gint video_height,
        gdouble video_pixel_aspect, gint *dest_width, gint *dest_height,
        gdouble *dest_pixel_aspect)
{

    XfmediaXine *xfx = (XfmediaXine *)xfx_gen;

    /* correct size with video_pixel_aspect */
    if(video_pixel_aspect >= xfx->priv->display_ratio)
        video_width = video_width * video_pixel_aspect / xfx->priv->display_ratio + .5;
    else
        video_height = video_height * xfx->priv->display_ratio / video_pixel_aspect + .5;
    
    if(xfx->priv->resize_factor == 0.0) {
        *dest_width = GTK_WIDGET(xfx)->allocation.width;
        *dest_height = GTK_WIDGET(xfx)->allocation.height;
    } else {
        *dest_width = video_width * xfx->priv->resize_factor;
        *dest_height = video_height * xfx->priv->resize_factor;
    }

    *dest_pixel_aspect = xfx->priv->display_ratio;
}

static gboolean
xfmedia_xine_idle_resize(gpointer user_data)
{
    XfmediaXine *xfx = user_data;
    GtkWindow *toplevel;
    gint video_width = xfx->priv->oldwidth;
    gint video_height = xfx->priv->oldheight;

    gdk_threads_enter();

    toplevel = GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(xfx)));

    gtk_window_set_resizable(toplevel, FALSE);
    GTK_WIDGET(xfx)->allocation.width = video_width;
    GTK_WIDGET(xfx)->allocation.height = video_height;
    gtk_widget_set_size_request(GTK_WIDGET(xfx), video_width, video_height);
    gtk_widget_queue_resize(gtk_widget_get_parent(GTK_WIDGET(xfx)));
    while(gtk_events_pending())
        gtk_main_iteration();

    gtk_window_set_resizable(toplevel, TRUE);

    gdk_threads_leave();

    return FALSE;
}

static void
frame_output_cb(gpointer xfx_gen, gint video_width, gint video_height,
        gdouble video_pixel_aspect, gint *dest_x, gint *dest_y,
        gint *dest_width, gint *dest_height, gdouble *dest_pixel_aspect,
        gint *win_x, gint *win_y)
{

    XfmediaXine *xfx = XFMEDIA_XINE(xfx_gen);

    g_return_if_fail(XFMEDIA_IS_XINE(xfx));

    /* correct size with video_pixel_aspect */
    if(video_pixel_aspect >= xfx->priv->display_ratio)
        video_width = video_width * video_pixel_aspect / xfx->priv->display_ratio + .5;
    else
        video_height = video_height * xfx->priv->display_ratio / video_pixel_aspect + .5;

    *dest_x = 0;
    *dest_y = 0;
    
    *win_x = xfx->priv->xpos;
    *win_y = xfx->priv->ypos;

    if(xfx->priv->resize_factor != 0.0) {    /* => auto-resize */
        video_width *= xfx->priv->resize_factor;
        video_height *= xfx->priv->resize_factor;

        /* size changed? */
        if((video_width != xfx->priv->oldwidth)
                || (video_height != xfx->priv->oldheight))
        {
            xfx->priv->oldwidth = video_width;
            xfx->priv->oldheight = video_height;

            /* why can't we do this right here? */
            g_idle_add_full(GTK_PRIORITY_RESIZE, xfmedia_xine_idle_resize,
                            xfx, NULL);
        }

        xfx->priv->resize_factor = 0.0;
    }

    *dest_width = GTK_WIDGET(xfx)->allocation.width;
    *dest_height = GTK_WIDGET(xfx)->allocation.height;
    
    *dest_pixel_aspect = xfx->priv->display_ratio;
}

static void
xfmedia_xine_init_driver_options(XfmediaXine *xfx)
{
    const gchar *const *driver_ids;
    gchar **choices;
    gint i;
    
    choices = g_malloc(sizeof(gchar *) * 100);
    choices[0] = "auto";
    
    /* audio */
    driver_ids = xine_list_audio_output_plugins(xfx->priv->xine);
    for(i = 0; driver_ids[i]; i++)
        choices[i+1] = (gchar *)driver_ids[i];
    choices[i+1] = 0;

    xine_config_register_enum(xfx->priv->xine, "audio.driver", 0, choices,
            "audio driver to use", NULL, 10, NULL, NULL);
    
    /* video */
    driver_ids = xine_list_video_output_plugins(xfx->priv->xine);
    for(i = 0; driver_ids[i]; i++)
        choices[i+1] = (gchar *)driver_ids[i];
    choices[i+1] = 0;
    
    xine_config_register_enum(xfx->priv->xine, "video.driver", 0, choices,
            "video driver to use", NULL, 10, NULL, NULL);
    
    g_free(choices);
}

static xine_video_port_t *
load_video_out_driver(XfmediaXine *xfx, const gchar *video_driver_id)
{
    gdouble res_h, res_v;
    x11_visual_t vis;
    xine_video_port_t *video_port;
    GdkScreen *gscreen = gtk_widget_get_screen(GTK_WIDGET(xfx));
    
    vis.display = xfx->priv->display;
    vis.screen = xfx->priv->screen;
    vis.d = xfx->priv->video_window;
    res_h = gdk_screen_get_width(gscreen) * 1000 / gdk_screen_get_width_mm(gscreen);
    res_v = gdk_screen_get_height(gscreen) * 1000 / gdk_screen_get_height_mm(gscreen);
    xfx->priv->display_ratio = res_v / res_h;

    if(fabs(xfx->priv->display_ratio - 1.0) < 0.01)
        xfx->priv->display_ratio = 1.0;

    vis.dest_size_cb = dest_size_cb;
    vis.frame_output_cb = frame_output_cb;
    vis.user_data = xfx;
    
    if(!video_driver_id) {
        if(xfx->priv->video_driver_id)
            video_driver_id = xfx->priv->video_driver_id;
        else {
            xine_cfg_entry_t cfg;
            
            if(xine_config_lookup_entry(xfx->priv->xine, "video.driver", &cfg))
                video_driver_id = cfg.enum_values[cfg.num_value];
            else
                video_driver_id = "auto";
        }
    }
    
    if(strcmp(video_driver_id, "auto")) {
        video_port = xine_open_video_driver(xfx->priv->xine,
                video_driver_id, XINE_VISUAL_TYPE_X11, (void *)&vis);
        if(video_port) {
            if(!xfx->priv->video_driver_id)
                xfx->priv->video_driver_id = g_strdup(video_driver_id);
            return video_port;
        } else
            g_warning("XfmediaXine: Video driver %s failed.  Attempting autodetection.", video_driver_id);
    }

    return xine_open_video_driver(xfx->priv->xine, NULL,
            XINE_VISUAL_TYPE_X11, (void *)&vis);
}

static xine_audio_port_t *
load_audio_out_driver(XfmediaXine *xfx, const gchar *audio_driver_id)
{
    xine_audio_port_t *audio_port;
    
    if(!audio_driver_id) {
        if(xfx->priv->audio_driver_id)
            audio_driver_id = xfx->priv->audio_driver_id;
        else {
            xine_cfg_entry_t cfg;
            
            if(xine_config_lookup_entry(xfx->priv->xine, "audio.driver", &cfg))
                audio_driver_id = cfg.enum_values[cfg.num_value];
            else
                audio_driver_id = "auto";
        }
    }

    DBG("trying audio driver %s", audio_driver_id);
    if(strcmp(audio_driver_id, "auto")) {
        audio_port = xine_open_audio_driver(xfx->priv->xine, audio_driver_id, NULL);
        if(audio_port) {
            if(!xfx->priv->audio_driver_id)
                xfx->priv->audio_driver_id = g_strdup(audio_driver_id);
            return audio_port;
        } else
            g_warning("XfmediaXine: Audio driver %s failed.  Attempting autodetection.", audio_driver_id);
    }

    /* autoprobe */
    return xine_open_audio_driver(xfx->priv->xine, NULL, NULL);
}

void
xfmedia_xine_port_send_gui_data(XfmediaXine *xfx, gint type, gpointer data)
{
    Display *dpy;
    xine_video_port_t *vo_port;
    
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    
    dpy = xfx->priv->display;
    
    XLockDisplay(dpy);
    
    if(xfx->priv->video_port)
        vo_port = xfx->priv->video_port;
    else
        vo_port = xfx->priv->none_video_port;
    
    xine_port_send_gui_data(vo_port, type, data);
    
    XUnlockDisplay(dpy);
}

static void
xfmedia_xine_fwd_event(XfmediaXine *xfx, const gchar *signal, GdkEvent *event)
{
    GtkWidget *toplevel;
    gboolean dummy = FALSE;
    
    gdk_threads_enter();
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(xfx));
    event->any.window = toplevel->window;
    
    if(event->type == GDK_EXPOSE)
        event->expose.region = gdk_region_rectangle(&event->expose.area);
    
    g_signal_emit_by_name(G_OBJECT(toplevel), signal, event, &dummy);
    
    if(event->type == GDK_EXPOSE)
        gdk_region_destroy(event->expose.region);
    
    gdk_threads_leave();
}

static gboolean
xine_display_event_handler(GIOChannel *source, GIOCondition condition,
        gpointer data)
{
    XfmediaXine *xfx = XFMEDIA_XINE(data);
    Display *dpy = xfx->priv->display;
    GValue gval;
    XEvent event;
    gint nevents, i;

    nevents = XPending(dpy);
    
    for(i = 0; i < nevents; i++) {
        XNextEvent(dpy, &event);

        /* printf ("xfmedia_xine: got an event (%d)\n", event.type);  */

        switch(event.type) {
            case Expose:
                if(event.xexpose.count != 0)
                    break;
                
                xfmedia_xine_port_send_gui_data(xfx,
                        XINE_GUI_SEND_EXPOSE_EVENT, &event);
                
                {
                    GdkEventExpose gdk_event;
                    XExposeEvent *eevent = (XExposeEvent *)&event;
                    
                    gdk_event.type = GDK_EXPOSE;
                    gdk_event.send_event = 0;
                    gdk_event.area.x = eevent->x;
                    gdk_event.area.y = eevent->y;
                    gdk_event.area.width = eevent->width;
                    gdk_event.area.height = eevent->height;
                    gdk_event.count = eevent->count;
                    
                    xfmedia_xine_fwd_event(xfx, "expose-event",
                                           (GdkEvent *)&gdk_event);
                }
                
                break;
    
            case MotionNotify:
                {
                    XMotionEvent *mevent = (XMotionEvent *)&event;
                    x11_rectangle_t rect;
                    xine_event_t event;
                    xine_input_data_t input;
                    
                    rect.x = mevent->x;
                    rect.y = mevent->y;
                    rect.w = 0;
                    rect.h = 0;
                    
                    xfmedia_xine_port_send_gui_data(xfx,
                            XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO, &rect);
                    
                    event.type = XINE_EVENT_INPUT_MOUSE_MOVE;
                    event.data = &input;
                    event.data_length = sizeof(input);
                    input.button = 0;    /* no buttons, just motion */
                    input.x = rect.x;
                    input.y = rect.y;
                    xine_event_send(xfx->priv->stream, &event);
                }
                {
                    GdkEventMotion gdk_event;
                    XMotionEvent *mevent = (XMotionEvent *)&event;
                    
                    gdk_event.type = GDK_MOTION_NOTIFY;
                    gdk_event.send_event = mevent->send_event;
                    gdk_event.time = mevent->time;
                    gdk_event.x = mevent->x;
                    gdk_event.y = mevent->y;
                    gdk_event.state = mevent->state;
                    gdk_event.is_hint = mevent->is_hint;
                    gdk_event.x_root = mevent->x_root;
                    gdk_event.y_root = mevent->y_root;
                    
                    xfmedia_xine_fwd_event(xfx, "motion-notify-event",
                                           (GdkEvent *)&gdk_event);
                }
                break;
    
            case ButtonPress:
                {
                    XButtonEvent *bevent = (XButtonEvent *)&event;
    
                    if(bevent->button == Button1) {
                        x11_rectangle_t rect;
                        xine_event_t event;
                        xine_input_data_t input;
                        
                        rect.x = bevent->x;
                        rect.y = bevent->y;
                        rect.w = 0;
                        rect.h = 0;
                        
                        xfmedia_xine_port_send_gui_data(xfx,
                                XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO,
                                &rect);
                        
                        event.type = XINE_EVENT_INPUT_MOUSE_BUTTON;
                        event.data = &input;
                        event.data_length = sizeof(input);
                        input.button = 1;
                        input.x = rect.x;
                        input.y = rect.y;
                        xine_event_send(xfx->priv->stream, &event);
                    }
                }
                {
                    GdkEventButton gdk_event;
                    XButtonEvent *bevent = (XButtonEvent *)&event;
                    gint dbl_click_time, dbl_click_dist;
                    
                    if(gdk_setting_get("Net/DoubleClickTime", &gval)) {
                        dbl_click_time = g_value_get_int(&gval);
                        g_value_unset(&gval);
                    } else
                        dbl_click_time = 250;
                    
                    if(gdk_setting_get("Net/DoubleClickDistance", &gval)) {
                        dbl_click_dist = g_value_get_int(&gval);
                        g_value_unset(&gval);
                    } else
                        dbl_click_dist = 5;
                    
                    gdk_event.type = GDK_BUTTON_PRESS;
                    if(bevent->button < 4) {
                        guint dx = abs(bevent->x - xfx->priv->last_x_coord[bevent->button-1]);
                        guint dy = abs(bevent->y - xfx->priv->last_y_coord[bevent->button-1]);
                        dx *= dx; dy *= dy;
                        if(bevent->time - xfx->priv->last_button_press[bevent->button-1] <= dbl_click_time
                                && sqrt(dx + dy) < dbl_click_dist)
                        {
                            gdk_event.type = GDK_2BUTTON_PRESS;
                        }
                        xfx->priv->last_button_press[bevent->button-1] = bevent->time;
                        xfx->priv->last_x_coord[bevent->button-1] = bevent->x;
                        xfx->priv->last_y_coord[bevent->button-1] = bevent->y;
                    }
                    gdk_event.send_event = bevent->send_event;
                    gdk_event.time = bevent->time;
                    gdk_event.x = bevent->x;
                    gdk_event.y = bevent->y;
                    gdk_event.state = bevent->state;
                    gdk_event.button = bevent->button;
                    gdk_event.x_root = bevent->x_root;
                    gdk_event.y_root = bevent->y_root;
                    
                    xfmedia_xine_fwd_event(xfx, "button-press-event",
                                           (GdkEvent *)&gdk_event);
                }
                break;
            
            case ButtonRelease:
                {
                    GdkEventButton gdk_event;
                    XButtonEvent *bevent = (XButtonEvent *)&event;
                    
                    gdk_event.type = GDK_BUTTON_RELEASE;
                    gdk_event.send_event = bevent->send_event;
                    gdk_event.time = bevent->time;
                    gdk_event.x = bevent->x;
                    gdk_event.y = bevent->y;
                    gdk_event.state = bevent->state;
                    gdk_event.button = bevent->button;
                    gdk_event.x_root = bevent->x_root;
                    gdk_event.y_root = bevent->y_root;
                    
                    xfmedia_xine_fwd_event(xfx, "button-release-event",
                                           (GdkEvent *)&gdk_event);
                }
                break;
    
            case KeyPress:
                {
                    GdkEventKey gdk_event;
                    XKeyEvent *kevent = (XKeyEvent *)&event;
                    gchar buffer[20];
                    gint bufsize = 20;
                    KeySym keysym;
                    XComposeStatus compose;
                    
                    memset(buffer, 0, sizeof(buffer));
                    XLookupString(kevent, buffer, bufsize, &keysym, &compose);
    
                    gdk_event.type = GDK_KEY_PRESS;
                    gdk_event.time = kevent->time;
                    gdk_event.keyval = keysym;
                    gdk_event.state = (GdkModifierType)kevent->state;
                    gdk_event.string = buffer;
                    gdk_event.length = strlen(buffer);
                    
                    xfmedia_xine_fwd_event(xfx, "key-press-event",
                                           (GdkEvent *)&gdk_event);
                }
    
                break;
            
            case KeyRelease:
                {
                    GdkEventKey gdk_event;
                    XKeyEvent *kevent = (XKeyEvent *)&event;
                    gchar buffer[20];
                    gint bufsize = 20;
                    KeySym keysym;
                    XComposeStatus compose;
                    
                    memset(buffer, 0, sizeof(buffer));
                    XLookupString(kevent, buffer, bufsize, &keysym, &compose);
    
                    gdk_event.type = GDK_KEY_RELEASE;
                    gdk_event.time = kevent->time;
                    gdk_event.keyval = keysym;
                    gdk_event.state = (GdkModifierType)kevent->state;
                    gdk_event.string = g_strdup(buffer);
                    gdk_event.length = strlen(buffer);
                    
                    xfmedia_xine_fwd_event(xfx, "key-release-event",
                                           (GdkEvent *)&gdk_event);
                }
    
                break;
            
            case MapNotify:
                {
                    int visible = 1;
                    DBG("videowin map notify");
                    xfmedia_xine_port_send_gui_data(xfx,
                            XINE_GUI_SEND_VIDEOWIN_VISIBLE, &visible);
                }
                
                break;
            
            case UnmapNotify:
                {
                    int visible = 0;
                    DBG("videowin unmap notify");
                    xfmedia_xine_port_send_gui_data(xfx,
                            XINE_GUI_SEND_VIDEOWIN_VISIBLE, &visible);
                }
                
                break;
        }
#ifdef XINE_GUI_SEND_COMPLETION_EVENT  /* deprecated? */
        if(event.type == xfx->priv->completion_event) {
            xfmedia_xine_port_send_gui_data(xfx,
                    XINE_GUI_SEND_COMPLETION_EVENT, &event);
        }
#endif
    }
    
    return TRUE;
}

static gboolean
configure_cb(GtkWidget *widget, GdkEventConfigure *event, gpointer user_data)
{
    XfmediaXine *xfx;

    xfx = XFMEDIA_XINE(user_data);

    xfx->priv->xpos = event->x + GTK_WIDGET(xfx)->allocation.x;
    xfx->priv->ypos = event->y + GTK_WIDGET(xfx)->allocation.y;

    return FALSE;
}

static void
reset_screen_saver(XfmediaXine *xfx)
{
#ifdef HAVE_XTESTEXTENSION
    gboolean do_fake_key = FALSE;
    
#ifdef HAVE_XSCREENSAVER_EXTENSION
    if(xfx->priv->have_xscr) {
        gint idle_time;
        GdkScreen *gscreen = gtk_widget_get_screen(GTK_WIDGET(xfx));
        GdkWindow *rootwin = gdk_screen_get_root_window(gscreen);
        idle_time = xfmedia_xine_get_x_idle_time(GDK_DISPLAY(),
                                                 GDK_WINDOW_XID(rootwin));
        if(idle_time == -1 || idle_time > 10000)
            do_fake_key = TRUE;
    } else
#else
    {
        do_fake_key = TRUE;
    }
#endif  /* HAVE_XSCREENSAVER_EXTENSION */
    
    if(do_fake_key && xfx->priv->have_xtest) {
        XTestFakeKeyEvent(GDK_DISPLAY(), xfx->priv->xtest_keycode,
                          True, gtk_get_current_event_time());
        XTestFakeKeyEvent(GDK_DISPLAY(), xfx->priv->xtest_keycode,
                          False, gtk_get_current_event_time());
        XSync(GDK_DISPLAY(), False);
    } else if(do_fake_key)
#endif  /* HAVE_XTESTEXTENSION */
    {
        XResetScreenSaver(GDK_DISPLAY());
    }
}

static gboolean
disable_screensaver_timeout(gpointer data)
{
    XfmediaXine *xfx = XFMEDIA_XINE(data);
    
    /* FIXME: need a way to enable/disable the screensaver auto-disabler now,
       so XfmediaVideoWindow can call it when we're in fullscreen mode.  or
       maybe just leave it this way.  can't decide.  */
    if(xfx->priv->video_port
       && xine_get_status(xfx->priv->stream) == XINE_STATUS_PLAY)
    {
        reset_screen_saver(xfx);
    }

    return TRUE;
}

static void
xfmedia_xine_realize(GtkWidget *widget)
{
    XfmediaXine *xfx;
    gint screen;
    gulong black_pixel;
    GdkScreen *gscreen;
    xine_cfg_entry_t cfgentry;
    gint real_bufsize = -1;
    
    g_return_if_fail(XFMEDIA_IS_XINE(widget));
    
    xfx = XFMEDIA_XINE(widget);
    
    /* set realized flag */
    GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);
    
    /* load audio driver */
    DBG("creating audio driver");
    xfx->priv->audio_port = load_audio_out_driver(xfx, NULL);
    if(!xfx->priv->audio_port)
        g_critical("XfmediaXine: Unable to load audio output driver.");

    /* create our own video window */
    gscreen = gtk_widget_get_screen(gtk_widget_get_toplevel(widget));
    screen = gdk_screen_get_number(gscreen);
    xfx->priv->screen = screen;
    black_pixel = BlackPixel(gdk_display, screen);
    
    DBG("creating video XWindow");
    xfx->priv->video_window = XCreateSimpleWindow(GDK_DISPLAY(),
            GDK_WINDOW_XWINDOW(gtk_widget_get_parent_window(widget)), 0, 0,
            widget->allocation.width, widget->allocation.height, 1,
            black_pixel, black_pixel);

    widget->window = gdk_window_foreign_new(xfx->priv->video_window);

    /* track configure events of toplevel window */
    g_signal_connect(gtk_widget_get_toplevel(widget),
            "configure-event", G_CALLBACK(configure_cb), xfx);
    
    DBG("opening a new XDisplay");
    xfx->priv->display = XOpenDisplay(gdk_display_get_name(gtk_widget_get_display(widget)));

    if(!xfx->priv->display) {
        printf("XfmediaXine: XOpenDisplay failed!\n");
        xfx->priv->video_port = xine_open_video_driver(xfx->priv->xine,
                "none", XINE_VISUAL_TYPE_NONE, NULL);
        goto create_stream;
    }
    
    DBG("locking display");
    XLockDisplay(xfx->priv->display);

#ifdef XINE_GUI_SEND_COMPLETION_EVENT
    if(XShmQueryExtension(xfx->priv->display) == True) {
        xfx->priv->completion_event = XShmGetEventBase(xfx->priv->display) + ShmCompletion;
    } else {
        xfx->priv->completion_event = -1;
    }
#endif

    XSelectInput(xfx->priv->display, xfx->priv->video_window,
            StructureNotifyMask | ExposureMask | ButtonPressMask
            | ButtonReleaseMask | PointerMotionMask | SubstructureNotifyMask
            | VisibilityChangeMask | KeyPressMask | KeyReleaseMask);
    
    /* load "none" driver.  it doesn't need a large buffer (saves RAM). */
    if(xine_config_lookup_entry(xfx->priv->xine, "video.num_buffers", &cfgentry)
            && cfgentry.type == XINE_CONFIG_TYPE_NUM)
    {
        DBG("got video buffer size (%d)", cfgentry.num_value);
        real_bufsize = cfgentry.num_value;
        cfgentry.num_value = 5;
        xine_config_update_entry(xfx->priv->xine, &cfgentry);
    }
    xfx->priv->none_video_port = xine_open_video_driver(xfx->priv->xine,
            "none", XINE_VISUAL_TYPE_NONE, NULL);
    if(real_bufsize != -1) {
        cfgentry.num_value = real_bufsize;
        xine_config_update_entry(xfx->priv->xine, &cfgentry);
    }
    
    DBG("unlocking display");
    XUnlockDisplay(xfx->priv->display);
    
    create_stream:
    
    /* create a stream object */
    DBG("creating stream");
    xfx->priv->stream = xine_stream_new(xfx->priv->xine,
            xfx->priv->audio_port, xfx->priv->none_video_port);
    
    /* create xine event handler thread */
    DBG("creating xine event queue");
    xfx->priv->ev_queue = xine_event_new_queue(xfx->priv->stream);
    xine_event_create_listener_thread(xfx->priv->ev_queue,
            event_listener_cb, xfx);
    
    /* install a mainloop listener on the xine display for events */
    xfx->priv->xine_display_ioc = g_io_channel_unix_new(XConnectionNumber(xfx->priv->display));
    g_io_channel_set_encoding(xfx->priv->xine_display_ioc, NULL, NULL);
    g_io_channel_set_buffered(xfx->priv->xine_display_ioc, FALSE);
    g_io_channel_set_close_on_unref(xfx->priv->xine_display_ioc, FALSE);
    xfx->priv->xine_display_src = g_io_add_watch_full(xfx->priv->xine_display_ioc,
            G_PRIORITY_DEFAULT, G_IO_IN, xine_display_event_handler, xfx, NULL);
    
    /* add timeout to disable screen saver */
    xfx->priv->screensaver_timeout = g_timeout_add(4000,
                                                   disable_screensaver_timeout,
                                                   xfx);
}

static void
xfmedia_xine_unrealize(GtkWidget *widget)
{
    XfmediaXine *xfx;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(widget));

    xfx = XFMEDIA_XINE(widget);
    
    g_source_remove(xfx->priv->screensaver_timeout);
    xfx->priv->screensaver_timeout = 0;
    
    g_signal_handlers_disconnect_by_func(gtk_widget_get_toplevel(widget),
                                         G_CALLBACK(configure_cb), xfx);
    
    xine_stop(xfx->priv->stream);
    
    /* destroy the xine display event listener */
    g_source_remove(xfx->priv->xine_display_src);
    xfx->priv->xine_display_src = 0;
    g_io_channel_unref(xfx->priv->xine_display_ioc);
    xfx->priv->xine_display_ioc = NULL;
    
    /* ditch xine event thread */
    xine_event_dispose_queue(xfx->priv->ev_queue);
    /* ditch stream */
    xine_close(xfx->priv->stream);
    xine_dispose(xfx->priv->stream);
    /* and a/v drivers */
    if(xfx->priv->video_port)
        xine_close_video_driver(xfx->priv->xine, xfx->priv->video_port);
    if(xfx->priv->audio_port)
        xine_close_audio_driver(xfx->priv->xine, xfx->priv->audio_port);
    if(xfx->priv->none_video_port)
        xine_close_video_driver(xfx->priv->xine, xfx->priv->none_video_port);
    
    /* Hide all windows */
    if(GTK_WIDGET_MAPPED(widget))
        gtk_widget_unmap(widget);

    GTK_WIDGET_UNSET_FLAGS(widget, GTK_MAPPED);

    /* This destroys widget->window and unsets the realized flag */
    if(GTK_WIDGET_CLASS(parent_class)->unrealize)
        (*GTK_WIDGET_CLASS(parent_class)->unrealize)(widget);
}


GtkWidget *
xfmedia_xine_new(const gchar *video_driver_id, const gchar *audio_driver_id)
{
    XfmediaXine *xfx = g_object_new(xfmedia_xine_get_type(), NULL);
    
    if(video_driver_id && *video_driver_id)
        xfx->priv->video_driver_id = strdup(video_driver_id);
    else
        xfx->priv->video_driver_id = NULL;

    if(audio_driver_id && audio_driver_id)
        xfx->priv->audio_driver_id = strdup(audio_driver_id);
    else
        xfx->priv->audio_driver_id = NULL;
    
    return GTK_WIDGET(xfx);
}

static void
xfmedia_xine_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
    g_return_if_fail(widget != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(widget));

    widget->allocation = *allocation;

    if(GTK_WIDGET_REALIZED(widget)) {
        gdk_window_move_resize(widget->window,
            allocation->x, allocation->y, allocation->width, allocation->height);
    }
}

gboolean
xfmedia_xine_open(XfmediaXine *xfx, const gchar *mrl)
{
    gint ret, has_video;
    
    g_return_val_if_fail(xfx != NULL, FALSE);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), FALSE);
    g_return_val_if_fail(xfx->priv->xine != NULL, FALSE);

    if(xfx->priv->mrl) {
        xine_stop(xfx->priv->stream);
        xine_close(xfx->priv->stream);
        g_free(xfx->priv->mrl);
        xfx->priv->mrl = NULL;
    }
    
    DBG("about to call xine_open()");
    ret = xine_open(xfx->priv->stream, mrl);
    DBG("xine_open() returned %d", ret);
    if(ret) {
        xfx->priv->mrl = g_strdup(mrl);
    
        has_video = xine_get_stream_info(xfx->priv->stream,
                XINE_STREAM_INFO_HAS_VIDEO);
        if((has_video || xfx->priv->vis_plugin_id) && !xfx->priv->video_port) {
            DBG("creating real vid out driver");
            XLockDisplay(xfx->priv->display);
            xfx->priv->video_port = load_video_out_driver(xfx, NULL);
            XUnlockDisplay(xfx->priv->display);
            if(xfx->priv->video_port) {
                xine_post_out_t *vid_src = xine_get_video_source(xfx->priv->stream);
                xine_post_wire_video_port(vid_src, xfx->priv->video_port);
                DBG("rewired stream to real vid out driver");
            }
        } else if(!has_video && xfx->priv->video_port && !xfx->priv->vis_plugin_id) {
            xine_post_out_t *vid_src = xine_get_video_source(xfx->priv->stream);
            DBG("wiring stream to none vid out driver and destroying real driver");
            xine_post_wire_video_port(vid_src, xfx->priv->none_video_port);
            xine_close_video_driver(xfx->priv->xine, xfx->priv->video_port);
            xfx->priv->video_port = NULL;
        }
    } else if(xfx->priv->video_port) {
        xine_post_out_t *vid_src = xine_get_video_source(xfx->priv->stream);
        DBG("wiring stream to none vid out driver and destroying real driver");
        xine_post_wire_video_port(vid_src, xfx->priv->none_video_port);
        xine_close_video_driver(xfx->priv->xine, xfx->priv->video_port);
        xfx->priv->video_port = NULL;
    }
    
#ifdef HAVE_SAFE_XINE_PLUGINS_GC
    xine_plugins_garbage_collector(xfx->priv->xine);
#endif

    return ret ? TRUE : FALSE;
}

gboolean
xfmedia_xine_play(XfmediaXine *xfx, gint pos, gint start_time)
{
    gint res;

    g_return_val_if_fail(xfx != NULL, FALSE);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), FALSE);
    g_return_val_if_fail(xfx->priv->xine != NULL, FALSE);

    /* visualization */
    if(!xine_get_stream_info(xfx->priv->stream, XINE_STREAM_INFO_HAS_VIDEO)) {
        if(!xfx->priv->vis_plugin && xfx->priv->vis_plugin_id) {
            xine_post_out_t *audio_source;
            xine_post_in_t *input;
            xine_cfg_entry_t cfg;

            xfx->priv->vis_plugin = xine_post_init(xfx->priv->xine, xfx->priv->vis_plugin_id, 0,
                &xfx->priv->audio_port, &xfx->priv->video_port);
            
            xine_config_lookup_entry(xfx->priv->xine, "post.goom_width", &cfg);
            cfg.num_value = DEFAULT_VIS_WIDTH;
            xine_config_update_entry(xfx->priv->xine, &cfg);
            
            xine_config_lookup_entry(xfx->priv->xine, "post.goom_height", &cfg);
            cfg.num_value = DEFAULT_VIS_HEIGHT;
            xine_config_update_entry(xfx->priv->xine, &cfg);
            
            audio_source = xine_get_audio_source(xfx->priv->stream);
            input = xine_post_input(xfx->priv->vis_plugin, "audio in");

            xine_post_wire(audio_source, input);
        }
    } else {
        if(xfx->priv->vis_plugin) {
            xine_post_out_t *pp;

            pp = xine_get_audio_source(xfx->priv->stream);
            xine_post_wire_audio_port(pp, xfx->priv->audio_port);
            xine_post_dispose(xfx->priv->xine, xfx->priv->vis_plugin);
            xfx->priv->vis_plugin = NULL;
        }
    }
    
    res = xine_play(xfx->priv->stream, pos, start_time);

    return res;
}

gboolean
xfmedia_xine_trick_mode(XfmediaXine * xfx, gint mode, gint value)
{

    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_trick_mode(xfx->priv->stream, mode, value);
}

gboolean
xfmedia_xine_get_pos_length(XfmediaXine *xfx, gint *pos_stream,
    gint *pos_time, gint *length_time)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_get_pos_length(xfx->priv->stream, pos_stream, pos_time,
                               length_time);
}

void
xfmedia_xine_stop(XfmediaXine *xfx)
{
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->stream != NULL);

    xine_stop(xfx->priv->stream);
}

gint
xfmedia_xine_get_error(XfmediaXine *xfx)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_get_error(xfx->priv->stream);
}

gint
xfmedia_xine_get_status(XfmediaXine *xfx)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_get_status(xfx->priv->stream);
}

void
xfmedia_xine_set_param(XfmediaXine *xfx, gint param, gint value)
{
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->stream != NULL);

    xine_set_param(xfx->priv->stream, param, value);
}

gint
xfmedia_xine_get_param(XfmediaXine *xfx, gint param)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_get_param(xfx->priv->stream, param);
}

void
xfmedia_xine_engine_set_param(XfmediaXine *xfx, gint param, gint value)
{
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->stream != NULL);

    xine_engine_set_param(xfx->priv->xine, param, value);
}

gint
xfmedia_xine_engine_get_param(XfmediaXine *xfx, gint param)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_engine_get_param(xfx->priv->xine, param);
}

gint
xfmedia_xine_get_audio_lang(XfmediaXine *xfx, gint channel, gchar * lang)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_get_audio_lang(xfx->priv->stream, channel, lang);
}

gint
xfmedia_xine_get_spu_lang(XfmediaXine *xfx, gint channel, gchar * lang)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_get_spu_lang(xfx->priv->stream, channel, lang);
}

gint
xfmedia_xine_get_stream_info(XfmediaXine *xfx, gint info)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_get_stream_info(xfx->priv->stream, info);
}

G_CONST_RETURN gchar *
xfmedia_xine_get_meta_info(XfmediaXine *xfx, gint info)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_get_meta_info(xfx->priv->stream, info);
}

G_CONST_RETURN gchar *
xfmedia_xine_get_current_mrl(XfmediaXine *xfx)
{
    g_return_val_if_fail(xfx, NULL);
    
    return xfx->priv->mrl;
}

gint
xfmedia_xine_get_current_frame(XfmediaXine *xfx, gint *width, gint *height,
        gint *ratio_code, gint *format, uint8_t *img)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->stream != NULL, 0);

    return xine_get_current_frame(xfx->priv->stream, width, height,
                                  ratio_code, format, img);
}

gint
xfmedia_xine_get_log_section_count(XfmediaXine *xfx)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->xine != NULL, 0);

    return xine_get_log_section_count(xfx->priv->xine);
}

gchar **
xfmedia_xine_get_log_names(XfmediaXine *xfx)
{
    g_return_val_if_fail(xfx != NULL, NULL);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    g_return_val_if_fail(xfx->priv->xine != NULL, NULL);

    return (gchar **)xine_get_log_names(xfx->priv->xine);
}

gchar **
xfmedia_xine_get_log(XfmediaXine *xfx, gint buf)
{
    g_return_val_if_fail(xfx != NULL, NULL);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    g_return_val_if_fail(xfx->priv->xine != NULL, NULL);

    return (gchar **)xine_get_log(xfx->priv->xine, buf);
}

void
xfmedia_xine_register_log_cb(XfmediaXine *xfx, xine_log_cb_t cb, void *user_data)
{
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->xine != NULL);

    return xine_register_log_cb(xfx->priv->xine, cb, user_data);
}

gchar **
xfmedia_xine_get_browsable_input_plugin_ids(XfmediaXine *xfx)
{

    g_return_val_if_fail(xfx != NULL, NULL);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    g_return_val_if_fail(xfx->priv->xine != NULL, NULL);

    return (gchar **)xine_get_browsable_input_plugin_ids(xfx->priv->xine);
}

xine_mrl_t **
xfmedia_xine_get_browse_mrls(XfmediaXine *xfx, const gchar *plugin_id,
        const gchar *start_mrl, gint *num_mrls)
{
    g_return_val_if_fail(xfx != NULL, NULL);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    g_return_val_if_fail(xfx->priv->xine != NULL, NULL);

    return (xine_mrl_t **)xine_get_browse_mrls(xfx->priv->xine, plugin_id,
                                               start_mrl, num_mrls);
}


GList *
xfmedia_xine_get_autoplay_input_plugin_ids(XfmediaXine *xfx)
{
    GList *list = NULL;
    const gchar * const *plugins;
    gint i;
    
    g_return_val_if_fail(xfx != NULL, NULL);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    g_return_val_if_fail(xfx->priv->xine != NULL, NULL);
    
    plugins = xine_get_autoplay_input_plugin_ids(xfx->priv->xine);
    if(!plugins)
        return NULL;
    
    for(i = 0; plugins[i]; i++) {
        list = g_list_insert_sorted(list, (gpointer)plugins[i],
                (GCompareFunc)g_ascii_strcasecmp);
    }

    return list;
}

GList *
xfmedia_xine_get_autoplay_mrls(XfmediaXine *xfx, const gchar *plugin_id)
{
    gchar **mrls;
    GList *list = NULL;
    gint i, num_mrls;
    
    g_return_val_if_fail(xfx != NULL, NULL);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    g_return_val_if_fail(xfx->priv->xine != NULL, NULL);
    
    mrls = xine_get_autoplay_mrls(xfx->priv->xine, plugin_id, &num_mrls);
    if(!mrls || num_mrls == 0)
        return NULL;
    
    for(i = 0; i < num_mrls; i++)
        list = g_list_prepend(list, mrls[i]);

    return g_list_reverse(list);
}

gchar *
xfmedia_xine_get_file_extensions(XfmediaXine *xfx)
{
    g_return_val_if_fail(xfx != NULL, NULL);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    g_return_val_if_fail(xfx->priv->xine != NULL, NULL);

    return (gchar *)xine_get_file_extensions(xfx->priv->xine);
}

gchar *
xfmedia_xine_get_mime_types(XfmediaXine *xfx)
{
    g_return_val_if_fail(xfx != NULL, NULL);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    g_return_val_if_fail(xfx->priv->xine != NULL, NULL);

    return (gchar *)xine_get_mime_types(xfx->priv->xine);
}

const gchar *
xfmedia_xine_config_register_string(XfmediaXine *xfx, const gchar *key,
        const gchar *def_value, const gchar *description,
        const gchar *help, gint exp_level, xine_config_cb_t changed_cb,
        void *cb_data)
{

    g_return_val_if_fail(xfx != NULL, NULL);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    g_return_val_if_fail(xfx->priv->xine != NULL, NULL);

    return (gchar *)xine_config_register_string(xfx->priv->xine, key,
            def_value, description, help, exp_level, changed_cb, cb_data);
}

gint
xfmedia_xine_config_register_range(XfmediaXine *xfx, const gchar *key, gint def_value,
        gint min, gint max, const gchar *description, const gchar *help,
        gint exp_level, xine_config_cb_t changed_cb, void *cb_data)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->xine != NULL, 0);

    return xine_config_register_range(xfx->priv->xine, key, def_value,
            min, max, description, help, exp_level, changed_cb, cb_data);
}

gint
xfmedia_xine_config_register_enum(XfmediaXine *xfx, const gchar *key, gint def_value,
        gchar **values, const gchar *description, const gchar *help,
        gint exp_level, xine_config_cb_t changed_cb, void *cb_data)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->xine != NULL, 0);

    return xine_config_register_enum(xfx->priv->xine, key, def_value, values,
            description, help, exp_level, changed_cb, cb_data);
}

gint
xfmedia_xine_config_register_num(XfmediaXine *xfx, const gchar *key, gint def_value,
        const gchar *description, const gchar *help, gint exp_level,
        xine_config_cb_t changed_cb, void *cb_data)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->xine != NULL, 0);

    return xine_config_register_num(xfx->priv->xine, key, def_value,
            description, help, exp_level, changed_cb, cb_data);
}

gint
xfmedia_xine_config_register_bool(XfmediaXine *xfx, const gchar *key, gint def_value,
        const gchar *description, const gchar *help, gint exp_level,
        xine_config_cb_t changed_cb, void *cb_data)
{

    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->xine != NULL, 0);

    return xine_config_register_bool(xfx->priv->xine, key, def_value,
            description, help, exp_level, changed_cb, cb_data);
}

int
xfmedia_xine_config_get_first_entry(XfmediaXine *xfx, xine_cfg_entry_t *entry)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->xine != NULL, 0);

    return xine_config_get_first_entry(xfx->priv->xine, entry);
}

int
xfmedia_xine_config_get_next_entry(XfmediaXine *xfx, xine_cfg_entry_t *entry)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->xine != NULL, 0);

    return xine_config_get_next_entry(xfx->priv->xine, entry);
}

int
xfmedia_xine_config_lookup_entry(XfmediaXine *xfx, const gchar *key,
        xine_cfg_entry_t *entry)
{
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->xine != NULL, 0);

    return xine_config_lookup_entry(xfx->priv->xine, key, entry);
}

void
xfmedia_xine_config_update_entry(XfmediaXine *xfx, xine_cfg_entry_t *entry)
{
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->xine != NULL);

    xine_config_update_entry(xfx->priv->xine, entry);
}

void
xfmedia_xine_config_load(XfmediaXine *xfx, const gchar *cfg_filename)
{
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->xine != NULL);

    xine_config_load(xfx->priv->xine, cfg_filename);
}

void
xfmedia_xine_config_save(XfmediaXine *xfx, const gchar *cfg_filename)
{
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->xine != NULL);

    xine_config_save(xfx->priv->xine, cfg_filename);
}

void
xfmedia_xine_config_reset(XfmediaXine *xfx)
{
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->xine != NULL);

    xine_config_reset(xfx->priv->xine);
}

void
xfmedia_xine_event_send(XfmediaXine *xfx, const xine_event_t *event)
{
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->stream != NULL);

    xine_event_send(xfx->priv->stream, event);
}

void
xfmedia_xine_set_resize_factor(XfmediaXine *xfx, gdouble factor)
{                                        /* 0.0 => don't resize */
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));
    g_return_if_fail(xfx->priv->xine != NULL);

    xfx->priv->resize_factor = factor;
}

void
xfmedia_xine_set_vis(XfmediaXine *xfx, const gchar *id)
{                              /* NULL to disable */
    g_return_if_fail(xfx != NULL);
    g_return_if_fail(XFMEDIA_IS_XINE(xfx));

    if(xfx->priv->vis_plugin_id)
        g_free(xfx->priv->vis_plugin_id);
    if(id)
        xfx->priv->vis_plugin_id = g_strdup(id);
    else
        xfx->priv->vis_plugin_id = NULL;
    
    if(xfx->priv->stream) {
        if(xfx->priv->vis_plugin) {
            xine_post_out_t *pp;

            pp = xine_get_audio_source(xfx->priv->stream);
            xine_post_wire_audio_port(pp, xfx->priv->audio_port);
            xine_post_dispose(xfx->priv->xine, xfx->priv->vis_plugin);
            xfx->priv->vis_plugin = NULL;
        }

        if(xfx->priv->vis_plugin_id) {
            xine_post_out_t *audio_source;
            xine_post_in_t *input;
            xine_cfg_entry_t cfg;
            
            if(!xfx->priv->video_port) {
                gboolean restart_after = (xine_get_status(xfx->priv->stream) == XINE_STATUS_PLAY);
                gint ps, pt, lt;
                
                if(restart_after) {
                    xine_get_pos_length(xfx->priv->stream, &ps, &pt, &lt);
                    xine_stop(xfx->priv->stream);
                }
                
                XLockDisplay(xfx->priv->display);
                xfx->priv->video_port = load_video_out_driver(xfx, NULL);
                XUnlockDisplay(xfx->priv->display);
                if(xfx->priv->video_port) {
                    xine_post_out_t *vid_src = xine_get_video_source(xfx->priv->stream);
                    xine_post_wire_video_port(vid_src, xfx->priv->video_port);
                    DBG("rewired stream to real vid out driver");
                }
                
                if(restart_after)
                    xine_play(xfx->priv->stream, ps, 0);
            }

            xfx->priv->vis_plugin = xine_post_init(xfx->priv->xine, 
                    xfx->priv->vis_plugin_id, 0,
                    &xfx->priv->audio_port, &xfx->priv->video_port);
            
            xine_config_lookup_entry(xfx->priv->xine, "post.goom_width", &cfg);
            cfg.num_value = DEFAULT_VIS_WIDTH;
            xine_config_update_entry(xfx->priv->xine, &cfg);
            
            xine_config_lookup_entry(xfx->priv->xine, "post.goom_height", &cfg);
            cfg.num_value = DEFAULT_VIS_HEIGHT;
            xine_config_update_entry(xfx->priv->xine, &cfg);
            
            audio_source = xine_get_audio_source(xfx->priv->stream);
            input = xine_post_input(xfx->priv->vis_plugin, "audio in");
            
            xine_post_wire(audio_source, input);
        }
    }
}

GList *
xfmedia_xine_list_post_plugins_typed(XfmediaXine *xfx, gint type)
{
    const gchar *const *plugins;
    GList *list = NULL;
    gint i;
    
    g_return_val_if_fail(xfx != NULL, 0);
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), 0);
    g_return_val_if_fail(xfx->priv->xine != NULL, 0);
    
    plugins = xine_list_post_plugins_typed(xfx->priv->xine, type);
    if(!plugins)
        return NULL;
    
    for(i = 0; plugins[i]; i++)
        list = g_list_append(list, (gpointer)plugins[i]);

    return list;
}

typedef enum
{
    PTYPE_AUDIO = 0,
    PTYPE_VIDEO,
} PTypePriv;

static GList *
xfmedia_xine_list_av_plugins(XfmediaXine *xfx, PTypePriv ptype)
{
    const char *const *plugins;
    GList *plugin_list = NULL;
    gint i;
    
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    
    switch(ptype) {
        case PTYPE_AUDIO:
            plugins = xine_list_audio_output_plugins(xfx->priv->xine);
            break;
        
        case PTYPE_VIDEO:
            plugins = xine_list_video_output_plugins(xfx->priv->xine);
            break;
        
        default:
            g_error("PTypePriv is invalid");
            return NULL; /* possibly NOTREACHED */
    }
    
    if(!plugins)
        return NULL;
    
    for(i = 0; plugins[i]; i++)
        plugin_list = g_list_prepend(plugin_list, (gpointer)plugins[i]);
    plugin_list = g_list_reverse(plugin_list);
    plugin_list = g_list_prepend(plugin_list, "auto");
    
    return plugin_list;
}

gboolean
xfmedia_xine_set_audio_output_plugin(XfmediaXine *xfx, const gchar *plugin_id)
{
    gint status, speed;
    gint pos_stream, pos_time = -1, length_time;
    xine_audio_port_t *old_port, *new_port;
    xine_post_out_t *aud_src;
    
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx) && plugin_id, FALSE);
    
    if(!strcmp(plugin_id, xfx->priv->audio_driver_id))
        return TRUE;
    
    new_port = load_audio_out_driver(xfx, plugin_id);
    if(!new_port)
        return FALSE;
    
    g_free(xfx->priv->audio_driver_id);
    xfx->priv->audio_driver_id = g_strdup(plugin_id);
    
    status = xfmedia_xine_get_status(xfx);
    speed = xfmedia_xine_get_param(xfx, XINE_PARAM_SPEED);
    if(status == XINE_STATUS_PLAY) {
        xfmedia_xine_get_pos_length(xfx, &pos_stream, &pos_time, &length_time);
        xfmedia_xine_stop(xfx);
    }
    
    old_port = xfx->priv->audio_port;
    xfx->priv->audio_port = new_port;
    
    aud_src = xine_get_audio_source(xfx->priv->stream);
    xine_post_wire_audio_port(aud_src, xfx->priv->audio_port);
    
    xine_close_audio_driver(xfx->priv->xine, old_port);
    
    if(status == XINE_STATUS_PLAY) {
        xfmedia_xine_play(xfx, pos_stream, 0);
        xfmedia_xine_set_param(xfx, XINE_PARAM_SPEED, speed);
    }
    
    return TRUE;
}

gboolean
xfmedia_xine_set_video_output_plugin(XfmediaXine *xfx, const gchar *plugin_id)
{
    gint status, speed;
    gint pos_stream, pos_time = -1, length_time;
    xine_video_port_t *new_port = NULL;
    
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx) && plugin_id, FALSE);
    
    if(!strcmp(plugin_id, xfx->priv->video_driver_id))
        return TRUE;
    
    if(xfx->priv->video_port) {
        XLockDisplay(xfx->priv->display);
        new_port = load_video_out_driver(xfx, plugin_id);
        if(!new_port) {
            XUnlockDisplay(xfx->priv->display);
            return FALSE;
        }
        XUnlockDisplay(xfx->priv->display);
    }
    
    g_free(xfx->priv->video_driver_id);
    xfx->priv->video_driver_id = g_strdup(plugin_id);
    
    status = xfmedia_xine_get_status(xfx);
    speed = xfmedia_xine_get_param(xfx, XINE_PARAM_SPEED);
    if(status == XINE_STATUS_PLAY) {
        xfmedia_xine_get_pos_length(xfx, &pos_stream, &pos_time, &length_time);
        xfmedia_xine_stop(xfx);
    }
    
    if(xfx->priv->video_port) {
        xine_post_out_t *vid_src;
        xine_video_port_t *old_port;
        
        XLockDisplay(xfx->priv->display);
        
        old_port = xfx->priv->video_port;
        xfx->priv->video_port = new_port;
        
        vid_src = xine_get_video_source(xfx->priv->stream);
        xine_post_wire_video_port(vid_src, xfx->priv->video_port);
        DBG("rewired stream to real vid out driver");
        
        xine_close_video_driver(xfx->priv->xine, old_port);
        
        XUnlockDisplay(xfx->priv->display);
    }
    
    if(status == XINE_STATUS_PLAY) {
        xfmedia_xine_play(xfx, pos_stream, 0);
        xfmedia_xine_set_param(xfx, XINE_PARAM_SPEED, speed);
    }
    
    return TRUE;
}

GList *
xfmedia_xine_list_audio_output_plugins(XfmediaXine *xfx)
{
    return xfmedia_xine_list_av_plugins(xfx, PTYPE_AUDIO);
}

GList *
xfmedia_xine_list_video_output_plugins(XfmediaXine *xfx)
{
    return xfmedia_xine_list_av_plugins(xfx, PTYPE_VIDEO);
}

xine_t *
xfmedia_xine_get_raw_xine_engine(XfmediaXine *xfx)
{
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    return xfx->priv->xine;
}

xine_stream_t *
xfmedia_xine_get_raw_stream(XfmediaXine *xfx)
{
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    return xfx->priv->stream;
}

xine_audio_port_t *
xfmedia_xine_get_raw_audio_port(XfmediaXine *xfx)
{
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    return xfx->priv->audio_port;
}

xine_video_port_t *
xfmedia_xine_get_raw_video_port(XfmediaXine *xfx)
{
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    return xfx->priv->video_port;
}

xine_post_t *
xfmedia_xine_get_raw_post_plugin(XfmediaXine *xfx)
{
    g_return_val_if_fail(XFMEDIA_IS_XINE(xfx), NULL);
    return xfx->priv->vis_plugin;
}
