#include "mysql_backend.h"

#if HAVE_LIBMYSQL

#include <syslog.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

MYSQL *mysql_con = NULL;
char mysql_connected = 0; // bool
char mysql_available = 0; // bool

struct t_sqlbe_config {
	char *db_hostname; // hostname of mysql server
	char *db_user; // connect as user
	char *db_pass; // password for 'user'
	char *db_name; // the database to use (e.g. mysql: 'use <db_name>')
} *sqlbe_config = NULL;

char sqlbe_query(const char *query_str)
{
	int tries;

	if ( !mysql_available ) return 1;
	if ( !mysql_connected ) sqlbe_reconnect();

	for ( tries = 2; tries--; )
	{
		if ( !mysql_query(mysql_con, query_str) )
			return 0; // success, finish.
		if ( mysql_errno(mysql_con) != CR_SERVER_GONE_ERROR )
		{
			#ifdef DEBUG
				syslog(LOG_WARNING, "mysql_backend: query '%s' failed: %s",
						query_str, mysql_error(mysql_con));
			#else
				syslog(LOG_WARNING, "mysql_backend: query failed: %s",
						mysql_error(mysql_con));
			#endif
			return 1; // failed
		}
		sqlbe_reconnect();
	}
	syslog(LOG_WARNING, "mysql_backend: query failed: %s",
					mysql_error(mysql_con));
	return 1;
}

char sqlbe_connected() { return mysql_connected; }
char sqlbe_available() { return mysql_available; }
char sqlbe_initialized() { return sqlbe_config != NULL; }

char sqlbe_connect()
{
	#define CONS_WARN(s)	if ( !count ) fprintf(stderr, s);
	#define CONS_WARN2(s, a)	if ( !count ) fprintf(stderr, s, a);

	static int count = 0;

	if ( !count )
	{
		mysql_available = sqlbe_config && *sqlbe_config->db_hostname;

		if ( !mysql_available )
		{
			fprintf(stderr, "MySQL not configured or not initialized!");
			return 1;
		}
	}

	if ( mysql_connected )
	{
		if ( mysql_con )
		{
			mysql_close(mysql_con);
			mysql_con = NULL;
		}
		mysql_connected = 0;
	}

	if ( !(mysql_con = mysql_init(NULL)) )
	{
		CONS_WARN("mysql_init(.) failed! MySQL logging not available.\n")
		count++;
		return 1;
	}
	
//  	mysql_options(mysql_con,MYSQL_READ_DEFAULT_GROUP,LCS_MYSQL_DEFGROUP);

// I get a segfault when using mysql_real_connect of libmysqlclient 10
//	if ( !mysql_real_connect(mysql_con, server->db_hostname, server->db_user, server->db_pass, server->db_name, 0, NULL, 0) )
	if ( !mysql_connect(mysql_con, sqlbe_config->db_hostname, sqlbe_config->db_user, sqlbe_config->db_pass) )
	{
		CONS_WARN2("Could not init the database: %s\n", mysql_error(mysql_con))
		count++;
		return 1;
	}

	if ( mysql_select_db(mysql_con, sqlbe_config->db_name) )
	{
		CONS_WARN2("mysql_select_db('%s') failed!", sqlbe_config->db_name)
		syslog(LOG_ERR, "mysql_select_db('%s') failed! MySQL backend switched off.", sqlbe_config->db_name);
		sqlbe_cleanup();
		mysql_available = 0;
	}

	CONS_WARN("MySQL logging initialised.\n")
	mysql_connected++;

	count++;
	return !mysql_connected;
}

char sqlbe_reconnect()
{
	if ( mysql_connected )
	{
		mysql_close(mysql_con);
		mysql_con = NULL;
		mysql_connected = 0;
	}
	return sqlbe_connect();
}

void sqlbe_cleanup()
{
	if ( mysql_connected )
	{
		mysql_close(mysql_con);
		mysql_con = NULL;
		mysql_connected = 0;
	}
	if ( sqlbe_config )
	{
		if ( sqlbe_config->db_hostname ) free(sqlbe_config->db_hostname);
		if ( sqlbe_config->db_user ) free(sqlbe_config->db_user);
		if ( sqlbe_config->db_pass ) free(sqlbe_config->db_pass);
		if ( sqlbe_config->db_name ) free(sqlbe_config->db_name);
		free(sqlbe_config);
		sqlbe_config = NULL;
	}
}

char sqlbe_init(
		const char *hostname,
		const char *user,
		const char *pass,
		const char *dbname
	)
{
	char init = 1;
	if ( sqlbe_config ) sqlbe_cleanup();
	sqlbe_config = malloc(sizeof(struct t_sqlbe_config));
	if ( !sqlbe_config ) return 1;
	init = init && ( sqlbe_config->db_hostname = strdup(hostname) );
	init = init && ( sqlbe_config->db_user = strdup(user) );
	init = init && ( sqlbe_config->db_pass = strdup(pass) );
	init = init && ( sqlbe_config->db_name = strdup(dbname) );
	if ( init ) return 0; // success
	sqlbe_cleanup();
	return 1; // err
}

#endif // HAVE_LIBMYSQL
