/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2005 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * utils.c - assorted utilities
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <string.h>
#include <gtk/gtk.h>

#if HAVE_SYS_IPC_H
#include <sys/ipc.h>
#include <sys/msg.h>
#endif

#if HAVE_SYS_SHM_H
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

#include "types.h"
#include "log.h"
#include "utils.h"
#include "preferences.h"
#include "support.h"
#include "history.h"
#include "dxcc.h"

#if WANT_HAMLIB
#	include <hamlib/rig.h>
# include "hamlib-utils.h"
#endif

gint statusbartimer;
extern GtkWidget *mainwindow;
extern GtkWidget *keyerwindow;
extern GtkWidget *b4dialog;
extern preferencestype preferences;
extern statetype state;
extern remotetype remote;
extern gchar **qso;
extern gchar *xlogdir;
extern gint remotetimer;
extern gint clocktimer, savetimer;
extern glong msgid;
extern void *shareCall;
extern GList *logwindowlist;
extern GtkUIManager *ui_manager;

/* calculate location string from latitude and longitude information,
 * returned value has to be freed */
gchar *
setlocation (gdouble lat, gint NS, gdouble lon, gint EW)
{
	gchar *location, *ns, *ew;

	if (NS == 1)
		ns = g_strdup (_("N"));
	else
		ns = g_strdup (_("S"));
	if (EW == 1)
		ew = g_strdup (_("E"));
	else
		ew = g_strdup (_("W"));
	location = g_strdup_printf ("%3.2f%s%3.2f%s", lat, ns, lon, ew);
	g_free (ns);
	g_free (ew);

	return (location);
}

/* update the menu items of an optionmenu */
void
makebandoptionmenu (gchar *bands)
{
	GtkWidget *bandhbox2, *bandoptionmenu, *old;
	gchar **split;
	gint index = 0;

	old = g_object_get_data (G_OBJECT (mainwindow), "bandoptionmenu");
	if (old) gtk_widget_destroy (old);

	bandoptionmenu = gtk_combo_box_new_text ();
	gtk_widget_show (bandoptionmenu);
	bandhbox2 = lookup_widget (mainwindow, "bandhbox2");
	gtk_box_pack_start (GTK_BOX (bandhbox2), bandoptionmenu, TRUE, TRUE, 0);

	split = g_strsplit (bands, ",", -1);
	for (;;)
	{
		if (split[index] == NULL)
			break;
		gtk_combo_box_append_text (GTK_COMBO_BOX(bandoptionmenu), split[index]);
		index++;
	}
	g_strfreev (split);
	gtk_combo_box_set_active (GTK_COMBO_BOX (bandoptionmenu),
		preferences.bandoptionmenu);
	GLADE_HOOKUP_OBJECT (mainwindow, bandoptionmenu, "bandoptionmenu");
}

void
makemodeoptionmenu (gchar *modes)
{
	GtkWidget *modehbox2, *modeoptionmenu, *old;
	gchar **split;
	gint index = 0;

	old = g_object_get_data (G_OBJECT (mainwindow), "modeoptionmenu");
	if (old) gtk_widget_destroy (old);

	modeoptionmenu = gtk_combo_box_new_text ();
	gtk_widget_show (modeoptionmenu);
	modehbox2 = lookup_widget (mainwindow, "modehbox2");
	gtk_box_pack_start (GTK_BOX (modehbox2), modeoptionmenu, TRUE, TRUE, 0);

	split = g_strsplit (modes, ",", -1);
	for (;;)
	{
		if (split[index] == NULL)
			break;
		gtk_combo_box_append_text (GTK_COMBO_BOX(modeoptionmenu), split[index]);
		index++;
	}
	g_strfreev (split);
	gtk_combo_box_set_active (GTK_COMBO_BOX (modeoptionmenu),
		preferences.modeoptionmenu);
	GLADE_HOOKUP_OBJECT (mainwindow, modeoptionmenu, "modeoptionmenu");
}

/* removing leading and trailing whitespaces from an array of strings */
void
deletespaces (gchar ** split)
{
	gint index = 0;

	for (;;)
	{
		if (split[index] == NULL)
			break;
		g_strstrip (split[index]);
		index++;
	}
}

/* clear statusbar */
static gint
statusbar_timeout (gpointer data)
{
	GtkWidget *statusbar;

	statusbar = lookup_widget (mainwindow, "statusbar");
	gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
	gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, _("Ready."));
	g_source_remove (statusbartimer);
	state.statustimer = FALSE;
	return FALSE;
}

/* change the statusbar */
void
update_statusbar (gchar * string)
{
	GtkWidget *statusbar;

	statusbar = lookup_widget (mainwindow, "statusbar");
	gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
	gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, string);
	if (state.statustimer)
		g_source_remove (statusbartimer);
	statusbartimer = g_timeout_add (30000, statusbar_timeout, NULL);
	state.statustimer = TRUE;
}

/* free mem in state struct */
static void
freestate (void)
{
	g_free (state.mylocation);
	g_free (state.rigrst);
	g_free (state.searchstr);
}

/* free mem in prefs struct */
static void
freeprefs (void)
{
	g_free (preferences.modes);
	g_free (preferences.bands);
	g_free (preferences.device);
	g_free (preferences.rigconf);
	g_free (preferences.logfont);
	g_free (preferences.savedir);
	g_free (preferences.backupdir);
	g_free (preferences.logstoload);
	g_free (preferences.locator);
	g_free (preferences.freefield1);
	g_free (preferences.freefield2);
	g_free (preferences.callsign);
	g_free (preferences.defaultmhz);
	g_free (preferences.defaultmode);
	g_free (preferences.defaulttxrst);
	g_free (preferences.defaultrxrst);
	g_free (preferences.defaultpower);
	g_free (preferences.defaultfreefield1);
	g_free (preferences.defaultfreefield2);
	g_free (preferences.defaultremarks);
	g_free (preferences.b4columns);
	g_free (preferences.logcwidths);
	g_free (preferences.saveastsv);
	g_free (preferences.cwf1);
	g_free (preferences.cwf2);
	g_free (preferences.cwf3);
	g_free (preferences.cwf4);
	g_free (preferences.cwf5);
	g_free (preferences.cwf6);
	g_free (preferences.cwf7);
	g_free (preferences.cwf8);
	g_free (preferences.cwf9);
	g_free (preferences.cwf10);
	g_free (preferences.cwf11);
	g_free (preferences.cwf12);
}

/* free mem in remote struct */
static void
freeremote (void)
{
	g_free (remote.program);
}

/* cleanup of variables, used at exit */
void
save_windowsize_and_cleanup (void)
{
	gint i;
	GtkWidget *bandoptionmenu, *modeoptionmenu, *handlebox, *hpaned;

	/* free the tables array */
	cleanup_dxcc ();

	/* free the qso array */
	for (i = 0; i < QSO_FIELDS; i++)
		g_free (qso[i]);
	g_free (qso);

	g_list_free (logwindowlist);

	gtk_window_get_size (GTK_WINDOW(mainwindow), 
		&preferences.width, &preferences.height);
	gtk_window_get_position (GTK_WINDOW(mainwindow), 
		&preferences.x, &preferences.y);
	bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
	modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
	preferences.modeoptionmenu = 
		gtk_combo_box_get_active (GTK_COMBO_BOX(modeoptionmenu));
	preferences.bandoptionmenu = 
		gtk_combo_box_get_active (GTK_COMBO_BOX(bandoptionmenu));

	handlebox = lookup_widget (mainwindow, "handlebox");
	if (GTK_WIDGET_VISIBLE(handlebox))
		preferences.viewtoolbar = 1;
	else
		preferences.viewtoolbar = 0;
	if (GTK_WIDGET_VISIBLE(b4dialog))
	{
		preferences.viewb4 = 1;
		gtk_window_get_size (GTK_WINDOW(b4dialog), 
			&preferences.b4width, &preferences.b4height);
		gtk_window_get_position (GTK_WINDOW(b4dialog), 
			&preferences.b4x, &preferences.b4y);
	}
	else
	{
		preferences.viewb4 = 0;
		preferences.b4x = 10;
		preferences.b4y = 30;
		preferences.b4width = 500;
		preferences.b4height = 300;
	}

	hpaned = lookup_widget (mainwindow, "hpaned");
	preferences.handlebarpos = gtk_paned_get_position (GTK_PANED(hpaned));
	if (keyerwindow)
		g_signal_emit_by_name (G_OBJECT(keyerwindow), "delete-event");

	savepreferences ();
	savehistory ();

#if WANT_HAMLIB
	if (preferences.hamlib > 0)
		stop_hamlib ();
#endif

	g_free (xlogdir);
	if (remotetimer != -1)
		g_source_remove (remotetimer);
	if (savetimer != -1)
		g_source_remove (savetimer);
	if (clocktimer != -1)
		g_source_remove (clocktimer);

	/* remove message queue */
#if HAVE_SYS_IPC_H
	if (msgid != -1)
		msgctl (msgid, IPC_RMID, 0);
#endif

	/* detach shared mem and destroy it */
#if HAVE_SYS_SHM_H
	if (state.shmid != -1)
		{
			shmdt (shareCall);
			shmctl (state.shmid, IPC_RMID, NULL);
		}
#endif

	freestate ();
	freeprefs ();
	freeremote ();
}

/* find dot in frequency and return correct band */
gchar *
finddot (gchar * str)
{
	gchar *qsoband, *end, *j, *found = NULL;

	qsoband = g_strdup (str);
	end = qsoband + strlen (qsoband);
	for (j = qsoband; j < end; ++j)
	{
		switch (*j)
		{
			case '.':
			*j = '\0';
			break;
		}
	}
	if (g_ascii_strcasecmp (qsoband, "1") == 0)
		found = "1.8";
	else if (g_ascii_strcasecmp (qsoband, "3") == 0)
		found = "3.5";
	else if (g_ascii_strcasecmp (qsoband, "5") == 0)
		found = "5.3";
	else if (g_ascii_strcasecmp (qsoband, "7") == 0)
		found = "7";
	else if (g_ascii_strcasecmp (qsoband, "29") == 0)
		found = "28";
	else found = qsoband;

	if (atoi (qsoband) > 30 )
	{
		if (atoi (qsoband) > 49 && atoi (qsoband) < 55)
			found = "50";
		else if (atoi (qsoband) > 143 && atoi (qsoband) < 149)
			found = "144";
		else if (atoi (qsoband) > 221 && atoi (qsoband) < 226)
			found = "222";
		else if (atoi (qsoband) > 419 && atoi (qsoband) < 451)
			found = "432";
		else if (atoi (qsoband) > 901 && atoi (qsoband) < 929)
			found = "902";
		else if (atoi (qsoband) > 1239 && atoi (qsoband) < 1326)
			found = "1296";
		else 
			found = qsoband;
	}
	return (g_strdup(found));
}

/* find dot in frequency and return string before the dot */
gchar *
findint (gchar * freq)
{
	gchar *end, *j;

	end = freq + strlen (freq);
	for (j = freq; j < end; ++j)
		{
			switch (*j)
	{
	case '.':
		*j = '\0';
		break;
	}
		}
	return (freq);
}

/* try to find the correct band for dupe checking */
gchar *
findband (gchar * arg)
{
	long long fr;
	gchar *str;

	str = g_new0 (gchar, 100);

	fr = strtoll(arg, (char **)NULL, 10);

	if (fr > 1325)		/* no dot */
	{
		sprintf (str, "%lld", fr);
		if (strlen (str) > 9)
			str[strlen (str) - 9] = '\0';
		else if (strlen (str) > 6)
			str[strlen (str) - 6] = '\0';
		else if (strlen (str) > 3)
			str[strlen (str) - 3] = '\0';
		fr = strtoll(str, (char **)NULL, 10);
	}

	switch (fr)
	{
		case 29:
			fr = 28;
			break;
		case 50 ... 54:
			fr = 50;
			break;
		case 144 ... 148:
			fr = 144;
			break;
		case 222 ... 225:
			fr = 222;
			break;
		case 420 ... 450:
			fr = 420;
			break;
		case 902 ... 928:
			fr = 902;
			break;
		case 1240 ... 1325:
			fr = 1250;
			break;
		default:
			break;
	}
	sprintf (str, "%lld", fr);
	return (str);
}

/* get the current date, returned value has to be freed */
gchar *
getdate (void)
{
	time_t current;
	struct tm *timestruct = NULL;
	gchar datenow[20], *date;
	GError *error;

	time (&current);
	timestruct = localtime (&current);
	strftime (datenow, 20, "%d %b %Y", timestruct);

	if (!g_utf8_validate (datenow, -1, NULL ))
	{
		date = g_locale_to_utf8 (datenow, -1, NULL, NULL, &error);
		if (!date) 
		{
				g_warning (_("Unable to convert '%s' to UTF-8: %s"), datenow, 
					error->message);
				g_error_free (error);
			}
 	}
	else date = g_strdup (datenow);
	
	return (date);
}

/* get the current time, returned value has to be freed */
gchar *
gettime (void)
{
	time_t current;
	struct tm *timestruct = NULL;
	gchar stimenow[20];

	time (&current);
	timestruct = localtime (&current);
	strftime (stimenow, 20, "%H%M", timestruct);
	return (g_strdup (stimenow));
}

/* look up mode in a list of modes */
gchar *
lookup_mode (gint index)
{
	gchar **s, *mode;

	s = g_strsplit (preferences.modes, ",", -1);
	mode = g_strdup (s[index]);
	g_strfreev (s);
	return (mode);
}

/* look up band in a list of bands */
gchar *
lookup_band (gint index)
{
	gchar **s, *band;

	s = g_strsplit (preferences.bands, ",", -1);
	band = g_strdup (s[index]);
	g_strfreev (s);
	return (band);
}

/* clock which updates a label in the statusbar */
gint
updateclock (void)
{
	GtkWidget *clocklabel;
	GString *timestr = g_string_new ("");
	gchar *nowdate, *nowtime;

	nowdate = getdate ();
	nowtime = gettime ();
	clocklabel = lookup_widget (mainwindow, "clocklabel");
	g_string_printf (timestr, "<b>%s", nowdate);
	timestr = g_string_append_c (timestr, ' ');
	timestr = g_string_append (timestr, nowtime);
	timestr = g_string_append (timestr, " GMT</b>");
	gtk_label_set_markup (GTK_LABEL (clocklabel), timestr->str);
	g_string_free (timestr, TRUE);
	g_free (nowdate);
	g_free (nowtime);
	return 1;
}

/* Put color in the right format for XParseColor, so it can be processed by gdk_color_parse */
gchar *
color_parse (gchar * value)
{
	gchar **valuesplit = NULL;
	gchar *rgbsyntax;

	valuesplit = g_strsplit (value, ",", 3);
	rgbsyntax =
		g_strconcat ("rgb:", valuesplit[0], "/", valuesplit[1], "/",
		 valuesplit[2], NULL);
	g_strfreev (valuesplit);
	return (rgbsyntax);
}



gint
autosave (void)
{
	gint i;
	logtype *logw;
	gchar *temp;
	gboolean message = FALSE;

	for (i = 0; i < g_list_length (logwindowlist); i++)
	{
		logw = g_list_nth_data (logwindowlist, i);
		if (logw->logchanged)
		{
			savelog (logw, logw->filename, TYPE_FLOG, 1, logw->qsos);
			logw->logchanged = FALSE;
			gtk_label_set_text (GTK_LABEL (logw->label), logw->logname);
			message = TRUE;
		}
	}

	if (message)
	{
		temp = g_strdup_printf (_("Log(s) autosaved"));
		update_statusbar (temp);
		g_free (temp);
	}
	return 1;
}

/*
 * check visibility of fields in the loglist and show/hide the corresponding
 * widgets in the QSO frame
 */
void
set_qsoframe (gpointer arg)
{
	gint i, j;
	GtkWidget *endhbox, *powerhbox, *namehbox, *locatorhbox, *qthhbox,
		*unknown1hbox, *unknown2hbox, *powerbutton, *powerlabel, *qslhbox,
		*locatorframe, *remarksvbox, *clearallmenu;
	logtype *logw = (logtype *) arg;

	endhbox = lookup_widget (mainwindow, "endhbox");
	powerhbox = lookup_widget (mainwindow, "powerhbox");
	qslhbox = lookup_widget (mainwindow, "qslhbox");
	namehbox = lookup_widget (mainwindow, "namehbox");
	qthhbox = lookup_widget (mainwindow, "qthhbox");
	locatorhbox = lookup_widget (mainwindow, "locatorhbox");
	locatorframe = lookup_widget (mainwindow, "locatorframe");
	unknown1hbox = lookup_widget (mainwindow, "unknown1hbox");
	unknown2hbox = lookup_widget (mainwindow, "unknown2hbox");
	remarksvbox = lookup_widget (mainwindow, "remarksvbox");

	if (logw) for (i = 0; i < QSO_FIELDS; i++)
	{
		for (j = 0; j < logw->columns; j++)
		{
		if (i == logw->logfields[j])
			break;
		}
		if (i == GMTEND)
		{
			if (j == logw->columns)
				gtk_widget_hide (endhbox);
			else
				gtk_widget_show (endhbox);
		}
		else if (i == POWER)
		{
			powerbutton = lookup_widget (mainwindow, "powerbutton");
			powerlabel = lookup_widget (mainwindow, "powerlabel");
			if (j == logw->columns)
				gtk_widget_hide (powerhbox);
			else
			{
				gtk_widget_show (powerhbox);
				if (preferences.hamlib > 0)
				{
					gtk_widget_show (powerbutton);
					gtk_widget_hide (powerlabel);
				}
				else
				{
					gtk_widget_hide (powerbutton);
					gtk_widget_show (powerlabel);
				}
			}
		}
		else if (i == QSLOUT)
		{
			if (j == logw->columns)
				gtk_widget_hide (qslhbox);
			else
				gtk_widget_show (qslhbox);
		}
		else if (i == NAME)
		{
			if (j == logw->columns)
				gtk_widget_hide (namehbox);
			else
				gtk_widget_show (namehbox);
		}
		else if (i == QTH)
		{
			if (j == logw->columns)
				gtk_widget_hide (qthhbox);
			else
				gtk_widget_show (qthhbox);
		}
		else if (i == LOCATOR)
		{
			if (j == logw->columns)
			{
				gtk_widget_hide (locatorhbox);
				gtk_widget_hide (locatorframe);
			}
			else
			{
				gtk_widget_show (locatorhbox);
				gtk_widget_show (locatorframe);
			}
		}
		else if (i == U1)
		{
			if (j == logw->columns)
				gtk_widget_hide (unknown1hbox);
			else
				gtk_widget_show (unknown1hbox);
		}
		else if (i == U2)
		{
			if (j == logw->columns)
				gtk_widget_hide (unknown2hbox);
			else
				gtk_widget_show (unknown2hbox);
		}
		else if (i == REMARKS)
		{
			if (j == logw->columns)
				gtk_widget_hide (remarksvbox);
			else
				gtk_widget_show (remarksvbox);
		}
	}
	else
	{
		gtk_widget_hide (endhbox);
		gtk_widget_show (qslhbox);
		gtk_widget_hide (powerhbox);
		gtk_widget_hide (namehbox);
		gtk_widget_hide (qthhbox);
		gtk_widget_hide (locatorhbox);
		gtk_widget_hide (locatorframe);
		gtk_widget_hide (endhbox);
		gtk_widget_hide (unknown1hbox);
		gtk_widget_hide (unknown2hbox);
		gtk_widget_show (remarksvbox);
	}
	clearallmenu = gtk_ui_manager_get_widget
         (ui_manager, "/MainMenu/EditMenu/Clear All");
	g_signal_emit_by_name (G_OBJECT (clearallmenu), "activate");
}

gboolean
fileexist (gchar * path)
{
	FILE *fp = NULL;

	fp = fopen (path, "r");

	if (fp)
		{
			fclose (fp);
			return (TRUE);
		}
	else
		return (FALSE);
}

gchar *
my_strreplace(const char *str, const char *delimiter, const char *replacement)
{
	gchar **split;
	gchar *ret;

	g_return_val_if_fail (str != NULL, NULL);
	g_return_val_if_fail (delimiter != NULL, NULL);
	g_return_val_if_fail (replacement != NULL, NULL);

	split = g_strsplit (str, delimiter, 0);
	ret = g_strjoinv (replacement, split);
	g_strfreev (split);

	return ret;
}
