/*
  Copyright (C) 2000-2007

  Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_argv.cc 1603 2007-11-18 18:47:12Z paultcochrane $
*/

/*! @file xmds_argv.cc
  @brief Command line argument vector parsing classes and methods

  More detailed explanation...
*/

#include <xmds_common.h>
#include <xmds_simulation.h>
#include <xmds_argv.h>
#include <xmds_arg_element.h>
#include <string>

extern bool debugFlag;

// **************************************************************************
// **************************************************************************
//                              xmdsArgv Public
// **************************************************************************
// **************************************************************************

// **************************************************************************
xmdsArgv::xmdsArgv(
                   const xmdsSimulation *const yourSimulation,
                   const bool& yourVerboseMode) :
  xmdsElement(yourSimulation, yourVerboseMode) {
  if (debugFlag) {
    printf("xmdsArgv::xmdsArgv\n");
  }
}

// **************************************************************************
xmdsArgv::~xmdsArgv() {
  if (debugFlag) {
    printf("xmdsArgv::~xmdsArgv\n");
  }

  // destroy xmdsArgs, since they are not managed by the xmdsElement class

  for (list<const xmdsArg*>::const_iterator
      ppxmdsArg = myArgsList.begin();
      ppxmdsArg != myArgsList.end();
      ppxmdsArg++) {
    delete(*ppxmdsArg);
  }
}

// **************************************************************************
void xmdsArgv::processElement(
                              const Element *const yourElement) {
  if (debugFlag) {
    printf("xmdsArgv::processElement\n");
  }

  if (verbose()) {
    printf("Processing argv element ...\n");
  }

  list<XMLString> anXMLStringList;

  // ************************************
  // find and process arg elements
  // ************************************

  const NodeList* candidateElements;

  candidateElements = yourElement->getElementsByTagName("arg", 0);

  if (candidateElements->length() == 0) {
    throw xmdsException(yourElement, "at least one argument expected");
  }

  for (unsigned long i=0; i<candidateElements->length(); i++) {

    const Element* nextElement =
      dynamic_cast<const Element*>(candidateElements->item(i));

    xmdsElement* newxmdsArgElement = createxmdsArgElement();

    newxmdsArgElement->processElement(nextElement);
  }
}

// **************************************************************************
void xmdsArgv::processArgs(
                           const list<XMLString>& argNamesList,
                           const unsigned long& space) const {
  if (debugFlag) {
    printf("xmdsArgv::processArgs\n");
  }

  for (list<XMLString>::const_iterator
      pXMLString = argNamesList.begin();
      pXMLString != argNamesList.end();
      pXMLString++) {

    const xmdsArg* nextArg;

    if (!getArg(*pXMLString, nextArg)) {
      sprintf(errorMessage(), "Arg '%s' unknown", pXMLString->c_str());
      throw xmdsException(errorMessage());
    }

    for (list<XMLString>::const_iterator
        pXMLString2 = argNamesList.begin();
        pXMLString2 != pXMLString;
        pXMLString2++) {
      if (*pXMLString == *pXMLString2) {
        sprintf(errorMessage(), "Duplicate arg '%s'", pXMLString->c_str());
        throw xmdsException(errorMessage());
      }
    }

    if (verbose()) {
      printf("adding arg '%s' to accessible args list\n", pXMLString->c_str());
    }
  }
}

// **************************************************************************
bool xmdsArgv::getArg(
                      const XMLString& argName,
                      const xmdsArg*& theArg) const {
  if (debugFlag) {
    printf("xmdsArgv::getArg\n");
  }

  theArg = 0;

  for (list<const xmdsArg*>::const_iterator
      ppxmdsArg = myArgsList.begin();
      ppxmdsArg != myArgsList.end();
      ppxmdsArg++) {
    if (*(*ppxmdsArg)->name() == argName) {
      theArg = *ppxmdsArg;
      return 1;
    }
  }
  return 0;
}

// **************************************************************************
void xmdsArgv::argNames(
                        list<XMLString>& argNamesList) const {
  if (debugFlag) {
    printf("xmdsArgv::argNames\n");
  }

  argNamesList.clear();

  for (list<const xmdsArg*>::const_iterator
      ppxmdsArg = myArgsList.begin();
      ppxmdsArg != myArgsList.end();
      ppxmdsArg++) {
    argNamesList.push_back(*(*ppxmdsArg)->name());
  }
}

// **************************************************************************
xmdsArg* xmdsArgv::createxmdsArg() {
  if (debugFlag) {
    printf("xmdsArgv::createxmdsArg\n");
  }

  xmdsArg* newArg = new xmdsArg(this);
  myArgsList.push_back(newArg);
  return newArg;
}

// **************************************************************************
// **************************************************************************
//                              xmdsArgv protected
// **************************************************************************
// **************************************************************************

void xmdsArgv::writeGlobals(FILE *const outfile) const
{
  if (debugFlag) {
    printf("xmdsArgv::writeGlobals\n");
  }

  if (simulation()->argStruct()->nameList.size() != 0) {
    fprintf(outfile, "\n"
        "// *********** Command line argument global variables *********\n");
    list<string>::const_iterator inameList =
      simulation()->argStruct()->nameList.begin();
    list<string>::const_iterator itypeList =
      simulation()->argStruct()->typeList.begin();
    list<string>::const_iterator idefaultValueList =
      simulation()->argStruct()->defaultValueList.begin();

    for (long unsigned int i = 0;
        i < simulation()->argStruct()->nameList.size(); i++) {
      fprintf(outfile, "%s %s = %s;\n",
          itypeList->c_str(), inameList->c_str(), idefaultValueList->c_str());
      itypeList++;
      inameList++;
      idefaultValueList++;
    }

    fprintf(outfile, "// **********************************************\n");
    fprintf(outfile, "\n");
  }
}

// **************************************************************************
// **************************************************************************
//                              xmdsArgv private
// **************************************************************************
// **************************************************************************

// **************************************************************************
xmdsArgElement* xmdsArgv::createxmdsArgElement() {
  if (debugFlag) {
    printf("xmdsArgv::createxmdsArgElement\n");
  }

  xmdsArgElement* newArgElement = new xmdsArgElement(simulation(), verbose(), this);
  myArgsList.push_back(newArgElement);
  return newArgElement;
}

/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
