
/*
  Copyright (C) 2000-2007

  Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_integrate_rk9.cc 1599 2007-11-17 13:42:49Z paultcochrane $
*/

/*! @file xmds_integrate_rk9.cc
  @brief Integrate element parsing classes and methods; ninth order Runge-Kutta

  More detailed explanation...
*/

#include <xmds_common.h>
#include <xmds_integrate_rk9.h>
#include <xmds_simulation.h>
#include <xmds_vector.h>

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateRK9 public
// **************************************************************************
// **************************************************************************

extern bool debugFlag;

long nxmdsIntegrateRK9s=0;   //!< Number of xmds integrate RK9 objects

// **************************************************************************
xmdsIntegrateRK9::xmdsIntegrateRK9(
                                   const xmdsSimulation *const yourSimulation,
                                   const bool& yourVerboseMode) :
  xmdsIntegrate(yourSimulation, yourVerboseMode, false, false) {
  if (debugFlag) {
    nxmdsIntegrateRK9s++;
    printf("xmdsIntegrateRK9::xmdsIntegrateRK9\n");
    printf("nxmdsIntegrateRK9s=%li\n", nxmdsIntegrateRK9s);
  }
}

// **************************************************************************
xmdsIntegrateRK9::~xmdsIntegrateRK9() {
  if (debugFlag) {
    nxmdsIntegrateRK9s--;
    printf("xmdsIntegrateRK9::~xmdsIntegrateRK9\n");
    printf("nxmdsIntegrateRK9s=%li\n", nxmdsIntegrateRK9s);
  }
}

// **************************************************************************
void xmdsIntegrateRK9::processElement(
                                      const Element *const yourElement) {
  if (debugFlag) {
    printf("xmdsIntegrateRK9::processElement\n");
  }

  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    printf("\n");
    printf("WARNING: RK9 methods may not always yield correct stochastic convergence.\n");
    printf("\n");
  }
}

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateRK9 protected
// **************************************************************************
// **************************************************************************

// **************************************************************************
void xmdsIntegrateRK9::writePrototypes(
                                       FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateRK9::writePrototypes\n");
  }

  fprintf(outfile, "void _segment%li(unsigned long cycle);\n", segmentNumber);
  fprintf(outfile, "\n");

  if (crossVectorNamesList()->size() > 0) {
    fprintf(outfile, "void _segment%li_calculate_cross_field(", segmentNumber);
    if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
      fprintf(outfile, "const double *const _noise_vector");
    }
    fprintf(outfile, ");\n");
    fprintf(outfile, "\n");
  }
}

// **************************************************************************
void xmdsIntegrateRK9::writeRoutines(
                                     FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateRK9::writeRoutines\n");
  }

  writeMainIntegrateRoutine(outfile);

  if (crossVectorNamesList()->size() > 0) {
    writeCalculateCrossFieldRoutine(outfile);
  }
}

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateRK9 private
// **************************************************************************
// **************************************************************************

// **************************************************************************
void xmdsIntegrateRK9::writeMainIntegrateRoutine(
                                                 FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateRK9::writeMainIntegrateRoutine\n");
  }

  const char *const fieldName = simulation()->field()->name()->c_str();

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateRK9::writeMainIntegrateRoutine: cannot find 'main' vector");
  }

  const char* typeName="";
  if (mainVector->vectorType()==COMPLEX) {
    typeName="complex";
  }
  else if (mainVector->vectorType()==DOUBLE) {
    typeName="double";
  }

  fprintf(outfile, "/* ******************************************** */\n");
  fprintf(outfile, "void _segment%li(unsigned long cycle) {\n", segmentNumber);
  fprintf(outfile, "\n");
  if ((simulation()->parameters()->usempi)&!(simulation()->parameters()->stochastic)){
    fprintf(outfile, "%s *_akafield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akbfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akcfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akdfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akefield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akffield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akgfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akhfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akifield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akjfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
  }
  else {
    fprintf(outfile, "%s *_akafield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akbfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akcfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akdfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akefield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akffield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akgfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akhfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akifield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akjfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);

  }
  fprintf(outfile, "\n");
  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "const double _var = 1");
    for (unsigned long i=0; i<simulation()->field()->geometry()->nDims(); i++) {
      fprintf(outfile, "/_%s_dx%li", fieldName, i);
    }
    fprintf(outfile, ";\n");
    fprintf(outfile, "double *_noise_vector = new double[_%s_size*_n_noises];\n", fieldName);
    if (simulation()->parameters()->errorCheck) {
      fprintf(outfile, "double *_noise_vector2 = new double[_%s_size*_n_noises];\n", fieldName);
    }
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "\n");
  fprintf(outfile, "// Runge Kutta method constants \n");
  fprintf(outfile, "double _a_raw[16];\n");
  fprintf(outfile, "double _a[16];\n");
  fprintf(outfile, "double _b[16][16];\n");
  fprintf(outfile, "double _c[16];\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "for (unsigned long _i0=0; _i0<16; _i0++) {\n");
  fprintf(outfile, "  _a_raw[_i0]=0.0;\n");
  fprintf(outfile, "  _c[_i0]=0.0;\n");
  fprintf(outfile, "  for (unsigned long _i1=0; _i1<16; _i1++)\n");
  fprintf(outfile, "   _b[_i0][_i1]=0.0;\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "_a_raw[1] = 0.02173913043478260869565217391304347;\n");
  fprintf(outfile, "_a_raw[2] = 0.09629581047800066670113001679819925;\n");
  fprintf(outfile, "_a_raw[3] = 0.14444371571700100005169502519729888;\n");
  fprintf(outfile, "_a_raw[4] = 0.52205882352941176470588235294117647;\n");
  fprintf(outfile, "_a_raw[5] = 0.22842443612863469578031459099794265;\n");
  fprintf(outfile, "_a_raw[6] = 0.54360353589933733219171338103002937;\n");
  fprintf(outfile, "_a_raw[7] = 0.64335664335664335664335664335664335;\n");
  fprintf(outfile, "_a_raw[8] = 0.48251748251748251748251748251748251;\n");
  fprintf(outfile, "_a_raw[9] = 0.06818181818181818181818181818181818;\n");
  fprintf(outfile, "_a_raw[10] = 0.25060827250608272506082725060827250;\n");
  fprintf(outfile, "_a_raw[11] = 0.66736715965600568968278165443304378;\n");
  fprintf(outfile, "_a_raw[12] = 0.85507246376811594202898550724637681;\n");
  fprintf(outfile, "_a_raw[13] = 0.89795918367346938775510204081632653;\n");
  fprintf(outfile, "_a_raw[14] = 1.0;\n");
  fprintf(outfile, "_a_raw[15] = 1.0;\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "_a[0]=0.0;\n");
  fprintf(outfile, "for (unsigned long _i0=1; _i0<16; _i0++)\n");
  fprintf(outfile, "  _a[_i0]=_a_raw[_i0]-_a_raw[_i0-1];\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "_b[1][0] = 1.0/46.0;\n");
  fprintf(outfile, "_b[2][0] =-0.11698050118114486205818241524969622;\n");
  fprintf(outfile, "_b[2][1] = 0.21327631165914552875931243204789548;\n");
  fprintf(outfile, "_b[3][0] = 0.03611092892925025001292375629932472;\n");
  fprintf(outfile, "_b[3][2] = 0.10833278678775075003877126889797416;\n");
  fprintf(outfile, "_b[4][0] = 1.57329743908138605107331820072051125;\n");
  fprintf(outfile, "_b[4][2] =-5.98400943754042002888532938159655553;\n");
  fprintf(outfile, "_b[4][3] = 4.93277082198844574251789353381722074;\n");
  fprintf(outfile, "_b[5][0] = 0.05052046351120380909008334360006234;\n");
  fprintf(outfile, "_b[5][3] = 0.17686653884807108146683657390397612;\n");
  fprintf(outfile, "_b[5][4] = 0.00103743376935980522339467349390418;\n");
  fprintf(outfile, "_b[6][0] = 0.10543148021953768958529340893598138;\n");
  fprintf(outfile, "_b[6][3] =-0.16042415162569842979496486916719383;\n");
  fprintf(outfile, "_b[6][4] = 0.11643956912829316045688724281285250;\n");
  fprintf(outfile, "_b[6][5] = 0.48215663817720491194449759844838932;\n");
  fprintf(outfile, "_b[7][0] = 0.07148407148407148407148407148407148;\n");
  fprintf(outfile, "_b[7][5] = 0.32971116090443908023196389566296464;\n");
  fprintf(outfile, "_b[7][6] = 0.24216141096813279233990867620960722;\n");
  fprintf(outfile, "_b[8][0] = 0.07162368881118881118881118881118881;\n");
  fprintf(outfile, "_b[8][5] = 0.32859867301674234161492268975519694;\n");
  fprintf(outfile, "_b[8][6] = 0.11622213117906185418927311444060725;\n");
  fprintf(outfile, "_b[8][7] =-0.03392701048951048951048951048951048;\n");
  fprintf(outfile, "_b[9][0] = 0.04861540768024729180628870095388582;\n");
  fprintf(outfile, "_b[9][5] = 0.03998502200331629058445317782406268;\n");
  fprintf(outfile, "_b[9][6] = 0.10715724786209388876739304914053506;\n");
  fprintf(outfile, "_b[9][7] =-0.02177735985419485163815426357369818;\n");
  fprintf(outfile, "_b[9][8] =-0.10579849950964443770179884616296721;\n");
  fprintf(outfile, "_b[10][0] =-0.02540141041535143673515871979014924;\n");
  fprintf(outfile, "_b[10][5] = 1.0/30.0;\n");
  fprintf(outfile, "_b[10][6] =-0.16404854760069182073503553020238782;\n");
  fprintf(outfile, "_b[10][7] = 0.03410548898794737788891414566528526;\n");
  fprintf(outfile, "_b[10][8] = 0.15836825014108792658008718465091487;\n");
  fprintf(outfile, "_b[10][9] = 0.21425115805975734472868683695127609;\n");
  fprintf(outfile, "_b[11][0] = 0.00584833331460742801095934302256470;\n");
  fprintf(outfile, "_b[11][5] =-0.53954170547283522916525526480339109;\n");
  fprintf(outfile, "_b[11][6] = 0.20128430845560909506500331018201158;\n");
  fprintf(outfile, "_b[11][7] = 0.04347222773254789483240207937678906;\n");
  fprintf(outfile, "_b[11][8] =-0.00402998571475307250775349983910179;\n");
  fprintf(outfile, "_b[11][9] = 0.16541535721570612771420482097898952;\n");
  fprintf(outfile, "_b[11][10] = 0.79491862412512344573322086551518180;\n");
  fprintf(outfile, "_b[12][0] =-0.39964965968794892497157706711861448;\n");
  fprintf(outfile, "_b[12][5] =-3.79096577568393158554742638116249372;\n");
  fprintf(outfile, "_b[12][6] =-0.40349325653530103387515807815498044;\n");
  fprintf(outfile, "_b[12][7] =-2.82463879530435263378049668286220715;\n");
  fprintf(outfile, "_b[12][8] = 1.04226892772185985533374283289821416;\n");
  fprintf(outfile, "_b[12][9] = 1.12510956420436603974237036536924078;\n");
  fprintf(outfile, "_b[12][10] = 3.32746188718986816186934832571938138;\n");
  fprintf(outfile, "_b[12][11] = 2.77897957186355606325818219255783627;\n");
  fprintf(outfile, "_b[13][0] = 0.39545306350085237157098218205756922;\n");
  fprintf(outfile, "_b[13][5] = 5.82534730759650564865380791881446903;\n");
  fprintf(outfile, "_b[13][6] =-0.36527452339161313311889856846974452;\n");
  fprintf(outfile, "_b[13][7] = 1.18860324058346533283780076203192232;\n");
  fprintf(outfile, "_b[13][8] = 0.57970467638357921347110271762687972;\n");
  fprintf(outfile, "_b[13][9] =-0.86824862589087693262676988867897834;\n");
  fprintf(outfile, "_b[13][10] =-5.20227677296454721392873650976792184;\n");
  fprintf(outfile, "_b[13][11] =-0.79895541420753382543211121058675915;\n");
  fprintf(outfile, "_b[13][12] = 0.14360623206363792632792463778889008;\n");
  fprintf(outfile, "_b[14][0] = 8.49173149061346398013352206978380938;\n");
  fprintf(outfile, "_b[14][5] = 86.32213734729036800877634194386790750;\n");
  fprintf(outfile, "_b[14][6] = 1.02560575501091662034511526187393241;\n");
  fprintf(outfile, "_b[14][7] = 85.77427969817339941806831550695235092;\n");
  fprintf(outfile, "_b[14][8] =-13.98699305104110611795532466113248067;\n");
  fprintf(outfile, "_b[14][9] =-20.71537405501426352265946477613161883;\n");
  fprintf(outfile, "_b[14][10] =-72.16597156619946800281180102605140463;\n");
  fprintf(outfile, "_b[14][11] =-76.71211139107806345587696023064419687;\n");
  fprintf(outfile, "_b[14][12] = 4.22319427707298828839851258893735507;\n");
  fprintf(outfile, "_b[14][13] =-1.25649850482823521641825667745565428;\n");
  fprintf(outfile, "_b[15][0] =-0.42892119881959353241190195318730008;\n");
  fprintf(outfile, "_b[15][5] =-9.16865700950084689999297912545025359;\n");
  fprintf(outfile, "_b[15][6] = 1.08317616770620939241547721530003920;\n");
  fprintf(outfile, "_b[15][7] =-1.23501525358323653198215832293981810;\n");
  fprintf(outfile, "_b[15][8] =-1.21438272617593906232943856422371019;\n");
  fprintf(outfile, "_b[15][9] = 1.37226168507232166621351243731869914;\n");
  fprintf(outfile, "_b[15][10] = 9.15723239697162418155377135344394113;\n");
  fprintf(outfile, "_b[15][11] = 1.30616301842220047563298585480401671;\n");
  fprintf(outfile, "_b[15][12] =-0.25285618808937955976690569433069974;\n");
  fprintf(outfile, "_b[15][13] = 0.38099910799663987066763679926508552;\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "_c[0] = 0.01490902081978461022483617102382552;\n");
  fprintf(outfile, "_c[7] =-0.20408044692054151258349120934134791;\n");
  fprintf(outfile, "_c[8] = 0.22901438600570447264772469337066476;\n");
  fprintf(outfile, "_c[9] = 0.12800558251147375669208211573729202;\n");
  fprintf(outfile, "_c[10] = 0.22380626846054143649770066956485937;\n");
  fprintf(outfile, "_c[11] = 0.39553165293700054420552389156421651;\n");
  fprintf(outfile, "_c[12] = 0.05416646758806981196568364538360743;\n");
  fprintf(outfile, "_c[13] = 0.12691439652445903685643385312168037;\n");
  fprintf(outfile, "_c[14] =-0.00052539244262118876455834655383035;\n");
  fprintf(outfile, "_c[15] = 1.0/31.0;\n");
  fprintf(outfile, "\n");


  fprintf(outfile, "for (unsigned long _i0=0; _i0<%li; _i0++) {\n", lattice());
  fprintf(outfile, "\n");
  writeRuntimeCheckingCode(outfile);

  if (simulation()->parameters()->errorCheck) {
    fprintf(outfile, " if (_half_step) {\n");
    fprintf(outfile, "\n");
    fprintf(outfile, "   const double _step = 0.5*%s/(double)%li;\n", interval()->c_str(), lattice());
    fprintf(outfile, "\n");
    if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
      if (simulation()->parameters()->noiseKind == "poissonian") {
        fprintf(outfile, "   _make_noises(_gen1, _var/_step, _noiseMeanRate*_step, _noise_vector, _%s_size*_n_noises);\n", fieldName);
      }
      else {
        fprintf(outfile, "   _make_noises(_gen1, _var/_step, _noise_vector, _%s_size*_n_noises);\n", fieldName);
      }
      fprintf(outfile, "\n");
    }
    writeSingleStepCode(outfile, FIRST_HALFSTEP);
    fprintf(outfile, "\n");
    if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
      if (simulation()->parameters()->noiseKind == "poissonian") {
        fprintf(outfile, "   _make_noises(_gen2, _var/_step, _noiseMeanRate*_step, _noise_vector, _%s_size*_n_noises);\n", fieldName);
      }
      else {
        fprintf(outfile, "   _make_noises(_gen2, _var/_step, _noise_vector, _%s_size*_n_noises);\n", fieldName);
      }
      fprintf(outfile, "\n");
    }
    writeSingleStepCode(outfile, SECOND_HALFSTEP);
    fprintf(outfile, "   }\n");
    fprintf(outfile, " else {\n");
    fprintf(outfile, "\n");
    fprintf(outfile, "   const double _step = %s/(double)%li;\n", interval()->c_str(), lattice());
    fprintf(outfile, "\n");
    if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
      if (simulation()->parameters()->noiseKind == "poissonian") {
        fprintf(outfile, "   _make_noises(_gen1, _var/_step/2, _noiseMeanRate*_step, _noise_vector, _%s_size*_n_noises);\n", fieldName);
        fprintf(outfile, "   _make_noises(_gen2, _var/_step/2, _noiseMeanRate*_step, _noise_vector2, _%s_size*_n_noises);\n", fieldName);
      }
      else {
        fprintf(outfile, "   _make_noises(_gen1, _var/_step/2, _noise_vector, _%s_size*_n_noises);\n", fieldName);
        fprintf(outfile, "   _make_noises(_gen2, _var/_step/2, _noise_vector2, _%s_size*_n_noises);\n", fieldName);
      }
      fprintf(outfile, "   for (unsigned long _i1=0; _i1<_%s_size*_n_noises; _i1++)\n", fieldName);
      fprintf(outfile, "     _noise_vector[_i1] += _noise_vector2[_i1];\n");
      fprintf(outfile, "\n");
    }
    writeSingleStepCode(outfile, FULLSTEP);
    fprintf(outfile, "   }\n");
  }
  else {
    fprintf(outfile, "\n");
    fprintf(outfile, " const double _step = %s/(double)%li;\n", interval()->c_str(), lattice());
    fprintf(outfile, "\n");
    if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
      if (simulation()->parameters()->noiseKind == "poissonian") {
        fprintf(outfile, " _make_noises(_gen, _var/_step, _noiseMeanRate*_step, _noise_vector, _%s_size*_n_noises);\n", fieldName);
      }
      else {
        fprintf(outfile, " _make_noises(_gen, _var/_step, _noise_vector, _%s_size*_n_noises);\n", fieldName);
      }
      fprintf(outfile, "\n");
    }
    writeSingleStepCode(outfile, FULLSTEP);
  }

  for (unsigned long i=0; i<simulation()->output()->nMomentGroups(); i++) {
    if (samples(i)!=0){
      fprintf(outfile, "\n");
      fprintf(outfile, " if (%li*((_i0+1)/%li)==(_i0+1))\n", lattice()/samples(i), lattice()/samples(i));
      fprintf(outfile, "   _mg%li_sample();\n", i);
    }
  }
  fprintf(outfile, " }\n");

  if (simulation()->parameters()->runTimeLimit > 0.0)
    fprintf(outfile, "_SEGMENT%li_END:\n", segmentNumber);

  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "    delete[] _noise_vector;\n");
    if (simulation()->parameters()->errorCheck)  {
      fprintf(outfile, "    delete[] _noise_vector2;\n");
    }
  }

  fprintf(outfile, "    fftw_free(_akafield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akbfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akcfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akdfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akefield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akffield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akgfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akhfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akifield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akjfield_%s_main);\n", fieldName);

  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

// **************************************************************************
void xmdsIntegrateRK9::writeCalculateCrossFieldRoutine(
                                                       FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateRK9IP::writeCalculateCrossFieldRoutine\n");
  }

  const unsigned long nDims = simulation()->field()->geometry()->nDims();
  const char *const fieldName = simulation()->field()->name()->c_str();

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_calculate_cross_field(", segmentNumber);
  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "const double *const _noise_vector");
  }
  fprintf(outfile, ") {\n");
  fprintf(outfile, "\n");

  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "const double* _noises = _noise_vector;\n");
    fprintf(outfile, "\n");
  }
  if (crossDimNumber()+1<nDims) {
    fprintf(outfile, "const unsigned long _%s_cross_size = _%s_lattice%li", fieldName, fieldName, crossDimNumber()+1);
    for (unsigned long i=crossDimNumber()+2; i<nDims; i++) {
      fprintf(outfile, "*_%s_lattice%li", fieldName, i);
    }
    fprintf(outfile, ";\n");
  }
  else
    fprintf(outfile, "const unsigned long _%s_cross_size = 1;\n", fieldName);
  fprintf(outfile, "\n");

  // need to create a vector list for ONLY main vectors

  list<XMLString> myMainVectorNamesList;

  for (list<XMLString>::const_iterator pXMLString = vectorNamesList()->begin(); pXMLString != vectorNamesList()->end(); pXMLString++) {
    list<XMLString>::const_iterator pXMLString2 = crossVectorNamesList()->begin();
    while ((pXMLString2 != crossVectorNamesList()->end()) && (*pXMLString2 != *pXMLString)) {
      pXMLString2++;
    }
    if (*pXMLString2 != *pXMLString) {
      myMainVectorNamesList.push_back(*pXMLString);
    }
  }

  const char* typeName;
  list<const xmdsVector*> mainVectorList;

  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {

    const xmdsVector* mainVector;

    if (!simulation()->field()->getVector(*pXMLString, mainVector)) {
      throw xmdsException("Internal error in xmdsIntegrateRK9::writeCalculateCrossFieldRoutine: cannot find main vector");
    }

    mainVectorList.push_back(mainVector);

    if (mainVector->vectorType()==DOUBLE) {
      typeName="double";
    }
    else {
      typeName="complex";
    }

    fprintf(outfile, "%s *_%s_%s_old = (%s*)fftw_malloc(sizeof(%s)*_%s_cross_size*_%s_%s_ncomponents);\n",
            typeName, fieldName, pXMLString->c_str(), typeName, typeName, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  list<const xmdsVector*> crossVectorList;

  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {

    const xmdsVector* crossVector;

    if (!simulation()->field()->getVector(*pXMLString, crossVector)) {
      throw xmdsException("Internal error in xmdsIntegrateRK9::writeCalculateCrossFieldRoutine: cannot find cross vector");
    }

    crossVectorList.push_back(crossVector);

    if (crossVector->vectorType()==DOUBLE) {
      typeName="double";
    }
    else {
      typeName="complex";
    }

    fprintf(outfile, "%s *_%s_%s_K = (%s*)fftw_malloc(sizeof(%s)*_%s_cross_size*_%s_%s_ncomponents);\n",
            typeName, fieldName, crossVector->name()->c_str(), typeName, typeName, fieldName, fieldName, crossVector->name()->c_str());
    fprintf(outfile, "%s *_%s_%s_I = (%s*)fftw_malloc(sizeof(%s)*_%s_cross_size*_%s_%s_ncomponents);\n",
            typeName, fieldName, crossVector->name()->c_str(), typeName, typeName, fieldName, fieldName, crossVector->name()->c_str());
    fprintf(outfile, "%s *_%s_%s_d = (%s*)fftw_malloc(sizeof(%s)*_%s_cross_size*_%s_%s_ncomponents);\n",
            typeName, fieldName, crossVector->name()->c_str(), typeName, typeName, fieldName, fieldName, crossVector->name()->c_str());
    fprintf(outfile, "\n");

    for (unsigned long i=0; i<crossVector->nComponents(); i++) {
      fprintf(outfile, "%s d%s_d%s;\n",
              typeName, crossVector->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str());
    }
    fprintf(outfile, "\n");
  }

  // add cross vectors to total vectors to use

  list<XMLString> myTotalVectorsList = myMainVectorNamesList;

  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    myTotalVectorsList.push_back(*pXMLString);
  }


  simulation()->field()->vectors2space(outfile, 0, myTotalVectorsList, "");

  // open outer loops

  for (unsigned long i=0; i<crossDimNumber(); i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, "double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, "for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  for (unsigned long j=0; j<crossDimNumber(); j++) {
    fprintf(outfile, " ");
  }
  fprintf(outfile, "double %s = _%s_xmin%li;\n",
          simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
  fprintf(outfile, "\n");

  for (unsigned long j=0; j<crossDimNumber(); j++) {
    fprintf(outfile, " ");
  }
  fprintf(outfile, "for (unsigned long _i%li=0; _i%li<_%s_lattice%li-1; _i%li++) {\n",
          crossDimNumber(), crossDimNumber(), fieldName, crossDimNumber(), crossDimNumber());
  fprintf(outfile, "\n");

  char indent[64];
  for (unsigned long i=0; i<crossDimNumber()+1; i++) {
    indent[i]=0x09;
  }
  indent[crossDimNumber()+1]=0;

  char indent2[64];
  for (unsigned long i=0; i<nDims; i++) {
    indent2[i]=0x09;
  }
  indent2[nDims]=0;

  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%sunsigned long _%s_%s_index_pointer_begin=0;\n", indent, fieldName, pXMLString->c_str());
    for (unsigned long i=0; i<crossDimNumber()+1; i++) {
      fprintf(outfile, "%s_%s_%s_index_pointer_begin += _i%li", indent, fieldName, pXMLString->c_str(), i);
      for (unsigned long j=i+1; j<nDims; j++) {
        fprintf(outfile, "*_%s_lattice%li", fieldName, j);
      }
      fprintf(outfile, "*_%s_%s_ncomponents;\n", fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// copy cross vectors into K and I vectors\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++) {\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] = _%s_%s[_%s_%s_index_pointer_begin + _j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_I[_j] = _%s_%s[_%s_%s_index_pointer_begin + _j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "%s }\n", indent);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// store main vectors into old\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_old[_j] = _active_%s_%s[_%s_%s_index_pointer_begin + _j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  // ****************************************************************
  fprintf(outfile, "%s// ********** step 1 ***************\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s {\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer=_%s_%s_index_pointer_begin;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s // calculate k1\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer_local=0;\n", indent, fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  // open inner loops
  for (unsigned long i=crossDimNumber()+1; i<nDims; i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "// *********** cross_propagation code ***********\n");
  fprintf(outfile, "%s\n", crossPropagationCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");
  for (list<const xmdsVector*>::const_iterator pxmdsVector = crossVectorList.begin(); pxmdsVector != crossVectorList.end(); pxmdsVector++) {
    for (unsigned long i=0; i<(*pxmdsVector)->nComponents(); i++) {
      fprintf(outfile, "%s _%s_%s_d[_%s_%s_index_pointer_local + %li] = d%s_d%s*_%s_dx%li;\n",
              indent2, fieldName, (*pxmdsVector)->name()->c_str(), fieldName, (*pxmdsVector)->name()->c_str(), i,
              (*pxmdsVector)->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
    }
    fprintf(outfile, "\n");
  }

  //close inner loops
  if (crossDimNumber()+1<nDims) {
    for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
    for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer_local += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  for (unsigned long i=nDims; i>crossDimNumber()+1; i--) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " %s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " }\n");
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s }\n", indent);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// add to K cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] += _%s_%s_d[_j]/6;\n", indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// create next cross vectors\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_I[_j] + _%s_%s_d[_j]/2;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// create midpoint main vectors for steps 2 and 3\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=_%s_%s_index_pointer_begin; _j < _%s_%s_index_pointer_begin + _%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _active_%s_%s[_j] = (_active_%s_%s[_j]+_active_%s_%s[_%s_cross_size*_%s_%s_ncomponents + _j])/2;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(),
            fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }
  fprintf(outfile, "%s// move cross dim to lattice midpoint for steps 2 and 3\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _%s_dx%li/2;\n",
          indent, simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
  fprintf(outfile, "\n");

  // ****************************************************************
  fprintf(outfile, "%s// ********** step 2 ***************\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s {\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer=_%s_%s_index_pointer_begin;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s // calculate k1\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer_local=0;\n", indent, fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  // open inner loops
  for (unsigned long i=crossDimNumber()+1; i<nDims; i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "// *********** cross_propagation code ***********\n");
  fprintf(outfile, "%s\n", crossPropagationCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");
  for (list<const xmdsVector*>::const_iterator pxmdsVector = crossVectorList.begin(); pxmdsVector != crossVectorList.end(); pxmdsVector++) {
    for (unsigned long i=0; i<(*pxmdsVector)->nComponents(); i++) {
      fprintf(outfile, "%s _%s_%s_d[_%s_%s_index_pointer_local + %li] = d%s_d%s*_%s_dx%li;\n",
              indent2, fieldName, (*pxmdsVector)->name()->c_str(), fieldName, (*pxmdsVector)->name()->c_str(), i,
              (*pxmdsVector)->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
    }
    fprintf(outfile, "\n");
  }

  //close inner loops
  if (crossDimNumber()+1<nDims) {
    for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
    for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer_local += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  for (unsigned long i=nDims; i>crossDimNumber()+1; i--) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " %s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " }\n");
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s }\n", indent);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// add to K cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] += _%s_%s_d[_j]/3;\n", indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// create next cross vectors\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_I[_j] + _%s_%s_d[_j]/2;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  // ****************************************************************
  fprintf(outfile, "%s// ********** step 3 ***************\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s {\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer=_%s_%s_index_pointer_begin;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s // calculate k1\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer_local=0;\n", indent, fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  // open inner loops
  for (unsigned long i=crossDimNumber()+1; i<nDims; i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "// *********** cross_propagation code ***********\n");
  fprintf(outfile, "%s\n", crossPropagationCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");
  for (list<const xmdsVector*>::const_iterator pxmdsVector = crossVectorList.begin(); pxmdsVector != crossVectorList.end(); pxmdsVector++) {
    for (unsigned long i=0; i<(*pxmdsVector)->nComponents(); i++) {
      fprintf(outfile, "%s _%s_%s_d[_%s_%s_index_pointer_local + %li] = d%s_d%s*_%s_dx%li;\n",
              indent2, fieldName, (*pxmdsVector)->name()->c_str(), fieldName, (*pxmdsVector)->name()->c_str(), i,
              (*pxmdsVector)->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
    }
    fprintf(outfile, "\n");
  }

  //close inner loops
  if (crossDimNumber()+1<nDims) {
    for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
    for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer_local += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  for (unsigned long i=nDims; i>crossDimNumber()+1; i--) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " %s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " }\n");
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s }\n", indent);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// add to K cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] += _%s_%s_d[_j]/3;\n", indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// create next cross vectors\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_I[_j] + _%s_%s_d[_j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// move main vectors to next lattice point for step 4\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=_%s_%s_index_pointer_begin; _j < _%s_%s_index_pointer_begin + _%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _active_%s_%s[_j] = _active_%s_%s[_%s_cross_size*_%s_%s_ncomponents + _j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(),
            fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }
  fprintf(outfile, "%s// move cross dim to next lattice point for step 4\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _%s_dx%li/2;\n",
          indent, simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
  fprintf(outfile, "\n");

  // ****************************************************************
  fprintf(outfile, "%s// ********** step 4 ***************\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s {\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer=_%s_%s_index_pointer_begin;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s // calculate k1\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer_local=0;\n", indent, fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  // open inner loops
  for (unsigned long i=crossDimNumber()+1; i<nDims; i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "// *********** cross_propagation code ***********\n");
  fprintf(outfile, "%s\n", crossPropagationCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");
  for (list<const xmdsVector*>::const_iterator pxmdsVector = crossVectorList.begin(); pxmdsVector != crossVectorList.end(); pxmdsVector++) {
    for (unsigned long i=0; i<(*pxmdsVector)->nComponents(); i++) {
      fprintf(outfile, "%s _%s_%s_d[_%s_%s_index_pointer_local + %li] = d%s_d%s*_%s_dx%li;\n",
              indent2, fieldName, (*pxmdsVector)->name()->c_str(), fieldName, (*pxmdsVector)->name()->c_str(), i,
              (*pxmdsVector)->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
    }
    fprintf(outfile, "\n");
  }

  //close inner loops
  if (crossDimNumber()+1<nDims) {
    for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
    for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer_local += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  for (unsigned long i=nDims; i>crossDimNumber()+1; i--) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " %s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " }\n");
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s }\n", indent);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// add to K cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] += _%s_%s_d[_j]/6;\n", indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// copy I cross vector back into old main cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_I[_j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// copy K cross vector into next main cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _%s_cross_size*_%s_%s_ncomponents + _j] = _%s_%s_K[_j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// copy old main vectors back into last lattice point\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _active_%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_old[_j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  // close outer loops

  fprintf(outfile, "%s}\n", indent);

  for (unsigned long i=crossDimNumber(); i>0; i--) {

    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, "%s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, "}\n");
  }


  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {

    fprintf(outfile, "    fftw_free(_%s_%s_old);\n", fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {

    fprintf(outfile, "fftw_free(_%s_%s_K);\n", fieldName, pXMLString->c_str());
    fprintf(outfile, "fftw_free(_%s_%s_I);\n", fieldName, pXMLString->c_str());
    fprintf(outfile, "fftw_free(_%s_%s_d);\n", fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");

  }

  fprintf(outfile, "\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
