/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  FM Radio plug-in v1.5
 *  Copyright (C) 1999 Julien Viard <Julien.Viard@enst.fr>
 *  
 *  Radio acces routines from Keith Wesolowski (wesolows@cs.unr.edu)
 *  GRadio 1.0.0
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifndef __FreeBSD__
#include "radio.h"

#include <errno.h>
#if defined __OpenBSD__ || defined __NetBSD__
#include <err.h>
#endif /* __OpenBSD__ || __NetBSD__ */

static void radio_init(void);
static int is_our_file(char *filename);
static void play_file(char *filename);
static void stop(void);
static void radio_pause(short p);
static int get_time(void);
static void get_song_info(char *filename, char **title, int *length);
static void get_volume(int *l, int *r);
static void set_volume(int l, int r);
static void radio_about(void);

InputPlugin radio_ip =
{
  NULL,
  NULL,
  "FM Radio player 1.5",
  radio_init,
  radio_about,
  radio_configure,
  is_our_file,
  NULL,
  play_file,
  stop,
  radio_pause,
  NULL,                   /* seek */
  NULL,
  get_time,
  get_volume,
  set_volume,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  get_song_info,
  radio_file_info_box,			/* file_info_box */
  NULL
};

RadioConfig radio_cfg;

gint radio_fd = -1;
gint freq,volume;
#ifdef BSD_RADIO
struct radio_info ri;
#endif /* BSD_RADIO */
#if defined __OpenBSD__ || defined __NetBSD__
static mixer_ctrl_t volctl;
#endif /* __OpenBSD__ || __NetBSD__ */
gboolean is_paused,tuned;


InputPlugin *get_iplugin_info(void)
{
  return &radio_ip;
}

static void radio_init(void)
{
  ConfigFile *cfgfile;
  gchar *filename;

  filename = g_strconcat(g_get_home_dir(), "/.xmms/config", NULL);
  
  radio_cfg.device = NULL;
  radio_cfg.directory = NULL;
	
#if defined(HAVE_SYS_SOUNDCARD_H) || defined(HAVE_MACHINE_SOUNDCARD_H)
  radio_cfg.use_oss_mixer = TRUE;
#endif
  
  if ((cfgfile = xmms_cfg_open_file(filename)) != NULL)
    {
      xmms_cfg_read_string(cfgfile, "Radio", "device", &radio_cfg.device);
      xmms_cfg_read_string(cfgfile, "Radio", "directory", &radio_cfg.directory);
      xmms_cfg_read_boolean(cfgfile, "Radio", "use_oss_mixer", &radio_cfg.use_oss_mixer);
      xmms_cfg_free(cfgfile);
    }
  g_free(filename);
  
  if (!radio_cfg.device)
#ifdef linux
    radio_cfg.device = g_strdup("/dev/radio0");
#elif defined BSD_RADIO
    radio_cfg.device = g_strdup("/dev/radio");
#endif /* linux || BSD_RADIO */
  if (!radio_cfg.directory)
    radio_cfg.directory = g_strdup(g_get_home_dir());
  
  volume=FMR_VOL_DEFAULT;
#ifdef BSD_RADIO
  SetGetRadioInfo(RIOCGINFO);
#endif /* BSD_RADIO */
}

static int is_our_file(char *filename)
{
  gchar *ext;
  
  ext = strrchr(filename, '.');
  if (ext)
    if (!strcasecmp(ext, ".fmr"))
      return TRUE;
  return FALSE;
}

int get_time(void){return 0;}

#ifdef linux
/* Determine and return the appropriate frequency multiplier for
   the first tuner on the open video device with handle FD. */
double fmradio_get_freq_fact (void)
{
  struct video_tuner tuner;
  gint fd = -2;
  tuner.tuner = 0;

  if (radio_fd == -2) {
    if( ( fd = open (radio_cfg.device, O_RDONLY) ) < 0 ) {
      printf( "Cannot open %s\n", radio_cfg.device, errno );
      return .016;
    }
  }else
    fd = radio_fd;
  if (ioctl (fd, VIDIOCGTUNER, &tuner) < 0)
    return .016;

  if (radio_fd == -2)
    close(fd);

  if ((tuner.flags & VIDEO_TUNER_LOW) == 0)
    return .016;
  return 16;
}
#elif defined BSD_RADIO
int
SetGetRadioInfo(int setinfo) {
	int rd;

	rd = open(radio_cfg.device, O_RDWR);
	if (rd < 0) {
		warn("%s open error", radio_cfg.device);
		return -1;
	}

	if (ioctl(rd, setinfo == RIOCSINFO ? RIOCSINFO : RIOCGINFO, &ri) < 0) {
		warn("%s", setinfo == RIOCSINFO ? "RIOCSINFO" : "RIOCGINFO");
		return -1;
	}

	if (close(rd) < 0)
		warn("%s close error", radio_cfg.device);

	return 0;
}
#endif /* linux || BSD_RADIO */


static void play_file(char *filename)
{
  gchar *tmp,*radioname;
#ifdef linux
  struct video_tuner v;
  struct video_audio va;
  unsigned long xl_freq;
#endif /* linux */
  FILE * file;
  
  freq=0;
  file=fopen(filename,"r");
  if(file!=NULL){
    radioname=(gchar*)malloc(STR_LEN_MAX*sizeof(gchar));
    fgets(radioname,STR_LEN_MAX,file);
    radioname[STR_LEN_MAX-1] = '\0';
    if ((tmp = strrchr(radioname, '\n')) != NULL)
      *tmp = '\0';
    if(!fscanf(file,"Freq:%d", &freq)) freq=0;
    fclose(file);
  }else{
    radioname=NULL;
  }
  

  if(freq==0){
    tmp = strrchr(filename, '/');
    if (tmp)
      tmp++;
    else
      tmp = filename;
    
    if (!sscanf(tmp, "Radio %d.fmr", &freq)){
      freq=0;
      if(radioname==NULL){
	radioname = g_strdup(filename);
	if ((tmp = strrchr(radioname, '.')) != NULL)
	  *tmp = '\0';
      }
    }
  }
  if(freq!=0){
#ifdef linux
    radio_fd = open (radio_cfg.device, O_RDONLY);
    if (radio_fd == -1) {
      printf( "Cannot open %s %ld\n", radio_cfg.device, errno );
      return;
    }
#endif /* linux */
    
    if (freq > FMR_FREQ_MAX)
      freq = FMR_FREQ_MAX;
    if (freq < FMR_FREQ_MIN)
      freq = FMR_FREQ_MIN;
    
    is_paused = FALSE;
    
#ifdef linux
    xl_freq = (unsigned long)(freq*fmradio_get_freq_fact () + 0.5);

    /* Get audio settings */
    if( ioctl (radio_fd, VIDIOCGAUDIO, &va) < 0 ) {
      printf( "VIDIOCGTUNER %ld\n", errno );
      return ;
    }

    va.volume = volume * (65535/(FMR_VOL_MAX-FMR_VOL_MIN));
    va.audio = 0;
    va.flags = VIDEO_AUDIO_VOLUME|(is_paused ? VIDEO_AUDIO_MUTE : 0);
    
    v.tuner = 0;
    if( ioctl (radio_fd, VIDIOCSFREQ, &xl_freq) < 0 ) {
      printf( "VIDIOCSFREQ %ld %ld %ld\n", xl_freq, freq, errno );
      return ;
    }
    if( ioctl (radio_fd, VIDIOCGTUNER, &v) < 0 ) {
      printf( "VIDIOCGTUNER %ld\n", errno );
      return ;
    }
    if( ioctl (radio_fd, VIDIOCSAUDIO, &va) < 0 ) {
      printf( "VIDIOCGTUNER %ld\n", errno );
      return ;
    }
    tuned = ((v.signal != 0) ? TRUE : FALSE);
#elif defined BSD_RADIO
    ri.freq = (unsigned long)freq;
    ri.mute = 0;
    ri.volume = volume * (255/(FMR_VOL_MAX - FMR_VOL_MIN));
    SetGetRadioInfo(RIOCSINFO);
    usleep(50000);
    SetGetRadioInfo(RIOCGINFO);
    tuned = ri.info ? TRUE : FALSE;
#endif /* linux || BSD_RADIO */
    
    if (radioname==NULL)
      tmp = g_strdup_printf("FM Radio %6.2fMHz", (float)freq/1000.0);
    else
      tmp = g_strdup_printf("FM Radio %6.2fMHz - %s", (float)freq/1000.0,radioname);
  
    radio_ip.set_info(tmp, -1, 16000 * (tuned ? 2 : 1 ) * 2 * 8, 16000, (tuned ? 2 : 1 ));
  }else{
#ifdef linux
    radio_fd = open (radio_cfg.device, O_RDONLY);
    if (radio_fd == -1) {
      printf( "Canot open %s %ld\n", radio_cfg.device, errno );
      return ;
    }
#endif /* linux */
    freq=FMR_FREQ_DEFAULT;
    
    is_paused = FALSE;

#ifdef linux
    xl_freq = (unsigned long)(freq*fmradio_get_freq_fact () + 0.5);
    va.volume = volume * (65535/(FMR_VOL_MAX-FMR_VOL_MIN));
    va.audio = 0;
    va.flags = VIDEO_AUDIO_VOLUME|(is_paused ? VIDEO_AUDIO_MUTE : 0);
    
    v.tuner = 0;
    if( ioctl (radio_fd, VIDIOCSFREQ, &xl_freq) < 0 ){
      printf( "VIDIOCSFREQ %ld %ld %ld\n", xl_freq, freq, errno );
      return ;
    }
    if( ioctl (radio_fd, VIDIOCGTUNER, &v) < 0 ) {
      printf( "VIDIOCGTUNER %ld\n", errno );
      return ;
    }
    if( ioctl (radio_fd, VIDIOCSAUDIO, &va) < 0 ) {
      printf( "VIDIOCGTUNER %ld\n", errno );
      return ;
    }
    tuned = ((v.signal != 0) ? TRUE : FALSE);
#elif defined BSD_RADIO
    ri.freq = (unsigned long)freq;
    ri.mute = 0;
    ri.volume = volume * (255/(FMR_VOL_MAX - FMR_VOL_MIN));
    SetGetRadioInfo(RIOCSINFO);
    usleep(50000);
    SetGetRadioInfo(RIOCGINFO);
    tuned = ri.info ? TRUE : FALSE;
#endif /* linux || BSD_RADIO */

    if (radioname==NULL)
      tmp = g_strdup_printf("FM Radio (undefined)");
    else
      tmp = g_strdup_printf("FM Radio (undefined) - %s", radioname);
    radio_ip.set_info(tmp, -1, 0, 16000, 0);
  }
#ifdef linux
  free(radioname);
  close(radio_fd);
  radio_fd = -2;
#endif /* linux */
}
static void stop(void)
{
#ifdef linux
  struct video_audio va;
  if (radio_fd == -2) {
    radio_fd = open (radio_cfg.device, O_RDONLY);
  }
  va.flags = VIDEO_AUDIO_MUTE;
  va.audio = 0;
  ioctl (radio_fd, VIDIOCSAUDIO, &va);
  close(radio_fd);
  radio_fd = -1;
#elif defined BSD_RADIO
  ri.mute = 1;
  SetGetRadioInfo(RIOCSINFO);
#endif /* linux || BSD_RADIO */
}

static void radio_pause(short p)
{
#ifdef linux
  struct video_audio va;
  if (radio_fd == -2) {
    if( ( radio_fd = open (radio_cfg.device, O_RDONLY) ) < 0 ) {
      printf( "Canot open %s %ld\n", radio_cfg.device, errno );
      return ;
    }
  }
  if (p)
    va.flags = VIDEO_AUDIO_MUTE;
  else {
    va.volume = rint (volume * (65535./(FMR_VOL_MAX-FMR_VOL_MIN)));
    va.flags = VIDEO_AUDIO_VOLUME;
  }
  va.audio = 0;
  if( ioctl (radio_fd, VIDIOCSAUDIO, &va) < 0 ) {
    printf( "VIDIOCSAUDIO %s %ld\n", radio_cfg.device, errno );
    return ;
  }
  is_paused = (p ? TRUE : FALSE);
  close(radio_fd);
  radio_fd = -2;
#elif defined BSD_RADIO
  if (p) {
	  ri.mute = 1;
	  is_paused = TRUE;
  } else {
	  ri.mute = 0;
	  is_paused = FALSE;
  }
  SetGetRadioInfo(RIOCSINFO);
#endif /* linux || BSD_RADIO */
}

static void get_song_info(char *filename, char **title, int *len)
{
  gint fr;
  gchar *tmp,*radioname;
  FILE * file;
  
  *title = NULL;
  *len = -1;
  
  fr=0;
  
  file=fopen(filename,"r");
  if(file!=NULL){
    radioname=(gchar*)malloc(STR_LEN_MAX*sizeof(gchar));
    fgets(radioname,STR_LEN_MAX,file);
    radioname[STR_LEN_MAX-1] = '\0';
    if ((tmp = strrchr(radioname, '\n')) != NULL)
      *tmp = '\0';
    if(!fscanf(file,"Freq:%d", &fr)) fr=0;
    fclose(file);
  }else{
    radioname=NULL;
  }
  
  if(fr==0){
    tmp = strrchr(filename, '/');
    if (tmp)
      tmp++;
    else
      tmp = filename;
    
    if (!sscanf(tmp, "Radio %d.fmr", &fr)){
      fr=0;
      if(radioname==NULL){
	radioname = g_strdup(filename);
	if ((tmp = strrchr(radioname, '.')) != NULL)
	  *tmp = '\0';
      }
    }
  }
  *len = -1;
  
  if(fr!=0){
    if (fr > FMR_FREQ_MAX)
      fr = FMR_FREQ_MAX;
    if (fr < FMR_FREQ_MIN)
	    fr = FMR_FREQ_MIN;
    if (radioname==NULL)
      *title = g_strdup_printf("FM Radio %6.2fMHz", (float)fr/1000.0);
    else
      *title= g_strdup_printf("FM Radio %6.2fMHz - %s", (float)fr/1000.0,radioname);
  }else{
    if (radioname==NULL)
      *title = g_strdup_printf("FM Radio (undefined)");
    else
      *title= g_strdup_printf("FM Radio (undefined) - %s", radioname);
  }
  free(radioname);
}

static void get_volume(int *l, int *r)
{
#ifdef linux
	int v;
#elif defined __OpenBSD__ || defined __NetBSD__
	mixer_devinfo_t devinfo;
#endif /* linux || __OpenBSD__ || __NetBSD__ */
  int fd, devs, cmd;

#if defined(HAVE_SYS_SOUNDCARD_H) || defined(HAVE_MACHINE_SOUNDCARD_H)
  if (radio_cfg.use_oss_mixer)
    {
#ifdef linux
      fd = open("/dev/mixer", O_RDONLY);
      if (fd != -1)
	{
	  if( ioctl(fd, SOUND_MIXER_READ_DEVMASK, &devs) < 0 ) {
	    printf( "SOUND_MIXER_READ_DEVMASK %ld\n", errno );
	    return;
	  }
	  if (devs & SOUND_MASK_LINE)
	    cmd = SOUND_MIXER_READ_LINE;
	  else if (devs & SOUND_MASK_VOLUME)
	    cmd = SOUND_MIXER_READ_VOLUME;
	  else
	    {
	      close(fd);
				return;
	    }
	  if( ioctl(fd, cmd, &v) < 0 ) {
	    printf( "CMD %ld\n", errno );
	    return;
	  }
	  close(fd);
	  *r = (v & 0xFF00) >> 8;
	  *l = (v & 0x00FF);
          /* MJT printf( "Volume %ld %ld\n", *r, *l ); */
	  volume = *r;
	} else {
	  printf( "Cannot open /dev/mixer %ld\n", errno );
	}
#elif defined __OpenBSD__ || defined __NetBSD__
	cmd = 1;
	fd = open("/dev/mixer", O_RDWR);
	if (fd < 0) {
		warn("/dev/mixer open error");
		return;
	}
	for (devs = 0; ; devs++) {
		devinfo.index = devs;
		if (ioctl(fd, AUDIO_MIXER_DEVINFO, &devinfo) < 0)
			break;
		if (strncmp(devinfo.label.name, AudioNmaster, 6) == 0) {
			volctl.dev = devs;
			volctl.type = devinfo.type;
			volctl.un.value.num_channels = 2;
			if(ioctl(fd, AUDIO_MIXER_READ, &volctl) < 0) {
				volctl.un.value.num_channels = 1;
				if(ioctl(fd, AUDIO_MIXER_READ, &volctl) < 0) {
					warn("AUDIO_MIXER_READ");
					return;
				}
			}
			switch (volctl.un.value.num_channels) {
			case 2:
				*l = volctl.un.value.level[AUDIO_MIXER_LEVEL_LEFT] * 100 / 255;
				*r = volctl.un.value.level[AUDIO_MIXER_LEVEL_RIGHT] * 100 / 255;
				break;
			case 1:
				*l = *r = volctl.un.value.level[AUDIO_MIXER_LEVEL_MONO] * 100 / 255;
				break;
			default:
				warnx("Don't know how to handle %d channels",
					volctl.un.value.num_channels);
				return;
			}
			volume = *r;
			cmd = 0;
			break;
		}
	}
	if (devs == 0)
		warn("No mixer devices");
	else if (cmd)
		warn("%s.%s not found", AudioCoutputs, AudioNmaster);

	if (close(fd) < 0)
		warn("/dev/mixer close error");
#endif /* linux || __OpenBSD__ || __NetBSD__ */
    }
  else
#endif

    if (!radio_cfg.use_oss_mixer)
      {
	*r = (volume*100L)/255L;
	*l = *r;
        /* printf( "GET CARD VOLUME %d %d\n", *l, *r ); */
      }
  
}

static void set_volume(int l, int r)
{
#ifdef linux
  struct video_audio va;
  int v;
#elif defined __OpenBSD__ || defined __NetBSD__
  mixer_devinfo_t devinfo;
#endif /* linux || __OpenBSD__ || __NetBSD__ */
  int fd, devs, cmd;
  
#if defined(HAVE_SYS_SOUNDCARD_H) || defined(HAVE_MACHINE_SOUNDCARD_H)
  if (radio_cfg.use_oss_mixer)
    {
#ifdef linux
      fd = open("/dev/mixer", O_RDONLY);
      if (fd != -1)
	{
	  if( ioctl(fd, SOUND_MIXER_READ_DEVMASK, &devs) < 0 ) {
	    printf( "SOUND_MIXER_READ_DEVMASK %ld\n", errno );
	    return;
	  }
	  if (devs & SOUND_MASK_LINE)
	    cmd = SOUND_MIXER_WRITE_LINE;
	  else if (devs & SOUND_MASK_VOLUME)
	    cmd = SOUND_MIXER_WRITE_VOLUME;
	  else
	    {
	      close(fd);
	      return;
	    }
	  /* MJT printf( "Set Volume %ld %ld\n", r, l ); */
	  v = (r << 8) | l;
	  if( ioctl(fd, cmd, &v) < 0 ) {
	    printf( "CMD %ld\n", errno );
	    return;
	  }
	  close(fd);
	} else {
	  printf( "Cannot open /dev/mixer %ld\n", errno );
	}
#elif defined __OpenBSD__ || defined __NetBSD__
	cmd = 1;
	fd = open("/dev/mixer", O_RDWR);
	if (fd < 0) {
		warn("/dev/mixer open error");
		return;
	}
	for (devs = 0; ; devs++) {
		devinfo.index = devs;
		if (ioctl(fd, AUDIO_MIXER_DEVINFO, &devinfo) < 0)
			break;
		if (strncmp(devinfo.label.name, AudioNmaster, 6) == 0) {
			volctl.dev = devs;
			volctl.type = devinfo.type;
			volctl.un.value.num_channels = 2;
			if(ioctl(fd, AUDIO_MIXER_READ, &volctl) < 0) {
				volctl.un.value.num_channels = 1;
				if(ioctl(fd, AUDIO_MIXER_READ, &volctl) < 0) {
					warn("AUDIO_MIXER_READ");
					return;
				}
			}
			switch (volctl.un.value.num_channels) {
			case 2:
				volctl.un.value.level[AUDIO_MIXER_LEVEL_LEFT] = l * 255 / 100;
				volctl.un.value.level[AUDIO_MIXER_LEVEL_RIGHT] = r * 255 / 100;
				break;
			case 1:
				volctl.un.value.level[AUDIO_MIXER_LEVEL_MONO] = l * 255 / 100;
				break;
			default:
				warnx("Don't know how to handle %d channels",
					volctl.un.value.num_channels);
				return;
			}
			if (ioctl(fd, AUDIO_MIXER_WRITE, &volctl) < 0) {
				warn("AUDIO_MIXER_WRITE");
				return;
			}
			cmd = 0;
		}
	}
	if (devs == 0)
		warn("No mixer devices");
	else if (cmd)
		warn("%s.%s not found", AudioCoutputs, AudioNmaster);

	if (close(fd) < 0)
		warn("/dev/mixer close error");
#endif /* linux || __OpenBSD__ || __NetBSD__ */
    }
  else
#endif
    {
      /* COnvert range to 0000 - FFFF */
      /* MJT printf( "GIVEN CARD VOLUME %d %d\n", l, r ); */

      r = MAX(l,r);
      l = ((long)r * 255L)/100L;
      r = ((long)r * 255L)/100L;

      /* MJT printf( "SET CARD VOLUME %d %d\n", l, r ); */
      
      volume= MAX(l,r);
      if (volume > FMR_VOL_MAX)
	volume = FMR_VOL_MAX;
      if (volume < FMR_VOL_MIN)
	volume = FMR_VOL_MIN;
      
#ifdef linux
      va.flags = VIDEO_AUDIO_VOLUME|(is_paused ? VIDEO_AUDIO_MUTE : 0);
      va.audio = 0;
      va.volume = rint (volume * (65535./(FMR_VOL_MAX-FMR_VOL_MIN)));
      if (radio_fd == -2) {
	radio_fd = open (radio_cfg.device, O_RDONLY);
      }
      if (radio_fd == -1) {
	printf( "Cannot open %s %ld\n", radio_cfg.device, errno );
	return;
      }
      if( ioctl (radio_fd, VIDIOCSAUDIO, &va) < 0 ) {
	printf( "VIDIOCSAUDIO %ld\n", errno );
	return;
      }
      close(radio_fd);
      radio_fd = -2;
#elif defined BSD_RADIO
    ri.volume = volume * (255 / (FMR_VOL_MAX - FMR_VOL_MIN));
    ri.mute = is_paused ? 1 : 0;
    SetGetRadioInfo(RIOCSINFO);
#endif /* linux || BSD_RADIO */
    }
}

static void radio_about(void)
{
	static GtkWidget *box;
	box = xmms_show_message(
		"About FM Radio",
		"FM Radio player by Julien Viard <silicone@free.fr>\n"
		"Find more about it on http://silicone.free.fr/xmms-FMRadio/\n"
                  "\n"
		  "Thanks to\n"
		  " Keith Wesolowski <wesolows@cs.unr.edu> \nfor the program GRadio 1.0.0\n"
		  " Wim <wim.delvaux@chello.be> \nfor correcting my code for the volume configuration through the radio device\n"
		  " Vladimir Popov <jumbo@narod.ru>\nfor the BSD port\n"
		  "\nTo use it, open a file with '.fmr' extension\n then edit the 'file info' to scan for a station and save the file.", "Ok",
		FALSE, NULL, NULL);
	gtk_signal_connect(GTK_OBJECT(box), "destroy",
			   gtk_widget_destroyed, &box);
}


#endif
